/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents information about the rule to be created for an associated condition. An example would be creating a new
 * rule for an entry condition, such as a rule that checks for a test result before allowing the run to enter the
 * deployment stage.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RuleDeclaration implements SdkPojo, Serializable, ToCopyableBuilder<RuleDeclaration.Builder, RuleDeclaration> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(RuleDeclaration::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<RuleTypeId> RULE_TYPE_ID_FIELD = SdkField.<RuleTypeId> builder(MarshallingType.SDK_POJO)
            .memberName("ruleTypeId").getter(getter(RuleDeclaration::ruleTypeId)).setter(setter(Builder::ruleTypeId))
            .constructor(RuleTypeId::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ruleTypeId").build()).build();

    private static final SdkField<Map<String, String>> CONFIGURATION_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("configuration")
            .getter(getter(RuleDeclaration::configuration))
            .setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<InputArtifact>> INPUT_ARTIFACTS_FIELD = SdkField
            .<List<InputArtifact>> builder(MarshallingType.LIST)
            .memberName("inputArtifacts")
            .getter(getter(RuleDeclaration::inputArtifacts))
            .setter(setter(Builder::inputArtifacts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inputArtifacts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InputArtifact> builder(MarshallingType.SDK_POJO)
                                            .constructor(InputArtifact::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(RuleDeclaration::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("region")
            .getter(getter(RuleDeclaration::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("region").build()).build();

    private static final SdkField<Integer> TIMEOUT_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("timeoutInMinutes").getter(getter(RuleDeclaration::timeoutInMinutes))
            .setter(setter(Builder::timeoutInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeoutInMinutes").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            RULE_TYPE_ID_FIELD, CONFIGURATION_FIELD, INPUT_ARTIFACTS_FIELD, ROLE_ARN_FIELD, REGION_FIELD,
            TIMEOUT_IN_MINUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final RuleTypeId ruleTypeId;

    private final Map<String, String> configuration;

    private final List<InputArtifact> inputArtifacts;

    private final String roleArn;

    private final String region;

    private final Integer timeoutInMinutes;

    private RuleDeclaration(BuilderImpl builder) {
        this.name = builder.name;
        this.ruleTypeId = builder.ruleTypeId;
        this.configuration = builder.configuration;
        this.inputArtifacts = builder.inputArtifacts;
        this.roleArn = builder.roleArn;
        this.region = builder.region;
        this.timeoutInMinutes = builder.timeoutInMinutes;
    }

    /**
     * <p>
     * The name of the rule that is created for the condition, such as CheckAllResults.
     * </p>
     * 
     * @return The name of the rule that is created for the condition, such as CheckAllResults.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ID for the rule type, which is made up of the combined values for category, owner, provider, and version.
     * </p>
     * 
     * @return The ID for the rule type, which is made up of the combined values for category, owner, provider, and
     *         version.
     */
    public final RuleTypeId ruleTypeId() {
        return ruleTypeId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Configuration property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConfiguration() {
        return configuration != null && !(configuration instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The action configuration fields for the rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConfiguration} method.
     * </p>
     * 
     * @return The action configuration fields for the rule.
     */
    public final Map<String, String> configuration() {
        return configuration;
    }

    /**
     * For responses, this returns true if the service returned a value for the InputArtifacts property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInputArtifacts() {
        return inputArtifacts != null && !(inputArtifacts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The input artifacts fields for the rule, such as specifying an input file for the rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInputArtifacts} method.
     * </p>
     * 
     * @return The input artifacts fields for the rule, such as specifying an input file for the rule.
     */
    public final List<InputArtifact> inputArtifacts() {
        return inputArtifacts;
    }

    /**
     * <p>
     * The pipeline role ARN associated with the rule.
     * </p>
     * 
     * @return The pipeline role ARN associated with the rule.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The Region for the condition associated with the rule.
     * </p>
     * 
     * @return The Region for the condition associated with the rule.
     */
    public final String region() {
        return region;
    }

    /**
     * <p>
     * The action timeout for the rule.
     * </p>
     * 
     * @return The action timeout for the rule.
     */
    public final Integer timeoutInMinutes() {
        return timeoutInMinutes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(ruleTypeId());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfiguration() ? configuration() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInputArtifacts() ? inputArtifacts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutInMinutes());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RuleDeclaration)) {
            return false;
        }
        RuleDeclaration other = (RuleDeclaration) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(ruleTypeId(), other.ruleTypeId())
                && hasConfiguration() == other.hasConfiguration() && Objects.equals(configuration(), other.configuration())
                && hasInputArtifacts() == other.hasInputArtifacts() && Objects.equals(inputArtifacts(), other.inputArtifacts())
                && Objects.equals(roleArn(), other.roleArn()) && Objects.equals(region(), other.region())
                && Objects.equals(timeoutInMinutes(), other.timeoutInMinutes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RuleDeclaration").add("Name", name()).add("RuleTypeId", ruleTypeId())
                .add("Configuration", hasConfiguration() ? configuration() : null)
                .add("InputArtifacts", hasInputArtifacts() ? inputArtifacts() : null).add("RoleArn", roleArn())
                .add("Region", region()).add("TimeoutInMinutes", timeoutInMinutes()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ruleTypeId":
            return Optional.ofNullable(clazz.cast(ruleTypeId()));
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "inputArtifacts":
            return Optional.ofNullable(clazz.cast(inputArtifacts()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "region":
            return Optional.ofNullable(clazz.cast(region()));
        case "timeoutInMinutes":
            return Optional.ofNullable(clazz.cast(timeoutInMinutes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RuleDeclaration, T> g) {
        return obj -> g.apply((RuleDeclaration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RuleDeclaration> {
        /**
         * <p>
         * The name of the rule that is created for the condition, such as CheckAllResults.
         * </p>
         * 
         * @param name
         *        The name of the rule that is created for the condition, such as CheckAllResults.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID for the rule type, which is made up of the combined values for category, owner, provider, and version.
         * </p>
         * 
         * @param ruleTypeId
         *        The ID for the rule type, which is made up of the combined values for category, owner, provider, and
         *        version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleTypeId(RuleTypeId ruleTypeId);

        /**
         * <p>
         * The ID for the rule type, which is made up of the combined values for category, owner, provider, and version.
         * </p>
         * This is a convenience method that creates an instance of the {@link RuleTypeId.Builder} avoiding the need to
         * create one manually via {@link RuleTypeId#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RuleTypeId.Builder#build()} is called immediately and its result
         * is passed to {@link #ruleTypeId(RuleTypeId)}.
         * 
         * @param ruleTypeId
         *        a consumer that will call methods on {@link RuleTypeId.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ruleTypeId(RuleTypeId)
         */
        default Builder ruleTypeId(Consumer<RuleTypeId.Builder> ruleTypeId) {
            return ruleTypeId(RuleTypeId.builder().applyMutation(ruleTypeId).build());
        }

        /**
         * <p>
         * The action configuration fields for the rule.
         * </p>
         * 
         * @param configuration
         *        The action configuration fields for the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(Map<String, String> configuration);

        /**
         * <p>
         * The input artifacts fields for the rule, such as specifying an input file for the rule.
         * </p>
         * 
         * @param inputArtifacts
         *        The input artifacts fields for the rule, such as specifying an input file for the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputArtifacts(Collection<InputArtifact> inputArtifacts);

        /**
         * <p>
         * The input artifacts fields for the rule, such as specifying an input file for the rule.
         * </p>
         * 
         * @param inputArtifacts
         *        The input artifacts fields for the rule, such as specifying an input file for the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputArtifacts(InputArtifact... inputArtifacts);

        /**
         * <p>
         * The input artifacts fields for the rule, such as specifying an input file for the rule.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codepipeline.model.InputArtifact.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codepipeline.model.InputArtifact#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codepipeline.model.InputArtifact.Builder#build()} is called
         * immediately and its result is passed to {@link #inputArtifacts(List<InputArtifact>)}.
         * 
         * @param inputArtifacts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codepipeline.model.InputArtifact.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputArtifacts(java.util.Collection<InputArtifact>)
         */
        Builder inputArtifacts(Consumer<InputArtifact.Builder>... inputArtifacts);

        /**
         * <p>
         * The pipeline role ARN associated with the rule.
         * </p>
         * 
         * @param roleArn
         *        The pipeline role ARN associated with the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The Region for the condition associated with the rule.
         * </p>
         * 
         * @param region
         *        The Region for the condition associated with the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);

        /**
         * <p>
         * The action timeout for the rule.
         * </p>
         * 
         * @param timeoutInMinutes
         *        The action timeout for the rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutInMinutes(Integer timeoutInMinutes);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private RuleTypeId ruleTypeId;

        private Map<String, String> configuration = DefaultSdkAutoConstructMap.getInstance();

        private List<InputArtifact> inputArtifacts = DefaultSdkAutoConstructList.getInstance();

        private String roleArn;

        private String region;

        private Integer timeoutInMinutes;

        private BuilderImpl() {
        }

        private BuilderImpl(RuleDeclaration model) {
            name(model.name);
            ruleTypeId(model.ruleTypeId);
            configuration(model.configuration);
            inputArtifacts(model.inputArtifacts);
            roleArn(model.roleArn);
            region(model.region);
            timeoutInMinutes(model.timeoutInMinutes);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final RuleTypeId.Builder getRuleTypeId() {
            return ruleTypeId != null ? ruleTypeId.toBuilder() : null;
        }

        public final void setRuleTypeId(RuleTypeId.BuilderImpl ruleTypeId) {
            this.ruleTypeId = ruleTypeId != null ? ruleTypeId.build() : null;
        }

        @Override
        public final Builder ruleTypeId(RuleTypeId ruleTypeId) {
            this.ruleTypeId = ruleTypeId;
            return this;
        }

        public final Map<String, String> getConfiguration() {
            if (configuration instanceof SdkAutoConstructMap) {
                return null;
            }
            return configuration;
        }

        public final void setConfiguration(Map<String, String> configuration) {
            this.configuration = RuleConfigurationMapCopier.copy(configuration);
        }

        @Override
        public final Builder configuration(Map<String, String> configuration) {
            this.configuration = RuleConfigurationMapCopier.copy(configuration);
            return this;
        }

        public final List<InputArtifact.Builder> getInputArtifacts() {
            List<InputArtifact.Builder> result = InputArtifactListCopier.copyToBuilder(this.inputArtifacts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInputArtifacts(Collection<InputArtifact.BuilderImpl> inputArtifacts) {
            this.inputArtifacts = InputArtifactListCopier.copyFromBuilder(inputArtifacts);
        }

        @Override
        public final Builder inputArtifacts(Collection<InputArtifact> inputArtifacts) {
            this.inputArtifacts = InputArtifactListCopier.copy(inputArtifacts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputArtifacts(InputArtifact... inputArtifacts) {
            inputArtifacts(Arrays.asList(inputArtifacts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputArtifacts(Consumer<InputArtifact.Builder>... inputArtifacts) {
            inputArtifacts(Stream.of(inputArtifacts).map(c -> InputArtifact.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final Integer getTimeoutInMinutes() {
            return timeoutInMinutes;
        }

        public final void setTimeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
        }

        @Override
        public final Builder timeoutInMinutes(Integer timeoutInMinutes) {
            this.timeoutInMinutes = timeoutInMinutes;
            return this;
        }

        @Override
        public RuleDeclaration build() {
            return new RuleDeclaration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
