/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that specifies the result, such as rollback, to occur upon stage failure.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FailureConditions implements SdkPojo, Serializable,
        ToCopyableBuilder<FailureConditions.Builder, FailureConditions> {
    private static final SdkField<String> RESULT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("result")
            .getter(getter(FailureConditions::resultAsString)).setter(setter(Builder::result))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("result").build()).build();

    private static final SdkField<RetryConfiguration> RETRY_CONFIGURATION_FIELD = SdkField
            .<RetryConfiguration> builder(MarshallingType.SDK_POJO).memberName("retryConfiguration")
            .getter(getter(FailureConditions::retryConfiguration)).setter(setter(Builder::retryConfiguration))
            .constructor(RetryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("retryConfiguration").build())
            .build();

    private static final SdkField<List<Condition>> CONDITIONS_FIELD = SdkField
            .<List<Condition>> builder(MarshallingType.LIST)
            .memberName("conditions")
            .getter(getter(FailureConditions::conditions))
            .setter(setter(Builder::conditions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("conditions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Condition> builder(MarshallingType.SDK_POJO)
                                            .constructor(Condition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESULT_FIELD,
            RETRY_CONFIGURATION_FIELD, CONDITIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String result;

    private final RetryConfiguration retryConfiguration;

    private final List<Condition> conditions;

    private FailureConditions(BuilderImpl builder) {
        this.result = builder.result;
        this.retryConfiguration = builder.retryConfiguration;
        this.conditions = builder.conditions;
    }

    /**
     * <p>
     * The specified result for when the failure conditions are met, such as rolling back the stage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #result} will
     * return {@link Result#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resultAsString}.
     * </p>
     * 
     * @return The specified result for when the failure conditions are met, such as rolling back the stage.
     * @see Result
     */
    public final Result result() {
        return Result.fromValue(result);
    }

    /**
     * <p>
     * The specified result for when the failure conditions are met, such as rolling back the stage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #result} will
     * return {@link Result#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resultAsString}.
     * </p>
     * 
     * @return The specified result for when the failure conditions are met, such as rolling back the stage.
     * @see Result
     */
    public final String resultAsString() {
        return result;
    }

    /**
     * <p>
     * The retry configuration specifies automatic retry for a failed stage, along with the configured retry mode.
     * </p>
     * 
     * @return The retry configuration specifies automatic retry for a failed stage, along with the configured retry
     *         mode.
     */
    public final RetryConfiguration retryConfiguration() {
        return retryConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Conditions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasConditions() {
        return conditions != null && !(conditions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The conditions that are configured as failure conditions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConditions} method.
     * </p>
     * 
     * @return The conditions that are configured as failure conditions.
     */
    public final List<Condition> conditions() {
        return conditions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resultAsString());
        hashCode = 31 * hashCode + Objects.hashCode(retryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasConditions() ? conditions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FailureConditions)) {
            return false;
        }
        FailureConditions other = (FailureConditions) obj;
        return Objects.equals(resultAsString(), other.resultAsString())
                && Objects.equals(retryConfiguration(), other.retryConfiguration()) && hasConditions() == other.hasConditions()
                && Objects.equals(conditions(), other.conditions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FailureConditions").add("Result", resultAsString())
                .add("RetryConfiguration", retryConfiguration()).add("Conditions", hasConditions() ? conditions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "result":
            return Optional.ofNullable(clazz.cast(resultAsString()));
        case "retryConfiguration":
            return Optional.ofNullable(clazz.cast(retryConfiguration()));
        case "conditions":
            return Optional.ofNullable(clazz.cast(conditions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FailureConditions, T> g) {
        return obj -> g.apply((FailureConditions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FailureConditions> {
        /**
         * <p>
         * The specified result for when the failure conditions are met, such as rolling back the stage.
         * </p>
         * 
         * @param result
         *        The specified result for when the failure conditions are met, such as rolling back the stage.
         * @see Result
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Result
         */
        Builder result(String result);

        /**
         * <p>
         * The specified result for when the failure conditions are met, such as rolling back the stage.
         * </p>
         * 
         * @param result
         *        The specified result for when the failure conditions are met, such as rolling back the stage.
         * @see Result
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Result
         */
        Builder result(Result result);

        /**
         * <p>
         * The retry configuration specifies automatic retry for a failed stage, along with the configured retry mode.
         * </p>
         * 
         * @param retryConfiguration
         *        The retry configuration specifies automatic retry for a failed stage, along with the configured retry
         *        mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retryConfiguration(RetryConfiguration retryConfiguration);

        /**
         * <p>
         * The retry configuration specifies automatic retry for a failed stage, along with the configured retry mode.
         * </p>
         * This is a convenience method that creates an instance of the {@link RetryConfiguration.Builder} avoiding the
         * need to create one manually via {@link RetryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RetryConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #retryConfiguration(RetryConfiguration)}.
         * 
         * @param retryConfiguration
         *        a consumer that will call methods on {@link RetryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #retryConfiguration(RetryConfiguration)
         */
        default Builder retryConfiguration(Consumer<RetryConfiguration.Builder> retryConfiguration) {
            return retryConfiguration(RetryConfiguration.builder().applyMutation(retryConfiguration).build());
        }

        /**
         * <p>
         * The conditions that are configured as failure conditions.
         * </p>
         * 
         * @param conditions
         *        The conditions that are configured as failure conditions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(Collection<Condition> conditions);

        /**
         * <p>
         * The conditions that are configured as failure conditions.
         * </p>
         * 
         * @param conditions
         *        The conditions that are configured as failure conditions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder conditions(Condition... conditions);

        /**
         * <p>
         * The conditions that are configured as failure conditions.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codepipeline.model.Condition.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.codepipeline.model.Condition#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codepipeline.model.Condition.Builder#build()} is called immediately
         * and its result is passed to {@link #conditions(List<Condition>)}.
         * 
         * @param conditions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codepipeline.model.Condition.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #conditions(java.util.Collection<Condition>)
         */
        Builder conditions(Consumer<Condition.Builder>... conditions);
    }

    static final class BuilderImpl implements Builder {
        private String result;

        private RetryConfiguration retryConfiguration;

        private List<Condition> conditions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(FailureConditions model) {
            result(model.result);
            retryConfiguration(model.retryConfiguration);
            conditions(model.conditions);
        }

        public final String getResult() {
            return result;
        }

        public final void setResult(String result) {
            this.result = result;
        }

        @Override
        public final Builder result(String result) {
            this.result = result;
            return this;
        }

        @Override
        public final Builder result(Result result) {
            this.result(result == null ? null : result.toString());
            return this;
        }

        public final RetryConfiguration.Builder getRetryConfiguration() {
            return retryConfiguration != null ? retryConfiguration.toBuilder() : null;
        }

        public final void setRetryConfiguration(RetryConfiguration.BuilderImpl retryConfiguration) {
            this.retryConfiguration = retryConfiguration != null ? retryConfiguration.build() : null;
        }

        @Override
        public final Builder retryConfiguration(RetryConfiguration retryConfiguration) {
            this.retryConfiguration = retryConfiguration;
            return this;
        }

        public final List<Condition.Builder> getConditions() {
            List<Condition.Builder> result = ConditionListCopier.copyToBuilder(this.conditions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setConditions(Collection<Condition.BuilderImpl> conditions) {
            this.conditions = ConditionListCopier.copyFromBuilder(conditions);
        }

        @Override
        public final Builder conditions(Collection<Condition> conditions) {
            this.conditions = ConditionListCopier.copy(conditions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditions(Condition... conditions) {
            conditions(Arrays.asList(conditions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder conditions(Consumer<Condition.Builder>... conditions) {
            conditions(Stream.of(conditions).map(c -> Condition.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public FailureConditions build() {
            return new FailureConditions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
