/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The S3 bucket where artifacts for the pipeline are stored.
 * </p>
 * <note>
 * <p>
 * You must include either <code>artifactStore</code> or <code>artifactStores</code> in your pipeline, but you cannot
 * use both. If you create a cross-region action in your pipeline, you must use <code>artifactStores</code>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ArtifactStore implements SdkPojo, Serializable, ToCopyableBuilder<ArtifactStore.Builder, ArtifactStore> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(ArtifactStore::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("location").getter(getter(ArtifactStore::location)).setter(setter(Builder::location))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("location").build()).build();

    private static final SdkField<EncryptionKey> ENCRYPTION_KEY_FIELD = SdkField
            .<EncryptionKey> builder(MarshallingType.SDK_POJO).memberName("encryptionKey")
            .getter(getter(ArtifactStore::encryptionKey)).setter(setter(Builder::encryptionKey))
            .constructor(EncryptionKey::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionKey").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, LOCATION_FIELD,
            ENCRYPTION_KEY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String location;

    private final EncryptionKey encryptionKey;

    private ArtifactStore(BuilderImpl builder) {
        this.type = builder.type;
        this.location = builder.location;
        this.encryptionKey = builder.encryptionKey;
    }

    /**
     * <p>
     * The type of the artifact store, such as S3.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ArtifactStoreType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the artifact store, such as S3.
     * @see ArtifactStoreType
     */
    public final ArtifactStoreType type() {
        return ArtifactStoreType.fromValue(type);
    }

    /**
     * <p>
     * The type of the artifact store, such as S3.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ArtifactStoreType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the artifact store, such as S3.
     * @see ArtifactStoreType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The S3 bucket used for storing the artifacts for a pipeline. You can specify the name of an S3 bucket but not a
     * folder in the bucket. A folder to contain the pipeline artifacts is created for you based on the name of the
     * pipeline. You can use any S3 bucket in the same Amazon Web Services Region as the pipeline to store your pipeline
     * artifacts.
     * </p>
     * 
     * @return The S3 bucket used for storing the artifacts for a pipeline. You can specify the name of an S3 bucket but
     *         not a folder in the bucket. A folder to contain the pipeline artifacts is created for you based on the
     *         name of the pipeline. You can use any S3 bucket in the same Amazon Web Services Region as the pipeline to
     *         store your pipeline artifacts.
     */
    public final String location() {
        return location;
    }

    /**
     * <p>
     * The encryption key used to encrypt the data in the artifact store, such as an Amazon Web Services Key Management
     * Service key. If this is undefined, the default key for Amazon S3 is used.
     * </p>
     * 
     * @return The encryption key used to encrypt the data in the artifact store, such as an Amazon Web Services Key
     *         Management Service key. If this is undefined, the default key for Amazon S3 is used.
     */
    public final EncryptionKey encryptionKey() {
        return encryptionKey;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(location());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKey());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ArtifactStore)) {
            return false;
        }
        ArtifactStore other = (ArtifactStore) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(location(), other.location())
                && Objects.equals(encryptionKey(), other.encryptionKey());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ArtifactStore").add("Type", typeAsString()).add("Location", location())
                .add("EncryptionKey", encryptionKey()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "location":
            return Optional.ofNullable(clazz.cast(location()));
        case "encryptionKey":
            return Optional.ofNullable(clazz.cast(encryptionKey()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("type", TYPE_FIELD);
        map.put("location", LOCATION_FIELD);
        map.put("encryptionKey", ENCRYPTION_KEY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ArtifactStore, T> g) {
        return obj -> g.apply((ArtifactStore) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ArtifactStore> {
        /**
         * <p>
         * The type of the artifact store, such as S3.
         * </p>
         * 
         * @param type
         *        The type of the artifact store, such as S3.
         * @see ArtifactStoreType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArtifactStoreType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the artifact store, such as S3.
         * </p>
         * 
         * @param type
         *        The type of the artifact store, such as S3.
         * @see ArtifactStoreType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArtifactStoreType
         */
        Builder type(ArtifactStoreType type);

        /**
         * <p>
         * The S3 bucket used for storing the artifacts for a pipeline. You can specify the name of an S3 bucket but not
         * a folder in the bucket. A folder to contain the pipeline artifacts is created for you based on the name of
         * the pipeline. You can use any S3 bucket in the same Amazon Web Services Region as the pipeline to store your
         * pipeline artifacts.
         * </p>
         * 
         * @param location
         *        The S3 bucket used for storing the artifacts for a pipeline. You can specify the name of an S3 bucket
         *        but not a folder in the bucket. A folder to contain the pipeline artifacts is created for you based on
         *        the name of the pipeline. You can use any S3 bucket in the same Amazon Web Services Region as the
         *        pipeline to store your pipeline artifacts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder location(String location);

        /**
         * <p>
         * The encryption key used to encrypt the data in the artifact store, such as an Amazon Web Services Key
         * Management Service key. If this is undefined, the default key for Amazon S3 is used.
         * </p>
         * 
         * @param encryptionKey
         *        The encryption key used to encrypt the data in the artifact store, such as an Amazon Web Services Key
         *        Management Service key. If this is undefined, the default key for Amazon S3 is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKey(EncryptionKey encryptionKey);

        /**
         * <p>
         * The encryption key used to encrypt the data in the artifact store, such as an Amazon Web Services Key
         * Management Service key. If this is undefined, the default key for Amazon S3 is used.
         * </p>
         * This is a convenience method that creates an instance of the {@link EncryptionKey.Builder} avoiding the need
         * to create one manually via {@link EncryptionKey#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EncryptionKey.Builder#build()} is called immediately and its
         * result is passed to {@link #encryptionKey(EncryptionKey)}.
         * 
         * @param encryptionKey
         *        a consumer that will call methods on {@link EncryptionKey.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionKey(EncryptionKey)
         */
        default Builder encryptionKey(Consumer<EncryptionKey.Builder> encryptionKey) {
            return encryptionKey(EncryptionKey.builder().applyMutation(encryptionKey).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String location;

        private EncryptionKey encryptionKey;

        private BuilderImpl() {
        }

        private BuilderImpl(ArtifactStore model) {
            type(model.type);
            location(model.location);
            encryptionKey(model.encryptionKey);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ArtifactStoreType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getLocation() {
            return location;
        }

        public final void setLocation(String location) {
            this.location = location;
        }

        @Override
        public final Builder location(String location) {
            this.location = location;
            return this;
        }

        public final EncryptionKey.Builder getEncryptionKey() {
            return encryptionKey != null ? encryptionKey.toBuilder() : null;
        }

        public final void setEncryptionKey(EncryptionKey.BuilderImpl encryptionKey) {
            this.encryptionKey = encryptionKey != null ? encryptionKey.build() : null;
        }

        @Override
        public final Builder encryptionKey(EncryptionKey encryptionKey) {
            this.encryptionKey = encryptionKey;
            return this;
        }

        @Override
        public ArtifactStore build() {
            return new ArtifactStore(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
