/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Summary information about a pipeline execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PipelineExecutionSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<PipelineExecutionSummary.Builder, PipelineExecutionSummary> {
    private static final SdkField<String> PIPELINE_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pipelineExecutionId").getter(getter(PipelineExecutionSummary::pipelineExecutionId))
            .setter(setter(Builder::pipelineExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineExecutionId").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(PipelineExecutionSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusSummary").getter(getter(PipelineExecutionSummary::statusSummary))
            .setter(setter(Builder::statusSummary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusSummary").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("startTime").getter(getter(PipelineExecutionSummary::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdateTime").getter(getter(PipelineExecutionSummary::lastUpdateTime))
            .setter(setter(Builder::lastUpdateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdateTime").build()).build();

    private static final SdkField<List<SourceRevision>> SOURCE_REVISIONS_FIELD = SdkField
            .<List<SourceRevision>> builder(MarshallingType.LIST)
            .memberName("sourceRevisions")
            .getter(getter(PipelineExecutionSummary::sourceRevisions))
            .setter(setter(Builder::sourceRevisions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceRevisions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SourceRevision> builder(MarshallingType.SDK_POJO)
                                            .constructor(SourceRevision::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ExecutionTrigger> TRIGGER_FIELD = SdkField.<ExecutionTrigger> builder(MarshallingType.SDK_POJO)
            .memberName("trigger").getter(getter(PipelineExecutionSummary::trigger)).setter(setter(Builder::trigger))
            .constructor(ExecutionTrigger::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("trigger").build()).build();

    private static final SdkField<StopExecutionTrigger> STOP_TRIGGER_FIELD = SdkField
            .<StopExecutionTrigger> builder(MarshallingType.SDK_POJO).memberName("stopTrigger")
            .getter(getter(PipelineExecutionSummary::stopTrigger)).setter(setter(Builder::stopTrigger))
            .constructor(StopExecutionTrigger::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stopTrigger").build()).build();

    private static final SdkField<String> EXECUTION_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionMode").getter(getter(PipelineExecutionSummary::executionModeAsString))
            .setter(setter(Builder::executionMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionMode").build()).build();

    private static final SdkField<String> EXECUTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionType").getter(getter(PipelineExecutionSummary::executionTypeAsString))
            .setter(setter(Builder::executionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionType").build()).build();

    private static final SdkField<PipelineRollbackMetadata> ROLLBACK_METADATA_FIELD = SdkField
            .<PipelineRollbackMetadata> builder(MarshallingType.SDK_POJO).memberName("rollbackMetadata")
            .getter(getter(PipelineExecutionSummary::rollbackMetadata)).setter(setter(Builder::rollbackMetadata))
            .constructor(PipelineRollbackMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rollbackMetadata").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PIPELINE_EXECUTION_ID_FIELD,
            STATUS_FIELD, STATUS_SUMMARY_FIELD, START_TIME_FIELD, LAST_UPDATE_TIME_FIELD, SOURCE_REVISIONS_FIELD, TRIGGER_FIELD,
            STOP_TRIGGER_FIELD, EXECUTION_MODE_FIELD, EXECUTION_TYPE_FIELD, ROLLBACK_METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String pipelineExecutionId;

    private final String status;

    private final String statusSummary;

    private final Instant startTime;

    private final Instant lastUpdateTime;

    private final List<SourceRevision> sourceRevisions;

    private final ExecutionTrigger trigger;

    private final StopExecutionTrigger stopTrigger;

    private final String executionMode;

    private final String executionType;

    private final PipelineRollbackMetadata rollbackMetadata;

    private PipelineExecutionSummary(BuilderImpl builder) {
        this.pipelineExecutionId = builder.pipelineExecutionId;
        this.status = builder.status;
        this.statusSummary = builder.statusSummary;
        this.startTime = builder.startTime;
        this.lastUpdateTime = builder.lastUpdateTime;
        this.sourceRevisions = builder.sourceRevisions;
        this.trigger = builder.trigger;
        this.stopTrigger = builder.stopTrigger;
        this.executionMode = builder.executionMode;
        this.executionType = builder.executionType;
        this.rollbackMetadata = builder.rollbackMetadata;
    }

    /**
     * <p>
     * The ID of the pipeline execution.
     * </p>
     * 
     * @return The ID of the pipeline execution.
     */
    public final String pipelineExecutionId() {
        return pipelineExecutionId;
    }

    /**
     * <p>
     * The status of the pipeline execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * InProgress: The pipeline execution is currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopped: The pipeline execution was manually stopped. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop mode,
     * the execution is either completing or abandoning in-progress actions. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Succeeded: The pipeline execution was completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
     * execution advanced and continued through the pipeline instead. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     * Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The pipeline execution was not completed successfully.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PipelineExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the pipeline execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         InProgress: The pipeline execution is currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopped: The pipeline execution was manually stopped. For more information, see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
     *         stop mode, the execution is either completing or abandoning in-progress actions. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Succeeded: The pipeline execution was completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
     *         pipeline execution advanced and continued through the pipeline instead. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     *         Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The pipeline execution was not completed successfully.
     *         </p>
     *         </li>
     * @see PipelineExecutionStatus
     */
    public final PipelineExecutionStatus status() {
        return PipelineExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the pipeline execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * InProgress: The pipeline execution is currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopped: The pipeline execution was manually stopped. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop mode,
     * the execution is either completing or abandoning in-progress actions. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Succeeded: The pipeline execution was completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
     * execution advanced and continued through the pipeline instead. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     * Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The pipeline execution was not completed successfully.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PipelineExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the pipeline execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         InProgress: The pipeline execution is currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopped: The pipeline execution was manually stopped. For more information, see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
     *         stop mode, the execution is either completing or abandoning in-progress actions. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Succeeded: The pipeline execution was completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
     *         pipeline execution advanced and continued through the pipeline instead. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     *         Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The pipeline execution was not completed successfully.
     *         </p>
     *         </li>
     * @see PipelineExecutionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Status summary for the pipeline.
     * </p>
     * 
     * @return Status summary for the pipeline.
     */
    public final String statusSummary() {
        return statusSummary;
    }

    /**
     * <p>
     * The date and time when the pipeline execution began, in timestamp format.
     * </p>
     * 
     * @return The date and time when the pipeline execution began, in timestamp format.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The date and time of the last change to the pipeline execution, in timestamp format.
     * </p>
     * 
     * @return The date and time of the last change to the pipeline execution, in timestamp format.
     */
    public final Instant lastUpdateTime() {
        return lastUpdateTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceRevisions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceRevisions() {
        return sourceRevisions != null && !(sourceRevisions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the source artifact revisions that initiated a pipeline execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceRevisions} method.
     * </p>
     * 
     * @return A list of the source artifact revisions that initiated a pipeline execution.
     */
    public final List<SourceRevision> sourceRevisions() {
        return sourceRevisions;
    }

    /**
     * <p>
     * The interaction or event that started a pipeline execution, such as automated change detection or a
     * <code>StartPipelineExecution</code> API call.
     * </p>
     * 
     * @return The interaction or event that started a pipeline execution, such as automated change detection or a
     *         <code>StartPipelineExecution</code> API call.
     */
    public final ExecutionTrigger trigger() {
        return trigger;
    }

    /**
     * <p>
     * The interaction that stopped a pipeline execution.
     * </p>
     * 
     * @return The interaction that stopped a pipeline execution.
     */
    public final StopExecutionTrigger stopTrigger() {
        return stopTrigger;
    }

    /**
     * <p>
     * The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionMode}
     * will return {@link ExecutionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #executionModeAsString}.
     * </p>
     * 
     * @return The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
     * @see ExecutionMode
     */
    public final ExecutionMode executionMode() {
        return ExecutionMode.fromValue(executionMode);
    }

    /**
     * <p>
     * The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionMode}
     * will return {@link ExecutionMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #executionModeAsString}.
     * </p>
     * 
     * @return The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
     * @see ExecutionMode
     */
    public final String executionModeAsString() {
        return executionMode;
    }

    /**
     * <p>
     * Type of the pipeline execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionType}
     * will return {@link ExecutionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #executionTypeAsString}.
     * </p>
     * 
     * @return Type of the pipeline execution.
     * @see ExecutionType
     */
    public final ExecutionType executionType() {
        return ExecutionType.fromValue(executionType);
    }

    /**
     * <p>
     * Type of the pipeline execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionType}
     * will return {@link ExecutionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #executionTypeAsString}.
     * </p>
     * 
     * @return Type of the pipeline execution.
     * @see ExecutionType
     */
    public final String executionTypeAsString() {
        return executionType;
    }

    /**
     * <p>
     * The metadata for the stage execution to be rolled back.
     * </p>
     * 
     * @return The metadata for the stage execution to be rolled back.
     */
    public final PipelineRollbackMetadata rollbackMetadata() {
        return rollbackMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(pipelineExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusSummary());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdateTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceRevisions() ? sourceRevisions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(trigger());
        hashCode = 31 * hashCode + Objects.hashCode(stopTrigger());
        hashCode = 31 * hashCode + Objects.hashCode(executionModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(executionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(rollbackMetadata());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PipelineExecutionSummary)) {
            return false;
        }
        PipelineExecutionSummary other = (PipelineExecutionSummary) obj;
        return Objects.equals(pipelineExecutionId(), other.pipelineExecutionId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusSummary(), other.statusSummary()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(lastUpdateTime(), other.lastUpdateTime()) && hasSourceRevisions() == other.hasSourceRevisions()
                && Objects.equals(sourceRevisions(), other.sourceRevisions()) && Objects.equals(trigger(), other.trigger())
                && Objects.equals(stopTrigger(), other.stopTrigger())
                && Objects.equals(executionModeAsString(), other.executionModeAsString())
                && Objects.equals(executionTypeAsString(), other.executionTypeAsString())
                && Objects.equals(rollbackMetadata(), other.rollbackMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PipelineExecutionSummary").add("PipelineExecutionId", pipelineExecutionId())
                .add("Status", statusAsString()).add("StatusSummary", statusSummary()).add("StartTime", startTime())
                .add("LastUpdateTime", lastUpdateTime()).add("SourceRevisions", hasSourceRevisions() ? sourceRevisions() : null)
                .add("Trigger", trigger()).add("StopTrigger", stopTrigger()).add("ExecutionMode", executionModeAsString())
                .add("ExecutionType", executionTypeAsString()).add("RollbackMetadata", rollbackMetadata()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "pipelineExecutionId":
            return Optional.ofNullable(clazz.cast(pipelineExecutionId()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusSummary":
            return Optional.ofNullable(clazz.cast(statusSummary()));
        case "startTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "lastUpdateTime":
            return Optional.ofNullable(clazz.cast(lastUpdateTime()));
        case "sourceRevisions":
            return Optional.ofNullable(clazz.cast(sourceRevisions()));
        case "trigger":
            return Optional.ofNullable(clazz.cast(trigger()));
        case "stopTrigger":
            return Optional.ofNullable(clazz.cast(stopTrigger()));
        case "executionMode":
            return Optional.ofNullable(clazz.cast(executionModeAsString()));
        case "executionType":
            return Optional.ofNullable(clazz.cast(executionTypeAsString()));
        case "rollbackMetadata":
            return Optional.ofNullable(clazz.cast(rollbackMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("pipelineExecutionId", PIPELINE_EXECUTION_ID_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("statusSummary", STATUS_SUMMARY_FIELD);
        map.put("startTime", START_TIME_FIELD);
        map.put("lastUpdateTime", LAST_UPDATE_TIME_FIELD);
        map.put("sourceRevisions", SOURCE_REVISIONS_FIELD);
        map.put("trigger", TRIGGER_FIELD);
        map.put("stopTrigger", STOP_TRIGGER_FIELD);
        map.put("executionMode", EXECUTION_MODE_FIELD);
        map.put("executionType", EXECUTION_TYPE_FIELD);
        map.put("rollbackMetadata", ROLLBACK_METADATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PipelineExecutionSummary, T> g) {
        return obj -> g.apply((PipelineExecutionSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PipelineExecutionSummary> {
        /**
         * <p>
         * The ID of the pipeline execution.
         * </p>
         * 
         * @param pipelineExecutionId
         *        The ID of the pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineExecutionId(String pipelineExecutionId);

        /**
         * <p>
         * The status of the pipeline execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * InProgress: The pipeline execution is currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopped: The pipeline execution was manually stopped. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         * >Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop
         * mode, the execution is either completing or abandoning in-progress actions. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped">
         * Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Succeeded: The pipeline execution was completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
         * execution advanced and continued through the pipeline instead. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
         * Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The pipeline execution was not completed successfully.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the pipeline execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        InProgress: The pipeline execution is currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopped: The pipeline execution was manually stopped. For more information, see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
         *        stop mode, the execution is either completing or abandoning in-progress actions. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Succeeded: The pipeline execution was completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
         *        pipeline execution advanced and continued through the pipeline instead. For more information, see <a
         *        href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">
         *        Superseded Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The pipeline execution was not completed successfully.
         *        </p>
         *        </li>
         * @see PipelineExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the pipeline execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * InProgress: The pipeline execution is currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopped: The pipeline execution was manually stopped. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         * >Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop
         * mode, the execution is either completing or abandoning in-progress actions. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped">
         * Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Succeeded: The pipeline execution was completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
         * execution advanced and continued through the pipeline instead. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
         * Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The pipeline execution was not completed successfully.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the pipeline execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        InProgress: The pipeline execution is currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopped: The pipeline execution was manually stopped. For more information, see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
         *        stop mode, the execution is either completing or abandoning in-progress actions. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Succeeded: The pipeline execution was completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
         *        pipeline execution advanced and continued through the pipeline instead. For more information, see <a
         *        href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">
         *        Superseded Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The pipeline execution was not completed successfully.
         *        </p>
         *        </li>
         * @see PipelineExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineExecutionStatus
         */
        Builder status(PipelineExecutionStatus status);

        /**
         * <p>
         * Status summary for the pipeline.
         * </p>
         * 
         * @param statusSummary
         *        Status summary for the pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusSummary(String statusSummary);

        /**
         * <p>
         * The date and time when the pipeline execution began, in timestamp format.
         * </p>
         * 
         * @param startTime
         *        The date and time when the pipeline execution began, in timestamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The date and time of the last change to the pipeline execution, in timestamp format.
         * </p>
         * 
         * @param lastUpdateTime
         *        The date and time of the last change to the pipeline execution, in timestamp format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdateTime(Instant lastUpdateTime);

        /**
         * <p>
         * A list of the source artifact revisions that initiated a pipeline execution.
         * </p>
         * 
         * @param sourceRevisions
         *        A list of the source artifact revisions that initiated a pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRevisions(Collection<SourceRevision> sourceRevisions);

        /**
         * <p>
         * A list of the source artifact revisions that initiated a pipeline execution.
         * </p>
         * 
         * @param sourceRevisions
         *        A list of the source artifact revisions that initiated a pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRevisions(SourceRevision... sourceRevisions);

        /**
         * <p>
         * A list of the source artifact revisions that initiated a pipeline execution.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codepipeline.model.SourceRevision.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.codepipeline.model.SourceRevision#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codepipeline.model.SourceRevision.Builder#build()} is called
         * immediately and its result is passed to {@link #sourceRevisions(List<SourceRevision>)}.
         * 
         * @param sourceRevisions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codepipeline.model.SourceRevision.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceRevisions(java.util.Collection<SourceRevision>)
         */
        Builder sourceRevisions(Consumer<SourceRevision.Builder>... sourceRevisions);

        /**
         * <p>
         * The interaction or event that started a pipeline execution, such as automated change detection or a
         * <code>StartPipelineExecution</code> API call.
         * </p>
         * 
         * @param trigger
         *        The interaction or event that started a pipeline execution, such as automated change detection or a
         *        <code>StartPipelineExecution</code> API call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trigger(ExecutionTrigger trigger);

        /**
         * <p>
         * The interaction or event that started a pipeline execution, such as automated change detection or a
         * <code>StartPipelineExecution</code> API call.
         * </p>
         * This is a convenience method that creates an instance of the {@link ExecutionTrigger.Builder} avoiding the
         * need to create one manually via {@link ExecutionTrigger#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ExecutionTrigger.Builder#build()} is called immediately and its
         * result is passed to {@link #trigger(ExecutionTrigger)}.
         * 
         * @param trigger
         *        a consumer that will call methods on {@link ExecutionTrigger.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #trigger(ExecutionTrigger)
         */
        default Builder trigger(Consumer<ExecutionTrigger.Builder> trigger) {
            return trigger(ExecutionTrigger.builder().applyMutation(trigger).build());
        }

        /**
         * <p>
         * The interaction that stopped a pipeline execution.
         * </p>
         * 
         * @param stopTrigger
         *        The interaction that stopped a pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopTrigger(StopExecutionTrigger stopTrigger);

        /**
         * <p>
         * The interaction that stopped a pipeline execution.
         * </p>
         * This is a convenience method that creates an instance of the {@link StopExecutionTrigger.Builder} avoiding
         * the need to create one manually via {@link StopExecutionTrigger#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StopExecutionTrigger.Builder#build()} is called immediately and
         * its result is passed to {@link #stopTrigger(StopExecutionTrigger)}.
         * 
         * @param stopTrigger
         *        a consumer that will call methods on {@link StopExecutionTrigger.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stopTrigger(StopExecutionTrigger)
         */
        default Builder stopTrigger(Consumer<StopExecutionTrigger.Builder> stopTrigger) {
            return stopTrigger(StopExecutionTrigger.builder().applyMutation(stopTrigger).build());
        }

        /**
         * <p>
         * The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
         * </p>
         * 
         * @param executionMode
         *        The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
         * @see ExecutionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionMode
         */
        Builder executionMode(String executionMode);

        /**
         * <p>
         * The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
         * </p>
         * 
         * @param executionMode
         *        The method that the pipeline will use to handle multiple executions. The default mode is SUPERSEDED.
         * @see ExecutionMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionMode
         */
        Builder executionMode(ExecutionMode executionMode);

        /**
         * <p>
         * Type of the pipeline execution.
         * </p>
         * 
         * @param executionType
         *        Type of the pipeline execution.
         * @see ExecutionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionType
         */
        Builder executionType(String executionType);

        /**
         * <p>
         * Type of the pipeline execution.
         * </p>
         * 
         * @param executionType
         *        Type of the pipeline execution.
         * @see ExecutionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionType
         */
        Builder executionType(ExecutionType executionType);

        /**
         * <p>
         * The metadata for the stage execution to be rolled back.
         * </p>
         * 
         * @param rollbackMetadata
         *        The metadata for the stage execution to be rolled back.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollbackMetadata(PipelineRollbackMetadata rollbackMetadata);

        /**
         * <p>
         * The metadata for the stage execution to be rolled back.
         * </p>
         * This is a convenience method that creates an instance of the {@link PipelineRollbackMetadata.Builder}
         * avoiding the need to create one manually via {@link PipelineRollbackMetadata#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PipelineRollbackMetadata.Builder#build()} is called immediately
         * and its result is passed to {@link #rollbackMetadata(PipelineRollbackMetadata)}.
         * 
         * @param rollbackMetadata
         *        a consumer that will call methods on {@link PipelineRollbackMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rollbackMetadata(PipelineRollbackMetadata)
         */
        default Builder rollbackMetadata(Consumer<PipelineRollbackMetadata.Builder> rollbackMetadata) {
            return rollbackMetadata(PipelineRollbackMetadata.builder().applyMutation(rollbackMetadata).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String pipelineExecutionId;

        private String status;

        private String statusSummary;

        private Instant startTime;

        private Instant lastUpdateTime;

        private List<SourceRevision> sourceRevisions = DefaultSdkAutoConstructList.getInstance();

        private ExecutionTrigger trigger;

        private StopExecutionTrigger stopTrigger;

        private String executionMode;

        private String executionType;

        private PipelineRollbackMetadata rollbackMetadata;

        private BuilderImpl() {
        }

        private BuilderImpl(PipelineExecutionSummary model) {
            pipelineExecutionId(model.pipelineExecutionId);
            status(model.status);
            statusSummary(model.statusSummary);
            startTime(model.startTime);
            lastUpdateTime(model.lastUpdateTime);
            sourceRevisions(model.sourceRevisions);
            trigger(model.trigger);
            stopTrigger(model.stopTrigger);
            executionMode(model.executionMode);
            executionType(model.executionType);
            rollbackMetadata(model.rollbackMetadata);
        }

        public final String getPipelineExecutionId() {
            return pipelineExecutionId;
        }

        public final void setPipelineExecutionId(String pipelineExecutionId) {
            this.pipelineExecutionId = pipelineExecutionId;
        }

        @Override
        public final Builder pipelineExecutionId(String pipelineExecutionId) {
            this.pipelineExecutionId = pipelineExecutionId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(PipelineExecutionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusSummary() {
            return statusSummary;
        }

        public final void setStatusSummary(String statusSummary) {
            this.statusSummary = statusSummary;
        }

        @Override
        public final Builder statusSummary(String statusSummary) {
            this.statusSummary = statusSummary;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getLastUpdateTime() {
            return lastUpdateTime;
        }

        public final void setLastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
        }

        @Override
        public final Builder lastUpdateTime(Instant lastUpdateTime) {
            this.lastUpdateTime = lastUpdateTime;
            return this;
        }

        public final List<SourceRevision.Builder> getSourceRevisions() {
            List<SourceRevision.Builder> result = SourceRevisionListCopier.copyToBuilder(this.sourceRevisions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceRevisions(Collection<SourceRevision.BuilderImpl> sourceRevisions) {
            this.sourceRevisions = SourceRevisionListCopier.copyFromBuilder(sourceRevisions);
        }

        @Override
        public final Builder sourceRevisions(Collection<SourceRevision> sourceRevisions) {
            this.sourceRevisions = SourceRevisionListCopier.copy(sourceRevisions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceRevisions(SourceRevision... sourceRevisions) {
            sourceRevisions(Arrays.asList(sourceRevisions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceRevisions(Consumer<SourceRevision.Builder>... sourceRevisions) {
            sourceRevisions(Stream.of(sourceRevisions).map(c -> SourceRevision.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ExecutionTrigger.Builder getTrigger() {
            return trigger != null ? trigger.toBuilder() : null;
        }

        public final void setTrigger(ExecutionTrigger.BuilderImpl trigger) {
            this.trigger = trigger != null ? trigger.build() : null;
        }

        @Override
        public final Builder trigger(ExecutionTrigger trigger) {
            this.trigger = trigger;
            return this;
        }

        public final StopExecutionTrigger.Builder getStopTrigger() {
            return stopTrigger != null ? stopTrigger.toBuilder() : null;
        }

        public final void setStopTrigger(StopExecutionTrigger.BuilderImpl stopTrigger) {
            this.stopTrigger = stopTrigger != null ? stopTrigger.build() : null;
        }

        @Override
        public final Builder stopTrigger(StopExecutionTrigger stopTrigger) {
            this.stopTrigger = stopTrigger;
            return this;
        }

        public final String getExecutionMode() {
            return executionMode;
        }

        public final void setExecutionMode(String executionMode) {
            this.executionMode = executionMode;
        }

        @Override
        public final Builder executionMode(String executionMode) {
            this.executionMode = executionMode;
            return this;
        }

        @Override
        public final Builder executionMode(ExecutionMode executionMode) {
            this.executionMode(executionMode == null ? null : executionMode.toString());
            return this;
        }

        public final String getExecutionType() {
            return executionType;
        }

        public final void setExecutionType(String executionType) {
            this.executionType = executionType;
        }

        @Override
        public final Builder executionType(String executionType) {
            this.executionType = executionType;
            return this;
        }

        @Override
        public final Builder executionType(ExecutionType executionType) {
            this.executionType(executionType == null ? null : executionType.toString());
            return this;
        }

        public final PipelineRollbackMetadata.Builder getRollbackMetadata() {
            return rollbackMetadata != null ? rollbackMetadata.toBuilder() : null;
        }

        public final void setRollbackMetadata(PipelineRollbackMetadata.BuilderImpl rollbackMetadata) {
            this.rollbackMetadata = rollbackMetadata != null ? rollbackMetadata.build() : null;
        }

        @Override
        public final Builder rollbackMetadata(PipelineRollbackMetadata rollbackMetadata) {
            this.rollbackMetadata = rollbackMetadata;
            return this;
        }

        @Override
        public PipelineExecutionSummary build() {
            return new PipelineExecutionSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
