/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestar;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codestar.model.AssociateTeamMemberRequest;
import software.amazon.awssdk.services.codestar.model.AssociateTeamMemberResponse;
import software.amazon.awssdk.services.codestar.model.CodeStarException;
import software.amazon.awssdk.services.codestar.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codestar.model.CreateProjectRequest;
import software.amazon.awssdk.services.codestar.model.CreateProjectResponse;
import software.amazon.awssdk.services.codestar.model.CreateUserProfileRequest;
import software.amazon.awssdk.services.codestar.model.CreateUserProfileResponse;
import software.amazon.awssdk.services.codestar.model.DeleteProjectRequest;
import software.amazon.awssdk.services.codestar.model.DeleteProjectResponse;
import software.amazon.awssdk.services.codestar.model.DeleteUserProfileRequest;
import software.amazon.awssdk.services.codestar.model.DeleteUserProfileResponse;
import software.amazon.awssdk.services.codestar.model.DescribeProjectRequest;
import software.amazon.awssdk.services.codestar.model.DescribeProjectResponse;
import software.amazon.awssdk.services.codestar.model.DescribeUserProfileRequest;
import software.amazon.awssdk.services.codestar.model.DescribeUserProfileResponse;
import software.amazon.awssdk.services.codestar.model.DisassociateTeamMemberRequest;
import software.amazon.awssdk.services.codestar.model.DisassociateTeamMemberResponse;
import software.amazon.awssdk.services.codestar.model.InvalidNextTokenException;
import software.amazon.awssdk.services.codestar.model.InvalidServiceRoleException;
import software.amazon.awssdk.services.codestar.model.LimitExceededException;
import software.amazon.awssdk.services.codestar.model.ListProjectsRequest;
import software.amazon.awssdk.services.codestar.model.ListProjectsResponse;
import software.amazon.awssdk.services.codestar.model.ListResourcesRequest;
import software.amazon.awssdk.services.codestar.model.ListResourcesResponse;
import software.amazon.awssdk.services.codestar.model.ListTagsForProjectRequest;
import software.amazon.awssdk.services.codestar.model.ListTagsForProjectResponse;
import software.amazon.awssdk.services.codestar.model.ListTeamMembersRequest;
import software.amazon.awssdk.services.codestar.model.ListTeamMembersResponse;
import software.amazon.awssdk.services.codestar.model.ListUserProfilesRequest;
import software.amazon.awssdk.services.codestar.model.ListUserProfilesResponse;
import software.amazon.awssdk.services.codestar.model.ProjectAlreadyExistsException;
import software.amazon.awssdk.services.codestar.model.ProjectConfigurationException;
import software.amazon.awssdk.services.codestar.model.ProjectCreationFailedException;
import software.amazon.awssdk.services.codestar.model.ProjectNotFoundException;
import software.amazon.awssdk.services.codestar.model.TagProjectRequest;
import software.amazon.awssdk.services.codestar.model.TagProjectResponse;
import software.amazon.awssdk.services.codestar.model.TeamMemberAlreadyAssociatedException;
import software.amazon.awssdk.services.codestar.model.TeamMemberNotFoundException;
import software.amazon.awssdk.services.codestar.model.UntagProjectRequest;
import software.amazon.awssdk.services.codestar.model.UntagProjectResponse;
import software.amazon.awssdk.services.codestar.model.UpdateProjectRequest;
import software.amazon.awssdk.services.codestar.model.UpdateProjectResponse;
import software.amazon.awssdk.services.codestar.model.UpdateTeamMemberRequest;
import software.amazon.awssdk.services.codestar.model.UpdateTeamMemberResponse;
import software.amazon.awssdk.services.codestar.model.UpdateUserProfileRequest;
import software.amazon.awssdk.services.codestar.model.UpdateUserProfileResponse;
import software.amazon.awssdk.services.codestar.model.UserProfileAlreadyExistsException;
import software.amazon.awssdk.services.codestar.model.UserProfileNotFoundException;
import software.amazon.awssdk.services.codestar.model.ValidationException;
import software.amazon.awssdk.services.codestar.transform.AssociateTeamMemberRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.CreateUserProfileRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DeleteUserProfileRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DescribeProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DescribeUserProfileRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DisassociateTeamMemberRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListTagsForProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListTeamMembersRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListUserProfilesRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.TagProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.UntagProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.UpdateTeamMemberRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.UpdateUserProfileRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CodeStarAsyncClient}.
 *
 * @see CodeStarAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeStarAsyncClient implements CodeStarAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCodeStarAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodeStarAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds an IAM user to the team for an AWS CodeStar project.
     * </p>
     *
     * @param associateTeamMemberRequest
     * @return A Java Future containing the result of the AssociateTeamMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException A resource limit has been exceeded.</li>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>TeamMemberAlreadyAssociatedException The team member is already associated with a role in this
     *         project.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>InvalidServiceRoleException The service role is not valid.</li>
     *         <li>ProjectConfigurationException Project configuration information is required but not specified.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.AssociateTeamMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/AssociateTeamMember" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateTeamMemberResponse> associateTeamMember(
            AssociateTeamMemberRequest associateTeamMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateTeamMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateTeamMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateTeamMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateTeamMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateTeamMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateTeamMemberRequest, AssociateTeamMemberResponse>()
                            .withOperationName("AssociateTeamMember")
                            .withMarshaller(new AssociateTeamMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateTeamMemberRequest));
            CompletableFuture<AssociateTeamMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a project, including project resources. This action creates a project based on a submitted project
     * request. A set of source code files and a toolchain template file can be included with the project request. If
     * these are not provided, an empty project is created.
     * </p>
     *
     * @param createProjectRequest
     * @return A Java Future containing the result of the CreateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectAlreadyExistsException An AWS CodeStar project with the same ID already exists in this region
     *         for the AWS account. AWS CodeStar project IDs must be unique within a region for the AWS account.</li>
     *         <li>LimitExceededException A resource limit has been exceeded.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>ProjectCreationFailedException The project creation request was valid, but a nonspecific exception or
     *         error occurred during project creation. The project could not be created in AWS CodeStar.</li>
     *         <li>InvalidServiceRoleException The service role is not valid.</li>
     *         <li>ProjectConfigurationException Project configuration information is required but not specified.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProjectResponse> createProject(CreateProjectRequest createProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                            .withOperationName("CreateProject")
                            .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProjectRequest));
            CompletableFuture<CreateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a profile for a user that includes user preferences, such as the display name and email address
     * assocciated with the user, in AWS CodeStar. The user profile is not project-specific. Information in the user
     * profile is displayed wherever the user's information appears to other users in AWS CodeStar.
     * </p>
     *
     * @param createUserProfileRequest
     * @return A Java Future containing the result of the CreateUserProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UserProfileAlreadyExistsException A user profile with that name already exists in this region for the
     *         AWS account. AWS CodeStar user profile names must be unique within a region for the AWS account.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.CreateUserProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/CreateUserProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUserProfileResponse> createUserProfile(CreateUserProfileRequest createUserProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUserProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUserProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateUserProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateUserProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUserProfileRequest, CreateUserProfileResponse>()
                            .withOperationName("CreateUserProfile")
                            .withMarshaller(new CreateUserProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createUserProfileRequest));
            CompletableFuture<CreateUserProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a project, including project resources. Does not delete users associated with the project, but does
     * delete the IAM roles that allowed access to the project.
     * </p>
     *
     * @param deleteProjectRequest
     * @return A Java Future containing the result of the DeleteProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>InvalidServiceRoleException The service role is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProjectResponse> deleteProject(DeleteProjectRequest deleteProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                            .withOperationName("DeleteProject")
                            .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProjectRequest));
            CompletableFuture<DeleteProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a user profile in AWS CodeStar, including all personal preference data associated with that profile, such
     * as display name and email address. It does not delete the history of that user, for example the history of
     * commits made by that user.
     * </p>
     *
     * @param deleteUserProfileRequest
     * @return A Java Future containing the result of the DeleteUserProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.DeleteUserProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DeleteUserProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteUserProfileResponse> deleteUserProfile(DeleteUserProfileRequest deleteUserProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUserProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteUserProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteUserProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteUserProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteUserProfileRequest, DeleteUserProfileResponse>()
                            .withOperationName("DeleteUserProfile")
                            .withMarshaller(new DeleteUserProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteUserProfileRequest));
            CompletableFuture<DeleteUserProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a project and its resources.
     * </p>
     *
     * @param describeProjectRequest
     * @return A Java Future containing the result of the DescribeProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>InvalidServiceRoleException The service role is not valid.</li>
     *         <li>ProjectConfigurationException Project configuration information is required but not specified.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.DescribeProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DescribeProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProjectResponse> describeProject(DescribeProjectRequest describeProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProjectRequest, DescribeProjectResponse>()
                            .withOperationName("DescribeProject")
                            .withMarshaller(new DescribeProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeProjectRequest));
            CompletableFuture<DescribeProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a user in AWS CodeStar and the user attributes across all projects.
     * </p>
     *
     * @param describeUserProfileRequest
     * @return A Java Future containing the result of the DescribeUserProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UserProfileNotFoundException The user profile was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.DescribeUserProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DescribeUserProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUserProfileResponse> describeUserProfile(
            DescribeUserProfileRequest describeUserProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUserProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUserProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeUserProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUserProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUserProfileRequest, DescribeUserProfileResponse>()
                            .withOperationName("DescribeUserProfile")
                            .withMarshaller(new DescribeUserProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeUserProfileRequest));
            CompletableFuture<DescribeUserProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a user from a project. Removing a user from a project also removes the IAM policies from that user that
     * allowed access to the project and its resources. Disassociating a team member does not remove that user's profile
     * from AWS CodeStar. It does not remove the user from IAM.
     * </p>
     *
     * @param disassociateTeamMemberRequest
     * @return A Java Future containing the result of the DisassociateTeamMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>InvalidServiceRoleException The service role is not valid.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.DisassociateTeamMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DisassociateTeamMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateTeamMemberResponse> disassociateTeamMember(
            DisassociateTeamMemberRequest disassociateTeamMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateTeamMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateTeamMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateTeamMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateTeamMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateTeamMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateTeamMemberRequest, DisassociateTeamMemberResponse>()
                            .withOperationName("DisassociateTeamMember")
                            .withMarshaller(new DisassociateTeamMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateTeamMemberRequest));
            CompletableFuture<DisassociateTeamMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all projects in AWS CodeStar associated with your AWS account.
     * </p>
     *
     * @param listProjectsRequest
     * @return A Java Future containing the result of the ListProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The next token is not valid.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProjectsResponse> listProjects(ListProjectsRequest listProjectsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                            .withOperationName("ListProjects").withMarshaller(new ListProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProjectsRequest));
            CompletableFuture<ListProjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists resources associated with a project in AWS CodeStar.
     * </p>
     *
     * @param listResourcesRequest
     * @return A Java Future containing the result of the ListResources operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>InvalidNextTokenException The next token is not valid.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesResponse> listResources(ListResourcesRequest listResourcesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                            .withOperationName("ListResources")
                            .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourcesRequest));
            CompletableFuture<ListResourcesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the tags for a project.
     * </p>
     *
     * @param listTagsForProjectRequest
     * @return A Java Future containing the result of the ListTagsForProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>InvalidNextTokenException The next token is not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.ListTagsForProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListTagsForProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForProjectResponse> listTagsForProject(ListTagsForProjectRequest listTagsForProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForProjectRequest, ListTagsForProjectResponse>()
                            .withOperationName("ListTagsForProject")
                            .withMarshaller(new ListTagsForProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForProjectRequest));
            CompletableFuture<ListTagsForProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all team members associated with a project.
     * </p>
     *
     * @param listTeamMembersRequest
     * @return A Java Future containing the result of the ListTeamMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>InvalidNextTokenException The next token is not valid.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.ListTeamMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListTeamMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTeamMembersResponse> listTeamMembers(ListTeamMembersRequest listTeamMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTeamMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTeamMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTeamMembersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTeamMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTeamMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTeamMembersRequest, ListTeamMembersResponse>()
                            .withOperationName("ListTeamMembers")
                            .withMarshaller(new ListTeamMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTeamMembersRequest));
            CompletableFuture<ListTeamMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the user profiles configured for your AWS account in AWS CodeStar.
     * </p>
     *
     * @param listUserProfilesRequest
     * @return A Java Future containing the result of the ListUserProfiles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The next token is not valid.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.ListUserProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListUserProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUserProfilesResponse> listUserProfiles(ListUserProfilesRequest listUserProfilesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUserProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUserProfiles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUserProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListUserProfilesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUserProfilesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUserProfilesRequest, ListUserProfilesResponse>()
                            .withOperationName("ListUserProfiles")
                            .withMarshaller(new ListUserProfilesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listUserProfilesRequest));
            CompletableFuture<ListUserProfilesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds tags to a project.
     * </p>
     *
     * @param tagProjectRequest
     * @return A Java Future containing the result of the TagProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>LimitExceededException A resource limit has been exceeded.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.TagProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/TagProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagProjectResponse> tagProject(TagProjectRequest tagProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagProjectRequest, TagProjectResponse>().withOperationName("TagProject")
                            .withMarshaller(new TagProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagProjectRequest));
            CompletableFuture<TagProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a project.
     * </p>
     *
     * @param untagProjectRequest
     * @return A Java Future containing the result of the UntagProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>LimitExceededException A resource limit has been exceeded.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.UntagProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/UntagProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagProjectResponse> untagProject(UntagProjectRequest untagProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagProjectRequest, UntagProjectResponse>()
                            .withOperationName("UntagProject").withMarshaller(new UntagProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagProjectRequest));
            CompletableFuture<UntagProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a project in AWS CodeStar.
     * </p>
     *
     * @param updateProjectRequest
     * @return A Java Future containing the result of the UpdateProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProjectResponse> updateProject(UpdateProjectRequest updateProjectRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                            .withOperationName("UpdateProject")
                            .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProjectRequest));
            CompletableFuture<UpdateProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a team member's attributes in an AWS CodeStar project. For example, you can change a team member's role
     * in the project, or change whether they have remote access to project resources.
     * </p>
     *
     * @param updateTeamMemberRequest
     * @return A Java Future containing the result of the UpdateTeamMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException A resource limit has been exceeded.</li>
     *         <li>ProjectNotFoundException The specified AWS CodeStar project was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>InvalidServiceRoleException The service role is not valid.</li>
     *         <li>ProjectConfigurationException Project configuration information is required but not specified.</li>
     *         <li>ConcurrentModificationException Another modification is being made. That modification must complete
     *         before you can make your change.</li>
     *         <li>TeamMemberNotFoundException The specified team member was not found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.UpdateTeamMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/UpdateTeamMember" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTeamMemberResponse> updateTeamMember(UpdateTeamMemberRequest updateTeamMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTeamMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTeamMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTeamMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTeamMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTeamMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTeamMemberRequest, UpdateTeamMemberResponse>()
                            .withOperationName("UpdateTeamMember")
                            .withMarshaller(new UpdateTeamMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTeamMemberRequest));
            CompletableFuture<UpdateTeamMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a user's profile in AWS CodeStar. The user profile is not project-specific. Information in the user
     * profile is displayed wherever the user's information appears to other users in AWS CodeStar.
     * </p>
     *
     * @param updateUserProfileRequest
     * @return A Java Future containing the result of the UpdateUserProfile operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UserProfileNotFoundException The user profile was not found.</li>
     *         <li>ValidationException The specified input is either not valid, or it could not be validated.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CodeStarAsyncClient.UpdateUserProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/UpdateUserProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserProfileResponse> updateUserProfile(UpdateUserProfileRequest updateUserProfileRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserProfile");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateUserProfileResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateUserProfileResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserProfileRequest, UpdateUserProfileResponse>()
                            .withOperationName("UpdateUserProfile")
                            .withMarshaller(new UpdateUserProfileRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateUserProfileRequest));
            CompletableFuture<UpdateUserProfileResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeStarException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidServiceRoleException")
                                .exceptionBuilderSupplier(InvalidServiceRoleException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProjectAlreadyExistsException")
                                .exceptionBuilderSupplier(ProjectAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserProfileAlreadyExistsException")
                                .exceptionBuilderSupplier(UserProfileAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProjectConfigurationException")
                                .exceptionBuilderSupplier(ProjectConfigurationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProjectNotFoundException")
                                .exceptionBuilderSupplier(ProjectNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProjectCreationFailedException")
                                .exceptionBuilderSupplier(ProjectCreationFailedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TeamMemberAlreadyAssociatedException")
                                .exceptionBuilderSupplier(TeamMemberAlreadyAssociatedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserProfileNotFoundException")
                                .exceptionBuilderSupplier(UserProfileNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TeamMemberNotFoundException")
                                .exceptionBuilderSupplier(TeamMemberNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
