/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestar;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codestar.model.AssociateTeamMemberRequest;
import software.amazon.awssdk.services.codestar.model.AssociateTeamMemberResponse;
import software.amazon.awssdk.services.codestar.model.CodeStarException;
import software.amazon.awssdk.services.codestar.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codestar.model.CreateProjectRequest;
import software.amazon.awssdk.services.codestar.model.CreateProjectResponse;
import software.amazon.awssdk.services.codestar.model.CreateUserProfileRequest;
import software.amazon.awssdk.services.codestar.model.CreateUserProfileResponse;
import software.amazon.awssdk.services.codestar.model.DeleteProjectRequest;
import software.amazon.awssdk.services.codestar.model.DeleteProjectResponse;
import software.amazon.awssdk.services.codestar.model.DeleteUserProfileRequest;
import software.amazon.awssdk.services.codestar.model.DeleteUserProfileResponse;
import software.amazon.awssdk.services.codestar.model.DescribeProjectRequest;
import software.amazon.awssdk.services.codestar.model.DescribeProjectResponse;
import software.amazon.awssdk.services.codestar.model.DescribeUserProfileRequest;
import software.amazon.awssdk.services.codestar.model.DescribeUserProfileResponse;
import software.amazon.awssdk.services.codestar.model.DisassociateTeamMemberRequest;
import software.amazon.awssdk.services.codestar.model.DisassociateTeamMemberResponse;
import software.amazon.awssdk.services.codestar.model.InvalidNextTokenException;
import software.amazon.awssdk.services.codestar.model.InvalidServiceRoleException;
import software.amazon.awssdk.services.codestar.model.LimitExceededException;
import software.amazon.awssdk.services.codestar.model.ListProjectsRequest;
import software.amazon.awssdk.services.codestar.model.ListProjectsResponse;
import software.amazon.awssdk.services.codestar.model.ListResourcesRequest;
import software.amazon.awssdk.services.codestar.model.ListResourcesResponse;
import software.amazon.awssdk.services.codestar.model.ListTagsForProjectRequest;
import software.amazon.awssdk.services.codestar.model.ListTagsForProjectResponse;
import software.amazon.awssdk.services.codestar.model.ListTeamMembersRequest;
import software.amazon.awssdk.services.codestar.model.ListTeamMembersResponse;
import software.amazon.awssdk.services.codestar.model.ListUserProfilesRequest;
import software.amazon.awssdk.services.codestar.model.ListUserProfilesResponse;
import software.amazon.awssdk.services.codestar.model.ProjectAlreadyExistsException;
import software.amazon.awssdk.services.codestar.model.ProjectConfigurationException;
import software.amazon.awssdk.services.codestar.model.ProjectCreationFailedException;
import software.amazon.awssdk.services.codestar.model.ProjectNotFoundException;
import software.amazon.awssdk.services.codestar.model.TagProjectRequest;
import software.amazon.awssdk.services.codestar.model.TagProjectResponse;
import software.amazon.awssdk.services.codestar.model.TeamMemberAlreadyAssociatedException;
import software.amazon.awssdk.services.codestar.model.TeamMemberNotFoundException;
import software.amazon.awssdk.services.codestar.model.UntagProjectRequest;
import software.amazon.awssdk.services.codestar.model.UntagProjectResponse;
import software.amazon.awssdk.services.codestar.model.UpdateProjectRequest;
import software.amazon.awssdk.services.codestar.model.UpdateProjectResponse;
import software.amazon.awssdk.services.codestar.model.UpdateTeamMemberRequest;
import software.amazon.awssdk.services.codestar.model.UpdateTeamMemberResponse;
import software.amazon.awssdk.services.codestar.model.UpdateUserProfileRequest;
import software.amazon.awssdk.services.codestar.model.UpdateUserProfileResponse;
import software.amazon.awssdk.services.codestar.model.UserProfileAlreadyExistsException;
import software.amazon.awssdk.services.codestar.model.UserProfileNotFoundException;
import software.amazon.awssdk.services.codestar.model.ValidationException;
import software.amazon.awssdk.services.codestar.transform.AssociateTeamMemberRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.CreateProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.CreateUserProfileRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DeleteProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DeleteUserProfileRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DescribeProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DescribeUserProfileRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.DisassociateTeamMemberRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListProjectsRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListTagsForProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListTeamMembersRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.ListUserProfilesRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.TagProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.UntagProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.UpdateProjectRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.UpdateTeamMemberRequestMarshaller;
import software.amazon.awssdk.services.codestar.transform.UpdateUserProfileRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CodeStarClient}.
 *
 * @see CodeStarClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodeStarClient implements CodeStarClient {
    private static final Logger log = Logger.loggerFor(DefaultCodeStarClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final CodeStarServiceClientConfiguration serviceClientConfiguration;

    protected DefaultCodeStarClient(CodeStarServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Adds an IAM user to the team for an AWS CodeStar project.
     * </p>
     *
     * @param associateTeamMemberRequest
     * @return Result of the AssociateTeamMember operation returned by the service.
     * @throws LimitExceededException
     *         A resource limit has been exceeded.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws TeamMemberAlreadyAssociatedException
     *         The team member is already associated with a role in this project.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws InvalidServiceRoleException
     *         The service role is not valid.
     * @throws ProjectConfigurationException
     *         Project configuration information is required but not specified.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.AssociateTeamMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/AssociateTeamMember" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public AssociateTeamMemberResponse associateTeamMember(AssociateTeamMemberRequest associateTeamMemberRequest)
            throws LimitExceededException, ProjectNotFoundException, TeamMemberAlreadyAssociatedException, ValidationException,
            InvalidServiceRoleException, ProjectConfigurationException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateTeamMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateTeamMemberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateTeamMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateTeamMember");

            return clientHandler.execute(new ClientExecutionParams<AssociateTeamMemberRequest, AssociateTeamMemberResponse>()
                    .withOperationName("AssociateTeamMember").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(associateTeamMemberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new AssociateTeamMemberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a project, including project resources. This action creates a project based on a submitted project
     * request. A set of source code files and a toolchain template file can be included with the project request. If
     * these are not provided, an empty project is created.
     * </p>
     *
     * @param createProjectRequest
     * @return Result of the CreateProject operation returned by the service.
     * @throws ProjectAlreadyExistsException
     *         An AWS CodeStar project with the same ID already exists in this region for the AWS account. AWS CodeStar
     *         project IDs must be unique within a region for the AWS account.
     * @throws LimitExceededException
     *         A resource limit has been exceeded.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws ProjectCreationFailedException
     *         The project creation request was valid, but a nonspecific exception or error occurred during project
     *         creation. The project could not be created in AWS CodeStar.
     * @throws InvalidServiceRoleException
     *         The service role is not valid.
     * @throws ProjectConfigurationException
     *         Project configuration information is required but not specified.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.CreateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/CreateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateProjectResponse createProject(CreateProjectRequest createProjectRequest) throws ProjectAlreadyExistsException,
            LimitExceededException, ValidationException, ProjectCreationFailedException, InvalidServiceRoleException,
            ProjectConfigurationException, ConcurrentModificationException, AwsServiceException, SdkClientException,
            CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProject");

            return clientHandler.execute(new ClientExecutionParams<CreateProjectRequest, CreateProjectResponse>()
                    .withOperationName("CreateProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a profile for a user that includes user preferences, such as the display name and email address
     * assocciated with the user, in AWS CodeStar. The user profile is not project-specific. Information in the user
     * profile is displayed wherever the user's information appears to other users in AWS CodeStar.
     * </p>
     *
     * @param createUserProfileRequest
     * @return Result of the CreateUserProfile operation returned by the service.
     * @throws UserProfileAlreadyExistsException
     *         A user profile with that name already exists in this region for the AWS account. AWS CodeStar user
     *         profile names must be unique within a region for the AWS account.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.CreateUserProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/CreateUserProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateUserProfileResponse createUserProfile(CreateUserProfileRequest createUserProfileRequest)
            throws UserProfileAlreadyExistsException, ValidationException, AwsServiceException, SdkClientException,
            CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateUserProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateUserProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUserProfile");

            return clientHandler.execute(new ClientExecutionParams<CreateUserProfileRequest, CreateUserProfileResponse>()
                    .withOperationName("CreateUserProfile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createUserProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateUserProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a project, including project resources. Does not delete users associated with the project, but does
     * delete the IAM roles that allowed access to the project.
     * </p>
     *
     * @param deleteProjectRequest
     * @return Result of the DeleteProject operation returned by the service.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws InvalidServiceRoleException
     *         The service role is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.DeleteProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DeleteProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteProjectResponse deleteProject(DeleteProjectRequest deleteProjectRequest) throws ConcurrentModificationException,
            ValidationException, InvalidServiceRoleException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProject");

            return clientHandler.execute(new ClientExecutionParams<DeleteProjectRequest, DeleteProjectResponse>()
                    .withOperationName("DeleteProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a user profile in AWS CodeStar, including all personal preference data associated with that profile, such
     * as display name and email address. It does not delete the history of that user, for example the history of
     * commits made by that user.
     * </p>
     *
     * @param deleteUserProfileRequest
     * @return Result of the DeleteUserProfile operation returned by the service.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.DeleteUserProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DeleteUserProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteUserProfileResponse deleteUserProfile(DeleteUserProfileRequest deleteUserProfileRequest)
            throws ValidationException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteUserProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteUserProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUserProfile");

            return clientHandler.execute(new ClientExecutionParams<DeleteUserProfileRequest, DeleteUserProfileResponse>()
                    .withOperationName("DeleteUserProfile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteUserProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteUserProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a project and its resources.
     * </p>
     *
     * @param describeProjectRequest
     * @return Result of the DescribeProject operation returned by the service.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws InvalidServiceRoleException
     *         The service role is not valid.
     * @throws ProjectConfigurationException
     *         Project configuration information is required but not specified.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.DescribeProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DescribeProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeProjectResponse describeProject(DescribeProjectRequest describeProjectRequest)
            throws ProjectNotFoundException, ValidationException, InvalidServiceRoleException, ProjectConfigurationException,
            ConcurrentModificationException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProject");

            return clientHandler.execute(new ClientExecutionParams<DescribeProjectRequest, DescribeProjectResponse>()
                    .withOperationName("DescribeProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a user in AWS CodeStar and the user attributes across all projects.
     * </p>
     *
     * @param describeUserProfileRequest
     * @return Result of the DescribeUserProfile operation returned by the service.
     * @throws UserProfileNotFoundException
     *         The user profile was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.DescribeUserProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DescribeUserProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeUserProfileResponse describeUserProfile(DescribeUserProfileRequest describeUserProfileRequest)
            throws UserProfileNotFoundException, ValidationException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeUserProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeUserProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUserProfile");

            return clientHandler.execute(new ClientExecutionParams<DescribeUserProfileRequest, DescribeUserProfileResponse>()
                    .withOperationName("DescribeUserProfile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeUserProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeUserProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a user from a project. Removing a user from a project also removes the IAM policies from that user that
     * allowed access to the project and its resources. Disassociating a team member does not remove that user's profile
     * from AWS CodeStar. It does not remove the user from IAM.
     * </p>
     *
     * @param disassociateTeamMemberRequest
     * @return Result of the DisassociateTeamMember operation returned by the service.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws InvalidServiceRoleException
     *         The service role is not valid.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.DisassociateTeamMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/DisassociateTeamMember"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateTeamMemberResponse disassociateTeamMember(DisassociateTeamMemberRequest disassociateTeamMemberRequest)
            throws ProjectNotFoundException, ValidationException, InvalidServiceRoleException, ConcurrentModificationException,
            AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateTeamMemberResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateTeamMemberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateTeamMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateTeamMember");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateTeamMemberRequest, DisassociateTeamMemberResponse>()
                            .withOperationName("DisassociateTeamMember").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(disassociateTeamMemberRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateTeamMemberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all projects in AWS CodeStar associated with your AWS account.
     * </p>
     *
     * @param listProjectsRequest
     * @return Result of the ListProjects operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token is not valid.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.ListProjects
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListProjects" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListProjectsResponse listProjects(ListProjectsRequest listProjectsRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProjectsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListProjectsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProjects");

            return clientHandler.execute(new ClientExecutionParams<ListProjectsRequest, ListProjectsResponse>()
                    .withOperationName("ListProjects").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listProjectsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListProjectsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists resources associated with a project in AWS CodeStar.
     * </p>
     *
     * @param listResourcesRequest
     * @return Result of the ListResources operation returned by the service.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws InvalidNextTokenException
     *         The next token is not valid.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListResources" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListResourcesResponse listResources(ListResourcesRequest listResourcesRequest) throws ProjectNotFoundException,
            InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");

            return clientHandler.execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                    .withOperationName("ListResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the tags for a project.
     * </p>
     *
     * @param listTagsForProjectRequest
     * @return Result of the ListTagsForProject operation returned by the service.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws InvalidNextTokenException
     *         The next token is not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.ListTagsForProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListTagsForProject" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForProjectResponse listTagsForProject(ListTagsForProjectRequest listTagsForProjectRequest)
            throws ProjectNotFoundException, ValidationException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForProject");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForProjectRequest, ListTagsForProjectResponse>()
                    .withOperationName("ListTagsForProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all team members associated with a project.
     * </p>
     *
     * @param listTeamMembersRequest
     * @return Result of the ListTeamMembers operation returned by the service.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws InvalidNextTokenException
     *         The next token is not valid.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.ListTeamMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListTeamMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTeamMembersResponse listTeamMembers(ListTeamMembersRequest listTeamMembersRequest)
            throws ProjectNotFoundException, InvalidNextTokenException, ValidationException, AwsServiceException,
            SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTeamMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTeamMembersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTeamMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTeamMembers");

            return clientHandler.execute(new ClientExecutionParams<ListTeamMembersRequest, ListTeamMembersResponse>()
                    .withOperationName("ListTeamMembers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTeamMembersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTeamMembersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the user profiles configured for your AWS account in AWS CodeStar.
     * </p>
     *
     * @param listUserProfilesRequest
     * @return Result of the ListUserProfiles operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token is not valid.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.ListUserProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/ListUserProfiles" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUserProfilesResponse listUserProfiles(ListUserProfilesRequest listUserProfilesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUserProfilesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUserProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUserProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUserProfiles");

            return clientHandler.execute(new ClientExecutionParams<ListUserProfilesRequest, ListUserProfilesResponse>()
                    .withOperationName("ListUserProfiles").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listUserProfilesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListUserProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds tags to a project.
     * </p>
     *
     * @param tagProjectRequest
     * @return Result of the TagProject operation returned by the service.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws LimitExceededException
     *         A resource limit has been exceeded.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.TagProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/TagProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagProjectResponse tagProject(TagProjectRequest tagProjectRequest) throws ProjectNotFoundException,
            ValidationException, LimitExceededException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagProject");

            return clientHandler
                    .execute(new ClientExecutionParams<TagProjectRequest, TagProjectResponse>().withOperationName("TagProject")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagProjectRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new TagProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from a project.
     * </p>
     *
     * @param untagProjectRequest
     * @return Result of the UntagProject operation returned by the service.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws LimitExceededException
     *         A resource limit has been exceeded.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.UntagProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/UntagProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagProjectResponse untagProject(UntagProjectRequest untagProjectRequest) throws ProjectNotFoundException,
            ValidationException, LimitExceededException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagProject");

            return clientHandler.execute(new ClientExecutionParams<UntagProjectRequest, UntagProjectResponse>()
                    .withOperationName("UntagProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a project in AWS CodeStar.
     * </p>
     *
     * @param updateProjectRequest
     * @return Result of the UpdateProject operation returned by the service.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.UpdateProject
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/UpdateProject" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateProjectResponse updateProject(UpdateProjectRequest updateProjectRequest) throws ProjectNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProjectResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateProjectResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProject");

            return clientHandler.execute(new ClientExecutionParams<UpdateProjectRequest, UpdateProjectResponse>()
                    .withOperationName("UpdateProject").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateProjectRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a team member's attributes in an AWS CodeStar project. For example, you can change a team member's role
     * in the project, or change whether they have remote access to project resources.
     * </p>
     *
     * @param updateTeamMemberRequest
     * @return Result of the UpdateTeamMember operation returned by the service.
     * @throws LimitExceededException
     *         A resource limit has been exceeded.
     * @throws ProjectNotFoundException
     *         The specified AWS CodeStar project was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws InvalidServiceRoleException
     *         The service role is not valid.
     * @throws ProjectConfigurationException
     *         Project configuration information is required but not specified.
     * @throws ConcurrentModificationException
     *         Another modification is being made. That modification must complete before you can make your change.
     * @throws TeamMemberNotFoundException
     *         The specified team member was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.UpdateTeamMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/UpdateTeamMember" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTeamMemberResponse updateTeamMember(UpdateTeamMemberRequest updateTeamMemberRequest)
            throws LimitExceededException, ProjectNotFoundException, ValidationException, InvalidServiceRoleException,
            ProjectConfigurationException, ConcurrentModificationException, TeamMemberNotFoundException, AwsServiceException,
            SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTeamMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTeamMemberResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTeamMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTeamMember");

            return clientHandler.execute(new ClientExecutionParams<UpdateTeamMemberRequest, UpdateTeamMemberResponse>()
                    .withOperationName("UpdateTeamMember").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateTeamMemberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTeamMemberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a user's profile in AWS CodeStar. The user profile is not project-specific. Information in the user
     * profile is displayed wherever the user's information appears to other users in AWS CodeStar.
     * </p>
     *
     * @param updateUserProfileRequest
     * @return Result of the UpdateUserProfile operation returned by the service.
     * @throws UserProfileNotFoundException
     *         The user profile was not found.
     * @throws ValidationException
     *         The specified input is either not valid, or it could not be validated.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodeStarException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodeStarClient.UpdateUserProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-2017-04-19/UpdateUserProfile" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateUserProfileResponse updateUserProfile(UpdateUserProfileRequest updateUserProfileRequest)
            throws UserProfileNotFoundException, ValidationException, AwsServiceException, SdkClientException, CodeStarException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserProfileResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateUserProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CodeStar");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserProfile");

            return clientHandler.execute(new ClientExecutionParams<UpdateUserProfileRequest, UpdateUserProfileResponse>()
                    .withOperationName("UpdateUserProfile").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateUserProfileRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateUserProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodeStarException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidServiceRoleException")
                                .exceptionBuilderSupplier(InvalidServiceRoleException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProjectAlreadyExistsException")
                                .exceptionBuilderSupplier(ProjectAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserProfileAlreadyExistsException")
                                .exceptionBuilderSupplier(UserProfileAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProjectConfigurationException")
                                .exceptionBuilderSupplier(ProjectConfigurationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProjectNotFoundException")
                                .exceptionBuilderSupplier(ProjectNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProjectCreationFailedException")
                                .exceptionBuilderSupplier(ProjectCreationFailedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TeamMemberAlreadyAssociatedException")
                                .exceptionBuilderSupplier(TeamMemberAlreadyAssociatedException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UserProfileNotFoundException")
                                .exceptionBuilderSupplier(UserProfileNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TeamMemberNotFoundException")
                                .exceptionBuilderSupplier(TeamMemberNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public final CodeStarServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
