/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestar.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The toolchain template file provided with the project request. AWS CodeStar uses the template to provision the
 * toolchain stack in AWS CloudFormation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Toolchain implements SdkPojo, Serializable, ToCopyableBuilder<Toolchain.Builder, Toolchain> {
    private static final SdkField<ToolchainSource> SOURCE_FIELD = SdkField.<ToolchainSource> builder(MarshallingType.SDK_POJO)
            .getter(getter(Toolchain::source)).setter(setter(Builder::source)).constructor(ToolchainSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Toolchain::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<Map<String, String>> STACK_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(Toolchain::stackParameters))
            .setter(setter(Builder::stackParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stackParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_FIELD, ROLE_ARN_FIELD,
            STACK_PARAMETERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final ToolchainSource source;

    private final String roleArn;

    private final Map<String, String> stackParameters;

    private Toolchain(BuilderImpl builder) {
        this.source = builder.source;
        this.roleArn = builder.roleArn;
        this.stackParameters = builder.stackParameters;
    }

    /**
     * <p>
     * The Amazon S3 location where the toolchain template file provided with the project request is stored. AWS
     * CodeStar retrieves the file during project creation.
     * </p>
     * 
     * @return The Amazon S3 location where the toolchain template file provided with the project request is stored. AWS
     *         CodeStar retrieves the file during project creation.
     */
    public ToolchainSource source() {
        return source;
    }

    /**
     * <p>
     * The service role ARN for AWS CodeStar to use for the toolchain template during stack provisioning.
     * </p>
     * 
     * @return The service role ARN for AWS CodeStar to use for the toolchain template during stack provisioning.
     */
    public String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The list of parameter overrides to be passed into the toolchain template during stack provisioning, if any.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The list of parameter overrides to be passed into the toolchain template during stack provisioning, if
     *         any.
     */
    public Map<String, String> stackParameters() {
        return stackParameters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(stackParameters());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Toolchain)) {
            return false;
        }
        Toolchain other = (Toolchain) obj;
        return Objects.equals(source(), other.source()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(stackParameters(), other.stackParameters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Toolchain").add("Source", source()).add("RoleArn", roleArn())
                .add("StackParameters", stackParameters() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "stackParameters":
            return Optional.ofNullable(clazz.cast(stackParameters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Toolchain, T> g) {
        return obj -> g.apply((Toolchain) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Toolchain> {
        /**
         * <p>
         * The Amazon S3 location where the toolchain template file provided with the project request is stored. AWS
         * CodeStar retrieves the file during project creation.
         * </p>
         * 
         * @param source
         *        The Amazon S3 location where the toolchain template file provided with the project request is stored.
         *        AWS CodeStar retrieves the file during project creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(ToolchainSource source);

        /**
         * <p>
         * The Amazon S3 location where the toolchain template file provided with the project request is stored. AWS
         * CodeStar retrieves the file during project creation.
         * </p>
         * This is a convenience that creates an instance of the {@link ToolchainSource.Builder} avoiding the need to
         * create one manually via {@link ToolchainSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link ToolchainSource.Builder#build()} is called immediately and its
         * result is passed to {@link #source(ToolchainSource)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link ToolchainSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(ToolchainSource)
         */
        default Builder source(Consumer<ToolchainSource.Builder> source) {
            return source(ToolchainSource.builder().applyMutation(source).build());
        }

        /**
         * <p>
         * The service role ARN for AWS CodeStar to use for the toolchain template during stack provisioning.
         * </p>
         * 
         * @param roleArn
         *        The service role ARN for AWS CodeStar to use for the toolchain template during stack provisioning.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The list of parameter overrides to be passed into the toolchain template during stack provisioning, if any.
         * </p>
         * 
         * @param stackParameters
         *        The list of parameter overrides to be passed into the toolchain template during stack provisioning, if
         *        any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackParameters(Map<String, String> stackParameters);
    }

    static final class BuilderImpl implements Builder {
        private ToolchainSource source;

        private String roleArn;

        private Map<String, String> stackParameters = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Toolchain model) {
            source(model.source);
            roleArn(model.roleArn);
            stackParameters(model.stackParameters);
        }

        public final ToolchainSource.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        @Override
        public final Builder source(ToolchainSource source) {
            this.source = source;
            return this;
        }

        public final void setSource(ToolchainSource.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final Map<String, String> getStackParameters() {
            return stackParameters;
        }

        @Override
        public final Builder stackParameters(Map<String, String> stackParameters) {
            this.stackParameters = TemplateParameterMapCopier.copy(stackParameters);
            return this;
        }

        public final void setStackParameters(Map<String, String> stackParameters) {
            this.stackParameters = TemplateParameterMapCopier.copy(stackParameters);
        }

        @Override
        public Toolchain build() {
            return new Toolchain(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
