/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestarconnections;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.codestarconnections.model.CreateConnectionRequest;
import software.amazon.awssdk.services.codestarconnections.model.CreateConnectionResponse;
import software.amazon.awssdk.services.codestarconnections.model.CreateHostRequest;
import software.amazon.awssdk.services.codestarconnections.model.CreateHostResponse;
import software.amazon.awssdk.services.codestarconnections.model.DeleteConnectionRequest;
import software.amazon.awssdk.services.codestarconnections.model.DeleteConnectionResponse;
import software.amazon.awssdk.services.codestarconnections.model.DeleteHostRequest;
import software.amazon.awssdk.services.codestarconnections.model.DeleteHostResponse;
import software.amazon.awssdk.services.codestarconnections.model.GetConnectionRequest;
import software.amazon.awssdk.services.codestarconnections.model.GetConnectionResponse;
import software.amazon.awssdk.services.codestarconnections.model.GetHostRequest;
import software.amazon.awssdk.services.codestarconnections.model.GetHostResponse;
import software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest;
import software.amazon.awssdk.services.codestarconnections.model.ListConnectionsResponse;
import software.amazon.awssdk.services.codestarconnections.model.ListHostsRequest;
import software.amazon.awssdk.services.codestarconnections.model.ListHostsResponse;
import software.amazon.awssdk.services.codestarconnections.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codestarconnections.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codestarconnections.model.TagResourceRequest;
import software.amazon.awssdk.services.codestarconnections.model.TagResourceResponse;
import software.amazon.awssdk.services.codestarconnections.model.UntagResourceRequest;
import software.amazon.awssdk.services.codestarconnections.model.UntagResourceResponse;
import software.amazon.awssdk.services.codestarconnections.model.UpdateHostRequest;
import software.amazon.awssdk.services.codestarconnections.model.UpdateHostResponse;
import software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsPublisher;
import software.amazon.awssdk.services.codestarconnections.paginators.ListHostsPublisher;

/**
 * Service client for accessing AWS CodeStar connections asynchronously. This can be created using the static
 * {@link #builder()} method.
 *
 * <fullname>AWS CodeStar Connections</fullname>
 * <p>
 * This AWS CodeStar Connections API Reference provides descriptions and usage examples of the operations and data types
 * for the AWS CodeStar Connections API. You can use the connections API to work with connections and installations.
 * </p>
 * <p>
 * <i>Connections</i> are configurations that you use to connect AWS resources to external code repositories. Each
 * connection is a resource that can be given to services such as CodePipeline to connect to a third-party repository
 * such as Bitbucket. For example, you can add the connection in CodePipeline so that it triggers your pipeline when a
 * code change is made to your third-party code repository. Each connection is named and associated with a unique ARN
 * that is used to reference the connection.
 * </p>
 * <p>
 * When you create a connection, the console initiates a third-party connection handshake. <i>Installations</i> are the
 * apps that are used to conduct this handshake. For example, the installation for the Bitbucket provider type is the
 * Bitbucket app. When you create a connection, you can choose an existing installation or create one.
 * </p>
 * <p>
 * When you want to create a connection to an installed provider type such as GitHub Enterprise Server, you create a
 * <i>host</i> for your connections.
 * </p>
 * <p>
 * You can work with connections by calling:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateConnection</a>, which creates a uniquely named connection that can be referenced by services such as
 * CodePipeline.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteConnection</a>, which deletes the specified connection.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetConnection</a>, which returns information about the connection, including the connection status.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListConnections</a>, which lists the connections associated with your account.
 * </p>
 * </li>
 * </ul>
 * <p>
 * You can work with hosts by calling:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreateHost</a>, which creates a host that represents the infrastructure where your provider is installed.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteHost</a>, which deletes the specified host.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>GetHost</a>, which returns information about the host, including the setup status.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>ListHosts</a>, which lists the hosts associated with your account.
 * </p>
 * </li>
 * </ul>
 * <p>
 * You can work with tags in AWS CodeStar Connections by calling the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>ListTagsForResource</a>, which gets information about AWS tags for a specified Amazon Resource Name (ARN) in AWS
 * CodeStar Connections.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>TagResource</a>, which adds or updates tags for a resource in AWS CodeStar Connections.
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UntagResource</a>, which removes tags for a resource in AWS CodeStar Connections.
 * </p>
 * </li>
 * </ul>
 * <p>
 * For information about how to use AWS CodeStar Connections, see the <a
 * href="https://docs.aws.amazon.com/dtconsole/latest/userguide/welcome-connections.html">Developer Tools User
 * Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface CodeStarConnectionsAsyncClient extends AwsClient {
    String SERVICE_NAME = "codestar-connections";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "codestar-connections";

    /**
     * <p>
     * Creates a connection that can then be given to other AWS services like CodePipeline so that it can access
     * third-party code repositories. The connection is in pending status until the third-party connection handshake is
     * completed from the console.
     * </p>
     *
     * @param createConnectionRequest
     * @return A Java Future containing the result of the CreateConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.CreateConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/CreateConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateConnectionResponse> createConnection(CreateConnectionRequest createConnectionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a connection that can then be given to other AWS services like CodePipeline so that it can access
     * third-party code repositories. The connection is in pending status until the third-party connection handshake is
     * completed from the console.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateConnectionRequest.Builder} avoiding the need
     * to create one manually via {@link CreateConnectionRequest#builder()}
     * </p>
     *
     * @param createConnectionRequest
     *        A {@link Consumer} that will call methods on {@link CreateConnectionInput.Builder} to create a request.
     * @return A Java Future containing the result of the CreateConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.CreateConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/CreateConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateConnectionResponse> createConnection(
            Consumer<CreateConnectionRequest.Builder> createConnectionRequest) {
        return createConnection(CreateConnectionRequest.builder().applyMutation(createConnectionRequest).build());
    }

    /**
     * <p>
     * Creates a resource that represents the infrastructure where a third-party provider is installed. The host is used
     * when you create connections to an installed third-party provider type, such as GitHub Enterprise Server. You
     * create one host for all connections to that provider.
     * </p>
     * <note>
     * <p>
     * A host created through the CLI or the SDK is in `PENDING` status by default. You can make its status `AVAILABLE`
     * by setting up the host in the console.
     * </p>
     * </note>
     *
     * @param createHostRequest
     * @return A Java Future containing the result of the CreateHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.CreateHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/CreateHost"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateHostResponse> createHost(CreateHostRequest createHostRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a resource that represents the infrastructure where a third-party provider is installed. The host is used
     * when you create connections to an installed third-party provider type, such as GitHub Enterprise Server. You
     * create one host for all connections to that provider.
     * </p>
     * <note>
     * <p>
     * A host created through the CLI or the SDK is in `PENDING` status by default. You can make its status `AVAILABLE`
     * by setting up the host in the console.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateHostRequest.Builder} avoiding the need to
     * create one manually via {@link CreateHostRequest#builder()}
     * </p>
     *
     * @param createHostRequest
     *        A {@link Consumer} that will call methods on {@link CreateHostInput.Builder} to create a request.
     * @return A Java Future containing the result of the CreateHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.CreateHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/CreateHost"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<CreateHostResponse> createHost(Consumer<CreateHostRequest.Builder> createHostRequest) {
        return createHost(CreateHostRequest.builder().applyMutation(createHostRequest).build());
    }

    /**
     * <p>
     * The connection to be deleted.
     * </p>
     *
     * @param deleteConnectionRequest
     * @return A Java Future containing the result of the DeleteConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/DeleteConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteConnectionResponse> deleteConnection(DeleteConnectionRequest deleteConnectionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The connection to be deleted.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteConnectionRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteConnectionRequest#builder()}
     * </p>
     *
     * @param deleteConnectionRequest
     *        A {@link Consumer} that will call methods on {@link DeleteConnectionInput.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.DeleteConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/DeleteConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteConnectionResponse> deleteConnection(
            Consumer<DeleteConnectionRequest.Builder> deleteConnectionRequest) {
        return deleteConnection(DeleteConnectionRequest.builder().applyMutation(deleteConnectionRequest).build());
    }

    /**
     * <p>
     * The host to be deleted. Before you delete a host, all connections associated to the host must be deleted.
     * </p>
     * <note>
     * <p>
     * A host cannot be deleted if it is in the VPC_CONFIG_INITIALIZING or VPC_CONFIG_DELETING state.
     * </p>
     * </note>
     *
     * @param deleteHostRequest
     * @return A Java Future containing the result of the DeleteHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.DeleteHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/DeleteHost"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteHostResponse> deleteHost(DeleteHostRequest deleteHostRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * The host to be deleted. Before you delete a host, all connections associated to the host must be deleted.
     * </p>
     * <note>
     * <p>
     * A host cannot be deleted if it is in the VPC_CONFIG_INITIALIZING or VPC_CONFIG_DELETING state.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteHostRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteHostRequest#builder()}
     * </p>
     *
     * @param deleteHostRequest
     *        A {@link Consumer} that will call methods on {@link DeleteHostInput.Builder} to create a request.
     * @return A Java Future containing the result of the DeleteHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.DeleteHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/DeleteHost"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<DeleteHostResponse> deleteHost(Consumer<DeleteHostRequest.Builder> deleteHostRequest) {
        return deleteHost(DeleteHostRequest.builder().applyMutation(deleteHostRequest).build());
    }

    /**
     * <p>
     * Returns the connection ARN and details such as status, owner, and provider type.
     * </p>
     *
     * @param getConnectionRequest
     * @return A Java Future containing the result of the GetConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.GetConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/GetConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetConnectionResponse> getConnection(GetConnectionRequest getConnectionRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the connection ARN and details such as status, owner, and provider type.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetConnectionRequest.Builder} avoiding the need to
     * create one manually via {@link GetConnectionRequest#builder()}
     * </p>
     *
     * @param getConnectionRequest
     *        A {@link Consumer} that will call methods on {@link GetConnectionInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetConnection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.GetConnection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/GetConnection"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<GetConnectionResponse> getConnection(Consumer<GetConnectionRequest.Builder> getConnectionRequest) {
        return getConnection(GetConnectionRequest.builder().applyMutation(getConnectionRequest).build());
    }

    /**
     * <p>
     * Returns the host ARN and details such as status, provider type, endpoint, and, if applicable, the VPC
     * configuration.
     * </p>
     *
     * @param getHostRequest
     * @return A Java Future containing the result of the GetHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.GetHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/GetHost" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetHostResponse> getHost(GetHostRequest getHostRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns the host ARN and details such as status, provider type, endpoint, and, if applicable, the VPC
     * configuration.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetHostRequest.Builder} avoiding the need to create
     * one manually via {@link GetHostRequest#builder()}
     * </p>
     *
     * @param getHostRequest
     *        A {@link Consumer} that will call methods on {@link GetHostInput.Builder} to create a request.
     * @return A Java Future containing the result of the GetHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.GetHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/GetHost" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<GetHostResponse> getHost(Consumer<GetHostRequest.Builder> getHostRequest) {
        return getHost(GetHostRequest.builder().applyMutation(getHostRequest).build());
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     *
     * @param listConnectionsRequest
     * @return A Java Future containing the result of the ListConnections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListConnectionsResponse> listConnections(ListConnectionsRequest listConnectionsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListConnectionsRequest.Builder} avoiding the need
     * to create one manually via {@link ListConnectionsRequest#builder()}
     * </p>
     *
     * @param listConnectionsRequest
     *        A {@link Consumer} that will call methods on {@link ListConnectionsInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListConnections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListConnectionsResponse> listConnections(
            Consumer<ListConnectionsRequest.Builder> listConnectionsRequest) {
        return listConnections(ListConnectionsRequest.builder().applyMutation(listConnectionsRequest).build());
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConnections(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsPublisher publisher = client.listConnectionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsPublisher publisher = client.listConnectionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codestarconnections.model.ListConnectionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConnections(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listConnectionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectionsPublisher listConnectionsPaginator(ListConnectionsRequest listConnectionsRequest) {
        return new ListConnectionsPublisher(this, listConnectionsRequest);
    }

    /**
     * <p>
     * Lists the connections associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listConnections(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsPublisher publisher = client.listConnectionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListConnectionsPublisher publisher = client.listConnectionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codestarconnections.model.ListConnectionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listConnections(software.amazon.awssdk.services.codestarconnections.model.ListConnectionsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListConnectionsRequest.Builder} avoiding the need
     * to create one manually via {@link ListConnectionsRequest#builder()}
     * </p>
     *
     * @param listConnectionsRequest
     *        A {@link Consumer} that will call methods on {@link ListConnectionsInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListConnections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListConnections"
     *      target="_top">AWS API Documentation</a>
     */
    default ListConnectionsPublisher listConnectionsPaginator(Consumer<ListConnectionsRequest.Builder> listConnectionsRequest) {
        return listConnectionsPaginator(ListConnectionsRequest.builder().applyMutation(listConnectionsRequest).build());
    }

    /**
     * <p>
     * Lists the hosts associated with your account.
     * </p>
     *
     * @param listHostsRequest
     * @return A Java Future containing the result of the ListHosts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListHosts"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListHostsResponse> listHosts(ListHostsRequest listHostsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the hosts associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListHostsRequest.Builder} avoiding the need to
     * create one manually via {@link ListHostsRequest#builder()}
     * </p>
     *
     * @param listHostsRequest
     *        A {@link Consumer} that will call methods on {@link ListHostsInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListHosts operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListHosts"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListHostsResponse> listHosts(Consumer<ListHostsRequest.Builder> listHostsRequest) {
        return listHosts(ListHostsRequest.builder().applyMutation(listHostsRequest).build());
    }

    /**
     * <p>
     * Lists the hosts associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listHosts(software.amazon.awssdk.services.codestarconnections.model.ListHostsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListHostsPublisher publisher = client.listHostsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListHostsPublisher publisher = client.listHostsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codestarconnections.model.ListHostsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codestarconnections.model.ListHostsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHosts(software.amazon.awssdk.services.codestarconnections.model.ListHostsRequest)} operation.</b>
     * </p>
     *
     * @param listHostsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListHosts"
     *      target="_top">AWS API Documentation</a>
     */
    default ListHostsPublisher listHostsPaginator(ListHostsRequest listHostsRequest) {
        return new ListHostsPublisher(this, listHostsRequest);
    }

    /**
     * <p>
     * Lists the hosts associated with your account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listHosts(software.amazon.awssdk.services.codestarconnections.model.ListHostsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListHostsPublisher publisher = client.listHostsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarconnections.paginators.ListHostsPublisher publisher = client.listHostsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codestarconnections.model.ListHostsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codestarconnections.model.ListHostsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listHosts(software.amazon.awssdk.services.codestarconnections.model.ListHostsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListHostsRequest.Builder} avoiding the need to
     * create one manually via {@link ListHostsRequest#builder()}
     * </p>
     *
     * @param listHostsRequest
     *        A {@link Consumer} that will call methods on {@link ListHostsInput.Builder} to create a request.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListHosts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListHosts"
     *      target="_top">AWS API Documentation</a>
     */
    default ListHostsPublisher listHostsPaginator(Consumer<ListHostsRequest.Builder> listHostsRequest) {
        return listHostsPaginator(ListHostsRequest.builder().applyMutation(listHostsRequest).build());
    }

    /**
     * <p>
     * Gets the set of key-value pairs (metadata) that are used to manage the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Gets the set of key-value pairs (metadata) that are used to manage the resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on {@link ListTagsForResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds to or modifies the tags of the given resource. Tags are metadata that can be used to manage a resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on {@link TagResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>LimitExceededException Exceeded the maximum limit for connections.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<TagResourceResponse> tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes tags from an AWS resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes tags from an AWS resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on {@link UntagResourceInput.Builder} to create a request.
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UntagResourceResponse> untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest) {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates a specified host with the provided configurations.
     * </p>
     *
     * @param updateHostRequest
     * @return A Java Future containing the result of the UpdateHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Two conflicting operations have been made on the same resource.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>UnsupportedOperationException The operation is not supported. Check the connection status and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.UpdateHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/UpdateHost"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateHostResponse> updateHost(UpdateHostRequest updateHostRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a specified host with the provided configurations.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateHostRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateHostRequest#builder()}
     * </p>
     *
     * @param updateHostRequest
     *        A {@link Consumer} that will call methods on {@link UpdateHostInput.Builder} to create a request.
     * @return A Java Future containing the result of the UpdateHost operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Two conflicting operations have been made on the same resource.</li>
     *         <li>ResourceNotFoundException Resource not found. Verify the connection resource ARN and try again.</li>
     *         <li>ResourceUnavailableException Resource not found. Verify the ARN for the host resource and try again.</li>
     *         <li>UnsupportedOperationException The operation is not supported. Check the connection status and try
     *         again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodeStarConnectionsException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CodeStarConnectionsAsyncClient.UpdateHost
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/codestar-connections-2019-12-01/UpdateHost"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<UpdateHostResponse> updateHost(Consumer<UpdateHostRequest.Builder> updateHostRequest) {
        return updateHost(UpdateHostRequest.builder().applyMutation(updateHostRequest).build());
    }

    @Override
    default CodeStarConnectionsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link CodeStarConnectionsAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static CodeStarConnectionsAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link CodeStarConnectionsAsyncClient}.
     */
    static CodeStarConnectionsAsyncClientBuilder builder() {
        return new DefaultCodeStarConnectionsAsyncClientBuilder();
    }
}
