/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestarconnections.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a repository sync attempt for a repository with a sync configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RepositorySyncAttempt implements SdkPojo, Serializable,
        ToCopyableBuilder<RepositorySyncAttempt.Builder, RepositorySyncAttempt> {
    private static final SdkField<Instant> STARTED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartedAt").getter(getter(RepositorySyncAttempt::startedAt)).setter(setter(Builder::startedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartedAt").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(RepositorySyncAttempt::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<List<RepositorySyncEvent>> EVENTS_FIELD = SdkField
            .<List<RepositorySyncEvent>> builder(MarshallingType.LIST)
            .memberName("Events")
            .getter(getter(RepositorySyncAttempt::events))
            .setter(setter(Builder::events))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Events").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RepositorySyncEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(RepositorySyncEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STARTED_AT_FIELD,
            STATUS_FIELD, EVENTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("StartedAt", STARTED_AT_FIELD);
                    put("Status", STATUS_FIELD);
                    put("Events", EVENTS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final Instant startedAt;

    private final String status;

    private final List<RepositorySyncEvent> events;

    private RepositorySyncAttempt(BuilderImpl builder) {
        this.startedAt = builder.startedAt;
        this.status = builder.status;
        this.events = builder.events;
    }

    /**
     * <p>
     * The start time of a specific sync attempt.
     * </p>
     * 
     * @return The start time of a specific sync attempt.
     */
    public final Instant startedAt() {
        return startedAt;
    }

    /**
     * <p>
     * The status of a specific sync attempt. The following are valid statuses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * INITIATED - A repository sync attempt has been created and will begin soon.
     * </p>
     * </li>
     * <li>
     * <p>
     * IN_PROGRESS - A repository sync attempt has started and work is being done to reconcile the branch.
     * </p>
     * </li>
     * <li>
     * <p>
     * SUCCEEDED - The repository sync attempt has completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED - The repository sync attempt has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * QUEUED - The repository sync attempt didn't execute and was queued.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link RepositorySyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a specific sync attempt. The following are valid statuses:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         INITIATED - A repository sync attempt has been created and will begin soon.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IN_PROGRESS - A repository sync attempt has started and work is being done to reconcile the branch.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SUCCEEDED - The repository sync attempt has completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FAILED - The repository sync attempt has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         QUEUED - The repository sync attempt didn't execute and was queued.
     *         </p>
     *         </li>
     * @see RepositorySyncStatus
     */
    public final RepositorySyncStatus status() {
        return RepositorySyncStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of a specific sync attempt. The following are valid statuses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * INITIATED - A repository sync attempt has been created and will begin soon.
     * </p>
     * </li>
     * <li>
     * <p>
     * IN_PROGRESS - A repository sync attempt has started and work is being done to reconcile the branch.
     * </p>
     * </li>
     * <li>
     * <p>
     * SUCCEEDED - The repository sync attempt has completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED - The repository sync attempt has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * QUEUED - The repository sync attempt didn't execute and was queued.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link RepositorySyncStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a specific sync attempt. The following are valid statuses:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         INITIATED - A repository sync attempt has been created and will begin soon.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         IN_PROGRESS - A repository sync attempt has started and work is being done to reconcile the branch.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SUCCEEDED - The repository sync attempt has completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FAILED - The repository sync attempt has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         QUEUED - The repository sync attempt didn't execute and was queued.
     *         </p>
     *         </li>
     * @see RepositorySyncStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Events property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEvents() {
        return events != null && !(events instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The events associated with a specific sync attempt.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEvents} method.
     * </p>
     * 
     * @return The events associated with a specific sync attempt.
     */
    public final List<RepositorySyncEvent> events() {
        return events;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasEvents() ? events() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RepositorySyncAttempt)) {
            return false;
        }
        RepositorySyncAttempt other = (RepositorySyncAttempt) obj;
        return Objects.equals(startedAt(), other.startedAt()) && Objects.equals(statusAsString(), other.statusAsString())
                && hasEvents() == other.hasEvents() && Objects.equals(events(), other.events());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RepositorySyncAttempt").add("StartedAt", startedAt()).add("Status", statusAsString())
                .add("Events", hasEvents() ? events() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StartedAt":
            return Optional.ofNullable(clazz.cast(startedAt()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Events":
            return Optional.ofNullable(clazz.cast(events()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<RepositorySyncAttempt, T> g) {
        return obj -> g.apply((RepositorySyncAttempt) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RepositorySyncAttempt> {
        /**
         * <p>
         * The start time of a specific sync attempt.
         * </p>
         * 
         * @param startedAt
         *        The start time of a specific sync attempt.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Instant startedAt);

        /**
         * <p>
         * The status of a specific sync attempt. The following are valid statuses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * INITIATED - A repository sync attempt has been created and will begin soon.
         * </p>
         * </li>
         * <li>
         * <p>
         * IN_PROGRESS - A repository sync attempt has started and work is being done to reconcile the branch.
         * </p>
         * </li>
         * <li>
         * <p>
         * SUCCEEDED - The repository sync attempt has completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * FAILED - The repository sync attempt has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * QUEUED - The repository sync attempt didn't execute and was queued.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of a specific sync attempt. The following are valid statuses:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        INITIATED - A repository sync attempt has been created and will begin soon.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IN_PROGRESS - A repository sync attempt has started and work is being done to reconcile the branch.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SUCCEEDED - The repository sync attempt has completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FAILED - The repository sync attempt has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        QUEUED - The repository sync attempt didn't execute and was queued.
         *        </p>
         *        </li>
         * @see RepositorySyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RepositorySyncStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a specific sync attempt. The following are valid statuses:
         * </p>
         * <ul>
         * <li>
         * <p>
         * INITIATED - A repository sync attempt has been created and will begin soon.
         * </p>
         * </li>
         * <li>
         * <p>
         * IN_PROGRESS - A repository sync attempt has started and work is being done to reconcile the branch.
         * </p>
         * </li>
         * <li>
         * <p>
         * SUCCEEDED - The repository sync attempt has completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * FAILED - The repository sync attempt has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * QUEUED - The repository sync attempt didn't execute and was queued.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of a specific sync attempt. The following are valid statuses:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        INITIATED - A repository sync attempt has been created and will begin soon.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        IN_PROGRESS - A repository sync attempt has started and work is being done to reconcile the branch.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SUCCEEDED - The repository sync attempt has completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FAILED - The repository sync attempt has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        QUEUED - The repository sync attempt didn't execute and was queued.
         *        </p>
         *        </li>
         * @see RepositorySyncStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RepositorySyncStatus
         */
        Builder status(RepositorySyncStatus status);

        /**
         * <p>
         * The events associated with a specific sync attempt.
         * </p>
         * 
         * @param events
         *        The events associated with a specific sync attempt.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(Collection<RepositorySyncEvent> events);

        /**
         * <p>
         * The events associated with a specific sync attempt.
         * </p>
         * 
         * @param events
         *        The events associated with a specific sync attempt.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(RepositorySyncEvent... events);

        /**
         * <p>
         * The events associated with a specific sync attempt.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codestarconnections.model.RepositorySyncEvent.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.codestarconnections.model.RepositorySyncEvent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codestarconnections.model.RepositorySyncEvent.Builder#build()} is
         * called immediately and its result is passed to {@link #events(List<RepositorySyncEvent>)}.
         * 
         * @param events
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codestarconnections.model.RepositorySyncEvent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #events(java.util.Collection<RepositorySyncEvent>)
         */
        Builder events(Consumer<RepositorySyncEvent.Builder>... events);
    }

    static final class BuilderImpl implements Builder {
        private Instant startedAt;

        private String status;

        private List<RepositorySyncEvent> events = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RepositorySyncAttempt model) {
            startedAt(model.startedAt);
            status(model.status);
            events(model.events);
        }

        public final Instant getStartedAt() {
            return startedAt;
        }

        public final void setStartedAt(Instant startedAt) {
            this.startedAt = startedAt;
        }

        @Override
        public final Builder startedAt(Instant startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(RepositorySyncStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<RepositorySyncEvent.Builder> getEvents() {
            List<RepositorySyncEvent.Builder> result = RepositorySyncEventListCopier.copyToBuilder(this.events);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEvents(Collection<RepositorySyncEvent.BuilderImpl> events) {
            this.events = RepositorySyncEventListCopier.copyFromBuilder(events);
        }

        @Override
        public final Builder events(Collection<RepositorySyncEvent> events) {
            this.events = RepositorySyncEventListCopier.copy(events);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder events(RepositorySyncEvent... events) {
            events(Arrays.asList(events));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder events(Consumer<RepositorySyncEvent.Builder>... events) {
            events(Stream.of(events).map(c -> RepositorySyncEvent.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public RepositorySyncAttempt build() {
            return new RepositorySyncAttempt(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
