/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestarnotifications;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codestarnotifications.model.AccessDeniedException;
import software.amazon.awssdk.services.codestarnotifications.model.CodestarNotificationsException;
import software.amazon.awssdk.services.codestarnotifications.model.CodestarNotificationsRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codestarnotifications.model.ConfigurationException;
import software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.InvalidNextTokenException;
import software.amazon.awssdk.services.codestarnotifications.model.LimitExceededException;
import software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.codestarnotifications.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codestarnotifications.model.SubscribeRequest;
import software.amazon.awssdk.services.codestarnotifications.model.SubscribeResponse;
import software.amazon.awssdk.services.codestarnotifications.model.TagResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.TagResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UntagResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UntagResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ValidationException;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesPublisher;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesPublisher;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsPublisher;
import software.amazon.awssdk.services.codestarnotifications.transform.CreateNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DeleteNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DeleteTargetRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DescribeNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListEventTypesRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListNotificationRulesRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListTargetsRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.SubscribeRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UnsubscribeRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UpdateNotificationRuleRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CodestarNotificationsAsyncClient}.
 *
 * @see CodestarNotificationsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodestarNotificationsAsyncClient implements CodestarNotificationsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCodestarNotificationsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodestarNotificationsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a notification rule for a resource. The rule specifies the events you want notifications about and the
     * targets (such as SNS topics) where you want to receive them.
     * </p>
     *
     * @param createNotificationRuleRequest
     * @return A Java Future containing the result of the CreateNotificationRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceAlreadyExistsException A resource with the same name or ID already exists. Notification rule
     *         names must be unique in your AWS account.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>LimitExceededException One of the AWS CodeStar Notifications limits has been exceeded. Limits apply
     *         to accounts, notification rules, notifications, resources, and targets. For more information, see Limits.
     *         </li>
     *         <li>ConfigurationException Some or all of the configuration is incomplete, missing, or not valid.</li>
     *         <li>ConcurrentModificationException AWS CodeStar Notifications can't complete the request because the
     *         resource is being modified by another process. Wait a few minutes and try again.</li>
     *         <li>AccessDeniedException AWS CodeStar Notifications can't create the notification rule because you do
     *         not have sufficient permissions.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.CreateNotificationRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/CreateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateNotificationRuleResponse> createNotificationRule(
            CreateNotificationRuleRequest createNotificationRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateNotificationRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateNotificationRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateNotificationRuleRequest, CreateNotificationRuleResponse>()
                            .withOperationName("CreateNotificationRule")
                            .withMarshaller(new CreateNotificationRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createNotificationRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a notification rule for a resource.
     * </p>
     *
     * @param deleteNotificationRuleRequest
     * @return A Java Future containing the result of the DeleteNotificationRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>LimitExceededException One of the AWS CodeStar Notifications limits has been exceeded. Limits apply
     *         to accounts, notification rules, notifications, resources, and targets. For more information, see Limits.
     *         </li>
     *         <li>ConcurrentModificationException AWS CodeStar Notifications can't complete the request because the
     *         resource is being modified by another process. Wait a few minutes and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.DeleteNotificationRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteNotificationRuleResponse> deleteNotificationRule(
            DeleteNotificationRuleRequest deleteNotificationRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteNotificationRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteNotificationRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteNotificationRuleRequest, DeleteNotificationRuleResponse>()
                            .withOperationName("DeleteNotificationRule")
                            .withMarshaller(new DeleteNotificationRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteNotificationRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified target for notifications.
     * </p>
     *
     * @param deleteTargetRequest
     * @return A Java Future containing the result of the DeleteTarget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.DeleteTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTargetResponse> deleteTarget(DeleteTargetRequest deleteTargetRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTargetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTargetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTargetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTargetRequest, DeleteTargetResponse>()
                            .withOperationName("DeleteTarget").withMarshaller(new DeleteTargetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteTargetRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specified notification rule.
     * </p>
     *
     * @param describeNotificationRuleRequest
     * @return A Java Future containing the result of the DescribeNotificationRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException AWS CodeStar Notifications can't find a resource that matches the provided
     *         ARN.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.DescribeNotificationRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DescribeNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeNotificationRuleResponse> describeNotificationRule(
            DescribeNotificationRuleRequest describeNotificationRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeNotificationRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeNotificationRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationRuleRequest, DescribeNotificationRuleResponse>()
                            .withOperationName("DescribeNotificationRule")
                            .withMarshaller(new DescribeNotificationRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeNotificationRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the event types available for configuring notifications.
     * </p>
     *
     * @param listEventTypesRequest
     * @return A Java Future containing the result of the ListEventTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value for the enumeration token used in the request to return the next
     *         batch of the results is not valid.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.ListEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventTypesResponse> listEventTypes(ListEventTypesRequest listEventTypesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEventTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventTypesRequest, ListEventTypesResponse>()
                            .withOperationName("ListEventTypes")
                            .withMarshaller(new ListEventTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listEventTypesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the event types available for configuring notifications.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEventTypes(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesPublisher publisher = client.listEventTypesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesPublisher publisher = client.listEventTypesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEventTypes(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEventTypesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value for the enumeration token used in the request to return the next
     *         batch of the results is not valid.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.ListEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    public ListEventTypesPublisher listEventTypesPaginator(ListEventTypesRequest listEventTypesRequest) {
        return new ListEventTypesPublisher(this, applyPaginatorUserAgent(listEventTypesRequest));
    }

    /**
     * <p>
     * Returns a list of the notification rules for an AWS account.
     * </p>
     *
     * @param listNotificationRulesRequest
     * @return A Java Future containing the result of the ListNotificationRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value for the enumeration token used in the request to return the next
     *         batch of the results is not valid.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.ListNotificationRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNotificationRulesResponse> listNotificationRules(
            ListNotificationRulesRequest listNotificationRulesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNotificationRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNotificationRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNotificationRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNotificationRulesRequest, ListNotificationRulesResponse>()
                            .withOperationName("ListNotificationRules")
                            .withMarshaller(new ListNotificationRulesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listNotificationRulesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the notification rules for an AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNotificationRules(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesPublisher publisher = client.listNotificationRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesPublisher publisher = client.listNotificationRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNotificationRules(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNotificationRulesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value for the enumeration token used in the request to return the next
     *         batch of the results is not valid.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.ListNotificationRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    public ListNotificationRulesPublisher listNotificationRulesPaginator(ListNotificationRulesRequest listNotificationRulesRequest) {
        return new ListNotificationRulesPublisher(this, applyPaginatorUserAgent(listNotificationRulesRequest));
    }

    /**
     * <p>
     * Returns a list of the tags associated with a notification rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException AWS CodeStar Notifications can't find a resource that matches the provided
     *         ARN.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTagsForResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the notification rule targets for an AWS account.
     * </p>
     *
     * @param listTargetsRequest
     * @return A Java Future containing the result of the ListTargets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value for the enumeration token used in the request to return the next
     *         batch of the results is not valid.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.ListTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTargetsResponse> listTargets(ListTargetsRequest listTargetsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTargetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTargetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTargetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTargetsRequest, ListTargetsResponse>()
                            .withOperationName("ListTargets").withMarshaller(new ListTargetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listTargetsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the notification rule targets for an AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTargets(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsPublisher publisher = client.listTargetsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsPublisher publisher = client.listTargetsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTargets(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTargetsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The value for the enumeration token used in the request to return the next
     *         batch of the results is not valid.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.ListTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    public ListTargetsPublisher listTargetsPaginator(ListTargetsRequest listTargetsRequest) {
        return new ListTargetsPublisher(this, applyPaginatorUserAgent(listTargetsRequest));
    }

    /**
     * <p>
     * Creates an association between a notification rule and an SNS topic so that the associated target can receive
     * notifications when the events described in the rule are triggered.
     * </p>
     *
     * @param subscribeRequest
     * @return A Java Future containing the result of the Subscribe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>ResourceNotFoundException AWS CodeStar Notifications can't find a resource that matches the provided
     *         ARN.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.Subscribe
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Subscribe"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SubscribeResponse> subscribe(SubscribeRequest subscribeRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SubscribeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SubscribeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SubscribeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SubscribeRequest, SubscribeResponse>().withOperationName("Subscribe")
                            .withMarshaller(new SubscribeRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(subscribeRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associates a set of provided tags with a notification rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException AWS CodeStar Notifications can't find a resource that matches the provided
     *         ARN.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>ConcurrentModificationException AWS CodeStar Notifications can't complete the request because the
     *         resource is being modified by another process. Wait a few minutes and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(tagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an association between a notification rule and an Amazon SNS topic so that subscribers to that topic stop
     * receiving notifications when the events described in the rule are triggered.
     * </p>
     *
     * @param unsubscribeRequest
     * @return A Java Future containing the result of the Unsubscribe operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.Unsubscribe
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Unsubscribe"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UnsubscribeResponse> unsubscribe(UnsubscribeRequest unsubscribeRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UnsubscribeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UnsubscribeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UnsubscribeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnsubscribeRequest, UnsubscribeResponse>()
                            .withOperationName("Unsubscribe").withMarshaller(new UnsubscribeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(unsubscribeRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association between one or more provided tags and a notification rule.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException AWS CodeStar Notifications can't find a resource that matches the provided
     *         ARN.</li>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>ConcurrentModificationException AWS CodeStar Notifications can't complete the request because the
     *         resource is being modified by another process. Wait a few minutes and try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(untagResourceRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a notification rule for a resource. You can change the events that trigger the notification rule, the
     * status of the rule, and the targets that receive the notifications.
     * </p>
     * <note>
     * <p>
     * To add or remove tags for a notification rule, you must use <a>TagResource</a> and <a>UntagResource</a>.
     * </p>
     * </note>
     *
     * @param updateNotificationRuleRequest
     * @return A Java Future containing the result of the UpdateNotificationRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException One or more parameter values are not valid.</li>
     *         <li>ResourceNotFoundException AWS CodeStar Notifications can't find a resource that matches the provided
     *         ARN.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CodestarNotificationsException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample CodestarNotificationsAsyncClient.UpdateNotificationRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UpdateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateNotificationRuleResponse> updateNotificationRule(
            UpdateNotificationRuleRequest updateNotificationRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateNotificationRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateNotificationRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateNotificationRuleRequest, UpdateNotificationRuleResponse>()
                            .withOperationName("UpdateNotificationRule")
                            .withMarshaller(new UpdateNotificationRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateNotificationRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodestarNotificationsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConfigurationException")
                                .exceptionBuilderSupplier(ConfigurationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    private <T extends CodestarNotificationsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
