/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codestarnotifications;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.codestarnotifications.model.AccessDeniedException;
import software.amazon.awssdk.services.codestarnotifications.model.CodestarNotificationsException;
import software.amazon.awssdk.services.codestarnotifications.model.CodestarNotificationsRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ConcurrentModificationException;
import software.amazon.awssdk.services.codestarnotifications.model.ConfigurationException;
import software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.CreateNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DeleteTargetResponse;
import software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.DescribeNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.InvalidNextTokenException;
import software.amazon.awssdk.services.codestarnotifications.model.LimitExceededException;
import software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest;
import software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.codestarnotifications.model.ResourceNotFoundException;
import software.amazon.awssdk.services.codestarnotifications.model.SubscribeRequest;
import software.amazon.awssdk.services.codestarnotifications.model.SubscribeResponse;
import software.amazon.awssdk.services.codestarnotifications.model.TagResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.TagResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UnsubscribeResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UntagResourceRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UntagResourceResponse;
import software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleRequest;
import software.amazon.awssdk.services.codestarnotifications.model.UpdateNotificationRuleResponse;
import software.amazon.awssdk.services.codestarnotifications.model.ValidationException;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable;
import software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable;
import software.amazon.awssdk.services.codestarnotifications.transform.CreateNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DeleteNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DeleteTargetRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.DescribeNotificationRuleRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListEventTypesRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListNotificationRulesRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.ListTargetsRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.SubscribeRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UnsubscribeRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.codestarnotifications.transform.UpdateNotificationRuleRequestMarshaller;

/**
 * Internal implementation of {@link CodestarNotificationsClient}.
 *
 * @see CodestarNotificationsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCodestarNotificationsClient implements CodestarNotificationsClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCodestarNotificationsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a notification rule for a resource. The rule specifies the events you want notifications about and the
     * targets (such as SNS topics) where you want to receive them.
     * </p>
     *
     * @param createNotificationRuleRequest
     * @return Result of the CreateNotificationRule operation returned by the service.
     * @throws ResourceAlreadyExistsException
     *         A resource with the same name or ID already exists. Notification rule names must be unique in your AWS
     *         account.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws LimitExceededException
     *         One of the AWS CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification
     *         rules, notifications, resources, and targets. For more information, see Limits.
     * @throws ConfigurationException
     *         Some or all of the configuration is incomplete, missing, or not valid.
     * @throws ConcurrentModificationException
     *         AWS CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws AccessDeniedException
     *         AWS CodeStar Notifications can't create the notification rule because you do not have sufficient
     *         permissions.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.CreateNotificationRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/CreateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateNotificationRuleResponse createNotificationRule(CreateNotificationRuleRequest createNotificationRuleRequest)
            throws ResourceAlreadyExistsException, ValidationException, LimitExceededException, ConfigurationException,
            ConcurrentModificationException, AccessDeniedException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateNotificationRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<CreateNotificationRuleRequest, CreateNotificationRuleResponse>()
                .withOperationName("CreateNotificationRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(createNotificationRuleRequest)
                .withMarshaller(new CreateNotificationRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a notification rule for a resource.
     * </p>
     *
     * @param deleteNotificationRuleRequest
     * @return Result of the DeleteNotificationRule operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws LimitExceededException
     *         One of the AWS CodeStar Notifications limits has been exceeded. Limits apply to accounts, notification
     *         rules, notifications, resources, and targets. For more information, see Limits.
     * @throws ConcurrentModificationException
     *         AWS CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DeleteNotificationRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteNotificationRuleResponse deleteNotificationRule(DeleteNotificationRuleRequest deleteNotificationRuleRequest)
            throws ValidationException, LimitExceededException, ConcurrentModificationException, AwsServiceException,
            SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteNotificationRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteNotificationRuleRequest, DeleteNotificationRuleResponse>()
                .withOperationName("DeleteNotificationRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteNotificationRuleRequest)
                .withMarshaller(new DeleteNotificationRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Deletes a specified target for notifications.
     * </p>
     *
     * @param deleteTargetRequest
     * @return Result of the DeleteTarget operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DeleteTarget
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DeleteTarget"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteTargetResponse deleteTarget(DeleteTargetRequest deleteTargetRequest) throws ValidationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTargetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTargetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<DeleteTargetRequest, DeleteTargetResponse>()
                .withOperationName("DeleteTarget").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(deleteTargetRequest)
                .withMarshaller(new DeleteTargetRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about a specified notification rule.
     * </p>
     *
     * @param describeNotificationRuleRequest
     * @return Result of the DescribeNotificationRule operation returned by the service.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.DescribeNotificationRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/DescribeNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeNotificationRuleResponse describeNotificationRule(
            DescribeNotificationRuleRequest describeNotificationRuleRequest) throws ResourceNotFoundException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeNotificationRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeNotificationRuleRequest, DescribeNotificationRuleResponse>()
                        .withOperationName("DescribeNotificationRule").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeNotificationRuleRequest)
                        .withMarshaller(new DescribeNotificationRuleRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about the event types available for configuring notifications.
     * </p>
     *
     * @param listEventTypesRequest
     * @return Result of the ListEventTypes operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEventTypesResponse listEventTypes(ListEventTypesRequest listEventTypesRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEventTypesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEventTypesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListEventTypesRequest, ListEventTypesResponse>()
                .withOperationName("ListEventTypes").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listEventTypesRequest)
                .withMarshaller(new ListEventTypesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns information about the event types available for configuring notifications.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listEventTypes(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client.listEventTypesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client
     *             .listEventTypesPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListEventTypesIterable responses = client.listEventTypesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEventTypes(software.amazon.awssdk.services.codestarnotifications.model.ListEventTypesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listEventTypesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListEventTypes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListEventTypes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListEventTypesIterable listEventTypesPaginator(ListEventTypesRequest listEventTypesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return new ListEventTypesIterable(this, applyPaginatorUserAgent(listEventTypesRequest));
    }

    /**
     * <p>
     * Returns a list of the notification rules for an AWS account.
     * </p>
     *
     * @param listNotificationRulesRequest
     * @return Result of the ListNotificationRules operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListNotificationRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationRulesResponse listNotificationRules(ListNotificationRulesRequest listNotificationRulesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListNotificationRulesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListNotificationRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListNotificationRulesRequest, ListNotificationRulesResponse>()
                .withOperationName("ListNotificationRules").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listNotificationRulesRequest)
                .withMarshaller(new ListNotificationRulesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the notification rules for an AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNotificationRules(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client.listNotificationRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client
     *             .listNotificationRulesPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListNotificationRulesIterable responses = client.listNotificationRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNotificationRules(software.amazon.awssdk.services.codestarnotifications.model.ListNotificationRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNotificationRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListNotificationRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListNotificationRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListNotificationRulesIterable listNotificationRulesPaginator(ListNotificationRulesRequest listNotificationRulesRequest)
            throws InvalidNextTokenException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        return new ListNotificationRulesIterable(this, applyPaginatorUserAgent(listNotificationRulesRequest));
    }

    /**
     * <p>
     * Returns a list of the tags associated with a notification rule.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the notification rule targets for an AWS account.
     * </p>
     *
     * @param listTargetsRequest
     * @return Result of the ListTargets operation returned by the service.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTargetsResponse listTargets(ListTargetsRequest listTargetsRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTargetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTargetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListTargetsRequest, ListTargetsResponse>()
                .withOperationName("ListTargets").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listTargetsRequest)
                .withMarshaller(new ListTargetsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns a list of the notification rule targets for an AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listTargets(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client.listTargetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client
     *             .listTargetsPaginator(request);
     *     for (software.amazon.awssdk.services.codestarnotifications.model.ListTargetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.codestarnotifications.paginators.ListTargetsIterable responses = client.listTargetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTargets(software.amazon.awssdk.services.codestarnotifications.model.ListTargetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listTargetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The value for the enumeration token used in the request to return the next batch of the results is not
     *         valid.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.ListTargets
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/ListTargets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTargetsIterable listTargetsPaginator(ListTargetsRequest listTargetsRequest) throws InvalidNextTokenException,
            ValidationException, AwsServiceException, SdkClientException, CodestarNotificationsException {
        return new ListTargetsIterable(this, applyPaginatorUserAgent(listTargetsRequest));
    }

    /**
     * <p>
     * Creates an association between a notification rule and an SNS topic so that the associated target can receive
     * notifications when the events described in the rule are triggered.
     * </p>
     *
     * @param subscribeRequest
     * @return Result of the Subscribe operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.Subscribe
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Subscribe"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SubscribeResponse subscribe(SubscribeRequest subscribeRequest) throws ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SubscribeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SubscribeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<SubscribeRequest, SubscribeResponse>()
                .withOperationName("Subscribe").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(subscribeRequest)
                .withMarshaller(new SubscribeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Associates a set of provided tags with a notification rule.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ConcurrentModificationException
     *         AWS CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, ConcurrentModificationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                .withOperationName("TagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes an association between a notification rule and an Amazon SNS topic so that subscribers to that topic stop
     * receiving notifications when the events described in the rule are triggered.
     * </p>
     *
     * @param unsubscribeRequest
     * @return Result of the Unsubscribe operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.Unsubscribe
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/Unsubscribe"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UnsubscribeResponse unsubscribe(UnsubscribeRequest unsubscribeRequest) throws ValidationException,
            AwsServiceException, SdkClientException, CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UnsubscribeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UnsubscribeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UnsubscribeRequest, UnsubscribeResponse>()
                .withOperationName("Unsubscribe").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(unsubscribeRequest)
                .withMarshaller(new UnsubscribeRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Removes the association between one or more provided tags and a notification rule.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ConcurrentModificationException
     *         AWS CodeStar Notifications can't complete the request because the resource is being modified by another
     *         process. Wait a few minutes and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, ConcurrentModificationException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                .withOperationName("UntagResource").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Updates a notification rule for a resource. You can change the events that trigger the notification rule, the
     * status of the rule, and the targets that receive the notifications.
     * </p>
     * <note>
     * <p>
     * To add or remove tags for a notification rule, you must use <a>TagResource</a> and <a>UntagResource</a>.
     * </p>
     * </note>
     *
     * @param updateNotificationRuleRequest
     * @return Result of the UpdateNotificationRule operation returned by the service.
     * @throws ValidationException
     *         One or more parameter values are not valid.
     * @throws ResourceNotFoundException
     *         AWS CodeStar Notifications can't find a resource that matches the provided ARN.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CodestarNotificationsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CodestarNotificationsClient.UpdateNotificationRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/codestar-notifications-2019-10-15/UpdateNotificationRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateNotificationRuleResponse updateNotificationRule(UpdateNotificationRuleRequest updateNotificationRuleRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            CodestarNotificationsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNotificationRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNotificationRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<UpdateNotificationRuleRequest, UpdateNotificationRuleResponse>()
                .withOperationName("UpdateNotificationRule").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(updateNotificationRuleRequest)
                .withMarshaller(new UpdateNotificationRuleRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CodestarNotificationsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConfigurationException")
                                .exceptionBuilderSupplier(ConfigurationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CodestarNotificationsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
