/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentity;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cognitoidentity.model.CognitoIdentityException;
import software.amazon.awssdk.services.cognitoidentity.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cognitoidentity.model.CreateIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.CreateIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DeleteIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.DescribeIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.DeveloperUserAlreadyRegisteredException;
import software.amazon.awssdk.services.cognitoidentity.model.ExternalServiceException;
import software.amazon.awssdk.services.cognitoidentity.model.GetCredentialsForIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetCredentialsForIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdentityPoolRolesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetIdentityPoolRolesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenForDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenForDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenRequest;
import software.amazon.awssdk.services.cognitoidentity.model.GetOpenIdTokenResponse;
import software.amazon.awssdk.services.cognitoidentity.model.InternalErrorException;
import software.amazon.awssdk.services.cognitoidentity.model.InvalidIdentityPoolConfigurationException;
import software.amazon.awssdk.services.cognitoidentity.model.InvalidParameterException;
import software.amazon.awssdk.services.cognitoidentity.model.LimitExceededException;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentityPoolsRequest;
import software.amazon.awssdk.services.cognitoidentity.model.ListIdentityPoolsResponse;
import software.amazon.awssdk.services.cognitoidentity.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cognitoidentity.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cognitoidentity.model.LookupDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.LookupDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.MergeDeveloperIdentitiesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.MergeDeveloperIdentitiesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.NotAuthorizedException;
import software.amazon.awssdk.services.cognitoidentity.model.ResourceConflictException;
import software.amazon.awssdk.services.cognitoidentity.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cognitoidentity.model.SetIdentityPoolRolesRequest;
import software.amazon.awssdk.services.cognitoidentity.model.SetIdentityPoolRolesResponse;
import software.amazon.awssdk.services.cognitoidentity.model.TagResourceRequest;
import software.amazon.awssdk.services.cognitoidentity.model.TagResourceResponse;
import software.amazon.awssdk.services.cognitoidentity.model.TooManyRequestsException;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkDeveloperIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkDeveloperIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkIdentityRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UnlinkIdentityResponse;
import software.amazon.awssdk.services.cognitoidentity.model.UntagResourceRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UntagResourceResponse;
import software.amazon.awssdk.services.cognitoidentity.model.UpdateIdentityPoolRequest;
import software.amazon.awssdk.services.cognitoidentity.model.UpdateIdentityPoolResponse;
import software.amazon.awssdk.services.cognitoidentity.transform.CreateIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DeleteIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DeleteIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DescribeIdentityPoolRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.DescribeIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetCredentialsForIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetIdRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetIdentityPoolRolesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetOpenIdTokenForDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.GetOpenIdTokenRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListIdentityPoolsRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.LookupDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.MergeDeveloperIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.SetIdentityPoolRolesRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UnlinkDeveloperIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UnlinkIdentityRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cognitoidentity.transform.UpdateIdentityPoolRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CognitoIdentityAsyncClient}.
 *
 * @see CognitoIdentityAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCognitoIdentityAsyncClient implements CognitoIdentityAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCognitoIdentityAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCognitoIdentityAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new identity pool. The identity pool is a store of user identity information that is specific to your
     * AWS account. The keys for <code>SupportedLoginProviders</code> are as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Facebook: <code>graph.facebook.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Google: <code>accounts.google.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon: <code>www.amazon.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Twitter: <code>api.twitter.com</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Digits: <code>www.digits.com</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param createIdentityPoolRequest
     *        Input to the CreateIdentityPool action.
     * @return A Java Future containing the result of the CreateIdentityPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>LimitExceededException Thrown when the total number of user pools has exceeded a preset limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.CreateIdentityPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/CreateIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateIdentityPoolResponse> createIdentityPool(CreateIdentityPoolRequest createIdentityPoolRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateIdentityPool");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateIdentityPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateIdentityPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateIdentityPoolRequest, CreateIdentityPoolResponse>()
                            .withOperationName("CreateIdentityPool")
                            .withMarshaller(new CreateIdentityPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createIdentityPoolRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createIdentityPoolRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateIdentityPoolResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createIdentityPoolRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes identities from an identity pool. You can specify a list of 1-60 identities that you want to delete.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param deleteIdentitiesRequest
     *        Input to the <code>DeleteIdentities</code> action.
     * @return A Java Future containing the result of the DeleteIdentities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.DeleteIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DeleteIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIdentitiesResponse> deleteIdentities(DeleteIdentitiesRequest deleteIdentitiesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIdentities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteIdentitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteIdentitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIdentitiesRequest, DeleteIdentitiesResponse>()
                            .withOperationName("DeleteIdentities")
                            .withMarshaller(new DeleteIdentitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteIdentitiesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteIdentitiesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteIdentitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIdentitiesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an identity pool. Once a pool is deleted, users will not be able to authenticate with the pool.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param deleteIdentityPoolRequest
     *        Input to the DeleteIdentityPool action.
     * @return A Java Future containing the result of the DeleteIdentityPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.DeleteIdentityPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DeleteIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteIdentityPoolResponse> deleteIdentityPool(DeleteIdentityPoolRequest deleteIdentityPoolRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteIdentityPool");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteIdentityPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteIdentityPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteIdentityPoolRequest, DeleteIdentityPoolResponse>()
                            .withOperationName("DeleteIdentityPool")
                            .withMarshaller(new DeleteIdentityPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteIdentityPoolRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteIdentityPoolRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteIdentityPoolResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteIdentityPoolRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata related to the given identity, including when the identity was created and any associated linked
     * logins.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param describeIdentityRequest
     *        Input to the <code>DescribeIdentity</code> action.
     * @return A Java Future containing the result of the DescribeIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.DescribeIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DescribeIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeIdentityResponse> describeIdentity(DescribeIdentityRequest describeIdentityRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeIdentityRequest, DescribeIdentityResponse>()
                            .withOperationName("DescribeIdentity")
                            .withMarshaller(new DescribeIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeIdentityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeIdentityRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeIdentityRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details about a particular identity pool, including the pool name, ID description, creation date, and
     * current number of users.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param describeIdentityPoolRequest
     *        Input to the DescribeIdentityPool action.
     * @return A Java Future containing the result of the DescribeIdentityPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.DescribeIdentityPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/DescribeIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeIdentityPoolResponse> describeIdentityPool(
            DescribeIdentityPoolRequest describeIdentityPoolRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeIdentityPool");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeIdentityPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeIdentityPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeIdentityPoolRequest, DescribeIdentityPoolResponse>()
                            .withOperationName("DescribeIdentityPool")
                            .withMarshaller(new DescribeIdentityPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeIdentityPoolRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeIdentityPoolRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeIdentityPoolResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeIdentityPoolRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns credentials for the provided identity ID. Any provided logins will be validated against supported login
     * providers. If the token is for cognito-identity.amazonaws.com, it will be passed through to AWS Security Token
     * Service with the appropriate role for the token.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getCredentialsForIdentityRequest
     *        Input to the <code>GetCredentialsForIdentity</code> action.
     * @return A Java Future containing the result of the GetCredentialsForIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InvalidIdentityPoolConfigurationException Thrown if the identity pool has no role associated for the
     *         given auth type (auth/unauth) or if the AssumeRole fails.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>ExternalServiceException An exception thrown when a dependent service such as Facebook or Twitter is
     *         not responding</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.GetCredentialsForIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetCredentialsForIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCredentialsForIdentityResponse> getCredentialsForIdentity(
            GetCredentialsForIdentityRequest getCredentialsForIdentityRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCredentialsForIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCredentialsForIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCredentialsForIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCredentialsForIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCredentialsForIdentityRequest, GetCredentialsForIdentityResponse>()
                            .withOperationName("GetCredentialsForIdentity")
                            .withMarshaller(new GetCredentialsForIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCredentialsForIdentityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getCredentialsForIdentityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetCredentialsForIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    getCredentialsForIdentityRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates (or retrieves) a Cognito ID. Supplying multiple logins will create an implicit linked account.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getIdRequest
     *        Input to the GetId action.
     * @return A Java Future containing the result of the GetId operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>LimitExceededException Thrown when the total number of user pools has exceeded a preset limit.</li>
     *         <li>ExternalServiceException An exception thrown when a dependent service such as Facebook or Twitter is
     *         not responding</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.GetId
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetId" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetIdResponse> getId(GetIdRequest getIdRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetId");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIdResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetIdResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIdResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIdRequest, GetIdResponse>().withOperationName("GetId")
                            .withMarshaller(new GetIdRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getIdRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getIdRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetIdResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIdRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the roles for an identity pool.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param getIdentityPoolRolesRequest
     *        Input to the <code>GetIdentityPoolRoles</code> action.
     * @return A Java Future containing the result of the GetIdentityPoolRoles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.GetIdentityPoolRoles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetIdentityPoolRoles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetIdentityPoolRolesResponse> getIdentityPoolRoles(
            GetIdentityPoolRolesRequest getIdentityPoolRolesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetIdentityPoolRoles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetIdentityPoolRolesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetIdentityPoolRolesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetIdentityPoolRolesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetIdentityPoolRolesRequest, GetIdentityPoolRolesResponse>()
                            .withOperationName("GetIdentityPoolRoles")
                            .withMarshaller(new GetIdentityPoolRolesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getIdentityPoolRolesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getIdentityPoolRolesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetIdentityPoolRolesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getIdentityPoolRolesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an OpenID token, using a known Cognito ID. This known Cognito ID is returned by <a>GetId</a>. You can
     * optionally add additional logins for the identity. Supplying multiple logins creates an implicit link.
     * </p>
     * <p>
     * The OpenId token is valid for 10 minutes.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param getOpenIdTokenRequest
     *        Input to the GetOpenIdToken action.
     * @return A Java Future containing the result of the GetOpenIdToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>ExternalServiceException An exception thrown when a dependent service such as Facebook or Twitter is
     *         not responding</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.GetOpenIdToken
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetOpenIdToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOpenIdTokenResponse> getOpenIdToken(GetOpenIdTokenRequest getOpenIdTokenRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOpenIdToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOpenIdTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetOpenIdTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOpenIdTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOpenIdTokenRequest, GetOpenIdTokenResponse>()
                            .withOperationName("GetOpenIdToken")
                            .withMarshaller(new GetOpenIdTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getOpenIdTokenRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getOpenIdTokenRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetOpenIdTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getOpenIdTokenRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers (or retrieves) a Cognito <code>IdentityId</code> and an OpenID Connect token for a user authenticated
     * by your backend authentication process. Supplying multiple logins will create an implicit linked account. You can
     * only specify one developer provider as part of the <code>Logins</code> map, which is linked to the identity pool.
     * The developer provider is the "domain" by which Cognito will refer to your users.
     * </p>
     * <p>
     * You can use <code>GetOpenIdTokenForDeveloperIdentity</code> to create a new identity and to link new logins (that
     * is, user credentials issued by a public provider or developer provider) to an existing identity. When you want to
     * create a new identity, the <code>IdentityId</code> should be null. When you want to associate a new login with an
     * existing authenticated/unauthenticated identity, you can do so by providing the existing <code>IdentityId</code>.
     * This API will create the identity in the specified <code>IdentityPoolId</code>.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param getOpenIdTokenForDeveloperIdentityRequest
     *        Input to the <code>GetOpenIdTokenForDeveloperIdentity</code> action.
     * @return A Java Future containing the result of the GetOpenIdTokenForDeveloperIdentity operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>DeveloperUserAlreadyRegisteredException The provided developer user identifier is already registered
     *         with Cognito under a different identity ID.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.GetOpenIdTokenForDeveloperIdentity
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/GetOpenIdTokenForDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetOpenIdTokenForDeveloperIdentityResponse> getOpenIdTokenForDeveloperIdentity(
            GetOpenIdTokenForDeveloperIdentityRequest getOpenIdTokenForDeveloperIdentityRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetOpenIdTokenForDeveloperIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOpenIdTokenForDeveloperIdentityResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetOpenIdTokenForDeveloperIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOpenIdTokenForDeveloperIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOpenIdTokenForDeveloperIdentityRequest, GetOpenIdTokenForDeveloperIdentityResponse>()
                            .withOperationName("GetOpenIdTokenForDeveloperIdentity")
                            .withMarshaller(new GetOpenIdTokenForDeveloperIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getOpenIdTokenForDeveloperIdentityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getOpenIdTokenForDeveloperIdentityRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetOpenIdTokenForDeveloperIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                    getOpenIdTokenForDeveloperIdentityRequest.overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the identities in an identity pool.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param listIdentitiesRequest
     *        Input to the ListIdentities action.
     * @return A Java Future containing the result of the ListIdentities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.ListIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/ListIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIdentitiesResponse> listIdentities(ListIdentitiesRequest listIdentitiesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIdentitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIdentitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIdentitiesRequest, ListIdentitiesResponse>()
                            .withOperationName("ListIdentities")
                            .withMarshaller(new ListIdentitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listIdentitiesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listIdentitiesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListIdentitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentitiesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the Cognito identity pools registered for your account.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param listIdentityPoolsRequest
     *        Input to the ListIdentityPools action.
     * @return A Java Future containing the result of the ListIdentityPools operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.ListIdentityPools
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/ListIdentityPools"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListIdentityPoolsResponse> listIdentityPools(ListIdentityPoolsRequest listIdentityPoolsRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListIdentityPools");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListIdentityPoolsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListIdentityPoolsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListIdentityPoolsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListIdentityPoolsRequest, ListIdentityPoolsResponse>()
                            .withOperationName("ListIdentityPools")
                            .withMarshaller(new ListIdentityPoolsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listIdentityPoolsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listIdentityPoolsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListIdentityPoolsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listIdentityPoolsRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags that are assigned to an Amazon Cognito identity pool.
     * </p>
     * <p>
     * A tag is a label that you can apply to identity pools to categorize and manage them in different ways, such as by
     * purpose, owner, environment, or other criteria.
     * </p>
     * <p>
     * You can use this action up to 10 times per second, per account.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.ListTagsForResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsForResourceRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <code>IdentityID</code> associated with a <code>DeveloperUserIdentifier</code> or the list of
     * <code>DeveloperUserIdentifier</code> values associated with an <code>IdentityId</code> for an existing identity.
     * Either <code>IdentityID</code> or <code>DeveloperUserIdentifier</code> must not be null. If you supply only one
     * of these values, the other value will be searched in the database and returned as a part of the response. If you
     * supply both, <code>DeveloperUserIdentifier</code> will be matched against <code>IdentityID</code>. If the values
     * are verified against the database, the response returns both values and is the same as the request. Otherwise a
     * <code>ResourceConflictException</code> is thrown.
     * </p>
     * <p>
     * <code>LookupDeveloperIdentity</code> is intended for low-throughput control plane operations: for example, to
     * enable customer service to locate an identity ID by username. If you are using it for higher-volume operations
     * such as user authentication, your requests are likely to be throttled. <a>GetOpenIdTokenForDeveloperIdentity</a>
     * is a better option for higher-volume operations for user authentication.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param lookupDeveloperIdentityRequest
     *        Input to the <code>LookupDeveloperIdentityInput</code> action.
     * @return A Java Future containing the result of the LookupDeveloperIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.LookupDeveloperIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/LookupDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<LookupDeveloperIdentityResponse> lookupDeveloperIdentity(
            LookupDeveloperIdentityRequest lookupDeveloperIdentityRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "LookupDeveloperIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<LookupDeveloperIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, LookupDeveloperIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<LookupDeveloperIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<LookupDeveloperIdentityRequest, LookupDeveloperIdentityResponse>()
                            .withOperationName("LookupDeveloperIdentity")
                            .withMarshaller(new LookupDeveloperIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(lookupDeveloperIdentityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = lookupDeveloperIdentityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<LookupDeveloperIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, lookupDeveloperIdentityRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Merges two users having different <code>IdentityId</code>s, existing in the same identity pool, and identified by
     * the same developer provider. You can use this action to request that discrete users be merged and identified as a
     * single user in the Cognito environment. Cognito associates the given source user (
     * <code>SourceUserIdentifier</code>) with the <code>IdentityId</code> of the <code>DestinationUserIdentifier</code>
     * . Only developer-authenticated users can be merged. If the users to be merged are associated with the same public
     * provider, but as two different users, an exception will be thrown.
     * </p>
     * <p>
     * The number of linked logins is limited to 20. So, the number of linked logins for the source user,
     * <code>SourceUserIdentifier</code>, and the destination user, <code>DestinationUserIdentifier</code>, together
     * should not be larger than 20. Otherwise, an exception will be thrown.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param mergeDeveloperIdentitiesRequest
     *        Input to the <code>MergeDeveloperIdentities</code> action.
     * @return A Java Future containing the result of the MergeDeveloperIdentities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.MergeDeveloperIdentities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/MergeDeveloperIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<MergeDeveloperIdentitiesResponse> mergeDeveloperIdentities(
            MergeDeveloperIdentitiesRequest mergeDeveloperIdentitiesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "MergeDeveloperIdentities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<MergeDeveloperIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, MergeDeveloperIdentitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<MergeDeveloperIdentitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<MergeDeveloperIdentitiesRequest, MergeDeveloperIdentitiesResponse>()
                            .withOperationName("MergeDeveloperIdentities")
                            .withMarshaller(new MergeDeveloperIdentitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(mergeDeveloperIdentitiesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = mergeDeveloperIdentitiesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<MergeDeveloperIdentitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, mergeDeveloperIdentitiesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the roles for an identity pool. These roles are used when making calls to <a>GetCredentialsForIdentity</a>
     * action.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param setIdentityPoolRolesRequest
     *        Input to the <code>SetIdentityPoolRoles</code> action.
     * @return A Java Future containing the result of the SetIdentityPoolRoles operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>ConcurrentModificationException Thrown if there are parallel requests to modify a resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.SetIdentityPoolRoles
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/SetIdentityPoolRoles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SetIdentityPoolRolesResponse> setIdentityPoolRoles(
            SetIdentityPoolRolesRequest setIdentityPoolRolesRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetIdentityPoolRoles");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SetIdentityPoolRolesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SetIdentityPoolRolesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SetIdentityPoolRolesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SetIdentityPoolRolesRequest, SetIdentityPoolRolesResponse>()
                            .withOperationName("SetIdentityPoolRoles")
                            .withMarshaller(new SetIdentityPoolRolesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(setIdentityPoolRolesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = setIdentityPoolRolesRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<SetIdentityPoolRolesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setIdentityPoolRolesRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Assigns a set of tags to an Amazon Cognito identity pool. A tag is a label that you can use to categorize and
     * manage identity pools in different ways, such as by purpose, owner, environment, or other criteria.
     * </p>
     * <p>
     * Each tag consists of a key and value, both of which you define. A key is a general category for more specific
     * values. For example, if you have two versions of an identity pool, one for testing and another for production,
     * you might assign an <code>Environment</code> tag key to both identity pools. The value of this key might be
     * <code>Test</code> for one identity pool and <code>Production</code> for the other.
     * </p>
     * <p>
     * Tags are useful for cost tracking and access control. You can activate your tags so that they appear on the
     * Billing and Cost Management console, where you can track the costs associated with your identity pools. In an IAM
     * policy, you can constrain permissions for identity pools based on specific tags or tag values.
     * </p>
     * <p>
     * You can use this action up to 5 times per second, per account. An identity pool can have as many as 50 tags.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Unlinks a <code>DeveloperUserIdentifier</code> from an existing identity. Unlinked developer users will be
     * considered new identities next time they are seen. If, for a given Cognito identity, you remove all federated
     * identities as well as the developer user identifier, the Cognito identity becomes inaccessible.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param unlinkDeveloperIdentityRequest
     *        Input to the <code>UnlinkDeveloperIdentity</code> action.
     * @return A Java Future containing the result of the UnlinkDeveloperIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.UnlinkDeveloperIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UnlinkDeveloperIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UnlinkDeveloperIdentityResponse> unlinkDeveloperIdentity(
            UnlinkDeveloperIdentityRequest unlinkDeveloperIdentityRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnlinkDeveloperIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UnlinkDeveloperIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UnlinkDeveloperIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UnlinkDeveloperIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnlinkDeveloperIdentityRequest, UnlinkDeveloperIdentityResponse>()
                            .withOperationName("UnlinkDeveloperIdentity")
                            .withMarshaller(new UnlinkDeveloperIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(unlinkDeveloperIdentityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = unlinkDeveloperIdentityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UnlinkDeveloperIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unlinkDeveloperIdentityRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Unlinks a federated identity from an existing account. Unlinked logins will be considered new identities next
     * time they are seen. Removing the last linked login will make this identity inaccessible.
     * </p>
     * <p>
     * This is a public API. You do not need any credentials to call this API.
     * </p>
     *
     * @param unlinkIdentityRequest
     *        Input to the UnlinkIdentity action.
     * @return A Java Future containing the result of the UnlinkIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>ExternalServiceException An exception thrown when a dependent service such as Facebook or Twitter is
     *         not responding</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.UnlinkIdentity
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UnlinkIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UnlinkIdentityResponse> unlinkIdentity(UnlinkIdentityRequest unlinkIdentityRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UnlinkIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UnlinkIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UnlinkIdentityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UnlinkIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UnlinkIdentityRequest, UnlinkIdentityResponse>()
                            .withOperationName("UnlinkIdentity")
                            .withMarshaller(new UnlinkIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(unlinkIdentityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = unlinkIdentityRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UnlinkIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, unlinkIdentityRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from an Amazon Cognito identity pool. You can use this action up to 5 times per
     * second, per account
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an identity pool.
     * </p>
     * <p>
     * You must use AWS Developer credentials to call this API.
     * </p>
     *
     * @param updateIdentityPoolRequest
     *        An object representing an Amazon Cognito identity pool.
     * @return A Java Future containing the result of the UpdateIdentityPool operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException Thrown for missing or bad input parameter(s).</li>
     *         <li>ResourceNotFoundException Thrown when the requested resource (for example, a dataset or record) does
     *         not exist.</li>
     *         <li>NotAuthorizedException Thrown when a user is not authorized to access the requested resource.</li>
     *         <li>ResourceConflictException Thrown when a user tries to use a login which is already linked to another
     *         account.</li>
     *         <li>TooManyRequestsException Thrown when a request is throttled.</li>
     *         <li>InternalErrorException Thrown when the service encounters an error during processing the request.</li>
     *         <li>ConcurrentModificationException Thrown if there are parallel requests to modify a resource.</li>
     *         <li>LimitExceededException Thrown when the total number of user pools has exceeded a preset limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CognitoIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample CognitoIdentityAsyncClient.UpdateIdentityPool
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cognito-identity-2014-06-30/UpdateIdentityPool"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateIdentityPoolResponse> updateIdentityPool(UpdateIdentityPoolRequest updateIdentityPoolRequest) {
        MetricCollector apiCallMetricCollector = MetricCollector.create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cognito Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateIdentityPool");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateIdentityPoolResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateIdentityPoolResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateIdentityPoolResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateIdentityPoolRequest, UpdateIdentityPoolResponse>()
                            .withOperationName("UpdateIdentityPool")
                            .withMarshaller(new UpdateIdentityPoolRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateIdentityPoolRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateIdentityPoolRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateIdentityPoolResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, requestOverrideConfig);
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateIdentityPoolRequest
                    .overrideConfiguration().orElse(null));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CognitoIdentityException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeveloperUserAlreadyRegisteredException")
                                .exceptionBuilderSupplier(DeveloperUserAlreadyRegisteredException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflictException")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExternalServiceException")
                                .exceptionBuilderSupplier(ExternalServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotAuthorizedException")
                                .exceptionBuilderSupplier(NotAuthorizedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidIdentityPoolConfigurationException")
                                .exceptionBuilderSupplier(InvalidIdentityPoolConfigurationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
