/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentity.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Input to the <code>SetIdentityPoolRoles</code> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SetIdentityPoolRolesRequest extends CognitoIdentityRequest implements
        ToCopyableBuilder<SetIdentityPoolRolesRequest.Builder, SetIdentityPoolRolesRequest> {
    private static final SdkField<String> IDENTITY_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SetIdentityPoolRolesRequest::identityPoolId)).setter(setter(Builder::identityPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentityPoolId").build()).build();

    private static final SdkField<Map<String, String>> ROLES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(SetIdentityPoolRolesRequest::roles))
            .setter(setter(Builder::roles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Roles").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, RoleMapping>> ROLE_MAPPINGS_FIELD = SdkField
            .<Map<String, RoleMapping>> builder(MarshallingType.MAP)
            .getter(getter(SetIdentityPoolRolesRequest::roleMappings))
            .setter(setter(Builder::roleMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleMappings").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<RoleMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(RoleMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTITY_POOL_ID_FIELD,
            ROLES_FIELD, ROLE_MAPPINGS_FIELD));

    private final String identityPoolId;

    private final Map<String, String> roles;

    private final Map<String, RoleMapping> roleMappings;

    private SetIdentityPoolRolesRequest(BuilderImpl builder) {
        super(builder);
        this.identityPoolId = builder.identityPoolId;
        this.roles = builder.roles;
        this.roleMappings = builder.roleMappings;
    }

    /**
     * <p>
     * An identity pool ID in the format REGION:GUID.
     * </p>
     * 
     * @return An identity pool ID in the format REGION:GUID.
     */
    public String identityPoolId() {
        return identityPoolId;
    }

    /**
     * Returns true if the Roles property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRoles() {
        return roles != null && !(roles instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The map of roles associated with this pool. For a given role, the key will be either "authenticated" or
     * "unauthenticated" and the value will be the Role ARN.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRoles()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The map of roles associated with this pool. For a given role, the key will be either "authenticated" or
     *         "unauthenticated" and the value will be the Role ARN.
     */
    public Map<String, String> roles() {
        return roles;
    }

    /**
     * Returns true if the RoleMappings property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRoleMappings() {
        return roleMappings != null && !(roleMappings instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * How users for a specific identity provider are to mapped to roles. This is a string to <a>RoleMapping</a> object
     * map. The string identifies the identity provider, for example, "graph.facebook.com" or
     * "cognito-idp-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id".
     * </p>
     * <p>
     * Up to 25 rules can be specified per identity provider.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRoleMappings()} to see if a value was sent in this field.
     * </p>
     * 
     * @return How users for a specific identity provider are to mapped to roles. This is a string to <a>RoleMapping</a>
     *         object map. The string identifies the identity provider, for example, "graph.facebook.com" or
     *         "cognito-idp-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id".</p>
     *         <p>
     *         Up to 25 rules can be specified per identity provider.
     */
    public Map<String, RoleMapping> roleMappings() {
        return roleMappings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(identityPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(roles());
        hashCode = 31 * hashCode + Objects.hashCode(roleMappings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SetIdentityPoolRolesRequest)) {
            return false;
        }
        SetIdentityPoolRolesRequest other = (SetIdentityPoolRolesRequest) obj;
        return Objects.equals(identityPoolId(), other.identityPoolId()) && Objects.equals(roles(), other.roles())
                && Objects.equals(roleMappings(), other.roleMappings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SetIdentityPoolRolesRequest").add("IdentityPoolId", identityPoolId()).add("Roles", roles())
                .add("RoleMappings", roleMappings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IdentityPoolId":
            return Optional.ofNullable(clazz.cast(identityPoolId()));
        case "Roles":
            return Optional.ofNullable(clazz.cast(roles()));
        case "RoleMappings":
            return Optional.ofNullable(clazz.cast(roleMappings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SetIdentityPoolRolesRequest, T> g) {
        return obj -> g.apply((SetIdentityPoolRolesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, SetIdentityPoolRolesRequest> {
        /**
         * <p>
         * An identity pool ID in the format REGION:GUID.
         * </p>
         * 
         * @param identityPoolId
         *        An identity pool ID in the format REGION:GUID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityPoolId(String identityPoolId);

        /**
         * <p>
         * The map of roles associated with this pool. For a given role, the key will be either "authenticated" or
         * "unauthenticated" and the value will be the Role ARN.
         * </p>
         * 
         * @param roles
         *        The map of roles associated with this pool. For a given role, the key will be either "authenticated"
         *        or "unauthenticated" and the value will be the Role ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Map<String, String> roles);

        /**
         * <p>
         * How users for a specific identity provider are to mapped to roles. This is a string to <a>RoleMapping</a>
         * object map. The string identifies the identity provider, for example, "graph.facebook.com" or
         * "cognito-idp-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id".
         * </p>
         * <p>
         * Up to 25 rules can be specified per identity provider.
         * </p>
         * 
         * @param roleMappings
         *        How users for a specific identity provider are to mapped to roles. This is a string to
         *        <a>RoleMapping</a> object map. The string identifies the identity provider, for example,
         *        "graph.facebook.com" or "cognito-idp-east-1.amazonaws.com/us-east-1_abcdefghi:app_client_id".</p>
         *        <p>
         *        Up to 25 rules can be specified per identity provider.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleMappings(Map<String, RoleMapping> roleMappings);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityRequest.BuilderImpl implements Builder {
        private String identityPoolId;

        private Map<String, String> roles = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, RoleMapping> roleMappings = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SetIdentityPoolRolesRequest model) {
            super(model);
            identityPoolId(model.identityPoolId);
            roles(model.roles);
            roleMappings(model.roleMappings);
        }

        public final String getIdentityPoolId() {
            return identityPoolId;
        }

        @Override
        public final Builder identityPoolId(String identityPoolId) {
            this.identityPoolId = identityPoolId;
            return this;
        }

        public final void setIdentityPoolId(String identityPoolId) {
            this.identityPoolId = identityPoolId;
        }

        public final Map<String, String> getRoles() {
            return roles;
        }

        @Override
        public final Builder roles(Map<String, String> roles) {
            this.roles = RolesMapCopier.copy(roles);
            return this;
        }

        public final void setRoles(Map<String, String> roles) {
            this.roles = RolesMapCopier.copy(roles);
        }

        public final Map<String, RoleMapping.Builder> getRoleMappings() {
            return roleMappings != null ? CollectionUtils.mapValues(roleMappings, RoleMapping::toBuilder) : null;
        }

        @Override
        public final Builder roleMappings(Map<String, RoleMapping> roleMappings) {
            this.roleMappings = RoleMappingMapCopier.copy(roleMappings);
            return this;
        }

        public final void setRoleMappings(Map<String, RoleMapping.BuilderImpl> roleMappings) {
            this.roleMappings = RoleMappingMapCopier.copyFromBuilder(roleMappings);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SetIdentityPoolRolesRequest build() {
            return new SetIdentityPoolRolesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
