/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.runtime.TypeConverter;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.cognitoidentityprovider.transform.UserPoolTypeMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A container for information about the user pool type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UserPoolType implements StructuredPojo, ToCopyableBuilder<UserPoolType.Builder, UserPoolType> {
    private final String id;

    private final String name;

    private final UserPoolPolicyType policies;

    private final LambdaConfigType lambdaConfig;

    private final String status;

    private final Instant lastModifiedDate;

    private final Instant creationDate;

    private final List<SchemaAttributeType> schemaAttributes;

    private final List<String> autoVerifiedAttributes;

    private final List<String> aliasAttributes;

    private final List<String> usernameAttributes;

    private final String smsVerificationMessage;

    private final String emailVerificationMessage;

    private final String emailVerificationSubject;

    private final VerificationMessageTemplateType verificationMessageTemplate;

    private final String smsAuthenticationMessage;

    private final String mfaConfiguration;

    private final DeviceConfigurationType deviceConfiguration;

    private final Integer estimatedNumberOfUsers;

    private final EmailConfigurationType emailConfiguration;

    private final SmsConfigurationType smsConfiguration;

    private final Map<String, String> userPoolTags;

    private final String smsConfigurationFailure;

    private final String emailConfigurationFailure;

    private final AdminCreateUserConfigType adminCreateUserConfig;

    private UserPoolType(BuilderImpl builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.policies = builder.policies;
        this.lambdaConfig = builder.lambdaConfig;
        this.status = builder.status;
        this.lastModifiedDate = builder.lastModifiedDate;
        this.creationDate = builder.creationDate;
        this.schemaAttributes = builder.schemaAttributes;
        this.autoVerifiedAttributes = builder.autoVerifiedAttributes;
        this.aliasAttributes = builder.aliasAttributes;
        this.usernameAttributes = builder.usernameAttributes;
        this.smsVerificationMessage = builder.smsVerificationMessage;
        this.emailVerificationMessage = builder.emailVerificationMessage;
        this.emailVerificationSubject = builder.emailVerificationSubject;
        this.verificationMessageTemplate = builder.verificationMessageTemplate;
        this.smsAuthenticationMessage = builder.smsAuthenticationMessage;
        this.mfaConfiguration = builder.mfaConfiguration;
        this.deviceConfiguration = builder.deviceConfiguration;
        this.estimatedNumberOfUsers = builder.estimatedNumberOfUsers;
        this.emailConfiguration = builder.emailConfiguration;
        this.smsConfiguration = builder.smsConfiguration;
        this.userPoolTags = builder.userPoolTags;
        this.smsConfigurationFailure = builder.smsConfigurationFailure;
        this.emailConfigurationFailure = builder.emailConfigurationFailure;
        this.adminCreateUserConfig = builder.adminCreateUserConfig;
    }

    /**
     * <p>
     * The ID of the user pool.
     * </p>
     * 
     * @return The ID of the user pool.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * The name of the user pool.
     * </p>
     * 
     * @return The name of the user pool.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A container for the policies associated with a user pool.
     * </p>
     * 
     * @return A container for the policies associated with a user pool.
     */
    public UserPoolPolicyType policies() {
        return policies;
    }

    /**
     * <p>
     * A container for the AWS Lambda triggers associated with a user pool.
     * </p>
     * 
     * @return A container for the AWS Lambda triggers associated with a user pool.
     */
    public LambdaConfigType lambdaConfig() {
        return lambdaConfig;
    }

    /**
     * <p>
     * The status of a user pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a user pool.
     * @see StatusType
     */
    public StatusType status() {
        return StatusType.fromValue(status);
    }

    /**
     * <p>
     * The status of a user pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link StatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a user pool.
     * @see StatusType
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date the user pool was last modified.
     * </p>
     * 
     * @return The date the user pool was last modified.
     */
    public Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    /**
     * <p>
     * The date the user pool was created.
     * </p>
     * 
     * @return The date the user pool was created.
     */
    public Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * A container with the schema attributes of a user pool.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A container with the schema attributes of a user pool.
     */
    public List<SchemaAttributeType> schemaAttributes() {
        return schemaAttributes;
    }

    /**
     * <p>
     * Specifies the attributes that are auto-verified in a user pool.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies the attributes that are auto-verified in a user pool.
     */
    public List<VerifiedAttributeType> autoVerifiedAttributes() {
        return TypeConverter.convert(autoVerifiedAttributes, VerifiedAttributeType::fromValue);
    }

    /**
     * <p>
     * Specifies the attributes that are auto-verified in a user pool.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies the attributes that are auto-verified in a user pool.
     */
    public List<String> autoVerifiedAttributesAsStrings() {
        return autoVerifiedAttributes;
    }

    /**
     * <p>
     * Specifies the attributes that are aliased in a user pool.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies the attributes that are aliased in a user pool.
     */
    public List<AliasAttributeType> aliasAttributes() {
        return TypeConverter.convert(aliasAttributes, AliasAttributeType::fromValue);
    }

    /**
     * <p>
     * Specifies the attributes that are aliased in a user pool.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies the attributes that are aliased in a user pool.
     */
    public List<String> aliasAttributesAsStrings() {
        return aliasAttributes;
    }

    /**
     * <p>
     * Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
     */
    public List<UsernameAttributeType> usernameAttributes() {
        return TypeConverter.convert(usernameAttributes, UsernameAttributeType::fromValue);
    }

    /**
     * <p>
     * Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
     */
    public List<String> usernameAttributesAsStrings() {
        return usernameAttributes;
    }

    /**
     * <p>
     * The contents of the SMS verification message.
     * </p>
     * 
     * @return The contents of the SMS verification message.
     */
    public String smsVerificationMessage() {
        return smsVerificationMessage;
    }

    /**
     * <p>
     * The contents of the email verification message.
     * </p>
     * 
     * @return The contents of the email verification message.
     */
    public String emailVerificationMessage() {
        return emailVerificationMessage;
    }

    /**
     * <p>
     * The subject of the email verification message.
     * </p>
     * 
     * @return The subject of the email verification message.
     */
    public String emailVerificationSubject() {
        return emailVerificationSubject;
    }

    /**
     * <p>
     * The template for verification messages.
     * </p>
     * 
     * @return The template for verification messages.
     */
    public VerificationMessageTemplateType verificationMessageTemplate() {
        return verificationMessageTemplate;
    }

    /**
     * <p>
     * The contents of the SMS authentication message.
     * </p>
     * 
     * @return The contents of the SMS authentication message.
     */
    public String smsAuthenticationMessage() {
        return smsAuthenticationMessage;
    }

    /**
     * <p>
     * Can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>OFF</code> - MFA tokens are not required and cannot be specified during user registration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ON</code> - MFA tokens are required for all user registrations. You can only specify required when you are
     * initially creating a user pool.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OPTIONAL</code> - Users have the option when registering to create an MFA token.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mfaConfiguration}
     * will return {@link UserPoolMfaType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #mfaConfigurationAsString}.
     * </p>
     * 
     * @return Can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>OFF</code> - MFA tokens are not required and cannot be specified during user registration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ON</code> - MFA tokens are required for all user registrations. You can only specify required when
     *         you are initially creating a user pool.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OPTIONAL</code> - Users have the option when registering to create an MFA token.
     *         </p>
     *         </li>
     * @see UserPoolMfaType
     */
    public UserPoolMfaType mfaConfiguration() {
        return UserPoolMfaType.fromValue(mfaConfiguration);
    }

    /**
     * <p>
     * Can be one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>OFF</code> - MFA tokens are not required and cannot be specified during user registration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ON</code> - MFA tokens are required for all user registrations. You can only specify required when you are
     * initially creating a user pool.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OPTIONAL</code> - Users have the option when registering to create an MFA token.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mfaConfiguration}
     * will return {@link UserPoolMfaType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #mfaConfigurationAsString}.
     * </p>
     * 
     * @return Can be one of the following values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>OFF</code> - MFA tokens are not required and cannot be specified during user registration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ON</code> - MFA tokens are required for all user registrations. You can only specify required when
     *         you are initially creating a user pool.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OPTIONAL</code> - Users have the option when registering to create an MFA token.
     *         </p>
     *         </li>
     * @see UserPoolMfaType
     */
    public String mfaConfigurationAsString() {
        return mfaConfiguration;
    }

    /**
     * <p>
     * The device configuration.
     * </p>
     * 
     * @return The device configuration.
     */
    public DeviceConfigurationType deviceConfiguration() {
        return deviceConfiguration;
    }

    /**
     * <p>
     * A number estimating the size of the user pool.
     * </p>
     * 
     * @return A number estimating the size of the user pool.
     */
    public Integer estimatedNumberOfUsers() {
        return estimatedNumberOfUsers;
    }

    /**
     * <p>
     * The email configuration.
     * </p>
     * 
     * @return The email configuration.
     */
    public EmailConfigurationType emailConfiguration() {
        return emailConfiguration;
    }

    /**
     * <p>
     * The SMS configuration.
     * </p>
     * 
     * @return The SMS configuration.
     */
    public SmsConfigurationType smsConfiguration() {
        return smsConfiguration;
    }

    /**
     * <p>
     * The cost allocation tags for the user pool. For more information, see <a
     * href="http://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-cost-allocation-tagging.html"
     * >Adding Cost Allocation Tags to Your User Pool</a>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The cost allocation tags for the user pool. For more information, see <a
     *         href="http://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-cost-allocation-tagging.html"
     *         >Adding Cost Allocation Tags to Your User Pool</a>
     */
    public Map<String, String> userPoolTags() {
        return userPoolTags;
    }

    /**
     * <p>
     * The reason why the SMS configuration cannot send the messages to your users.
     * </p>
     * 
     * @return The reason why the SMS configuration cannot send the messages to your users.
     */
    public String smsConfigurationFailure() {
        return smsConfigurationFailure;
    }

    /**
     * <p>
     * The reason why the email configuration cannot send the messages to your users.
     * </p>
     * 
     * @return The reason why the email configuration cannot send the messages to your users.
     */
    public String emailConfigurationFailure() {
        return emailConfigurationFailure;
    }

    /**
     * <p>
     * The configuration for <code>AdminCreateUser</code> requests.
     * </p>
     * 
     * @return The configuration for <code>AdminCreateUser</code> requests.
     */
    public AdminCreateUserConfigType adminCreateUserConfig() {
        return adminCreateUserConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(policies());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaConfig());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(schemaAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(autoVerifiedAttributesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(aliasAttributesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(usernameAttributesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(smsVerificationMessage());
        hashCode = 31 * hashCode + Objects.hashCode(emailVerificationMessage());
        hashCode = 31 * hashCode + Objects.hashCode(emailVerificationSubject());
        hashCode = 31 * hashCode + Objects.hashCode(verificationMessageTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(smsAuthenticationMessage());
        hashCode = 31 * hashCode + Objects.hashCode(mfaConfigurationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(deviceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(estimatedNumberOfUsers());
        hashCode = 31 * hashCode + Objects.hashCode(emailConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(smsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(userPoolTags());
        hashCode = 31 * hashCode + Objects.hashCode(smsConfigurationFailure());
        hashCode = 31 * hashCode + Objects.hashCode(emailConfigurationFailure());
        hashCode = 31 * hashCode + Objects.hashCode(adminCreateUserConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UserPoolType)) {
            return false;
        }
        UserPoolType other = (UserPoolType) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(name(), other.name())
                && Objects.equals(policies(), other.policies()) && Objects.equals(lambdaConfig(), other.lambdaConfig())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(lastModifiedDate(), other.lastModifiedDate())
                && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(schemaAttributes(), other.schemaAttributes())
                && Objects.equals(autoVerifiedAttributesAsStrings(), other.autoVerifiedAttributesAsStrings())
                && Objects.equals(aliasAttributesAsStrings(), other.aliasAttributesAsStrings())
                && Objects.equals(usernameAttributesAsStrings(), other.usernameAttributesAsStrings())
                && Objects.equals(smsVerificationMessage(), other.smsVerificationMessage())
                && Objects.equals(emailVerificationMessage(), other.emailVerificationMessage())
                && Objects.equals(emailVerificationSubject(), other.emailVerificationSubject())
                && Objects.equals(verificationMessageTemplate(), other.verificationMessageTemplate())
                && Objects.equals(smsAuthenticationMessage(), other.smsAuthenticationMessage())
                && Objects.equals(mfaConfigurationAsString(), other.mfaConfigurationAsString())
                && Objects.equals(deviceConfiguration(), other.deviceConfiguration())
                && Objects.equals(estimatedNumberOfUsers(), other.estimatedNumberOfUsers())
                && Objects.equals(emailConfiguration(), other.emailConfiguration())
                && Objects.equals(smsConfiguration(), other.smsConfiguration())
                && Objects.equals(userPoolTags(), other.userPoolTags())
                && Objects.equals(smsConfigurationFailure(), other.smsConfigurationFailure())
                && Objects.equals(emailConfigurationFailure(), other.emailConfigurationFailure())
                && Objects.equals(adminCreateUserConfig(), other.adminCreateUserConfig());
    }

    @Override
    public String toString() {
        return ToString.builder("UserPoolType").add("Id", id()).add("Name", name()).add("Policies", policies())
                .add("LambdaConfig", lambdaConfig()).add("Status", statusAsString()).add("LastModifiedDate", lastModifiedDate())
                .add("CreationDate", creationDate()).add("SchemaAttributes", schemaAttributes())
                .add("AutoVerifiedAttributes", autoVerifiedAttributesAsStrings())
                .add("AliasAttributes", aliasAttributesAsStrings()).add("UsernameAttributes", usernameAttributesAsStrings())
                .add("SmsVerificationMessage", smsVerificationMessage())
                .add("EmailVerificationMessage", emailVerificationMessage())
                .add("EmailVerificationSubject", emailVerificationSubject())
                .add("VerificationMessageTemplate", verificationMessageTemplate())
                .add("SmsAuthenticationMessage", smsAuthenticationMessage()).add("MfaConfiguration", mfaConfigurationAsString())
                .add("DeviceConfiguration", deviceConfiguration()).add("EstimatedNumberOfUsers", estimatedNumberOfUsers())
                .add("EmailConfiguration", emailConfiguration()).add("SmsConfiguration", smsConfiguration())
                .add("UserPoolTags", userPoolTags()).add("SmsConfigurationFailure", smsConfigurationFailure())
                .add("EmailConfigurationFailure", emailConfigurationFailure())
                .add("AdminCreateUserConfig", adminCreateUserConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Policies":
            return Optional.ofNullable(clazz.cast(policies()));
        case "LambdaConfig":
            return Optional.ofNullable(clazz.cast(lambdaConfig()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "LastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "SchemaAttributes":
            return Optional.ofNullable(clazz.cast(schemaAttributes()));
        case "AutoVerifiedAttributes":
            return Optional.ofNullable(clazz.cast(autoVerifiedAttributesAsStrings()));
        case "AliasAttributes":
            return Optional.ofNullable(clazz.cast(aliasAttributesAsStrings()));
        case "UsernameAttributes":
            return Optional.ofNullable(clazz.cast(usernameAttributesAsStrings()));
        case "SmsVerificationMessage":
            return Optional.ofNullable(clazz.cast(smsVerificationMessage()));
        case "EmailVerificationMessage":
            return Optional.ofNullable(clazz.cast(emailVerificationMessage()));
        case "EmailVerificationSubject":
            return Optional.ofNullable(clazz.cast(emailVerificationSubject()));
        case "VerificationMessageTemplate":
            return Optional.ofNullable(clazz.cast(verificationMessageTemplate()));
        case "SmsAuthenticationMessage":
            return Optional.ofNullable(clazz.cast(smsAuthenticationMessage()));
        case "MfaConfiguration":
            return Optional.ofNullable(clazz.cast(mfaConfigurationAsString()));
        case "DeviceConfiguration":
            return Optional.ofNullable(clazz.cast(deviceConfiguration()));
        case "EstimatedNumberOfUsers":
            return Optional.ofNullable(clazz.cast(estimatedNumberOfUsers()));
        case "EmailConfiguration":
            return Optional.ofNullable(clazz.cast(emailConfiguration()));
        case "SmsConfiguration":
            return Optional.ofNullable(clazz.cast(smsConfiguration()));
        case "UserPoolTags":
            return Optional.ofNullable(clazz.cast(userPoolTags()));
        case "SmsConfigurationFailure":
            return Optional.ofNullable(clazz.cast(smsConfigurationFailure()));
        case "EmailConfigurationFailure":
            return Optional.ofNullable(clazz.cast(emailConfigurationFailure()));
        case "AdminCreateUserConfig":
            return Optional.ofNullable(clazz.cast(adminCreateUserConfig()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        UserPoolTypeMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, UserPoolType> {
        /**
         * <p>
         * The ID of the user pool.
         * </p>
         * 
         * @param id
         *        The ID of the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the user pool.
         * </p>
         * 
         * @param name
         *        The name of the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A container for the policies associated with a user pool.
         * </p>
         * 
         * @param policies
         *        A container for the policies associated with a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(UserPoolPolicyType policies);

        /**
         * <p>
         * A container for the policies associated with a user pool.
         * </p>
         * This is a convenience that creates an instance of the {@link UserPoolPolicyType.Builder} avoiding the need to
         * create one manually via {@link UserPoolPolicyType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UserPoolPolicyType.Builder#build()} is called immediately and its
         * result is passed to {@link #policies(UserPoolPolicyType)}.
         * 
         * @param policies
         *        a consumer that will call methods on {@link UserPoolPolicyType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #policies(UserPoolPolicyType)
         */
        default Builder policies(Consumer<UserPoolPolicyType.Builder> policies) {
            return policies(UserPoolPolicyType.builder().applyMutation(policies).build());
        }

        /**
         * <p>
         * A container for the AWS Lambda triggers associated with a user pool.
         * </p>
         * 
         * @param lambdaConfig
         *        A container for the AWS Lambda triggers associated with a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaConfig(LambdaConfigType lambdaConfig);

        /**
         * <p>
         * A container for the AWS Lambda triggers associated with a user pool.
         * </p>
         * This is a convenience that creates an instance of the {@link LambdaConfigType.Builder} avoiding the need to
         * create one manually via {@link LambdaConfigType#builder()}.
         *
         * When the {@link Consumer} completes, {@link LambdaConfigType.Builder#build()} is called immediately and its
         * result is passed to {@link #lambdaConfig(LambdaConfigType)}.
         * 
         * @param lambdaConfig
         *        a consumer that will call methods on {@link LambdaConfigType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lambdaConfig(LambdaConfigType)
         */
        default Builder lambdaConfig(Consumer<LambdaConfigType.Builder> lambdaConfig) {
            return lambdaConfig(LambdaConfigType.builder().applyMutation(lambdaConfig).build());
        }

        /**
         * <p>
         * The status of a user pool.
         * </p>
         * 
         * @param status
         *        The status of a user pool.
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a user pool.
         * </p>
         * 
         * @param status
         *        The status of a user pool.
         * @see StatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusType
         */
        Builder status(StatusType status);

        /**
         * <p>
         * The date the user pool was last modified.
         * </p>
         * 
         * @param lastModifiedDate
         *        The date the user pool was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);

        /**
         * <p>
         * The date the user pool was created.
         * </p>
         * 
         * @param creationDate
         *        The date the user pool was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * A container with the schema attributes of a user pool.
         * </p>
         * 
         * @param schemaAttributes
         *        A container with the schema attributes of a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaAttributes(Collection<SchemaAttributeType> schemaAttributes);

        /**
         * <p>
         * A container with the schema attributes of a user pool.
         * </p>
         * 
         * @param schemaAttributes
         *        A container with the schema attributes of a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaAttributes(SchemaAttributeType... schemaAttributes);

        /**
         * <p>
         * A container with the schema attributes of a user pool.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SchemaAttributeType>.Builder} avoiding the
         * need to create one manually via {@link List<SchemaAttributeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SchemaAttributeType>.Builder#build()} is called immediately
         * and its result is passed to {@link #schemaAttributes(List<SchemaAttributeType>)}.
         * 
         * @param schemaAttributes
         *        a consumer that will call methods on {@link List<SchemaAttributeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schemaAttributes(List<SchemaAttributeType>)
         */
        Builder schemaAttributes(Consumer<SchemaAttributeType.Builder>... schemaAttributes);

        /**
         * <p>
         * Specifies the attributes that are auto-verified in a user pool.
         * </p>
         * 
         * @param autoVerifiedAttributes
         *        Specifies the attributes that are auto-verified in a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoVerifiedAttributesWithStrings(Collection<String> autoVerifiedAttributes);

        /**
         * <p>
         * Specifies the attributes that are auto-verified in a user pool.
         * </p>
         * 
         * @param autoVerifiedAttributes
         *        Specifies the attributes that are auto-verified in a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoVerifiedAttributesWithStrings(String... autoVerifiedAttributes);

        /**
         * <p>
         * Specifies the attributes that are auto-verified in a user pool.
         * </p>
         * 
         * @param autoVerifiedAttributes
         *        Specifies the attributes that are auto-verified in a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoVerifiedAttributes(Collection<VerifiedAttributeType> autoVerifiedAttributes);

        /**
         * <p>
         * Specifies the attributes that are auto-verified in a user pool.
         * </p>
         * 
         * @param autoVerifiedAttributes
         *        Specifies the attributes that are auto-verified in a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoVerifiedAttributes(VerifiedAttributeType... autoVerifiedAttributes);

        /**
         * <p>
         * Specifies the attributes that are aliased in a user pool.
         * </p>
         * 
         * @param aliasAttributes
         *        Specifies the attributes that are aliased in a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasAttributesWithStrings(Collection<String> aliasAttributes);

        /**
         * <p>
         * Specifies the attributes that are aliased in a user pool.
         * </p>
         * 
         * @param aliasAttributes
         *        Specifies the attributes that are aliased in a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasAttributesWithStrings(String... aliasAttributes);

        /**
         * <p>
         * Specifies the attributes that are aliased in a user pool.
         * </p>
         * 
         * @param aliasAttributes
         *        Specifies the attributes that are aliased in a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasAttributes(Collection<AliasAttributeType> aliasAttributes);

        /**
         * <p>
         * Specifies the attributes that are aliased in a user pool.
         * </p>
         * 
         * @param aliasAttributes
         *        Specifies the attributes that are aliased in a user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasAttributes(AliasAttributeType... aliasAttributes);

        /**
         * <p>
         * Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
         * </p>
         * 
         * @param usernameAttributes
         *        Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameAttributesWithStrings(Collection<String> usernameAttributes);

        /**
         * <p>
         * Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
         * </p>
         * 
         * @param usernameAttributes
         *        Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameAttributesWithStrings(String... usernameAttributes);

        /**
         * <p>
         * Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
         * </p>
         * 
         * @param usernameAttributes
         *        Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameAttributes(Collection<UsernameAttributeType> usernameAttributes);

        /**
         * <p>
         * Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
         * </p>
         * 
         * @param usernameAttributes
         *        Specifies whether email addresses or phone numbers can be specified as usernames when a user signs up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameAttributes(UsernameAttributeType... usernameAttributes);

        /**
         * <p>
         * The contents of the SMS verification message.
         * </p>
         * 
         * @param smsVerificationMessage
         *        The contents of the SMS verification message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smsVerificationMessage(String smsVerificationMessage);

        /**
         * <p>
         * The contents of the email verification message.
         * </p>
         * 
         * @param emailVerificationMessage
         *        The contents of the email verification message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailVerificationMessage(String emailVerificationMessage);

        /**
         * <p>
         * The subject of the email verification message.
         * </p>
         * 
         * @param emailVerificationSubject
         *        The subject of the email verification message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailVerificationSubject(String emailVerificationSubject);

        /**
         * <p>
         * The template for verification messages.
         * </p>
         * 
         * @param verificationMessageTemplate
         *        The template for verification messages.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder verificationMessageTemplate(VerificationMessageTemplateType verificationMessageTemplate);

        /**
         * <p>
         * The template for verification messages.
         * </p>
         * This is a convenience that creates an instance of the {@link VerificationMessageTemplateType.Builder}
         * avoiding the need to create one manually via {@link VerificationMessageTemplateType#builder()}.
         *
         * When the {@link Consumer} completes, {@link VerificationMessageTemplateType.Builder#build()} is called
         * immediately and its result is passed to {@link #verificationMessageTemplate(VerificationMessageTemplateType)}
         * .
         * 
         * @param verificationMessageTemplate
         *        a consumer that will call methods on {@link VerificationMessageTemplateType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #verificationMessageTemplate(VerificationMessageTemplateType)
         */
        default Builder verificationMessageTemplate(Consumer<VerificationMessageTemplateType.Builder> verificationMessageTemplate) {
            return verificationMessageTemplate(VerificationMessageTemplateType.builder()
                    .applyMutation(verificationMessageTemplate).build());
        }

        /**
         * <p>
         * The contents of the SMS authentication message.
         * </p>
         * 
         * @param smsAuthenticationMessage
         *        The contents of the SMS authentication message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smsAuthenticationMessage(String smsAuthenticationMessage);

        /**
         * <p>
         * Can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>OFF</code> - MFA tokens are not required and cannot be specified during user registration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ON</code> - MFA tokens are required for all user registrations. You can only specify required when you
         * are initially creating a user pool.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OPTIONAL</code> - Users have the option when registering to create an MFA token.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mfaConfiguration
         *        Can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>OFF</code> - MFA tokens are not required and cannot be specified during user registration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ON</code> - MFA tokens are required for all user registrations. You can only specify required
         *        when you are initially creating a user pool.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OPTIONAL</code> - Users have the option when registering to create an MFA token.
         *        </p>
         *        </li>
         * @see UserPoolMfaType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserPoolMfaType
         */
        Builder mfaConfiguration(String mfaConfiguration);

        /**
         * <p>
         * Can be one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>OFF</code> - MFA tokens are not required and cannot be specified during user registration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ON</code> - MFA tokens are required for all user registrations. You can only specify required when you
         * are initially creating a user pool.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OPTIONAL</code> - Users have the option when registering to create an MFA token.
         * </p>
         * </li>
         * </ul>
         * 
         * @param mfaConfiguration
         *        Can be one of the following values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>OFF</code> - MFA tokens are not required and cannot be specified during user registration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ON</code> - MFA tokens are required for all user registrations. You can only specify required
         *        when you are initially creating a user pool.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OPTIONAL</code> - Users have the option when registering to create an MFA token.
         *        </p>
         *        </li>
         * @see UserPoolMfaType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserPoolMfaType
         */
        Builder mfaConfiguration(UserPoolMfaType mfaConfiguration);

        /**
         * <p>
         * The device configuration.
         * </p>
         * 
         * @param deviceConfiguration
         *        The device configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceConfiguration(DeviceConfigurationType deviceConfiguration);

        /**
         * <p>
         * The device configuration.
         * </p>
         * This is a convenience that creates an instance of the {@link DeviceConfigurationType.Builder} avoiding the
         * need to create one manually via {@link DeviceConfigurationType#builder()}.
         *
         * When the {@link Consumer} completes, {@link DeviceConfigurationType.Builder#build()} is called immediately
         * and its result is passed to {@link #deviceConfiguration(DeviceConfigurationType)}.
         * 
         * @param deviceConfiguration
         *        a consumer that will call methods on {@link DeviceConfigurationType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deviceConfiguration(DeviceConfigurationType)
         */
        default Builder deviceConfiguration(Consumer<DeviceConfigurationType.Builder> deviceConfiguration) {
            return deviceConfiguration(DeviceConfigurationType.builder().applyMutation(deviceConfiguration).build());
        }

        /**
         * <p>
         * A number estimating the size of the user pool.
         * </p>
         * 
         * @param estimatedNumberOfUsers
         *        A number estimating the size of the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder estimatedNumberOfUsers(Integer estimatedNumberOfUsers);

        /**
         * <p>
         * The email configuration.
         * </p>
         * 
         * @param emailConfiguration
         *        The email configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailConfiguration(EmailConfigurationType emailConfiguration);

        /**
         * <p>
         * The email configuration.
         * </p>
         * This is a convenience that creates an instance of the {@link EmailConfigurationType.Builder} avoiding the
         * need to create one manually via {@link EmailConfigurationType#builder()}.
         *
         * When the {@link Consumer} completes, {@link EmailConfigurationType.Builder#build()} is called immediately and
         * its result is passed to {@link #emailConfiguration(EmailConfigurationType)}.
         * 
         * @param emailConfiguration
         *        a consumer that will call methods on {@link EmailConfigurationType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #emailConfiguration(EmailConfigurationType)
         */
        default Builder emailConfiguration(Consumer<EmailConfigurationType.Builder> emailConfiguration) {
            return emailConfiguration(EmailConfigurationType.builder().applyMutation(emailConfiguration).build());
        }

        /**
         * <p>
         * The SMS configuration.
         * </p>
         * 
         * @param smsConfiguration
         *        The SMS configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smsConfiguration(SmsConfigurationType smsConfiguration);

        /**
         * <p>
         * The SMS configuration.
         * </p>
         * This is a convenience that creates an instance of the {@link SmsConfigurationType.Builder} avoiding the need
         * to create one manually via {@link SmsConfigurationType#builder()}.
         *
         * When the {@link Consumer} completes, {@link SmsConfigurationType.Builder#build()} is called immediately and
         * its result is passed to {@link #smsConfiguration(SmsConfigurationType)}.
         * 
         * @param smsConfiguration
         *        a consumer that will call methods on {@link SmsConfigurationType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #smsConfiguration(SmsConfigurationType)
         */
        default Builder smsConfiguration(Consumer<SmsConfigurationType.Builder> smsConfiguration) {
            return smsConfiguration(SmsConfigurationType.builder().applyMutation(smsConfiguration).build());
        }

        /**
         * <p>
         * The cost allocation tags for the user pool. For more information, see <a href=
         * "http://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-cost-allocation-tagging.html"
         * >Adding Cost Allocation Tags to Your User Pool</a>
         * </p>
         * 
         * @param userPoolTags
         *        The cost allocation tags for the user pool. For more information, see <a href=
         *        "http://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-cost-allocation-tagging.html"
         *        >Adding Cost Allocation Tags to Your User Pool</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolTags(Map<String, String> userPoolTags);

        /**
         * <p>
         * The reason why the SMS configuration cannot send the messages to your users.
         * </p>
         * 
         * @param smsConfigurationFailure
         *        The reason why the SMS configuration cannot send the messages to your users.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smsConfigurationFailure(String smsConfigurationFailure);

        /**
         * <p>
         * The reason why the email configuration cannot send the messages to your users.
         * </p>
         * 
         * @param emailConfigurationFailure
         *        The reason why the email configuration cannot send the messages to your users.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailConfigurationFailure(String emailConfigurationFailure);

        /**
         * <p>
         * The configuration for <code>AdminCreateUser</code> requests.
         * </p>
         * 
         * @param adminCreateUserConfig
         *        The configuration for <code>AdminCreateUser</code> requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adminCreateUserConfig(AdminCreateUserConfigType adminCreateUserConfig);

        /**
         * <p>
         * The configuration for <code>AdminCreateUser</code> requests.
         * </p>
         * This is a convenience that creates an instance of the {@link AdminCreateUserConfigType.Builder} avoiding the
         * need to create one manually via {@link AdminCreateUserConfigType#builder()}.
         *
         * When the {@link Consumer} completes, {@link AdminCreateUserConfigType.Builder#build()} is called immediately
         * and its result is passed to {@link #adminCreateUserConfig(AdminCreateUserConfigType)}.
         * 
         * @param adminCreateUserConfig
         *        a consumer that will call methods on {@link AdminCreateUserConfigType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #adminCreateUserConfig(AdminCreateUserConfigType)
         */
        default Builder adminCreateUserConfig(Consumer<AdminCreateUserConfigType.Builder> adminCreateUserConfig) {
            return adminCreateUserConfig(AdminCreateUserConfigType.builder().applyMutation(adminCreateUserConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String name;

        private UserPoolPolicyType policies;

        private LambdaConfigType lambdaConfig;

        private String status;

        private Instant lastModifiedDate;

        private Instant creationDate;

        private List<SchemaAttributeType> schemaAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<String> autoVerifiedAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<String> aliasAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<String> usernameAttributes = DefaultSdkAutoConstructList.getInstance();

        private String smsVerificationMessage;

        private String emailVerificationMessage;

        private String emailVerificationSubject;

        private VerificationMessageTemplateType verificationMessageTemplate;

        private String smsAuthenticationMessage;

        private String mfaConfiguration;

        private DeviceConfigurationType deviceConfiguration;

        private Integer estimatedNumberOfUsers;

        private EmailConfigurationType emailConfiguration;

        private SmsConfigurationType smsConfiguration;

        private Map<String, String> userPoolTags = DefaultSdkAutoConstructMap.getInstance();

        private String smsConfigurationFailure;

        private String emailConfigurationFailure;

        private AdminCreateUserConfigType adminCreateUserConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(UserPoolType model) {
            id(model.id);
            name(model.name);
            policies(model.policies);
            lambdaConfig(model.lambdaConfig);
            status(model.status);
            lastModifiedDate(model.lastModifiedDate);
            creationDate(model.creationDate);
            schemaAttributes(model.schemaAttributes);
            autoVerifiedAttributesWithStrings(model.autoVerifiedAttributes);
            aliasAttributesWithStrings(model.aliasAttributes);
            usernameAttributesWithStrings(model.usernameAttributes);
            smsVerificationMessage(model.smsVerificationMessage);
            emailVerificationMessage(model.emailVerificationMessage);
            emailVerificationSubject(model.emailVerificationSubject);
            verificationMessageTemplate(model.verificationMessageTemplate);
            smsAuthenticationMessage(model.smsAuthenticationMessage);
            mfaConfiguration(model.mfaConfiguration);
            deviceConfiguration(model.deviceConfiguration);
            estimatedNumberOfUsers(model.estimatedNumberOfUsers);
            emailConfiguration(model.emailConfiguration);
            smsConfiguration(model.smsConfiguration);
            userPoolTags(model.userPoolTags);
            smsConfigurationFailure(model.smsConfigurationFailure);
            emailConfigurationFailure(model.emailConfigurationFailure);
            adminCreateUserConfig(model.adminCreateUserConfig);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final UserPoolPolicyType.Builder getPolicies() {
            return policies != null ? policies.toBuilder() : null;
        }

        @Override
        public final Builder policies(UserPoolPolicyType policies) {
            this.policies = policies;
            return this;
        }

        public final void setPolicies(UserPoolPolicyType.BuilderImpl policies) {
            this.policies = policies != null ? policies.build() : null;
        }

        public final LambdaConfigType.Builder getLambdaConfig() {
            return lambdaConfig != null ? lambdaConfig.toBuilder() : null;
        }

        @Override
        public final Builder lambdaConfig(LambdaConfigType lambdaConfig) {
            this.lambdaConfig = lambdaConfig;
            return this;
        }

        public final void setLambdaConfig(LambdaConfigType.BuilderImpl lambdaConfig) {
            this.lambdaConfig = lambdaConfig != null ? lambdaConfig.build() : null;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(StatusType status) {
            this.status(status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        public final Collection<SchemaAttributeType.Builder> getSchemaAttributes() {
            return schemaAttributes != null ? schemaAttributes.stream().map(SchemaAttributeType::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder schemaAttributes(Collection<SchemaAttributeType> schemaAttributes) {
            this.schemaAttributes = SchemaAttributesListTypeCopier.copy(schemaAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schemaAttributes(SchemaAttributeType... schemaAttributes) {
            schemaAttributes(Arrays.asList(schemaAttributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schemaAttributes(Consumer<SchemaAttributeType.Builder>... schemaAttributes) {
            schemaAttributes(Stream.of(schemaAttributes).map(c -> SchemaAttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSchemaAttributes(Collection<SchemaAttributeType.BuilderImpl> schemaAttributes) {
            this.schemaAttributes = SchemaAttributesListTypeCopier.copyFromBuilder(schemaAttributes);
        }

        public final Collection<String> getAutoVerifiedAttributes() {
            return autoVerifiedAttributes;
        }

        @Override
        public final Builder autoVerifiedAttributesWithStrings(Collection<String> autoVerifiedAttributes) {
            this.autoVerifiedAttributes = VerifiedAttributesListTypeCopier.copy(autoVerifiedAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autoVerifiedAttributesWithStrings(String... autoVerifiedAttributes) {
            autoVerifiedAttributesWithStrings(Arrays.asList(autoVerifiedAttributes));
            return this;
        }

        @Override
        public final Builder autoVerifiedAttributes(Collection<VerifiedAttributeType> autoVerifiedAttributes) {
            this.autoVerifiedAttributes = VerifiedAttributesListTypeCopier.copyEnumToString(autoVerifiedAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autoVerifiedAttributes(VerifiedAttributeType... autoVerifiedAttributes) {
            autoVerifiedAttributes(Arrays.asList(autoVerifiedAttributes));
            return this;
        }

        public final void setAutoVerifiedAttributes(Collection<String> autoVerifiedAttributes) {
            this.autoVerifiedAttributes = VerifiedAttributesListTypeCopier.copy(autoVerifiedAttributes);
        }

        public final Collection<String> getAliasAttributes() {
            return aliasAttributes;
        }

        @Override
        public final Builder aliasAttributesWithStrings(Collection<String> aliasAttributes) {
            this.aliasAttributes = AliasAttributesListTypeCopier.copy(aliasAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aliasAttributesWithStrings(String... aliasAttributes) {
            aliasAttributesWithStrings(Arrays.asList(aliasAttributes));
            return this;
        }

        @Override
        public final Builder aliasAttributes(Collection<AliasAttributeType> aliasAttributes) {
            this.aliasAttributes = AliasAttributesListTypeCopier.copyEnumToString(aliasAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aliasAttributes(AliasAttributeType... aliasAttributes) {
            aliasAttributes(Arrays.asList(aliasAttributes));
            return this;
        }

        public final void setAliasAttributes(Collection<String> aliasAttributes) {
            this.aliasAttributes = AliasAttributesListTypeCopier.copy(aliasAttributes);
        }

        public final Collection<String> getUsernameAttributes() {
            return usernameAttributes;
        }

        @Override
        public final Builder usernameAttributesWithStrings(Collection<String> usernameAttributes) {
            this.usernameAttributes = UsernameAttributesListTypeCopier.copy(usernameAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder usernameAttributesWithStrings(String... usernameAttributes) {
            usernameAttributesWithStrings(Arrays.asList(usernameAttributes));
            return this;
        }

        @Override
        public final Builder usernameAttributes(Collection<UsernameAttributeType> usernameAttributes) {
            this.usernameAttributes = UsernameAttributesListTypeCopier.copyEnumToString(usernameAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder usernameAttributes(UsernameAttributeType... usernameAttributes) {
            usernameAttributes(Arrays.asList(usernameAttributes));
            return this;
        }

        public final void setUsernameAttributes(Collection<String> usernameAttributes) {
            this.usernameAttributes = UsernameAttributesListTypeCopier.copy(usernameAttributes);
        }

        public final String getSmsVerificationMessage() {
            return smsVerificationMessage;
        }

        @Override
        public final Builder smsVerificationMessage(String smsVerificationMessage) {
            this.smsVerificationMessage = smsVerificationMessage;
            return this;
        }

        public final void setSmsVerificationMessage(String smsVerificationMessage) {
            this.smsVerificationMessage = smsVerificationMessage;
        }

        public final String getEmailVerificationMessage() {
            return emailVerificationMessage;
        }

        @Override
        public final Builder emailVerificationMessage(String emailVerificationMessage) {
            this.emailVerificationMessage = emailVerificationMessage;
            return this;
        }

        public final void setEmailVerificationMessage(String emailVerificationMessage) {
            this.emailVerificationMessage = emailVerificationMessage;
        }

        public final String getEmailVerificationSubject() {
            return emailVerificationSubject;
        }

        @Override
        public final Builder emailVerificationSubject(String emailVerificationSubject) {
            this.emailVerificationSubject = emailVerificationSubject;
            return this;
        }

        public final void setEmailVerificationSubject(String emailVerificationSubject) {
            this.emailVerificationSubject = emailVerificationSubject;
        }

        public final VerificationMessageTemplateType.Builder getVerificationMessageTemplate() {
            return verificationMessageTemplate != null ? verificationMessageTemplate.toBuilder() : null;
        }

        @Override
        public final Builder verificationMessageTemplate(VerificationMessageTemplateType verificationMessageTemplate) {
            this.verificationMessageTemplate = verificationMessageTemplate;
            return this;
        }

        public final void setVerificationMessageTemplate(VerificationMessageTemplateType.BuilderImpl verificationMessageTemplate) {
            this.verificationMessageTemplate = verificationMessageTemplate != null ? verificationMessageTemplate.build() : null;
        }

        public final String getSmsAuthenticationMessage() {
            return smsAuthenticationMessage;
        }

        @Override
        public final Builder smsAuthenticationMessage(String smsAuthenticationMessage) {
            this.smsAuthenticationMessage = smsAuthenticationMessage;
            return this;
        }

        public final void setSmsAuthenticationMessage(String smsAuthenticationMessage) {
            this.smsAuthenticationMessage = smsAuthenticationMessage;
        }

        public final String getMfaConfiguration() {
            return mfaConfiguration;
        }

        @Override
        public final Builder mfaConfiguration(String mfaConfiguration) {
            this.mfaConfiguration = mfaConfiguration;
            return this;
        }

        @Override
        public final Builder mfaConfiguration(UserPoolMfaType mfaConfiguration) {
            this.mfaConfiguration(mfaConfiguration.toString());
            return this;
        }

        public final void setMfaConfiguration(String mfaConfiguration) {
            this.mfaConfiguration = mfaConfiguration;
        }

        public final DeviceConfigurationType.Builder getDeviceConfiguration() {
            return deviceConfiguration != null ? deviceConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder deviceConfiguration(DeviceConfigurationType deviceConfiguration) {
            this.deviceConfiguration = deviceConfiguration;
            return this;
        }

        public final void setDeviceConfiguration(DeviceConfigurationType.BuilderImpl deviceConfiguration) {
            this.deviceConfiguration = deviceConfiguration != null ? deviceConfiguration.build() : null;
        }

        public final Integer getEstimatedNumberOfUsers() {
            return estimatedNumberOfUsers;
        }

        @Override
        public final Builder estimatedNumberOfUsers(Integer estimatedNumberOfUsers) {
            this.estimatedNumberOfUsers = estimatedNumberOfUsers;
            return this;
        }

        public final void setEstimatedNumberOfUsers(Integer estimatedNumberOfUsers) {
            this.estimatedNumberOfUsers = estimatedNumberOfUsers;
        }

        public final EmailConfigurationType.Builder getEmailConfiguration() {
            return emailConfiguration != null ? emailConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder emailConfiguration(EmailConfigurationType emailConfiguration) {
            this.emailConfiguration = emailConfiguration;
            return this;
        }

        public final void setEmailConfiguration(EmailConfigurationType.BuilderImpl emailConfiguration) {
            this.emailConfiguration = emailConfiguration != null ? emailConfiguration.build() : null;
        }

        public final SmsConfigurationType.Builder getSmsConfiguration() {
            return smsConfiguration != null ? smsConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder smsConfiguration(SmsConfigurationType smsConfiguration) {
            this.smsConfiguration = smsConfiguration;
            return this;
        }

        public final void setSmsConfiguration(SmsConfigurationType.BuilderImpl smsConfiguration) {
            this.smsConfiguration = smsConfiguration != null ? smsConfiguration.build() : null;
        }

        public final Map<String, String> getUserPoolTags() {
            return userPoolTags;
        }

        @Override
        public final Builder userPoolTags(Map<String, String> userPoolTags) {
            this.userPoolTags = UserPoolTagsTypeCopier.copy(userPoolTags);
            return this;
        }

        public final void setUserPoolTags(Map<String, String> userPoolTags) {
            this.userPoolTags = UserPoolTagsTypeCopier.copy(userPoolTags);
        }

        public final String getSmsConfigurationFailure() {
            return smsConfigurationFailure;
        }

        @Override
        public final Builder smsConfigurationFailure(String smsConfigurationFailure) {
            this.smsConfigurationFailure = smsConfigurationFailure;
            return this;
        }

        public final void setSmsConfigurationFailure(String smsConfigurationFailure) {
            this.smsConfigurationFailure = smsConfigurationFailure;
        }

        public final String getEmailConfigurationFailure() {
            return emailConfigurationFailure;
        }

        @Override
        public final Builder emailConfigurationFailure(String emailConfigurationFailure) {
            this.emailConfigurationFailure = emailConfigurationFailure;
            return this;
        }

        public final void setEmailConfigurationFailure(String emailConfigurationFailure) {
            this.emailConfigurationFailure = emailConfigurationFailure;
        }

        public final AdminCreateUserConfigType.Builder getAdminCreateUserConfig() {
            return adminCreateUserConfig != null ? adminCreateUserConfig.toBuilder() : null;
        }

        @Override
        public final Builder adminCreateUserConfig(AdminCreateUserConfigType adminCreateUserConfig) {
            this.adminCreateUserConfig = adminCreateUserConfig;
            return this;
        }

        public final void setAdminCreateUserConfig(AdminCreateUserConfigType.BuilderImpl adminCreateUserConfig) {
            this.adminCreateUserConfig = adminCreateUserConfig != null ? adminCreateUserConfig.build() : null;
        }

        @Override
        public UserPoolType build() {
            return new UserPoolType(this);
        }
    }
}
