/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Initiates the authorization request, as an administrator.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdminInitiateAuthRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<AdminInitiateAuthRequest.Builder, AdminInitiateAuthRequest> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AdminInitiateAuthRequest::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AdminInitiateAuthRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<String> AUTH_FLOW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AdminInitiateAuthRequest::authFlowAsString)).setter(setter(Builder::authFlow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthFlow").build()).build();

    private static final SdkField<Map<String, String>> AUTH_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(AdminInitiateAuthRequest::authParameters))
            .setter(setter(Builder::authParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(AdminInitiateAuthRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            CLIENT_ID_FIELD, AUTH_FLOW_FIELD, AUTH_PARAMETERS_FIELD, CLIENT_METADATA_FIELD));

    private final String userPoolId;

    private final String clientId;

    private final String authFlow;

    private final Map<String, String> authParameters;

    private final Map<String, String> clientMetadata;

    private AdminInitiateAuthRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.clientId = builder.clientId;
        this.authFlow = builder.authFlow;
        this.authParameters = builder.authParameters;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The ID of the Amazon Cognito user pool.
     * </p>
     * 
     * @return The ID of the Amazon Cognito user pool.
     */
    public String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The app client ID.
     * </p>
     * 
     * @return The app client ID.
     */
    public String clientId() {
        return clientId;
    }

    /**
     * <p>
     * The authentication flow for this call to execute. The API action will depend on this value. For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>REFRESH_TOKEN_AUTH</code> will take in a valid refresh token and return new tokens.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_SRP_AUTH</code> will take in <code>USERNAME</code> and <code>SRP_A</code> and return the SRP variables
     * to be used for next challenge execution.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access token
     * and ID token by supplying a valid refresh token.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOM_AUTH</code>: Custom authentication flow.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADMIN_NO_SRP_AUTH</code>: Non-SRP authentication flow; you can pass in the USERNAME and PASSWORD directly
     * if the flow is enabled for calling the app client.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authFlow} will
     * return {@link AuthFlowType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #authFlowAsString}.
     * </p>
     * 
     * @return The authentication flow for this call to execute. The API action will depend on this value. For
     *         example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>REFRESH_TOKEN_AUTH</code> will take in a valid refresh token and return new tokens.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_SRP_AUTH</code> will take in <code>USERNAME</code> and <code>SRP_A</code> and return the SRP
     *         variables to be used for next challenge execution.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access
     *         token and ID token by supplying a valid refresh token.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOM_AUTH</code>: Custom authentication flow.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADMIN_NO_SRP_AUTH</code>: Non-SRP authentication flow; you can pass in the USERNAME and PASSWORD
     *         directly if the flow is enabled for calling the app client.
     *         </p>
     *         </li>
     * @see AuthFlowType
     */
    public AuthFlowType authFlow() {
        return AuthFlowType.fromValue(authFlow);
    }

    /**
     * <p>
     * The authentication flow for this call to execute. The API action will depend on this value. For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>REFRESH_TOKEN_AUTH</code> will take in a valid refresh token and return new tokens.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_SRP_AUTH</code> will take in <code>USERNAME</code> and <code>SRP_A</code> and return the SRP variables
     * to be used for next challenge execution.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access token
     * and ID token by supplying a valid refresh token.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOM_AUTH</code>: Custom authentication flow.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADMIN_NO_SRP_AUTH</code>: Non-SRP authentication flow; you can pass in the USERNAME and PASSWORD directly
     * if the flow is enabled for calling the app client.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authFlow} will
     * return {@link AuthFlowType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #authFlowAsString}.
     * </p>
     * 
     * @return The authentication flow for this call to execute. The API action will depend on this value. For
     *         example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>REFRESH_TOKEN_AUTH</code> will take in a valid refresh token and return new tokens.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_SRP_AUTH</code> will take in <code>USERNAME</code> and <code>SRP_A</code> and return the SRP
     *         variables to be used for next challenge execution.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access
     *         token and ID token by supplying a valid refresh token.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOM_AUTH</code>: Custom authentication flow.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADMIN_NO_SRP_AUTH</code>: Non-SRP authentication flow; you can pass in the USERNAME and PASSWORD
     *         directly if the flow is enabled for calling the app client.
     *         </p>
     *         </li>
     * @see AuthFlowType
     */
    public String authFlowAsString() {
        return authFlow;
    }

    /**
     * <p>
     * The authentication parameters. These are inputs corresponding to the <code>AuthFlow</code> that you are invoking.
     * The required values depend on the value of <code>AuthFlow</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>USER_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SRP_A</code> (required),
     * <code>SECRET_HASH</code> (required if the app client is configured with a client secret), <code>DEVICE_KEY</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>REFRESH_TOKEN_AUTH/REFRESH_TOKEN</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code>
     * (required if the app client is configured with a client secret), <code>REFRESH_TOKEN</code> (required),
     * <code>DEVICE_KEY</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>ADMIN_NO_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app client is
     * configured with client secret), <code>PASSWORD</code> (required), <code>DEVICE_KEY</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>CUSTOM_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app client is
     * configured with client secret), <code>DEVICE_KEY</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The authentication parameters. These are inputs corresponding to the <code>AuthFlow</code> that you are
     *         invoking. The required values depend on the value of <code>AuthFlow</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <code>USER_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SRP_A</code> (required),
     *         <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
     *         <code>DEVICE_KEY</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>REFRESH_TOKEN_AUTH/REFRESH_TOKEN</code>: <code>USERNAME</code> (required),
     *         <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
     *         <code>REFRESH_TOKEN</code> (required), <code>DEVICE_KEY</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>ADMIN_NO_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app
     *         client is configured with client secret), <code>PASSWORD</code> (required), <code>DEVICE_KEY</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>CUSTOM_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app client
     *         is configured with client secret), <code>DEVICE_KEY</code>
     *         </p>
     *         </li>
     */
    public Map<String, String> authParameters() {
        return authParameters;
    }

    /**
     * <p>
     * This is a random key-value pair map which can contain any key and will be passed to your PreAuthentication Lambda
     * trigger as-is. It can be used to implement additional validations around authentication.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return This is a random key-value pair map which can contain any key and will be passed to your
     *         PreAuthentication Lambda trigger as-is. It can be used to implement additional validations around
     *         authentication.
     */
    public Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(authFlowAsString());
        hashCode = 31 * hashCode + Objects.hashCode(authParameters());
        hashCode = 31 * hashCode + Objects.hashCode(clientMetadata());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdminInitiateAuthRequest)) {
            return false;
        }
        AdminInitiateAuthRequest other = (AdminInitiateAuthRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(clientId(), other.clientId())
                && Objects.equals(authFlowAsString(), other.authFlowAsString())
                && Objects.equals(authParameters(), other.authParameters())
                && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    @Override
    public String toString() {
        return ToString.builder("AdminInitiateAuthRequest").add("UserPoolId", userPoolId()).add("ClientId", clientId())
                .add("AuthFlow", authFlowAsString()).add("AuthParameters", authParameters())
                .add("ClientMetadata", clientMetadata()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "AuthFlow":
            return Optional.ofNullable(clazz.cast(authFlowAsString()));
        case "AuthParameters":
            return Optional.ofNullable(clazz.cast(authParameters()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AdminInitiateAuthRequest, T> g) {
        return obj -> g.apply((AdminInitiateAuthRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, AdminInitiateAuthRequest> {
        /**
         * <p>
         * The ID of the Amazon Cognito user pool.
         * </p>
         * 
         * @param userPoolId
         *        The ID of the Amazon Cognito user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The app client ID.
         * </p>
         * 
         * @param clientId
         *        The app client ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * The authentication flow for this call to execute. The API action will depend on this value. For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>REFRESH_TOKEN_AUTH</code> will take in a valid refresh token and return new tokens.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_SRP_AUTH</code> will take in <code>USERNAME</code> and <code>SRP_A</code> and return the SRP
         * variables to be used for next challenge execution.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access
         * token and ID token by supplying a valid refresh token.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOM_AUTH</code>: Custom authentication flow.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADMIN_NO_SRP_AUTH</code>: Non-SRP authentication flow; you can pass in the USERNAME and PASSWORD
         * directly if the flow is enabled for calling the app client.
         * </p>
         * </li>
         * </ul>
         * 
         * @param authFlow
         *        The authentication flow for this call to execute. The API action will depend on this value. For
         *        example:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>REFRESH_TOKEN_AUTH</code> will take in a valid refresh token and return new tokens.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_SRP_AUTH</code> will take in <code>USERNAME</code> and <code>SRP_A</code> and return the
         *        SRP variables to be used for next challenge execution.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the
         *        access token and ID token by supplying a valid refresh token.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOM_AUTH</code>: Custom authentication flow.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADMIN_NO_SRP_AUTH</code>: Non-SRP authentication flow; you can pass in the USERNAME and PASSWORD
         *        directly if the flow is enabled for calling the app client.
         *        </p>
         *        </li>
         * @see AuthFlowType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthFlowType
         */
        Builder authFlow(String authFlow);

        /**
         * <p>
         * The authentication flow for this call to execute. The API action will depend on this value. For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>REFRESH_TOKEN_AUTH</code> will take in a valid refresh token and return new tokens.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_SRP_AUTH</code> will take in <code>USERNAME</code> and <code>SRP_A</code> and return the SRP
         * variables to be used for next challenge execution.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access
         * token and ID token by supplying a valid refresh token.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOM_AUTH</code>: Custom authentication flow.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADMIN_NO_SRP_AUTH</code>: Non-SRP authentication flow; you can pass in the USERNAME and PASSWORD
         * directly if the flow is enabled for calling the app client.
         * </p>
         * </li>
         * </ul>
         * 
         * @param authFlow
         *        The authentication flow for this call to execute. The API action will depend on this value. For
         *        example:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>REFRESH_TOKEN_AUTH</code> will take in a valid refresh token and return new tokens.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_SRP_AUTH</code> will take in <code>USERNAME</code> and <code>SRP_A</code> and return the
         *        SRP variables to be used for next challenge execution.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the
         *        access token and ID token by supplying a valid refresh token.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOM_AUTH</code>: Custom authentication flow.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADMIN_NO_SRP_AUTH</code>: Non-SRP authentication flow; you can pass in the USERNAME and PASSWORD
         *        directly if the flow is enabled for calling the app client.
         *        </p>
         *        </li>
         * @see AuthFlowType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthFlowType
         */
        Builder authFlow(AuthFlowType authFlow);

        /**
         * <p>
         * The authentication parameters. These are inputs corresponding to the <code>AuthFlow</code> that you are
         * invoking. The required values depend on the value of <code>AuthFlow</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>USER_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SRP_A</code> (required),
         * <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
         * <code>DEVICE_KEY</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>REFRESH_TOKEN_AUTH/REFRESH_TOKEN</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code>
         * (required if the app client is configured with a client secret), <code>REFRESH_TOKEN</code> (required),
         * <code>DEVICE_KEY</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>ADMIN_NO_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app client
         * is configured with client secret), <code>PASSWORD</code> (required), <code>DEVICE_KEY</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>CUSTOM_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app client is
         * configured with client secret), <code>DEVICE_KEY</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param authParameters
         *        The authentication parameters. These are inputs corresponding to the <code>AuthFlow</code> that you
         *        are invoking. The required values depend on the value of <code>AuthFlow</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <code>USER_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SRP_A</code> (required),
         *        <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
         *        <code>DEVICE_KEY</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>REFRESH_TOKEN_AUTH/REFRESH_TOKEN</code>: <code>USERNAME</code> (required),
         *        <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
         *        <code>REFRESH_TOKEN</code> (required), <code>DEVICE_KEY</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>ADMIN_NO_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app
         *        client is configured with client secret), <code>PASSWORD</code> (required), <code>DEVICE_KEY</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>CUSTOM_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app
         *        client is configured with client secret), <code>DEVICE_KEY</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authParameters(Map<String, String> authParameters);

        /**
         * <p>
         * This is a random key-value pair map which can contain any key and will be passed to your PreAuthentication
         * Lambda trigger as-is. It can be used to implement additional validations around authentication.
         * </p>
         * 
         * @param clientMetadata
         *        This is a random key-value pair map which can contain any key and will be passed to your
         *        PreAuthentication Lambda trigger as-is. It can be used to implement additional validations around
         *        authentication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String clientId;

        private String authFlow;

        private Map<String, String> authParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdminInitiateAuthRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            clientId(model.clientId);
            authFlow(model.authFlow);
            authParameters(model.authParameters);
            clientMetadata(model.clientMetadata);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        public final String getClientId() {
            return clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        public final String getAuthFlow() {
            return authFlow;
        }

        @Override
        public final Builder authFlow(String authFlow) {
            this.authFlow = authFlow;
            return this;
        }

        @Override
        public final Builder authFlow(AuthFlowType authFlow) {
            this.authFlow(authFlow.toString());
            return this;
        }

        public final void setAuthFlow(String authFlow) {
            this.authFlow = authFlow;
        }

        public final Map<String, String> getAuthParameters() {
            return authParameters;
        }

        @Override
        public final Builder authParameters(Map<String, String> authParameters) {
            this.authParameters = AuthParametersTypeCopier.copy(authParameters);
            return this;
        }

        public final void setAuthParameters(Map<String, String> authParameters) {
            this.authParameters = AuthParametersTypeCopier.copy(authParameters);
        }

        public final Map<String, String> getClientMetadata() {
            return clientMetadata;
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AdminInitiateAuthRequest build() {
            return new AdminInitiateAuthRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
