/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to register a user.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SignUpRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<SignUpRequest.Builder, SignUpRequest> {
    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignUpRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<String> SECRET_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignUpRequest::secretHash)).setter(setter(Builder::secretHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretHash").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignUpRequest::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SignUpRequest::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<List<AttributeType>> USER_ATTRIBUTES_FIELD = SdkField
            .<List<AttributeType>> builder(MarshallingType.LIST)
            .getter(getter(SignUpRequest::userAttributes))
            .setter(setter(Builder::userAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AttributeType>> VALIDATION_DATA_FIELD = SdkField
            .<List<AttributeType>> builder(MarshallingType.LIST)
            .getter(getter(SignUpRequest::validationData))
            .setter(setter(Builder::validationData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidationData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AnalyticsMetadataType> ANALYTICS_METADATA_FIELD = SdkField
            .<AnalyticsMetadataType> builder(MarshallingType.SDK_POJO).getter(getter(SignUpRequest::analyticsMetadata))
            .setter(setter(Builder::analyticsMetadata)).constructor(AnalyticsMetadataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalyticsMetadata").build()).build();

    private static final SdkField<UserContextDataType> USER_CONTEXT_DATA_FIELD = SdkField
            .<UserContextDataType> builder(MarshallingType.SDK_POJO).getter(getter(SignUpRequest::userContextData))
            .setter(setter(Builder::userContextData)).constructor(UserContextDataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserContextData").build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(SignUpRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_ID_FIELD,
            SECRET_HASH_FIELD, USERNAME_FIELD, PASSWORD_FIELD, USER_ATTRIBUTES_FIELD, VALIDATION_DATA_FIELD,
            ANALYTICS_METADATA_FIELD, USER_CONTEXT_DATA_FIELD, CLIENT_METADATA_FIELD));

    private final String clientId;

    private final String secretHash;

    private final String username;

    private final String password;

    private final List<AttributeType> userAttributes;

    private final List<AttributeType> validationData;

    private final AnalyticsMetadataType analyticsMetadata;

    private final UserContextDataType userContextData;

    private final Map<String, String> clientMetadata;

    private SignUpRequest(BuilderImpl builder) {
        super(builder);
        this.clientId = builder.clientId;
        this.secretHash = builder.secretHash;
        this.username = builder.username;
        this.password = builder.password;
        this.userAttributes = builder.userAttributes;
        this.validationData = builder.validationData;
        this.analyticsMetadata = builder.analyticsMetadata;
        this.userContextData = builder.userContextData;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The ID of the client associated with the user pool.
     * </p>
     * 
     * @return The ID of the client associated with the user pool.
     */
    public String clientId() {
        return clientId;
    }

    /**
     * <p>
     * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     * username plus the client ID in the message.
     * </p>
     * 
     * @return A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     *         username plus the client ID in the message.
     */
    public String secretHash() {
        return secretHash;
    }

    /**
     * <p>
     * The user name of the user you wish to register.
     * </p>
     * 
     * @return The user name of the user you wish to register.
     */
    public String username() {
        return username;
    }

    /**
     * <p>
     * The password of the user you wish to register.
     * </p>
     * 
     * @return The password of the user you wish to register.
     */
    public String password() {
        return password;
    }

    /**
     * Returns true if the UserAttributes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasUserAttributes() {
        return userAttributes != null && !(userAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of name-value pairs representing user attributes.
     * </p>
     * <p>
     * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUserAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of name-value pairs representing user attributes.</p>
     *         <p>
     *         For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     */
    public List<AttributeType> userAttributes() {
        return userAttributes;
    }

    /**
     * Returns true if the ValidationData property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasValidationData() {
        return validationData != null && !(validationData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The validation data in the request to register a user.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValidationData()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The validation data in the request to register a user.
     */
    public List<AttributeType> validationData() {
        return validationData;
    }

    /**
     * <p>
     * The Amazon Pinpoint analytics metadata for collecting metrics for <code>SignUp</code> calls.
     * </p>
     * 
     * @return The Amazon Pinpoint analytics metadata for collecting metrics for <code>SignUp</code> calls.
     */
    public AnalyticsMetadataType analyticsMetadata() {
        return analyticsMetadata;
    }

    /**
     * <p>
     * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk of an
     * unexpected event by Amazon Cognito advanced security.
     * </p>
     * 
     * @return Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the
     *         risk of an unexpected event by Amazon Cognito advanced security.
     */
    public UserContextDataType userContextData() {
        return userContextData;
    }

    /**
     * Returns true if the ClientMetadata property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning AWS Lambda functions to user pool triggers. When you use the SignUp API
     * action, Amazon Cognito invokes any functions that are assigned to the following triggers: <i>pre sign-up</i>,
     * <i>custom message</i>, and <i>post confirmation</i>. When Amazon Cognito invokes any of these functions, it
     * passes a JSON payload, which the function receives as input. This payload contains a <code>clientMetadata</code>
     * attribute, which provides the data that you assigned to the ClientMetadata parameter in your SignUp request. In
     * your function code in AWS Lambda, you can process the <code>clientMetadata</code> value to enhance your workflow
     * for your specific needs.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * >Customizing User Pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * Take the following limitations into consideration when you use the ClientMetadata parameter:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Cognito does not store the ClientMetadata value. This data is available only to AWS Lambda triggers that
     * are assigned to a user pool to support custom workflows. If your user pool configuration does not include
     * triggers, the ClientMetadata parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Cognito does not validate the ClientMetadata value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Cognito does not encrypt the the ClientMetadata value, so don't use it to provide sensitive information.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasClientMetadata()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for any custom workflows that this action
     *         triggers. </p>
     *         <p>
     *         You create custom workflows by assigning AWS Lambda functions to user pool triggers. When you use the
     *         SignUp API action, Amazon Cognito invokes any functions that are assigned to the following triggers:
     *         <i>pre sign-up</i>, <i>custom message</i>, and <i>post confirmation</i>. When Amazon Cognito invokes any
     *         of these functions, it passes a JSON payload, which the function receives as input. This payload contains
     *         a <code>clientMetadata</code> attribute, which provides the data that you assigned to the ClientMetadata
     *         parameter in your SignUp request. In your function code in AWS Lambda, you can process the
     *         <code>clientMetadata</code> value to enhance your workflow for your specific needs.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         >Customizing User Pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         Take the following limitations into consideration when you use the ClientMetadata parameter:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Cognito does not store the ClientMetadata value. This data is available only to AWS Lambda
     *         triggers that are assigned to a user pool to support custom workflows. If your user pool configuration
     *         does not include triggers, the ClientMetadata parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Cognito does not validate the ClientMetadata value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Cognito does not encrypt the the ClientMetadata value, so don't use it to provide sensitive
     *         information.
     *         </p>
     *         </li>
     *         </ul>
     */
    public Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(secretHash());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(userAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(validationData());
        hashCode = 31 * hashCode + Objects.hashCode(analyticsMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(userContextData());
        hashCode = 31 * hashCode + Objects.hashCode(clientMetadata());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SignUpRequest)) {
            return false;
        }
        SignUpRequest other = (SignUpRequest) obj;
        return Objects.equals(clientId(), other.clientId()) && Objects.equals(secretHash(), other.secretHash())
                && Objects.equals(username(), other.username()) && Objects.equals(password(), other.password())
                && Objects.equals(userAttributes(), other.userAttributes())
                && Objects.equals(validationData(), other.validationData())
                && Objects.equals(analyticsMetadata(), other.analyticsMetadata())
                && Objects.equals(userContextData(), other.userContextData())
                && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SignUpRequest").add("ClientId", clientId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretHash", secretHash() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Username", username() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***")
                .add("UserAttributes", userAttributes()).add("ValidationData", validationData())
                .add("AnalyticsMetadata", analyticsMetadata()).add("UserContextData", userContextData())
                .add("ClientMetadata", clientMetadata()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "SecretHash":
            return Optional.ofNullable(clazz.cast(secretHash()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "UserAttributes":
            return Optional.ofNullable(clazz.cast(userAttributes()));
        case "ValidationData":
            return Optional.ofNullable(clazz.cast(validationData()));
        case "AnalyticsMetadata":
            return Optional.ofNullable(clazz.cast(analyticsMetadata()));
        case "UserContextData":
            return Optional.ofNullable(clazz.cast(userContextData()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SignUpRequest, T> g) {
        return obj -> g.apply((SignUpRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo, CopyableBuilder<Builder, SignUpRequest> {
        /**
         * <p>
         * The ID of the client associated with the user pool.
         * </p>
         * 
         * @param clientId
         *        The ID of the client associated with the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
         * username plus the client ID in the message.
         * </p>
         * 
         * @param secretHash
         *        A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client
         *        and username plus the client ID in the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretHash(String secretHash);

        /**
         * <p>
         * The user name of the user you wish to register.
         * </p>
         * 
         * @param username
         *        The user name of the user you wish to register.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * The password of the user you wish to register.
         * </p>
         * 
         * @param password
         *        The password of the user you wish to register.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * 
         * @param userAttributes
         *        An array of name-value pairs representing user attributes.</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(Collection<AttributeType> userAttributes);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * 
         * @param userAttributes
         *        An array of name-value pairs representing user attributes.</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(AttributeType... userAttributes);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeType>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeType>.Builder#build()} is called immediately and
         * its result is passed to {@link #userAttributes(List<AttributeType>)}.
         * 
         * @param userAttributes
         *        a consumer that will call methods on {@link List<AttributeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAttributes(List<AttributeType>)
         */
        Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes);

        /**
         * <p>
         * The validation data in the request to register a user.
         * </p>
         * 
         * @param validationData
         *        The validation data in the request to register a user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(Collection<AttributeType> validationData);

        /**
         * <p>
         * The validation data in the request to register a user.
         * </p>
         * 
         * @param validationData
         *        The validation data in the request to register a user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(AttributeType... validationData);

        /**
         * <p>
         * The validation data in the request to register a user.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeType>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeType>.Builder#build()} is called immediately and
         * its result is passed to {@link #validationData(List<AttributeType>)}.
         * 
         * @param validationData
         *        a consumer that will call methods on {@link List<AttributeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationData(List<AttributeType>)
         */
        Builder validationData(Consumer<AttributeType.Builder>... validationData);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>SignUp</code> calls.
         * </p>
         * 
         * @param analyticsMetadata
         *        The Amazon Pinpoint analytics metadata for collecting metrics for <code>SignUp</code> calls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>SignUp</code> calls.
         * </p>
         * This is a convenience that creates an instance of the {@link AnalyticsMetadataType.Builder} avoiding the need
         * to create one manually via {@link AnalyticsMetadataType#builder()}.
         *
         * When the {@link Consumer} completes, {@link AnalyticsMetadataType.Builder#build()} is called immediately and
         * its result is passed to {@link #analyticsMetadata(AnalyticsMetadataType)}.
         * 
         * @param analyticsMetadata
         *        a consumer that will call methods on {@link AnalyticsMetadataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #analyticsMetadata(AnalyticsMetadataType)
         */
        default Builder analyticsMetadata(Consumer<AnalyticsMetadataType.Builder> analyticsMetadata) {
            return analyticsMetadata(AnalyticsMetadataType.builder().applyMutation(analyticsMetadata).build());
        }

        /**
         * <p>
         * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk
         * of an unexpected event by Amazon Cognito advanced security.
         * </p>
         * 
         * @param userContextData
         *        Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the
         *        risk of an unexpected event by Amazon Cognito advanced security.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userContextData(UserContextDataType userContextData);

        /**
         * <p>
         * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk
         * of an unexpected event by Amazon Cognito advanced security.
         * </p>
         * This is a convenience that creates an instance of the {@link UserContextDataType.Builder} avoiding the need
         * to create one manually via {@link UserContextDataType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UserContextDataType.Builder#build()} is called immediately and
         * its result is passed to {@link #userContextData(UserContextDataType)}.
         * 
         * @param userContextData
         *        a consumer that will call methods on {@link UserContextDataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userContextData(UserContextDataType)
         */
        default Builder userContextData(Consumer<UserContextDataType.Builder> userContextData) {
            return userContextData(UserContextDataType.builder().applyMutation(userContextData).build());
        }

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for any custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning AWS Lambda functions to user pool triggers. When you use the SignUp
         * API action, Amazon Cognito invokes any functions that are assigned to the following triggers: <i>pre
         * sign-up</i>, <i>custom message</i>, and <i>post confirmation</i>. When Amazon Cognito invokes any of these
         * functions, it passes a JSON payload, which the function receives as input. This payload contains a
         * <code>clientMetadata</code> attribute, which provides the data that you assigned to the ClientMetadata
         * parameter in your SignUp request. In your function code in AWS Lambda, you can process the
         * <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * >Customizing User Pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * Take the following limitations into consideration when you use the ClientMetadata parameter:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Cognito does not store the ClientMetadata value. This data is available only to AWS Lambda triggers
         * that are assigned to a user pool to support custom workflows. If your user pool configuration does not
         * include triggers, the ClientMetadata parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Cognito does not validate the ClientMetadata value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Cognito does not encrypt the the ClientMetadata value, so don't use it to provide sensitive
         * information.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for any custom workflows that this
         *        action triggers. </p>
         *        <p>
         *        You create custom workflows by assigning AWS Lambda functions to user pool triggers. When you use the
         *        SignUp API action, Amazon Cognito invokes any functions that are assigned to the following triggers:
         *        <i>pre sign-up</i>, <i>custom message</i>, and <i>post confirmation</i>. When Amazon Cognito invokes
         *        any of these functions, it passes a JSON payload, which the function receives as input. This payload
         *        contains a <code>clientMetadata</code> attribute, which provides the data that you assigned to the
         *        ClientMetadata parameter in your SignUp request. In your function code in AWS Lambda, you can process
         *        the <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        >Customizing User Pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        Take the following limitations into consideration when you use the ClientMetadata parameter:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Cognito does not store the ClientMetadata value. This data is available only to AWS Lambda
         *        triggers that are assigned to a user pool to support custom workflows. If your user pool configuration
         *        does not include triggers, the ClientMetadata parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Cognito does not validate the ClientMetadata value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Cognito does not encrypt the the ClientMetadata value, so don't use it to provide sensitive
         *        information.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String clientId;

        private String secretHash;

        private String username;

        private String password;

        private List<AttributeType> userAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<AttributeType> validationData = DefaultSdkAutoConstructList.getInstance();

        private AnalyticsMetadataType analyticsMetadata;

        private UserContextDataType userContextData;

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SignUpRequest model) {
            super(model);
            clientId(model.clientId);
            secretHash(model.secretHash);
            username(model.username);
            password(model.password);
            userAttributes(model.userAttributes);
            validationData(model.validationData);
            analyticsMetadata(model.analyticsMetadata);
            userContextData(model.userContextData);
            clientMetadata(model.clientMetadata);
        }

        public final String getClientId() {
            return clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        public final String getSecretHash() {
            return secretHash;
        }

        @Override
        public final Builder secretHash(String secretHash) {
            this.secretHash = secretHash;
            return this;
        }

        public final void setSecretHash(String secretHash) {
            this.secretHash = secretHash;
        }

        public final String getUsername() {
            return username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        public final String getPassword() {
            return password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        public final Collection<AttributeType.Builder> getUserAttributes() {
            return userAttributes != null ? userAttributes.stream().map(AttributeType::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder userAttributes(Collection<AttributeType> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copy(userAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(AttributeType... userAttributes) {
            userAttributes(Arrays.asList(userAttributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes) {
            userAttributes(Stream.of(userAttributes).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUserAttributes(Collection<AttributeType.BuilderImpl> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copyFromBuilder(userAttributes);
        }

        public final Collection<AttributeType.Builder> getValidationData() {
            return validationData != null ? validationData.stream().map(AttributeType::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder validationData(Collection<AttributeType> validationData) {
            this.validationData = AttributeListTypeCopier.copy(validationData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(AttributeType... validationData) {
            validationData(Arrays.asList(validationData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(Consumer<AttributeType.Builder>... validationData) {
            validationData(Stream.of(validationData).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setValidationData(Collection<AttributeType.BuilderImpl> validationData) {
            this.validationData = AttributeListTypeCopier.copyFromBuilder(validationData);
        }

        public final AnalyticsMetadataType.Builder getAnalyticsMetadata() {
            return analyticsMetadata != null ? analyticsMetadata.toBuilder() : null;
        }

        @Override
        public final Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata;
            return this;
        }

        public final void setAnalyticsMetadata(AnalyticsMetadataType.BuilderImpl analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata != null ? analyticsMetadata.build() : null;
        }

        public final UserContextDataType.Builder getUserContextData() {
            return userContextData != null ? userContextData.toBuilder() : null;
        }

        @Override
        public final Builder userContextData(UserContextDataType userContextData) {
            this.userContextData = userContextData;
            return this;
        }

        public final void setUserContextData(UserContextDataType.BuilderImpl userContextData) {
            this.userContextData = userContextData != null ? userContextData.build() : null;
        }

        public final Map<String, String> getClientMetadata() {
            return clientMetadata;
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SignUpRequest build() {
            return new SignUpRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
