/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to delete user attributes as an administrator.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdminDeleteUserAttributesRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<AdminDeleteUserAttributesRequest.Builder, AdminDeleteUserAttributesRequest> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AdminDeleteUserAttributesRequest::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(AdminDeleteUserAttributesRequest::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<List<String>> USER_ATTRIBUTE_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(AdminDeleteUserAttributesRequest::userAttributeNames))
            .setter(setter(Builder::userAttributeNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAttributeNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            USERNAME_FIELD, USER_ATTRIBUTE_NAMES_FIELD));

    private final String userPoolId;

    private final String username;

    private final List<String> userAttributeNames;

    private AdminDeleteUserAttributesRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.username = builder.username;
        this.userAttributeNames = builder.userAttributeNames;
    }

    /**
     * <p>
     * The user pool ID for the user pool where you want to delete user attributes.
     * </p>
     * 
     * @return The user pool ID for the user pool where you want to delete user attributes.
     */
    public String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The user name of the user from which you would like to delete attributes.
     * </p>
     * 
     * @return The user name of the user from which you would like to delete attributes.
     */
    public String username() {
        return username;
    }

    /**
     * Returns true if the UserAttributeNames property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasUserAttributeNames() {
        return userAttributeNames != null && !(userAttributeNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of strings representing the user attribute names you wish to delete.
     * </p>
     * <p>
     * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUserAttributeNames()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of strings representing the user attribute names you wish to delete.</p>
     *         <p>
     *         For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     */
    public List<String> userAttributeNames() {
        return userAttributeNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(userAttributeNames());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdminDeleteUserAttributesRequest)) {
            return false;
        }
        AdminDeleteUserAttributesRequest other = (AdminDeleteUserAttributesRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(username(), other.username())
                && Objects.equals(userAttributeNames(), other.userAttributeNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AdminDeleteUserAttributesRequest").add("UserPoolId", userPoolId())
                .add("Username", username() == null ? null : "*** Sensitive Data Redacted ***")
                .add("UserAttributeNames", userAttributeNames()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "UserAttributeNames":
            return Optional.ofNullable(clazz.cast(userAttributeNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AdminDeleteUserAttributesRequest, T> g) {
        return obj -> g.apply((AdminDeleteUserAttributesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, AdminDeleteUserAttributesRequest> {
        /**
         * <p>
         * The user pool ID for the user pool where you want to delete user attributes.
         * </p>
         * 
         * @param userPoolId
         *        The user pool ID for the user pool where you want to delete user attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The user name of the user from which you would like to delete attributes.
         * </p>
         * 
         * @param username
         *        The user name of the user from which you would like to delete attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * An array of strings representing the user attribute names you wish to delete.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * 
         * @param userAttributeNames
         *        An array of strings representing the user attribute names you wish to delete.</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributeNames(Collection<String> userAttributeNames);

        /**
         * <p>
         * An array of strings representing the user attribute names you wish to delete.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * 
         * @param userAttributeNames
         *        An array of strings representing the user attribute names you wish to delete.</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributeNames(String... userAttributeNames);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String username;

        private List<String> userAttributeNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdminDeleteUserAttributesRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            username(model.username);
            userAttributeNames(model.userAttributeNames);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        public final String getUsername() {
            return username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        public final Collection<String> getUserAttributeNames() {
            return userAttributeNames;
        }

        @Override
        public final Builder userAttributeNames(Collection<String> userAttributeNames) {
            this.userAttributeNames = AttributeNameListTypeCopier.copy(userAttributeNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributeNames(String... userAttributeNames) {
            userAttributeNames(Arrays.asList(userAttributeNames));
            return this;
        }

        public final void setUserAttributeNames(Collection<String> userAttributeNames) {
            this.userAttributeNames = AttributeNameListTypeCopier.copy(userAttributeNames);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AdminDeleteUserAttributesRequest build() {
            return new AdminDeleteUserAttributesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
