/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to reset a user's password.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ForgotPasswordRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<ForgotPasswordRequest.Builder, ForgotPasswordRequest> {
    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientId").getter(getter(ForgotPasswordRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<String> SECRET_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretHash").getter(getter(ForgotPasswordRequest::secretHash)).setter(setter(Builder::secretHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretHash").build()).build();

    private static final SdkField<UserContextDataType> USER_CONTEXT_DATA_FIELD = SdkField
            .<UserContextDataType> builder(MarshallingType.SDK_POJO).memberName("UserContextData")
            .getter(getter(ForgotPasswordRequest::userContextData)).setter(setter(Builder::userContextData))
            .constructor(UserContextDataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserContextData").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(ForgotPasswordRequest::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<AnalyticsMetadataType> ANALYTICS_METADATA_FIELD = SdkField
            .<AnalyticsMetadataType> builder(MarshallingType.SDK_POJO).memberName("AnalyticsMetadata")
            .getter(getter(ForgotPasswordRequest::analyticsMetadata)).setter(setter(Builder::analyticsMetadata))
            .constructor(AnalyticsMetadataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalyticsMetadata").build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(ForgotPasswordRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_ID_FIELD,
            SECRET_HASH_FIELD, USER_CONTEXT_DATA_FIELD, USERNAME_FIELD, ANALYTICS_METADATA_FIELD, CLIENT_METADATA_FIELD));

    private final String clientId;

    private final String secretHash;

    private final UserContextDataType userContextData;

    private final String username;

    private final AnalyticsMetadataType analyticsMetadata;

    private final Map<String, String> clientMetadata;

    private ForgotPasswordRequest(BuilderImpl builder) {
        super(builder);
        this.clientId = builder.clientId;
        this.secretHash = builder.secretHash;
        this.userContextData = builder.userContextData;
        this.username = builder.username;
        this.analyticsMetadata = builder.analyticsMetadata;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The ID of the client associated with the user pool.
     * </p>
     * 
     * @return The ID of the client associated with the user pool.
     */
    public final String clientId() {
        return clientId;
    }

    /**
     * <p>
     * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     * username plus the client ID in the message.
     * </p>
     * 
     * @return A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     *         username plus the client ID in the message.
     */
    public final String secretHash() {
        return secretHash;
    }

    /**
     * <p>
     * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk of an
     * unexpected event by Amazon Cognito advanced security.
     * </p>
     * 
     * @return Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the
     *         risk of an unexpected event by Amazon Cognito advanced security.
     */
    public final UserContextDataType userContextData() {
        return userContextData;
    }

    /**
     * <p>
     * The user name of the user for whom you want to enter a code to reset a forgotten password.
     * </p>
     * 
     * @return The user name of the user for whom you want to enter a code to reset a forgotten password.
     */
    public final String username() {
        return username;
    }

    /**
     * <p>
     * The Amazon Pinpoint analytics metadata for collecting metrics for <code>ForgotPassword</code> calls.
     * </p>
     * 
     * @return The Amazon Pinpoint analytics metadata for collecting metrics for <code>ForgotPassword</code> calls.
     */
    public final AnalyticsMetadataType analyticsMetadata() {
        return analyticsMetadata;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the ForgotPassword
     * API action, Amazon Cognito invokes any functions that are assigned to the following triggers: <i>pre sign-up</i>,
     * <i>custom message</i>, and <i>user migration</i>. When Amazon Cognito invokes any of these functions, it passes a
     * JSON payload, which the function receives as input. This payload contains a <code>clientMetadata</code>
     * attribute, which provides the data that you assigned to the ClientMetadata parameter in your ForgotPassword
     * request. In your function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your
     * workflow for your specific needs.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a user pool
     * to support custom workflows. If your user pool configuration doesn't include triggers, the ClientMetadata
     * parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Validate the ClientMetadata value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientMetadata} method.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for any custom workflows that this action
     *         triggers.</p>
     *         <p>
     *         You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     *         ForgotPassword API action, Amazon Cognito invokes any functions that are assigned to the following
     *         triggers: <i>pre sign-up</i>, <i>custom message</i>, and <i>user migration</i>. When Amazon Cognito
     *         invokes any of these functions, it passes a JSON payload, which the function receives as input. This
     *         payload contains a <code>clientMetadata</code> attribute, which provides the data that you assigned to
     *         the ClientMetadata parameter in your ForgotPassword request. In your function code in Lambda, you can
     *         process the <code>clientMetadata</code> value to enhance your workflow for your specific needs.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a
     *         user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
     *         ClientMetadata parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Validate the ClientMetadata value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(secretHash());
        hashCode = 31 * hashCode + Objects.hashCode(userContextData());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(analyticsMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ForgotPasswordRequest)) {
            return false;
        }
        ForgotPasswordRequest other = (ForgotPasswordRequest) obj;
        return Objects.equals(clientId(), other.clientId()) && Objects.equals(secretHash(), other.secretHash())
                && Objects.equals(userContextData(), other.userContextData()) && Objects.equals(username(), other.username())
                && Objects.equals(analyticsMetadata(), other.analyticsMetadata())
                && hasClientMetadata() == other.hasClientMetadata() && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ForgotPasswordRequest")
                .add("ClientId", clientId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretHash", secretHash() == null ? null : "*** Sensitive Data Redacted ***")
                .add("UserContextData", userContextData())
                .add("Username", username() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AnalyticsMetadata", analyticsMetadata())
                .add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "SecretHash":
            return Optional.ofNullable(clazz.cast(secretHash()));
        case "UserContextData":
            return Optional.ofNullable(clazz.cast(userContextData()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "AnalyticsMetadata":
            return Optional.ofNullable(clazz.cast(analyticsMetadata()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ForgotPasswordRequest, T> g) {
        return obj -> g.apply((ForgotPasswordRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, ForgotPasswordRequest> {
        /**
         * <p>
         * The ID of the client associated with the user pool.
         * </p>
         * 
         * @param clientId
         *        The ID of the client associated with the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
         * username plus the client ID in the message.
         * </p>
         * 
         * @param secretHash
         *        A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client
         *        and username plus the client ID in the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretHash(String secretHash);

        /**
         * <p>
         * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk
         * of an unexpected event by Amazon Cognito advanced security.
         * </p>
         * 
         * @param userContextData
         *        Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the
         *        risk of an unexpected event by Amazon Cognito advanced security.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userContextData(UserContextDataType userContextData);

        /**
         * <p>
         * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk
         * of an unexpected event by Amazon Cognito advanced security.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserContextDataType.Builder} avoiding the
         * need to create one manually via {@link UserContextDataType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UserContextDataType.Builder#build()} is called immediately and
         * its result is passed to {@link #userContextData(UserContextDataType)}.
         * 
         * @param userContextData
         *        a consumer that will call methods on {@link UserContextDataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userContextData(UserContextDataType)
         */
        default Builder userContextData(Consumer<UserContextDataType.Builder> userContextData) {
            return userContextData(UserContextDataType.builder().applyMutation(userContextData).build());
        }

        /**
         * <p>
         * The user name of the user for whom you want to enter a code to reset a forgotten password.
         * </p>
         * 
         * @param username
         *        The user name of the user for whom you want to enter a code to reset a forgotten password.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>ForgotPassword</code> calls.
         * </p>
         * 
         * @param analyticsMetadata
         *        The Amazon Pinpoint analytics metadata for collecting metrics for <code>ForgotPassword</code> calls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>ForgotPassword</code> calls.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalyticsMetadataType.Builder} avoiding
         * the need to create one manually via {@link AnalyticsMetadataType#builder()}.
         *
         * When the {@link Consumer} completes, {@link AnalyticsMetadataType.Builder#build()} is called immediately and
         * its result is passed to {@link #analyticsMetadata(AnalyticsMetadataType)}.
         * 
         * @param analyticsMetadata
         *        a consumer that will call methods on {@link AnalyticsMetadataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #analyticsMetadata(AnalyticsMetadataType)
         */
        default Builder analyticsMetadata(Consumer<AnalyticsMetadataType.Builder> analyticsMetadata) {
            return analyticsMetadata(AnalyticsMetadataType.builder().applyMutation(analyticsMetadata).build());
        }

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for any custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         * ForgotPassword API action, Amazon Cognito invokes any functions that are assigned to the following triggers:
         * <i>pre sign-up</i>, <i>custom message</i>, and <i>user migration</i>. When Amazon Cognito invokes any of
         * these functions, it passes a JSON payload, which the function receives as input. This payload contains a
         * <code>clientMetadata</code> attribute, which provides the data that you assigned to the ClientMetadata
         * parameter in your ForgotPassword request. In your function code in Lambda, you can process the
         * <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a user
         * pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         * ClientMetadata parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Validate the ClientMetadata value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for any custom workflows that this
         *        action triggers.</p>
         *        <p>
         *        You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         *        ForgotPassword API action, Amazon Cognito invokes any functions that are assigned to the following
         *        triggers: <i>pre sign-up</i>, <i>custom message</i>, and <i>user migration</i>. When Amazon Cognito
         *        invokes any of these functions, it passes a JSON payload, which the function receives as input. This
         *        payload contains a <code>clientMetadata</code> attribute, which provides the data that you assigned to
         *        the ClientMetadata parameter in your ForgotPassword request. In your function code in Lambda, you can
         *        process the <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a
         *        user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         *        ClientMetadata parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Validate the ClientMetadata value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String clientId;

        private String secretHash;

        private UserContextDataType userContextData;

        private String username;

        private AnalyticsMetadataType analyticsMetadata;

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ForgotPasswordRequest model) {
            super(model);
            clientId(model.clientId);
            secretHash(model.secretHash);
            userContextData(model.userContextData);
            username(model.username);
            analyticsMetadata(model.analyticsMetadata);
            clientMetadata(model.clientMetadata);
        }

        public final String getClientId() {
            return clientId;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        @Override
        @Transient
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final String getSecretHash() {
            return secretHash;
        }

        public final void setSecretHash(String secretHash) {
            this.secretHash = secretHash;
        }

        @Override
        @Transient
        public final Builder secretHash(String secretHash) {
            this.secretHash = secretHash;
            return this;
        }

        public final UserContextDataType.Builder getUserContextData() {
            return userContextData != null ? userContextData.toBuilder() : null;
        }

        public final void setUserContextData(UserContextDataType.BuilderImpl userContextData) {
            this.userContextData = userContextData != null ? userContextData.build() : null;
        }

        @Override
        @Transient
        public final Builder userContextData(UserContextDataType userContextData) {
            this.userContextData = userContextData;
            return this;
        }

        public final String getUsername() {
            return username;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        @Override
        @Transient
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final AnalyticsMetadataType.Builder getAnalyticsMetadata() {
            return analyticsMetadata != null ? analyticsMetadata.toBuilder() : null;
        }

        public final void setAnalyticsMetadata(AnalyticsMetadataType.BuilderImpl analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata != null ? analyticsMetadata.build() : null;
        }

        @Override
        @Transient
        public final Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata;
            return this;
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        @Transient
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ForgotPasswordRequest build() {
            return new ForgotPasswordRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
