/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The email configuration of your user pool. The email configuration type sets your preferred sending method, Amazon
 * Web Services Region, and sender for messages from your user pool.
 * </p>
 * <note>
 * <p>
 * Amazon Cognito can send email messages with Amazon Simple Email Service resources in the Amazon Web Services Region
 * where you created your user pool, and in alternate Regions in some cases. For more information on the supported
 * Regions, see <a href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html">Email settings
 * for Amazon Cognito user pools</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EmailConfigurationType implements SdkPojo, Serializable,
        ToCopyableBuilder<EmailConfigurationType.Builder, EmailConfigurationType> {
    private static final SdkField<String> SOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceArn").getter(getter(EmailConfigurationType::sourceArn)).setter(setter(Builder::sourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceArn").build()).build();

    private static final SdkField<String> REPLY_TO_EMAIL_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplyToEmailAddress").getter(getter(EmailConfigurationType::replyToEmailAddress))
            .setter(setter(Builder::replyToEmailAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplyToEmailAddress").build())
            .build();

    private static final SdkField<String> EMAIL_SENDING_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EmailSendingAccount").getter(getter(EmailConfigurationType::emailSendingAccountAsString))
            .setter(setter(Builder::emailSendingAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmailSendingAccount").build())
            .build();

    private static final SdkField<String> FROM_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("From")
            .getter(getter(EmailConfigurationType::from)).setter(setter(Builder::from))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("From").build()).build();

    private static final SdkField<String> CONFIGURATION_SET_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationSet").getter(getter(EmailConfigurationType::configurationSet))
            .setter(setter(Builder::configurationSet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationSet").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_ARN_FIELD,
            REPLY_TO_EMAIL_ADDRESS_FIELD, EMAIL_SENDING_ACCOUNT_FIELD, FROM_FIELD, CONFIGURATION_SET_FIELD));

    private static final long serialVersionUID = 1L;

    private final String sourceArn;

    private final String replyToEmailAddress;

    private final String emailSendingAccount;

    private final String from;

    private final String configurationSet;

    private EmailConfigurationType(BuilderImpl builder) {
        this.sourceArn = builder.sourceArn;
        this.replyToEmailAddress = builder.replyToEmailAddress;
        this.emailSendingAccount = builder.emailSendingAccount;
        this.from = builder.from;
        this.configurationSet = builder.configurationSet;
    }

    /**
     * <p>
     * The ARN of a verified email address in Amazon SES. Amazon Cognito uses this email address in one of the following
     * ways, depending on the value that you specify for the <code>EmailSendingAccount</code> parameter:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you specify <code>COGNITO_DEFAULT</code>, Amazon Cognito uses this address as the custom FROM address when it
     * emails your users using its built-in email account.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you specify <code>DEVELOPER</code>, Amazon Cognito emails your users with this address by calling Amazon SES
     * on your behalf.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The Region value of the <code>SourceArn</code> parameter must indicate a supported Amazon Web Services Region of
     * your user pool. Typically, the Region in the <code>SourceArn</code> and the user pool Region are the same. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html#user-pool-email-developer-region-mapping"
     * >Amazon SES email configuration regions</a> in the <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html">Amazon Cognito
     * Developer Guide</a>.
     * </p>
     * 
     * @return The ARN of a verified email address in Amazon SES. Amazon Cognito uses this email address in one of the
     *         following ways, depending on the value that you specify for the <code>EmailSendingAccount</code>
     *         parameter:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you specify <code>COGNITO_DEFAULT</code>, Amazon Cognito uses this address as the custom FROM address
     *         when it emails your users using its built-in email account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If you specify <code>DEVELOPER</code>, Amazon Cognito emails your users with this address by calling
     *         Amazon SES on your behalf.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The Region value of the <code>SourceArn</code> parameter must indicate a supported Amazon Web Services
     *         Region of your user pool. Typically, the Region in the <code>SourceArn</code> and the user pool Region
     *         are the same. For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html#user-pool-email-developer-region-mapping"
     *         >Amazon SES email configuration regions</a> in the <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html">Amazon
     *         Cognito Developer Guide</a>.
     */
    public final String sourceArn() {
        return sourceArn;
    }

    /**
     * <p>
     * The destination to which the receiver of the email should reply.
     * </p>
     * 
     * @return The destination to which the receiver of the email should reply.
     */
    public final String replyToEmailAddress() {
        return replyToEmailAddress;
    }

    /**
     * <p>
     * Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or uses your
     * Amazon Simple Email Service email configuration. Specify one of the following values:
     * </p>
     * <dl>
     * <dt>COGNITO_DEFAULT</dt>
     * <dd>
     * <p>
     * When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the default option,
     * Amazon Cognito allows only a limited number of emails each day for your user pool. For typical production
     * environments, the default email limit is less than the required delivery volume. To achieve a higher delivery
     * volume, specify DEVELOPER to use your Amazon SES email configuration.
     * </p>
     * <p>
     * To look up the email delivery limit for the default option, see <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html">Limits in </a> in the <i> Developer
     * Guide</i>.
     * </p>
     * <p>
     * The default FROM address is <code>no-reply@verificationemail.com</code>. To customize the FROM address, provide
     * the Amazon Resource Name (ARN) of an Amazon SES verified email address for the <code>SourceArn</code> parameter.
     * </p>
     * <p>
     * If EmailSendingAccount is COGNITO_DEFAULT, you can't use the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * EmailVerificationMessage
     * </p>
     * </li>
     * <li>
     * <p>
     * EmailVerificationSubject
     * </p>
     * </li>
     * <li>
     * <p>
     * InviteMessageTemplate.EmailMessage
     * </p>
     * </li>
     * <li>
     * <p>
     * InviteMessageTemplate.EmailSubject
     * </p>
     * </li>
     * <li>
     * <p>
     * VerificationMessageTemplate.EmailMessage
     * </p>
     * </li>
     * <li>
     * <p>
     * VerificationMessageTemplate.EmailMessageByLink
     * </p>
     * </li>
     * <li>
     * <p>
     * VerificationMessageTemplate.EmailSubject,
     * </p>
     * </li>
     * <li>
     * <p>
     * VerificationMessageTemplate.EmailSubjectByLink
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * DEVELOPER EmailSendingAccount is required.
     * </p>
     * </note></dd>
     * <dt>DEVELOPER</dt>
     * <dd>
     * <p>
     * When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls Amazon SES on
     * your behalf to send email from your verified email address. When you use this option, the email delivery limits
     * are the same limits that apply to your Amazon SES verified email address in your Amazon Web Services account.
     * </p>
     * <p>
     * If you use this option, you must provide the ARN of an Amazon SES verified email address for the
     * <code>SourceArn</code> parameter.
     * </p>
     * <p>
     * Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on your behalf.
     * When you update your user pool with this option, Amazon Cognito creates a <i>service-linked role</i>, which is a
     * type of role, in your Amazon Web Services account. This role contains the permissions that allow to access Amazon
     * SES and send email messages with your address. For more information about the service-linked role that Amazon
     * Cognito creates, see <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html">Using
     * Service-Linked Roles for Amazon Cognito</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #emailSendingAccount} will return {@link EmailSendingAccountType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #emailSendingAccountAsString}.
     * </p>
     * 
     * @return Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or
     *         uses your Amazon Simple Email Service email configuration. Specify one of the following values:</p>
     *         <dl>
     *         <dt>COGNITO_DEFAULT</dt>
     *         <dd>
     *         <p>
     *         When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the default
     *         option, Amazon Cognito allows only a limited number of emails each day for your user pool. For typical
     *         production environments, the default email limit is less than the required delivery volume. To achieve a
     *         higher delivery volume, specify DEVELOPER to use your Amazon SES email configuration.
     *         </p>
     *         <p>
     *         To look up the email delivery limit for the default option, see <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html">Limits in </a> in the <i>
     *         Developer Guide</i>.
     *         </p>
     *         <p>
     *         The default FROM address is <code>no-reply@verificationemail.com</code>. To customize the FROM address,
     *         provide the Amazon Resource Name (ARN) of an Amazon SES verified email address for the
     *         <code>SourceArn</code> parameter.
     *         </p>
     *         <p>
     *         If EmailSendingAccount is COGNITO_DEFAULT, you can't use the following parameters:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         EmailVerificationMessage
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EmailVerificationSubject
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         InviteMessageTemplate.EmailMessage
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         InviteMessageTemplate.EmailSubject
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         VerificationMessageTemplate.EmailMessage
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         VerificationMessageTemplate.EmailMessageByLink
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         VerificationMessageTemplate.EmailSubject,
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         VerificationMessageTemplate.EmailSubjectByLink
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         DEVELOPER EmailSendingAccount is required.
     *         </p>
     *         </note></dd>
     *         <dt>DEVELOPER</dt>
     *         <dd>
     *         <p>
     *         When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls Amazon
     *         SES on your behalf to send email from your verified email address. When you use this option, the email
     *         delivery limits are the same limits that apply to your Amazon SES verified email address in your Amazon
     *         Web Services account.
     *         </p>
     *         <p>
     *         If you use this option, you must provide the ARN of an Amazon SES verified email address for the
     *         <code>SourceArn</code> parameter.
     *         </p>
     *         <p>
     *         Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on your
     *         behalf. When you update your user pool with this option, Amazon Cognito creates a <i>service-linked
     *         role</i>, which is a type of role, in your Amazon Web Services account. This role contains the
     *         permissions that allow to access Amazon SES and send email messages with your address. For more
     *         information about the service-linked role that Amazon Cognito creates, see <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html">Using
     *         Service-Linked Roles for Amazon Cognito</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         </dd>
     * @see EmailSendingAccountType
     */
    public final EmailSendingAccountType emailSendingAccount() {
        return EmailSendingAccountType.fromValue(emailSendingAccount);
    }

    /**
     * <p>
     * Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or uses your
     * Amazon Simple Email Service email configuration. Specify one of the following values:
     * </p>
     * <dl>
     * <dt>COGNITO_DEFAULT</dt>
     * <dd>
     * <p>
     * When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the default option,
     * Amazon Cognito allows only a limited number of emails each day for your user pool. For typical production
     * environments, the default email limit is less than the required delivery volume. To achieve a higher delivery
     * volume, specify DEVELOPER to use your Amazon SES email configuration.
     * </p>
     * <p>
     * To look up the email delivery limit for the default option, see <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html">Limits in </a> in the <i> Developer
     * Guide</i>.
     * </p>
     * <p>
     * The default FROM address is <code>no-reply@verificationemail.com</code>. To customize the FROM address, provide
     * the Amazon Resource Name (ARN) of an Amazon SES verified email address for the <code>SourceArn</code> parameter.
     * </p>
     * <p>
     * If EmailSendingAccount is COGNITO_DEFAULT, you can't use the following parameters:
     * </p>
     * <ul>
     * <li>
     * <p>
     * EmailVerificationMessage
     * </p>
     * </li>
     * <li>
     * <p>
     * EmailVerificationSubject
     * </p>
     * </li>
     * <li>
     * <p>
     * InviteMessageTemplate.EmailMessage
     * </p>
     * </li>
     * <li>
     * <p>
     * InviteMessageTemplate.EmailSubject
     * </p>
     * </li>
     * <li>
     * <p>
     * VerificationMessageTemplate.EmailMessage
     * </p>
     * </li>
     * <li>
     * <p>
     * VerificationMessageTemplate.EmailMessageByLink
     * </p>
     * </li>
     * <li>
     * <p>
     * VerificationMessageTemplate.EmailSubject,
     * </p>
     * </li>
     * <li>
     * <p>
     * VerificationMessageTemplate.EmailSubjectByLink
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * DEVELOPER EmailSendingAccount is required.
     * </p>
     * </note></dd>
     * <dt>DEVELOPER</dt>
     * <dd>
     * <p>
     * When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls Amazon SES on
     * your behalf to send email from your verified email address. When you use this option, the email delivery limits
     * are the same limits that apply to your Amazon SES verified email address in your Amazon Web Services account.
     * </p>
     * <p>
     * If you use this option, you must provide the ARN of an Amazon SES verified email address for the
     * <code>SourceArn</code> parameter.
     * </p>
     * <p>
     * Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on your behalf.
     * When you update your user pool with this option, Amazon Cognito creates a <i>service-linked role</i>, which is a
     * type of role, in your Amazon Web Services account. This role contains the permissions that allow to access Amazon
     * SES and send email messages with your address. For more information about the service-linked role that Amazon
     * Cognito creates, see <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html">Using
     * Service-Linked Roles for Amazon Cognito</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #emailSendingAccount} will return {@link EmailSendingAccountType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #emailSendingAccountAsString}.
     * </p>
     * 
     * @return Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or
     *         uses your Amazon Simple Email Service email configuration. Specify one of the following values:</p>
     *         <dl>
     *         <dt>COGNITO_DEFAULT</dt>
     *         <dd>
     *         <p>
     *         When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the default
     *         option, Amazon Cognito allows only a limited number of emails each day for your user pool. For typical
     *         production environments, the default email limit is less than the required delivery volume. To achieve a
     *         higher delivery volume, specify DEVELOPER to use your Amazon SES email configuration.
     *         </p>
     *         <p>
     *         To look up the email delivery limit for the default option, see <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html">Limits in </a> in the <i>
     *         Developer Guide</i>.
     *         </p>
     *         <p>
     *         The default FROM address is <code>no-reply@verificationemail.com</code>. To customize the FROM address,
     *         provide the Amazon Resource Name (ARN) of an Amazon SES verified email address for the
     *         <code>SourceArn</code> parameter.
     *         </p>
     *         <p>
     *         If EmailSendingAccount is COGNITO_DEFAULT, you can't use the following parameters:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         EmailVerificationMessage
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EmailVerificationSubject
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         InviteMessageTemplate.EmailMessage
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         InviteMessageTemplate.EmailSubject
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         VerificationMessageTemplate.EmailMessage
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         VerificationMessageTemplate.EmailMessageByLink
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         VerificationMessageTemplate.EmailSubject,
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         VerificationMessageTemplate.EmailSubjectByLink
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         DEVELOPER EmailSendingAccount is required.
     *         </p>
     *         </note></dd>
     *         <dt>DEVELOPER</dt>
     *         <dd>
     *         <p>
     *         When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls Amazon
     *         SES on your behalf to send email from your verified email address. When you use this option, the email
     *         delivery limits are the same limits that apply to your Amazon SES verified email address in your Amazon
     *         Web Services account.
     *         </p>
     *         <p>
     *         If you use this option, you must provide the ARN of an Amazon SES verified email address for the
     *         <code>SourceArn</code> parameter.
     *         </p>
     *         <p>
     *         Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on your
     *         behalf. When you update your user pool with this option, Amazon Cognito creates a <i>service-linked
     *         role</i>, which is a type of role, in your Amazon Web Services account. This role contains the
     *         permissions that allow to access Amazon SES and send email messages with your address. For more
     *         information about the service-linked role that Amazon Cognito creates, see <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html">Using
     *         Service-Linked Roles for Amazon Cognito</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         </dd>
     * @see EmailSendingAccountType
     */
    public final String emailSendingAccountAsString() {
        return emailSendingAccount;
    }

    /**
     * <p>
     * Either the sender’s email address or the sender’s name with their email address. For example,
     * <code>testuser@example.com</code> or <code>Test User &lt;testuser@example.com&gt;</code>. This address appears
     * before the body of the email.
     * </p>
     * 
     * @return Either the sender’s email address or the sender’s name with their email address. For example,
     *         <code>testuser@example.com</code> or <code>Test User &lt;testuser@example.com&gt;</code>. This address
     *         appears before the body of the email.
     */
    public final String from() {
        return from;
    }

    /**
     * <p>
     * The set of configuration rules that can be applied to emails sent using Amazon Simple Email Service. A
     * configuration set is applied to an email by including a reference to the configuration set in the headers of the
     * email. Once applied, all of the rules in that configuration set are applied to the email. Configuration sets can
     * be used to apply the following types of rules to emails:
     * </p>
     * <dl>
     * <dt>Event publishing</dt>
     * <dd>
     * <p>
     * Amazon Simple Email Service can track the number of send, delivery, open, click, bounce, and complaint events for
     * each email sent. Use event publishing to send information about these events to other Amazon Web Services
     * services such as and Amazon CloudWatch
     * </p>
     * </dd>
     * <dt>IP pool management</dt>
     * <dd>
     * <p>
     * When leasing dedicated IP addresses with Amazon Simple Email Service, you can create groups of IP addresses,
     * called dedicated IP pools. You can then associate the dedicated IP pools with configuration sets.
     * </p>
     * </dd>
     * </dl>
     * 
     * @return The set of configuration rules that can be applied to emails sent using Amazon Simple Email Service. A
     *         configuration set is applied to an email by including a reference to the configuration set in the headers
     *         of the email. Once applied, all of the rules in that configuration set are applied to the email.
     *         Configuration sets can be used to apply the following types of rules to emails: </p>
     *         <dl>
     *         <dt>Event publishing</dt>
     *         <dd>
     *         <p>
     *         Amazon Simple Email Service can track the number of send, delivery, open, click, bounce, and complaint
     *         events for each email sent. Use event publishing to send information about these events to other Amazon
     *         Web Services services such as and Amazon CloudWatch
     *         </p>
     *         </dd>
     *         <dt>IP pool management</dt>
     *         <dd>
     *         <p>
     *         When leasing dedicated IP addresses with Amazon Simple Email Service, you can create groups of IP
     *         addresses, called dedicated IP pools. You can then associate the dedicated IP pools with configuration
     *         sets.
     *         </p>
     *         </dd>
     */
    public final String configurationSet() {
        return configurationSet;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(replyToEmailAddress());
        hashCode = 31 * hashCode + Objects.hashCode(emailSendingAccountAsString());
        hashCode = 31 * hashCode + Objects.hashCode(from());
        hashCode = 31 * hashCode + Objects.hashCode(configurationSet());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EmailConfigurationType)) {
            return false;
        }
        EmailConfigurationType other = (EmailConfigurationType) obj;
        return Objects.equals(sourceArn(), other.sourceArn())
                && Objects.equals(replyToEmailAddress(), other.replyToEmailAddress())
                && Objects.equals(emailSendingAccountAsString(), other.emailSendingAccountAsString())
                && Objects.equals(from(), other.from()) && Objects.equals(configurationSet(), other.configurationSet());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EmailConfigurationType").add("SourceArn", sourceArn())
                .add("ReplyToEmailAddress", replyToEmailAddress()).add("EmailSendingAccount", emailSendingAccountAsString())
                .add("From", from()).add("ConfigurationSet", configurationSet()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceArn":
            return Optional.ofNullable(clazz.cast(sourceArn()));
        case "ReplyToEmailAddress":
            return Optional.ofNullable(clazz.cast(replyToEmailAddress()));
        case "EmailSendingAccount":
            return Optional.ofNullable(clazz.cast(emailSendingAccountAsString()));
        case "From":
            return Optional.ofNullable(clazz.cast(from()));
        case "ConfigurationSet":
            return Optional.ofNullable(clazz.cast(configurationSet()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EmailConfigurationType, T> g) {
        return obj -> g.apply((EmailConfigurationType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EmailConfigurationType> {
        /**
         * <p>
         * The ARN of a verified email address in Amazon SES. Amazon Cognito uses this email address in one of the
         * following ways, depending on the value that you specify for the <code>EmailSendingAccount</code> parameter:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you specify <code>COGNITO_DEFAULT</code>, Amazon Cognito uses this address as the custom FROM address when
         * it emails your users using its built-in email account.
         * </p>
         * </li>
         * <li>
         * <p>
         * If you specify <code>DEVELOPER</code>, Amazon Cognito emails your users with this address by calling Amazon
         * SES on your behalf.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The Region value of the <code>SourceArn</code> parameter must indicate a supported Amazon Web Services Region
         * of your user pool. Typically, the Region in the <code>SourceArn</code> and the user pool Region are the same.
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html#user-pool-email-developer-region-mapping"
         * >Amazon SES email configuration regions</a> in the <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html">Amazon
         * Cognito Developer Guide</a>.
         * </p>
         * 
         * @param sourceArn
         *        The ARN of a verified email address in Amazon SES. Amazon Cognito uses this email address in one of
         *        the following ways, depending on the value that you specify for the <code>EmailSendingAccount</code>
         *        parameter:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you specify <code>COGNITO_DEFAULT</code>, Amazon Cognito uses this address as the custom FROM
         *        address when it emails your users using its built-in email account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If you specify <code>DEVELOPER</code>, Amazon Cognito emails your users with this address by calling
         *        Amazon SES on your behalf.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The Region value of the <code>SourceArn</code> parameter must indicate a supported Amazon Web Services
         *        Region of your user pool. Typically, the Region in the <code>SourceArn</code> and the user pool Region
         *        are the same. For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-email.html#user-pool-email-developer-region-mapping"
         *        >Amazon SES email configuration regions</a> in the <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools.html"
         *        >Amazon Cognito Developer Guide</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceArn(String sourceArn);

        /**
         * <p>
         * The destination to which the receiver of the email should reply.
         * </p>
         * 
         * @param replyToEmailAddress
         *        The destination to which the receiver of the email should reply.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replyToEmailAddress(String replyToEmailAddress);

        /**
         * <p>
         * Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or uses
         * your Amazon Simple Email Service email configuration. Specify one of the following values:
         * </p>
         * <dl>
         * <dt>COGNITO_DEFAULT</dt>
         * <dd>
         * <p>
         * When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the default
         * option, Amazon Cognito allows only a limited number of emails each day for your user pool. For typical
         * production environments, the default email limit is less than the required delivery volume. To achieve a
         * higher delivery volume, specify DEVELOPER to use your Amazon SES email configuration.
         * </p>
         * <p>
         * To look up the email delivery limit for the default option, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html">Limits in </a> in the <i>
         * Developer Guide</i>.
         * </p>
         * <p>
         * The default FROM address is <code>no-reply@verificationemail.com</code>. To customize the FROM address,
         * provide the Amazon Resource Name (ARN) of an Amazon SES verified email address for the <code>SourceArn</code>
         * parameter.
         * </p>
         * <p>
         * If EmailSendingAccount is COGNITO_DEFAULT, you can't use the following parameters:
         * </p>
         * <ul>
         * <li>
         * <p>
         * EmailVerificationMessage
         * </p>
         * </li>
         * <li>
         * <p>
         * EmailVerificationSubject
         * </p>
         * </li>
         * <li>
         * <p>
         * InviteMessageTemplate.EmailMessage
         * </p>
         * </li>
         * <li>
         * <p>
         * InviteMessageTemplate.EmailSubject
         * </p>
         * </li>
         * <li>
         * <p>
         * VerificationMessageTemplate.EmailMessage
         * </p>
         * </li>
         * <li>
         * <p>
         * VerificationMessageTemplate.EmailMessageByLink
         * </p>
         * </li>
         * <li>
         * <p>
         * VerificationMessageTemplate.EmailSubject,
         * </p>
         * </li>
         * <li>
         * <p>
         * VerificationMessageTemplate.EmailSubjectByLink
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * DEVELOPER EmailSendingAccount is required.
         * </p>
         * </note></dd>
         * <dt>DEVELOPER</dt>
         * <dd>
         * <p>
         * When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls Amazon SES
         * on your behalf to send email from your verified email address. When you use this option, the email delivery
         * limits are the same limits that apply to your Amazon SES verified email address in your Amazon Web Services
         * account.
         * </p>
         * <p>
         * If you use this option, you must provide the ARN of an Amazon SES verified email address for the
         * <code>SourceArn</code> parameter.
         * </p>
         * <p>
         * Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on your
         * behalf. When you update your user pool with this option, Amazon Cognito creates a <i>service-linked role</i>,
         * which is a type of role, in your Amazon Web Services account. This role contains the permissions that allow
         * to access Amazon SES and send email messages with your address. For more information about the service-linked
         * role that Amazon Cognito creates, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html">Using
         * Service-Linked Roles for Amazon Cognito</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param emailSendingAccount
         *        Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or
         *        uses your Amazon Simple Email Service email configuration. Specify one of the following values:</p>
         *        <dl>
         *        <dt>COGNITO_DEFAULT</dt>
         *        <dd>
         *        <p>
         *        When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the
         *        default option, Amazon Cognito allows only a limited number of emails each day for your user pool. For
         *        typical production environments, the default email limit is less than the required delivery volume. To
         *        achieve a higher delivery volume, specify DEVELOPER to use your Amazon SES email configuration.
         *        </p>
         *        <p>
         *        To look up the email delivery limit for the default option, see <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html">Limits in </a> in the <i>
         *        Developer Guide</i>.
         *        </p>
         *        <p>
         *        The default FROM address is <code>no-reply@verificationemail.com</code>. To customize the FROM
         *        address, provide the Amazon Resource Name (ARN) of an Amazon SES verified email address for the
         *        <code>SourceArn</code> parameter.
         *        </p>
         *        <p>
         *        If EmailSendingAccount is COGNITO_DEFAULT, you can't use the following parameters:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        EmailVerificationMessage
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EmailVerificationSubject
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        InviteMessageTemplate.EmailMessage
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        InviteMessageTemplate.EmailSubject
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        VerificationMessageTemplate.EmailMessage
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        VerificationMessageTemplate.EmailMessageByLink
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        VerificationMessageTemplate.EmailSubject,
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        VerificationMessageTemplate.EmailSubjectByLink
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        DEVELOPER EmailSendingAccount is required.
         *        </p>
         *        </note></dd>
         *        <dt>DEVELOPER</dt>
         *        <dd>
         *        <p>
         *        When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls
         *        Amazon SES on your behalf to send email from your verified email address. When you use this option,
         *        the email delivery limits are the same limits that apply to your Amazon SES verified email address in
         *        your Amazon Web Services account.
         *        </p>
         *        <p>
         *        If you use this option, you must provide the ARN of an Amazon SES verified email address for the
         *        <code>SourceArn</code> parameter.
         *        </p>
         *        <p>
         *        Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on
         *        your behalf. When you update your user pool with this option, Amazon Cognito creates a
         *        <i>service-linked role</i>, which is a type of role, in your Amazon Web Services account. This role
         *        contains the permissions that allow to access Amazon SES and send email messages with your address.
         *        For more information about the service-linked role that Amazon Cognito creates, see <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html">Using
         *        Service-Linked Roles for Amazon Cognito</a> in the <i>Amazon Cognito Developer Guide</i>.
         *        </p>
         *        </dd>
         * @see EmailSendingAccountType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EmailSendingAccountType
         */
        Builder emailSendingAccount(String emailSendingAccount);

        /**
         * <p>
         * Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or uses
         * your Amazon Simple Email Service email configuration. Specify one of the following values:
         * </p>
         * <dl>
         * <dt>COGNITO_DEFAULT</dt>
         * <dd>
         * <p>
         * When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the default
         * option, Amazon Cognito allows only a limited number of emails each day for your user pool. For typical
         * production environments, the default email limit is less than the required delivery volume. To achieve a
         * higher delivery volume, specify DEVELOPER to use your Amazon SES email configuration.
         * </p>
         * <p>
         * To look up the email delivery limit for the default option, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html">Limits in </a> in the <i>
         * Developer Guide</i>.
         * </p>
         * <p>
         * The default FROM address is <code>no-reply@verificationemail.com</code>. To customize the FROM address,
         * provide the Amazon Resource Name (ARN) of an Amazon SES verified email address for the <code>SourceArn</code>
         * parameter.
         * </p>
         * <p>
         * If EmailSendingAccount is COGNITO_DEFAULT, you can't use the following parameters:
         * </p>
         * <ul>
         * <li>
         * <p>
         * EmailVerificationMessage
         * </p>
         * </li>
         * <li>
         * <p>
         * EmailVerificationSubject
         * </p>
         * </li>
         * <li>
         * <p>
         * InviteMessageTemplate.EmailMessage
         * </p>
         * </li>
         * <li>
         * <p>
         * InviteMessageTemplate.EmailSubject
         * </p>
         * </li>
         * <li>
         * <p>
         * VerificationMessageTemplate.EmailMessage
         * </p>
         * </li>
         * <li>
         * <p>
         * VerificationMessageTemplate.EmailMessageByLink
         * </p>
         * </li>
         * <li>
         * <p>
         * VerificationMessageTemplate.EmailSubject,
         * </p>
         * </li>
         * <li>
         * <p>
         * VerificationMessageTemplate.EmailSubjectByLink
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * DEVELOPER EmailSendingAccount is required.
         * </p>
         * </note></dd>
         * <dt>DEVELOPER</dt>
         * <dd>
         * <p>
         * When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls Amazon SES
         * on your behalf to send email from your verified email address. When you use this option, the email delivery
         * limits are the same limits that apply to your Amazon SES verified email address in your Amazon Web Services
         * account.
         * </p>
         * <p>
         * If you use this option, you must provide the ARN of an Amazon SES verified email address for the
         * <code>SourceArn</code> parameter.
         * </p>
         * <p>
         * Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on your
         * behalf. When you update your user pool with this option, Amazon Cognito creates a <i>service-linked role</i>,
         * which is a type of role, in your Amazon Web Services account. This role contains the permissions that allow
         * to access Amazon SES and send email messages with your address. For more information about the service-linked
         * role that Amazon Cognito creates, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html">Using
         * Service-Linked Roles for Amazon Cognito</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param emailSendingAccount
         *        Specifies whether Amazon Cognito uses its built-in functionality to send your users email messages, or
         *        uses your Amazon Simple Email Service email configuration. Specify one of the following values:</p>
         *        <dl>
         *        <dt>COGNITO_DEFAULT</dt>
         *        <dd>
         *        <p>
         *        When Amazon Cognito emails your users, it uses its built-in email functionality. When you use the
         *        default option, Amazon Cognito allows only a limited number of emails each day for your user pool. For
         *        typical production environments, the default email limit is less than the required delivery volume. To
         *        achieve a higher delivery volume, specify DEVELOPER to use your Amazon SES email configuration.
         *        </p>
         *        <p>
         *        To look up the email delivery limit for the default option, see <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/limits.html">Limits in </a> in the <i>
         *        Developer Guide</i>.
         *        </p>
         *        <p>
         *        The default FROM address is <code>no-reply@verificationemail.com</code>. To customize the FROM
         *        address, provide the Amazon Resource Name (ARN) of an Amazon SES verified email address for the
         *        <code>SourceArn</code> parameter.
         *        </p>
         *        <p>
         *        If EmailSendingAccount is COGNITO_DEFAULT, you can't use the following parameters:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        EmailVerificationMessage
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EmailVerificationSubject
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        InviteMessageTemplate.EmailMessage
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        InviteMessageTemplate.EmailSubject
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        VerificationMessageTemplate.EmailMessage
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        VerificationMessageTemplate.EmailMessageByLink
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        VerificationMessageTemplate.EmailSubject,
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        VerificationMessageTemplate.EmailSubjectByLink
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        DEVELOPER EmailSendingAccount is required.
         *        </p>
         *        </note></dd>
         *        <dt>DEVELOPER</dt>
         *        <dd>
         *        <p>
         *        When Amazon Cognito emails your users, it uses your Amazon SES configuration. Amazon Cognito calls
         *        Amazon SES on your behalf to send email from your verified email address. When you use this option,
         *        the email delivery limits are the same limits that apply to your Amazon SES verified email address in
         *        your Amazon Web Services account.
         *        </p>
         *        <p>
         *        If you use this option, you must provide the ARN of an Amazon SES verified email address for the
         *        <code>SourceArn</code> parameter.
         *        </p>
         *        <p>
         *        Before Amazon Cognito can email your users, it requires additional permissions to call Amazon SES on
         *        your behalf. When you update your user pool with this option, Amazon Cognito creates a
         *        <i>service-linked role</i>, which is a type of role, in your Amazon Web Services account. This role
         *        contains the permissions that allow to access Amazon SES and send email messages with your address.
         *        For more information about the service-linked role that Amazon Cognito creates, see <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/using-service-linked-roles.html">Using
         *        Service-Linked Roles for Amazon Cognito</a> in the <i>Amazon Cognito Developer Guide</i>.
         *        </p>
         *        </dd>
         * @see EmailSendingAccountType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EmailSendingAccountType
         */
        Builder emailSendingAccount(EmailSendingAccountType emailSendingAccount);

        /**
         * <p>
         * Either the sender’s email address or the sender’s name with their email address. For example,
         * <code>testuser@example.com</code> or <code>Test User &lt;testuser@example.com&gt;</code>. This address
         * appears before the body of the email.
         * </p>
         * 
         * @param from
         *        Either the sender’s email address or the sender’s name with their email address. For example,
         *        <code>testuser@example.com</code> or <code>Test User &lt;testuser@example.com&gt;</code>. This address
         *        appears before the body of the email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder from(String from);

        /**
         * <p>
         * The set of configuration rules that can be applied to emails sent using Amazon Simple Email Service. A
         * configuration set is applied to an email by including a reference to the configuration set in the headers of
         * the email. Once applied, all of the rules in that configuration set are applied to the email. Configuration
         * sets can be used to apply the following types of rules to emails:
         * </p>
         * <dl>
         * <dt>Event publishing</dt>
         * <dd>
         * <p>
         * Amazon Simple Email Service can track the number of send, delivery, open, click, bounce, and complaint events
         * for each email sent. Use event publishing to send information about these events to other Amazon Web Services
         * services such as and Amazon CloudWatch
         * </p>
         * </dd>
         * <dt>IP pool management</dt>
         * <dd>
         * <p>
         * When leasing dedicated IP addresses with Amazon Simple Email Service, you can create groups of IP addresses,
         * called dedicated IP pools. You can then associate the dedicated IP pools with configuration sets.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param configurationSet
         *        The set of configuration rules that can be applied to emails sent using Amazon Simple Email Service. A
         *        configuration set is applied to an email by including a reference to the configuration set in the
         *        headers of the email. Once applied, all of the rules in that configuration set are applied to the
         *        email. Configuration sets can be used to apply the following types of rules to emails: </p>
         *        <dl>
         *        <dt>Event publishing</dt>
         *        <dd>
         *        <p>
         *        Amazon Simple Email Service can track the number of send, delivery, open, click, bounce, and complaint
         *        events for each email sent. Use event publishing to send information about these events to other
         *        Amazon Web Services services such as and Amazon CloudWatch
         *        </p>
         *        </dd>
         *        <dt>IP pool management</dt>
         *        <dd>
         *        <p>
         *        When leasing dedicated IP addresses with Amazon Simple Email Service, you can create groups of IP
         *        addresses, called dedicated IP pools. You can then associate the dedicated IP pools with configuration
         *        sets.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationSet(String configurationSet);
    }

    static final class BuilderImpl implements Builder {
        private String sourceArn;

        private String replyToEmailAddress;

        private String emailSendingAccount;

        private String from;

        private String configurationSet;

        private BuilderImpl() {
        }

        private BuilderImpl(EmailConfigurationType model) {
            sourceArn(model.sourceArn);
            replyToEmailAddress(model.replyToEmailAddress);
            emailSendingAccount(model.emailSendingAccount);
            from(model.from);
            configurationSet(model.configurationSet);
        }

        public final String getSourceArn() {
            return sourceArn;
        }

        public final void setSourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
        }

        @Override
        public final Builder sourceArn(String sourceArn) {
            this.sourceArn = sourceArn;
            return this;
        }

        public final String getReplyToEmailAddress() {
            return replyToEmailAddress;
        }

        public final void setReplyToEmailAddress(String replyToEmailAddress) {
            this.replyToEmailAddress = replyToEmailAddress;
        }

        @Override
        public final Builder replyToEmailAddress(String replyToEmailAddress) {
            this.replyToEmailAddress = replyToEmailAddress;
            return this;
        }

        public final String getEmailSendingAccount() {
            return emailSendingAccount;
        }

        public final void setEmailSendingAccount(String emailSendingAccount) {
            this.emailSendingAccount = emailSendingAccount;
        }

        @Override
        public final Builder emailSendingAccount(String emailSendingAccount) {
            this.emailSendingAccount = emailSendingAccount;
            return this;
        }

        @Override
        public final Builder emailSendingAccount(EmailSendingAccountType emailSendingAccount) {
            this.emailSendingAccount(emailSendingAccount == null ? null : emailSendingAccount.toString());
            return this;
        }

        public final String getFrom() {
            return from;
        }

        public final void setFrom(String from) {
            this.from = from;
        }

        @Override
        public final Builder from(String from) {
            this.from = from;
            return this;
        }

        public final String getConfigurationSet() {
            return configurationSet;
        }

        public final void setConfigurationSet(String configurationSet) {
            this.configurationSet = configurationSet;
        }

        @Override
        public final Builder configurationSet(String configurationSet) {
            this.configurationSet = configurationSet;
            return this;
        }

        @Override
        public EmailConfigurationType build() {
            return new EmailConfigurationType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
