/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Initiates the authentication request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InitiateAuthRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<InitiateAuthRequest.Builder, InitiateAuthRequest> {
    private static final SdkField<String> AUTH_FLOW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthFlow").getter(getter(InitiateAuthRequest::authFlowAsString)).setter(setter(Builder::authFlow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthFlow").build()).build();

    private static final SdkField<Map<String, String>> AUTH_PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AuthParameters")
            .getter(getter(InitiateAuthRequest::authParameters))
            .setter(setter(Builder::authParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthParameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(InitiateAuthRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientId").getter(getter(InitiateAuthRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<AnalyticsMetadataType> ANALYTICS_METADATA_FIELD = SdkField
            .<AnalyticsMetadataType> builder(MarshallingType.SDK_POJO).memberName("AnalyticsMetadata")
            .getter(getter(InitiateAuthRequest::analyticsMetadata)).setter(setter(Builder::analyticsMetadata))
            .constructor(AnalyticsMetadataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalyticsMetadata").build()).build();

    private static final SdkField<UserContextDataType> USER_CONTEXT_DATA_FIELD = SdkField
            .<UserContextDataType> builder(MarshallingType.SDK_POJO).memberName("UserContextData")
            .getter(getter(InitiateAuthRequest::userContextData)).setter(setter(Builder::userContextData))
            .constructor(UserContextDataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserContextData").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTH_FLOW_FIELD,
            AUTH_PARAMETERS_FIELD, CLIENT_METADATA_FIELD, CLIENT_ID_FIELD, ANALYTICS_METADATA_FIELD, USER_CONTEXT_DATA_FIELD));

    private final String authFlow;

    private final Map<String, String> authParameters;

    private final Map<String, String> clientMetadata;

    private final String clientId;

    private final AnalyticsMetadataType analyticsMetadata;

    private final UserContextDataType userContextData;

    private InitiateAuthRequest(BuilderImpl builder) {
        super(builder);
        this.authFlow = builder.authFlow;
        this.authParameters = builder.authParameters;
        this.clientMetadata = builder.clientMetadata;
        this.clientId = builder.clientId;
        this.analyticsMetadata = builder.analyticsMetadata;
        this.userContextData = builder.userContextData;
    }

    /**
     * <p>
     * The authentication flow for this call to run. The API action will depend on this value. For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>REFRESH_TOKEN_AUTH</code> takes in a valid refresh token and returns new tokens.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_SRP_AUTH</code> takes in <code>USERNAME</code> and <code>SRP_A</code> and returns the SRP variables to
     * be used for next challenge execution.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_PASSWORD_AUTH</code> takes in <code>USERNAME</code> and <code>PASSWORD</code> and returns the next
     * challenge or tokens.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access token
     * and ID token by supplying a valid refresh token.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOM_AUTH</code>: Custom authentication flow.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_PASSWORD_AUTH</code>: Non-SRP authentication flow; USERNAME and PASSWORD are passed directly. If a
     * user migration Lambda trigger is set, this flow will invoke the user migration Lambda if it doesn't find the
     * USERNAME in the user pool.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>ADMIN_NO_SRP_AUTH</code> isn't a valid value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authFlow} will
     * return {@link AuthFlowType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #authFlowAsString}.
     * </p>
     * 
     * @return The authentication flow for this call to run. The API action will depend on this value. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>REFRESH_TOKEN_AUTH</code> takes in a valid refresh token and returns new tokens.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_SRP_AUTH</code> takes in <code>USERNAME</code> and <code>SRP_A</code> and returns the SRP
     *         variables to be used for next challenge execution.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_PASSWORD_AUTH</code> takes in <code>USERNAME</code> and <code>PASSWORD</code> and returns the
     *         next challenge or tokens.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access
     *         token and ID token by supplying a valid refresh token.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOM_AUTH</code>: Custom authentication flow.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_PASSWORD_AUTH</code>: Non-SRP authentication flow; USERNAME and PASSWORD are passed directly.
     *         If a user migration Lambda trigger is set, this flow will invoke the user migration Lambda if it doesn't
     *         find the USERNAME in the user pool.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>ADMIN_NO_SRP_AUTH</code> isn't a valid value.
     * @see AuthFlowType
     */
    public final AuthFlowType authFlow() {
        return AuthFlowType.fromValue(authFlow);
    }

    /**
     * <p>
     * The authentication flow for this call to run. The API action will depend on this value. For example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>REFRESH_TOKEN_AUTH</code> takes in a valid refresh token and returns new tokens.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_SRP_AUTH</code> takes in <code>USERNAME</code> and <code>SRP_A</code> and returns the SRP variables to
     * be used for next challenge execution.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_PASSWORD_AUTH</code> takes in <code>USERNAME</code> and <code>PASSWORD</code> and returns the next
     * challenge or tokens.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Valid values include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access token
     * and ID token by supplying a valid refresh token.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CUSTOM_AUTH</code>: Custom authentication flow.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>USER_PASSWORD_AUTH</code>: Non-SRP authentication flow; USERNAME and PASSWORD are passed directly. If a
     * user migration Lambda trigger is set, this flow will invoke the user migration Lambda if it doesn't find the
     * USERNAME in the user pool.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>ADMIN_NO_SRP_AUTH</code> isn't a valid value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authFlow} will
     * return {@link AuthFlowType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #authFlowAsString}.
     * </p>
     * 
     * @return The authentication flow for this call to run. The API action will depend on this value. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>REFRESH_TOKEN_AUTH</code> takes in a valid refresh token and returns new tokens.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_SRP_AUTH</code> takes in <code>USERNAME</code> and <code>SRP_A</code> and returns the SRP
     *         variables to be used for next challenge execution.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_PASSWORD_AUTH</code> takes in <code>USERNAME</code> and <code>PASSWORD</code> and returns the
     *         next challenge or tokens.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Valid values include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access
     *         token and ID token by supplying a valid refresh token.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CUSTOM_AUTH</code>: Custom authentication flow.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>USER_PASSWORD_AUTH</code>: Non-SRP authentication flow; USERNAME and PASSWORD are passed directly.
     *         If a user migration Lambda trigger is set, this flow will invoke the user migration Lambda if it doesn't
     *         find the USERNAME in the user pool.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <code>ADMIN_NO_SRP_AUTH</code> isn't a valid value.
     * @see AuthFlowType
     */
    public final String authFlowAsString() {
        return authFlow;
    }

    /**
     * For responses, this returns true if the service returned a value for the AuthParameters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAuthParameters() {
        return authParameters != null && !(authParameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The authentication parameters. These are inputs corresponding to the <code>AuthFlow</code> that you're invoking.
     * The required values depend on the value of <code>AuthFlow</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>USER_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SRP_A</code> (required),
     * <code>SECRET_HASH</code> (required if the app client is configured with a client secret), <code>DEVICE_KEY</code>
     * .
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>REFRESH_TOKEN_AUTH/REFRESH_TOKEN</code>: <code>REFRESH_TOKEN</code> (required),
     * <code>SECRET_HASH</code> (required if the app client is configured with a client secret), <code>DEVICE_KEY</code>
     * .
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>CUSTOM_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app client is
     * configured with client secret), <code>DEVICE_KEY</code>. To start the authentication flow with password
     * verification, include <code>ChallengeName: SRP_A</code> and <code>SRP_A: (The SRP_A Value)</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAuthParameters} method.
     * </p>
     * 
     * @return The authentication parameters. These are inputs corresponding to the <code>AuthFlow</code> that you're
     *         invoking. The required values depend on the value of <code>AuthFlow</code>:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <code>USER_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SRP_A</code> (required),
     *         <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
     *         <code>DEVICE_KEY</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>REFRESH_TOKEN_AUTH/REFRESH_TOKEN</code>: <code>REFRESH_TOKEN</code> (required),
     *         <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
     *         <code>DEVICE_KEY</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>CUSTOM_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app client
     *         is configured with client secret), <code>DEVICE_KEY</code>. To start the authentication flow with
     *         password verification, include <code>ChallengeName: SRP_A</code> and
     *         <code>SRP_A: (The SRP_A Value)</code>.
     *         </p>
     *         </li>
     */
    public final Map<String, String> authParameters() {
        return authParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for certain custom workflows that this action
     * triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the InitiateAuth
     * API action, Amazon Cognito invokes the Lambda functions that are specified for various triggers. The
     * ClientMetadata value is passed as input to the functions for only the following triggers:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pre signup
     * </p>
     * </li>
     * <li>
     * <p>
     * Pre authentication
     * </p>
     * </li>
     * <li>
     * <p>
     * User migration
     * </p>
     * </li>
     * </ul>
     * <p>
     * When Amazon Cognito invokes the functions for these triggers, it passes a JSON payload, which the function
     * receives as input. This payload contains a <code>validationData</code> attribute, which provides the data that
     * you assigned to the ClientMetadata parameter in your InitiateAuth request. In your function code in Lambda, you
     * can process the <code>validationData</code> value to enhance your workflow for your specific needs.
     * </p>
     * <p>
     * When you use the InitiateAuth API action, Amazon Cognito also invokes the functions for the following triggers,
     * but it doesn't provide the ClientMetadata value as input:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Post authentication
     * </p>
     * </li>
     * <li>
     * <p>
     * Custom message
     * </p>
     * </li>
     * <li>
     * <p>
     * Pre token generation
     * </p>
     * </li>
     * <li>
     * <p>
     * Create auth challenge
     * </p>
     * </li>
     * <li>
     * <p>
     * Define auth challenge
     * </p>
     * </li>
     * <li>
     * <p>
     * Verify auth challenge
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a user pool
     * to support custom workflows. If your user pool configuration doesn't include triggers, the ClientMetadata
     * parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Validate the ClientMetadata value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientMetadata} method.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for certain custom workflows that this
     *         action triggers.</p>
     *         <p>
     *         You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     *         InitiateAuth API action, Amazon Cognito invokes the Lambda functions that are specified for various
     *         triggers. The ClientMetadata value is passed as input to the functions for only the following triggers:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pre signup
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Pre authentication
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         User migration
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When Amazon Cognito invokes the functions for these triggers, it passes a JSON payload, which the
     *         function receives as input. This payload contains a <code>validationData</code> attribute, which provides
     *         the data that you assigned to the ClientMetadata parameter in your InitiateAuth request. In your function
     *         code in Lambda, you can process the <code>validationData</code> value to enhance your workflow for your
     *         specific needs.
     *         </p>
     *         <p>
     *         When you use the InitiateAuth API action, Amazon Cognito also invokes the functions for the following
     *         triggers, but it doesn't provide the ClientMetadata value as input:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Post authentication
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Custom message
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Pre token generation
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Create auth challenge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Define auth challenge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Verify auth challenge
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a
     *         user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
     *         ClientMetadata parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Validate the ClientMetadata value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    /**
     * <p>
     * The app client ID.
     * </p>
     * 
     * @return The app client ID.
     */
    public final String clientId() {
        return clientId;
    }

    /**
     * <p>
     * The Amazon Pinpoint analytics metadata for collecting metrics for <code>InitiateAuth</code> calls.
     * </p>
     * 
     * @return The Amazon Pinpoint analytics metadata for collecting metrics for <code>InitiateAuth</code> calls.
     */
    public final AnalyticsMetadataType analyticsMetadata() {
        return analyticsMetadata;
    }

    /**
     * <p>
     * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk of an
     * unexpected event by Amazon Cognito advanced security.
     * </p>
     * 
     * @return Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the
     *         risk of an unexpected event by Amazon Cognito advanced security.
     */
    public final UserContextDataType userContextData() {
        return userContextData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(authFlowAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAuthParameters() ? authParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(analyticsMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(userContextData());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InitiateAuthRequest)) {
            return false;
        }
        InitiateAuthRequest other = (InitiateAuthRequest) obj;
        return Objects.equals(authFlowAsString(), other.authFlowAsString()) && hasAuthParameters() == other.hasAuthParameters()
                && Objects.equals(authParameters(), other.authParameters()) && hasClientMetadata() == other.hasClientMetadata()
                && Objects.equals(clientMetadata(), other.clientMetadata()) && Objects.equals(clientId(), other.clientId())
                && Objects.equals(analyticsMetadata(), other.analyticsMetadata())
                && Objects.equals(userContextData(), other.userContextData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InitiateAuthRequest").add("AuthFlow", authFlowAsString())
                .add("AuthParameters", authParameters() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null)
                .add("ClientId", clientId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AnalyticsMetadata", analyticsMetadata()).add("UserContextData", userContextData()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AuthFlow":
            return Optional.ofNullable(clazz.cast(authFlowAsString()));
        case "AuthParameters":
            return Optional.ofNullable(clazz.cast(authParameters()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "AnalyticsMetadata":
            return Optional.ofNullable(clazz.cast(analyticsMetadata()));
        case "UserContextData":
            return Optional.ofNullable(clazz.cast(userContextData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InitiateAuthRequest, T> g) {
        return obj -> g.apply((InitiateAuthRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, InitiateAuthRequest> {
        /**
         * <p>
         * The authentication flow for this call to run. The API action will depend on this value. For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>REFRESH_TOKEN_AUTH</code> takes in a valid refresh token and returns new tokens.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_SRP_AUTH</code> takes in <code>USERNAME</code> and <code>SRP_A</code> and returns the SRP
         * variables to be used for next challenge execution.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_PASSWORD_AUTH</code> takes in <code>USERNAME</code> and <code>PASSWORD</code> and returns the next
         * challenge or tokens.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access
         * token and ID token by supplying a valid refresh token.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOM_AUTH</code>: Custom authentication flow.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_PASSWORD_AUTH</code>: Non-SRP authentication flow; USERNAME and PASSWORD are passed directly. If a
         * user migration Lambda trigger is set, this flow will invoke the user migration Lambda if it doesn't find the
         * USERNAME in the user pool.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>ADMIN_NO_SRP_AUTH</code> isn't a valid value.
         * </p>
         * 
         * @param authFlow
         *        The authentication flow for this call to run. The API action will depend on this value. For
         *        example:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>REFRESH_TOKEN_AUTH</code> takes in a valid refresh token and returns new tokens.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_SRP_AUTH</code> takes in <code>USERNAME</code> and <code>SRP_A</code> and returns the SRP
         *        variables to be used for next challenge execution.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_PASSWORD_AUTH</code> takes in <code>USERNAME</code> and <code>PASSWORD</code> and returns
         *        the next challenge or tokens.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the
         *        access token and ID token by supplying a valid refresh token.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOM_AUTH</code>: Custom authentication flow.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_PASSWORD_AUTH</code>: Non-SRP authentication flow; USERNAME and PASSWORD are passed
         *        directly. If a user migration Lambda trigger is set, this flow will invoke the user migration Lambda
         *        if it doesn't find the USERNAME in the user pool.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>ADMIN_NO_SRP_AUTH</code> isn't a valid value.
         * @see AuthFlowType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthFlowType
         */
        Builder authFlow(String authFlow);

        /**
         * <p>
         * The authentication flow for this call to run. The API action will depend on this value. For example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>REFRESH_TOKEN_AUTH</code> takes in a valid refresh token and returns new tokens.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_SRP_AUTH</code> takes in <code>USERNAME</code> and <code>SRP_A</code> and returns the SRP
         * variables to be used for next challenge execution.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_PASSWORD_AUTH</code> takes in <code>USERNAME</code> and <code>PASSWORD</code> and returns the next
         * challenge or tokens.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Valid values include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the access
         * token and ID token by supplying a valid refresh token.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CUSTOM_AUTH</code>: Custom authentication flow.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>USER_PASSWORD_AUTH</code>: Non-SRP authentication flow; USERNAME and PASSWORD are passed directly. If a
         * user migration Lambda trigger is set, this flow will invoke the user migration Lambda if it doesn't find the
         * USERNAME in the user pool.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <code>ADMIN_NO_SRP_AUTH</code> isn't a valid value.
         * </p>
         * 
         * @param authFlow
         *        The authentication flow for this call to run. The API action will depend on this value. For
         *        example:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>REFRESH_TOKEN_AUTH</code> takes in a valid refresh token and returns new tokens.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_SRP_AUTH</code> takes in <code>USERNAME</code> and <code>SRP_A</code> and returns the SRP
         *        variables to be used for next challenge execution.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_PASSWORD_AUTH</code> takes in <code>USERNAME</code> and <code>PASSWORD</code> and returns
         *        the next challenge or tokens.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Valid values include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>USER_SRP_AUTH</code>: Authentication flow for the Secure Remote Password (SRP) protocol.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REFRESH_TOKEN_AUTH</code>/<code>REFRESH_TOKEN</code>: Authentication flow for refreshing the
         *        access token and ID token by supplying a valid refresh token.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CUSTOM_AUTH</code>: Custom authentication flow.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>USER_PASSWORD_AUTH</code>: Non-SRP authentication flow; USERNAME and PASSWORD are passed
         *        directly. If a user migration Lambda trigger is set, this flow will invoke the user migration Lambda
         *        if it doesn't find the USERNAME in the user pool.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <code>ADMIN_NO_SRP_AUTH</code> isn't a valid value.
         * @see AuthFlowType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthFlowType
         */
        Builder authFlow(AuthFlowType authFlow);

        /**
         * <p>
         * The authentication parameters. These are inputs corresponding to the <code>AuthFlow</code> that you're
         * invoking. The required values depend on the value of <code>AuthFlow</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>USER_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SRP_A</code> (required),
         * <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
         * <code>DEVICE_KEY</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>REFRESH_TOKEN_AUTH/REFRESH_TOKEN</code>: <code>REFRESH_TOKEN</code> (required),
         * <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
         * <code>DEVICE_KEY</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>CUSTOM_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app client is
         * configured with client secret), <code>DEVICE_KEY</code>. To start the authentication flow with password
         * verification, include <code>ChallengeName: SRP_A</code> and <code>SRP_A: (The SRP_A Value)</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param authParameters
         *        The authentication parameters. These are inputs corresponding to the <code>AuthFlow</code> that you're
         *        invoking. The required values depend on the value of <code>AuthFlow</code>:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <code>USER_SRP_AUTH</code>: <code>USERNAME</code> (required), <code>SRP_A</code> (required),
         *        <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
         *        <code>DEVICE_KEY</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>REFRESH_TOKEN_AUTH/REFRESH_TOKEN</code>: <code>REFRESH_TOKEN</code> (required),
         *        <code>SECRET_HASH</code> (required if the app client is configured with a client secret),
         *        <code>DEVICE_KEY</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>CUSTOM_AUTH</code>: <code>USERNAME</code> (required), <code>SECRET_HASH</code> (if app
         *        client is configured with client secret), <code>DEVICE_KEY</code>. To start the authentication flow
         *        with password verification, include <code>ChallengeName: SRP_A</code> and
         *        <code>SRP_A: (The SRP_A Value)</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authParameters(Map<String, String> authParameters);

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for certain custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         * InitiateAuth API action, Amazon Cognito invokes the Lambda functions that are specified for various triggers.
         * The ClientMetadata value is passed as input to the functions for only the following triggers:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pre signup
         * </p>
         * </li>
         * <li>
         * <p>
         * Pre authentication
         * </p>
         * </li>
         * <li>
         * <p>
         * User migration
         * </p>
         * </li>
         * </ul>
         * <p>
         * When Amazon Cognito invokes the functions for these triggers, it passes a JSON payload, which the function
         * receives as input. This payload contains a <code>validationData</code> attribute, which provides the data
         * that you assigned to the ClientMetadata parameter in your InitiateAuth request. In your function code in
         * Lambda, you can process the <code>validationData</code> value to enhance your workflow for your specific
         * needs.
         * </p>
         * <p>
         * When you use the InitiateAuth API action, Amazon Cognito also invokes the functions for the following
         * triggers, but it doesn't provide the ClientMetadata value as input:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Post authentication
         * </p>
         * </li>
         * <li>
         * <p>
         * Custom message
         * </p>
         * </li>
         * <li>
         * <p>
         * Pre token generation
         * </p>
         * </li>
         * <li>
         * <p>
         * Create auth challenge
         * </p>
         * </li>
         * <li>
         * <p>
         * Define auth challenge
         * </p>
         * </li>
         * <li>
         * <p>
         * Verify auth challenge
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a user
         * pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         * ClientMetadata parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Validate the ClientMetadata value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for certain custom workflows that this
         *        action triggers.</p>
         *        <p>
         *        You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         *        InitiateAuth API action, Amazon Cognito invokes the Lambda functions that are specified for various
         *        triggers. The ClientMetadata value is passed as input to the functions for only the following
         *        triggers:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pre signup
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Pre authentication
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        User migration
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When Amazon Cognito invokes the functions for these triggers, it passes a JSON payload, which the
         *        function receives as input. This payload contains a <code>validationData</code> attribute, which
         *        provides the data that you assigned to the ClientMetadata parameter in your InitiateAuth request. In
         *        your function code in Lambda, you can process the <code>validationData</code> value to enhance your
         *        workflow for your specific needs.
         *        </p>
         *        <p>
         *        When you use the InitiateAuth API action, Amazon Cognito also invokes the functions for the following
         *        triggers, but it doesn't provide the ClientMetadata value as input:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Post authentication
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Custom message
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Pre token generation
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Create auth challenge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Define auth challenge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Verify auth challenge
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a
         *        user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         *        ClientMetadata parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Validate the ClientMetadata value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        /**
         * <p>
         * The app client ID.
         * </p>
         * 
         * @param clientId
         *        The app client ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>InitiateAuth</code> calls.
         * </p>
         * 
         * @param analyticsMetadata
         *        The Amazon Pinpoint analytics metadata for collecting metrics for <code>InitiateAuth</code> calls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>InitiateAuth</code> calls.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalyticsMetadataType.Builder} avoiding
         * the need to create one manually via {@link AnalyticsMetadataType#builder()}.
         *
         * When the {@link Consumer} completes, {@link AnalyticsMetadataType.Builder#build()} is called immediately and
         * its result is passed to {@link #analyticsMetadata(AnalyticsMetadataType)}.
         * 
         * @param analyticsMetadata
         *        a consumer that will call methods on {@link AnalyticsMetadataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #analyticsMetadata(AnalyticsMetadataType)
         */
        default Builder analyticsMetadata(Consumer<AnalyticsMetadataType.Builder> analyticsMetadata) {
            return analyticsMetadata(AnalyticsMetadataType.builder().applyMutation(analyticsMetadata).build());
        }

        /**
         * <p>
         * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk
         * of an unexpected event by Amazon Cognito advanced security.
         * </p>
         * 
         * @param userContextData
         *        Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the
         *        risk of an unexpected event by Amazon Cognito advanced security.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userContextData(UserContextDataType userContextData);

        /**
         * <p>
         * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk
         * of an unexpected event by Amazon Cognito advanced security.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserContextDataType.Builder} avoiding the
         * need to create one manually via {@link UserContextDataType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UserContextDataType.Builder#build()} is called immediately and
         * its result is passed to {@link #userContextData(UserContextDataType)}.
         * 
         * @param userContextData
         *        a consumer that will call methods on {@link UserContextDataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userContextData(UserContextDataType)
         */
        default Builder userContextData(Consumer<UserContextDataType.Builder> userContextData) {
            return userContextData(UserContextDataType.builder().applyMutation(userContextData).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String authFlow;

        private Map<String, String> authParameters = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private String clientId;

        private AnalyticsMetadataType analyticsMetadata;

        private UserContextDataType userContextData;

        private BuilderImpl() {
        }

        private BuilderImpl(InitiateAuthRequest model) {
            super(model);
            authFlow(model.authFlow);
            authParameters(model.authParameters);
            clientMetadata(model.clientMetadata);
            clientId(model.clientId);
            analyticsMetadata(model.analyticsMetadata);
            userContextData(model.userContextData);
        }

        public final String getAuthFlow() {
            return authFlow;
        }

        public final void setAuthFlow(String authFlow) {
            this.authFlow = authFlow;
        }

        @Override
        public final Builder authFlow(String authFlow) {
            this.authFlow = authFlow;
            return this;
        }

        @Override
        public final Builder authFlow(AuthFlowType authFlow) {
            this.authFlow(authFlow == null ? null : authFlow.toString());
            return this;
        }

        public final Map<String, String> getAuthParameters() {
            if (authParameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return authParameters;
        }

        public final void setAuthParameters(Map<String, String> authParameters) {
            this.authParameters = AuthParametersTypeCopier.copy(authParameters);
        }

        @Override
        public final Builder authParameters(Map<String, String> authParameters) {
            this.authParameters = AuthParametersTypeCopier.copy(authParameters);
            return this;
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        public final String getClientId() {
            return clientId;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final AnalyticsMetadataType.Builder getAnalyticsMetadata() {
            return analyticsMetadata != null ? analyticsMetadata.toBuilder() : null;
        }

        public final void setAnalyticsMetadata(AnalyticsMetadataType.BuilderImpl analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata != null ? analyticsMetadata.build() : null;
        }

        @Override
        public final Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata;
            return this;
        }

        public final UserContextDataType.Builder getUserContextData() {
            return userContextData != null ? userContextData.toBuilder() : null;
        }

        public final void setUserContextData(UserContextDataType.BuilderImpl userContextData) {
            this.userContextData = userContextData != null ? userContextData.build() : null;
        }

        @Override
        public final Builder userContextData(UserContextDataType userContextData) {
            this.userContextData = userContextData;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public InitiateAuthRequest build() {
            return new InitiateAuthRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
