/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to create a user in the specified user pool.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdminCreateUserRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<AdminCreateUserRequest.Builder, AdminCreateUserRequest> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(AdminCreateUserRequest::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(AdminCreateUserRequest::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<List<AttributeType>> USER_ATTRIBUTES_FIELD = SdkField
            .<List<AttributeType>> builder(MarshallingType.LIST)
            .memberName("UserAttributes")
            .getter(getter(AdminCreateUserRequest::userAttributes))
            .setter(setter(Builder::userAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AttributeType>> VALIDATION_DATA_FIELD = SdkField
            .<List<AttributeType>> builder(MarshallingType.LIST)
            .memberName("ValidationData")
            .getter(getter(AdminCreateUserRequest::validationData))
            .setter(setter(Builder::validationData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidationData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TEMPORARY_PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemporaryPassword").getter(getter(AdminCreateUserRequest::temporaryPassword))
            .setter(setter(Builder::temporaryPassword))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemporaryPassword").build()).build();

    private static final SdkField<Boolean> FORCE_ALIAS_CREATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ForceAliasCreation").getter(getter(AdminCreateUserRequest::forceAliasCreation))
            .setter(setter(Builder::forceAliasCreation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForceAliasCreation").build())
            .build();

    private static final SdkField<String> MESSAGE_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageAction").getter(getter(AdminCreateUserRequest::messageActionAsString))
            .setter(setter(Builder::messageAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageAction").build()).build();

    private static final SdkField<List<String>> DESIRED_DELIVERY_MEDIUMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DesiredDeliveryMediums")
            .getter(getter(AdminCreateUserRequest::desiredDeliveryMediumsAsStrings))
            .setter(setter(Builder::desiredDeliveryMediumsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DesiredDeliveryMediums").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(AdminCreateUserRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            USERNAME_FIELD, USER_ATTRIBUTES_FIELD, VALIDATION_DATA_FIELD, TEMPORARY_PASSWORD_FIELD, FORCE_ALIAS_CREATION_FIELD,
            MESSAGE_ACTION_FIELD, DESIRED_DELIVERY_MEDIUMS_FIELD, CLIENT_METADATA_FIELD));

    private final String userPoolId;

    private final String username;

    private final List<AttributeType> userAttributes;

    private final List<AttributeType> validationData;

    private final String temporaryPassword;

    private final Boolean forceAliasCreation;

    private final String messageAction;

    private final List<String> desiredDeliveryMediums;

    private final Map<String, String> clientMetadata;

    private AdminCreateUserRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.username = builder.username;
        this.userAttributes = builder.userAttributes;
        this.validationData = builder.validationData;
        this.temporaryPassword = builder.temporaryPassword;
        this.forceAliasCreation = builder.forceAliasCreation;
        this.messageAction = builder.messageAction;
        this.desiredDeliveryMediums = builder.desiredDeliveryMediums;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The user pool ID for the user pool where the user will be created.
     * </p>
     * 
     * @return The user pool ID for the user pool where the user will be created.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The username for the user. Must be unique within the user pool. Must be a UTF-8 string between 1 and 128
     * characters. After the user is created, the username cannot be changed.
     * </p>
     * 
     * @return The username for the user. Must be unique within the user pool. Must be a UTF-8 string between 1 and 128
     *         characters. After the user is created, the username cannot be changed.
     */
    public final String username() {
        return username;
    }

    /**
     * Returns true if the UserAttributes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasUserAttributes() {
        return userAttributes != null && !(userAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of name-value pairs that contain user attributes and attribute values to be set for the user to be
     * created. You can create a user without specifying any attributes other than <code>Username</code>. However, any
     * attributes that you specify as required (when creating a user pool or in the <b>Attributes</b> tab of the
     * console) must be supplied either by you (in your call to <code>AdminCreateUser</code>) or by the user (when he or
     * she signs up in response to your welcome message).
     * </p>
     * <p>
     * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     * </p>
     * <p>
     * To send a message inviting the user to sign up, you must specify the user's email address or phone number. This
     * can be done in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito console for managing
     * your user pools.
     * </p>
     * <p>
     * In your call to <code>AdminCreateUser</code>, you can set the <code>email_verified</code> attribute to
     * <code>True</code>, and you can set the <code>phone_number_verified</code> attribute to <code>True</code>. (You
     * can also do this by calling <a href=
     * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html"
     * >AdminUpdateUserAttributes</a>.)
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>email</b>: The email address of the user to whom the message that contains the code and username will be sent.
     * Required if the <code>email_verified</code> attribute is set to <code>True</code>, or if <code>"EMAIL"</code> is
     * specified in the <code>DesiredDeliveryMediums</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>phone_number</b>: The phone number of the user to whom the message that contains the code and username will be
     * sent. Required if the <code>phone_number_verified</code> attribute is set to <code>True</code>, or if
     * <code>"SMS"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUserAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of name-value pairs that contain user attributes and attribute values to be set for the user to
     *         be created. You can create a user without specifying any attributes other than <code>Username</code>.
     *         However, any attributes that you specify as required (when creating a user pool or in the
     *         <b>Attributes</b> tab of the console) must be supplied either by you (in your call to
     *         <code>AdminCreateUser</code>) or by the user (when he or she signs up in response to your welcome
     *         message).</p>
     *         <p>
     *         For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     *         </p>
     *         <p>
     *         To send a message inviting the user to sign up, you must specify the user's email address or phone
     *         number. This can be done in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito
     *         console for managing your user pools.
     *         </p>
     *         <p>
     *         In your call to <code>AdminCreateUser</code>, you can set the <code>email_verified</code> attribute to
     *         <code>True</code>, and you can set the <code>phone_number_verified</code> attribute to <code>True</code>.
     *         (You can also do this by calling <a href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html"
     *         >AdminUpdateUserAttributes</a>.)
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>email</b>: The email address of the user to whom the message that contains the code and username will
     *         be sent. Required if the <code>email_verified</code> attribute is set to <code>True</code>, or if
     *         <code>"EMAIL"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>phone_number</b>: The phone number of the user to whom the message that contains the code and username
     *         will be sent. Required if the <code>phone_number_verified</code> attribute is set to <code>True</code>,
     *         or if <code>"SMS"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
     *         </p>
     *         </li>
     */
    public final List<AttributeType> userAttributes() {
        return userAttributes;
    }

    /**
     * Returns true if the ValidationData property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasValidationData() {
        return validationData != null && !(validationData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The user's validation data. This is an array of name-value pairs that contain user attributes and attribute
     * values that you can use for custom validation, such as restricting the types of user accounts that can be
     * registered. For example, you might choose to allow or disallow user sign-up based on the user's domain.
     * </p>
     * <p>
     * To configure custom validation, you must create a Pre Sign-up Lambda trigger for the user pool as described in
     * the Amazon Cognito Developer Guide. The Lambda trigger receives the validation data and uses it in the validation
     * process.
     * </p>
     * <p>
     * The user's validation data is not persisted.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValidationData()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The user's validation data. This is an array of name-value pairs that contain user attributes and
     *         attribute values that you can use for custom validation, such as restricting the types of user accounts
     *         that can be registered. For example, you might choose to allow or disallow user sign-up based on the
     *         user's domain.</p>
     *         <p>
     *         To configure custom validation, you must create a Pre Sign-up Lambda trigger for the user pool as
     *         described in the Amazon Cognito Developer Guide. The Lambda trigger receives the validation data and uses
     *         it in the validation process.
     *         </p>
     *         <p>
     *         The user's validation data is not persisted.
     */
    public final List<AttributeType> validationData() {
        return validationData;
    }

    /**
     * <p>
     * The user's temporary password. This password must conform to the password policy that you specified when you
     * created the user pool.
     * </p>
     * <p>
     * The temporary password is valid only once. To complete the Admin Create User flow, the user must enter the
     * temporary password in the sign-in page along with a new password to be used in all future sign-ins.
     * </p>
     * <p>
     * This parameter is not required. If you do not specify a value, Amazon Cognito generates one for you.
     * </p>
     * <p>
     * The temporary password can only be used until the user account expiration limit that you specified when you
     * created the user pool. To reset the account after that time limit, you must call <code>AdminCreateUser</code>
     * again, specifying <code>"RESEND"</code> for the <code>MessageAction</code> parameter.
     * </p>
     * 
     * @return The user's temporary password. This password must conform to the password policy that you specified when
     *         you created the user pool.</p>
     *         <p>
     *         The temporary password is valid only once. To complete the Admin Create User flow, the user must enter
     *         the temporary password in the sign-in page along with a new password to be used in all future sign-ins.
     *         </p>
     *         <p>
     *         This parameter is not required. If you do not specify a value, Amazon Cognito generates one for you.
     *         </p>
     *         <p>
     *         The temporary password can only be used until the user account expiration limit that you specified when
     *         you created the user pool. To reset the account after that time limit, you must call
     *         <code>AdminCreateUser</code> again, specifying <code>"RESEND"</code> for the <code>MessageAction</code>
     *         parameter.
     */
    public final String temporaryPassword() {
        return temporaryPassword;
    }

    /**
     * <p>
     * This parameter is only used if the <code>phone_number_verified</code> or <code>email_verified</code> attribute is
     * set to <code>True</code>. Otherwise, it is ignored.
     * </p>
     * <p>
     * If this parameter is set to <code>True</code> and the phone number or email address specified in the
     * UserAttributes parameter already exists as an alias with a different user, the API call will migrate the alias
     * from the previous user to the newly created user. The previous user will no longer be able to log in using that
     * alias.
     * </p>
     * <p>
     * If this parameter is set to <code>False</code>, the API throws an <code>AliasExistsException</code> error if the
     * alias already exists. The default value is <code>False</code>.
     * </p>
     * 
     * @return This parameter is only used if the <code>phone_number_verified</code> or <code>email_verified</code>
     *         attribute is set to <code>True</code>. Otherwise, it is ignored.</p>
     *         <p>
     *         If this parameter is set to <code>True</code> and the phone number or email address specified in the
     *         UserAttributes parameter already exists as an alias with a different user, the API call will migrate the
     *         alias from the previous user to the newly created user. The previous user will no longer be able to log
     *         in using that alias.
     *         </p>
     *         <p>
     *         If this parameter is set to <code>False</code>, the API throws an <code>AliasExistsException</code> error
     *         if the alias already exists. The default value is <code>False</code>.
     */
    public final Boolean forceAliasCreation() {
        return forceAliasCreation;
    }

    /**
     * <p>
     * Set to <code>"RESEND"</code> to resend the invitation message to a user that already exists and reset the
     * expiration limit on the user's account. Set to <code>"SUPPRESS"</code> to suppress sending the message. Only one
     * value can be specified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageAction}
     * will return {@link MessageActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageActionAsString}.
     * </p>
     * 
     * @return Set to <code>"RESEND"</code> to resend the invitation message to a user that already exists and reset the
     *         expiration limit on the user's account. Set to <code>"SUPPRESS"</code> to suppress sending the message.
     *         Only one value can be specified.
     * @see MessageActionType
     */
    public final MessageActionType messageAction() {
        return MessageActionType.fromValue(messageAction);
    }

    /**
     * <p>
     * Set to <code>"RESEND"</code> to resend the invitation message to a user that already exists and reset the
     * expiration limit on the user's account. Set to <code>"SUPPRESS"</code> to suppress sending the message. Only one
     * value can be specified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageAction}
     * will return {@link MessageActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageActionAsString}.
     * </p>
     * 
     * @return Set to <code>"RESEND"</code> to resend the invitation message to a user that already exists and reset the
     *         expiration limit on the user's account. Set to <code>"SUPPRESS"</code> to suppress sending the message.
     *         Only one value can be specified.
     * @see MessageActionType
     */
    public final String messageActionAsString() {
        return messageAction;
    }

    /**
     * <p>
     * Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify <code>"SMS"</code> if the
     * phone number will be used. The default value is <code>"SMS"</code>. More than one value can be specified.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDesiredDeliveryMediums()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify
     *         <code>"SMS"</code> if the phone number will be used. The default value is <code>"SMS"</code>. More than
     *         one value can be specified.
     */
    public final List<DeliveryMediumType> desiredDeliveryMediums() {
        return DeliveryMediumListTypeCopier.copyStringToEnum(desiredDeliveryMediums);
    }

    /**
     * Returns true if the DesiredDeliveryMediums property was specified by the sender (it may be empty), or false if
     * the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasDesiredDeliveryMediums() {
        return desiredDeliveryMediums != null && !(desiredDeliveryMediums instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify <code>"SMS"</code> if the
     * phone number will be used. The default value is <code>"SMS"</code>. More than one value can be specified.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDesiredDeliveryMediums()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify
     *         <code>"SMS"</code> if the phone number will be used. The default value is <code>"SMS"</code>. More than
     *         one value can be specified.
     */
    public final List<String> desiredDeliveryMediumsAsStrings() {
        return desiredDeliveryMediums;
    }

    /**
     * Returns true if the ClientMetadata property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the AdminCreateUser
     * API action, Amazon Cognito invokes the function that is assigned to the <i>pre sign-up</i> trigger. When Amazon
     * Cognito invokes this function, it passes a JSON payload, which the function receives as input. This payload
     * contains a <code>clientMetadata</code> attribute, which provides the data that you assigned to the ClientMetadata
     * parameter in your AdminCreateUser request. In your function code in Lambda, you can process the
     * <code>clientMetadata</code> value to enhance your workflow for your specific needs.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * >Customizing User Pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * Take the following limitations into consideration when you use the ClientMetadata parameter:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Cognito does not store the ClientMetadata value. This data is available only to Lambda triggers that are
     * assigned to a user pool to support custom workflows. If your user pool configuration does not include triggers,
     * the ClientMetadata parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Cognito does not validate the ClientMetadata value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Cognito does not encrypt the the ClientMetadata value, so don't use it to provide sensitive information.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasClientMetadata()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for any custom workflows that this action
     *         triggers. </p>
     *         <p>
     *         You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     *         AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the <i>pre
     *         sign-up</i> trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the
     *         function receives as input. This payload contains a <code>clientMetadata</code> attribute, which provides
     *         the data that you assigned to the ClientMetadata parameter in your AdminCreateUser request. In your
     *         function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your workflow
     *         for your specific needs.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         >Customizing User Pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         Take the following limitations into consideration when you use the ClientMetadata parameter:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Cognito does not store the ClientMetadata value. This data is available only to Lambda triggers
     *         that are assigned to a user pool to support custom workflows. If your user pool configuration does not
     *         include triggers, the ClientMetadata parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Cognito does not validate the ClientMetadata value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Cognito does not encrypt the the ClientMetadata value, so don't use it to provide sensitive
     *         information.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(hasUserAttributes() ? userAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasValidationData() ? validationData() : null);
        hashCode = 31 * hashCode + Objects.hashCode(temporaryPassword());
        hashCode = 31 * hashCode + Objects.hashCode(forceAliasCreation());
        hashCode = 31 * hashCode + Objects.hashCode(messageActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDesiredDeliveryMediums() ? desiredDeliveryMediumsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdminCreateUserRequest)) {
            return false;
        }
        AdminCreateUserRequest other = (AdminCreateUserRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(username(), other.username())
                && hasUserAttributes() == other.hasUserAttributes() && Objects.equals(userAttributes(), other.userAttributes())
                && hasValidationData() == other.hasValidationData() && Objects.equals(validationData(), other.validationData())
                && Objects.equals(temporaryPassword(), other.temporaryPassword())
                && Objects.equals(forceAliasCreation(), other.forceAliasCreation())
                && Objects.equals(messageActionAsString(), other.messageActionAsString())
                && hasDesiredDeliveryMediums() == other.hasDesiredDeliveryMediums()
                && Objects.equals(desiredDeliveryMediumsAsStrings(), other.desiredDeliveryMediumsAsStrings())
                && hasClientMetadata() == other.hasClientMetadata() && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdminCreateUserRequest").add("UserPoolId", userPoolId())
                .add("Username", username() == null ? null : "*** Sensitive Data Redacted ***")
                .add("UserAttributes", hasUserAttributes() ? userAttributes() : null)
                .add("ValidationData", hasValidationData() ? validationData() : null)
                .add("TemporaryPassword", temporaryPassword() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ForceAliasCreation", forceAliasCreation()).add("MessageAction", messageActionAsString())
                .add("DesiredDeliveryMediums", hasDesiredDeliveryMediums() ? desiredDeliveryMediumsAsStrings() : null)
                .add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "UserAttributes":
            return Optional.ofNullable(clazz.cast(userAttributes()));
        case "ValidationData":
            return Optional.ofNullable(clazz.cast(validationData()));
        case "TemporaryPassword":
            return Optional.ofNullable(clazz.cast(temporaryPassword()));
        case "ForceAliasCreation":
            return Optional.ofNullable(clazz.cast(forceAliasCreation()));
        case "MessageAction":
            return Optional.ofNullable(clazz.cast(messageActionAsString()));
        case "DesiredDeliveryMediums":
            return Optional.ofNullable(clazz.cast(desiredDeliveryMediumsAsStrings()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AdminCreateUserRequest, T> g) {
        return obj -> g.apply((AdminCreateUserRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, AdminCreateUserRequest> {
        /**
         * <p>
         * The user pool ID for the user pool where the user will be created.
         * </p>
         * 
         * @param userPoolId
         *        The user pool ID for the user pool where the user will be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The username for the user. Must be unique within the user pool. Must be a UTF-8 string between 1 and 128
         * characters. After the user is created, the username cannot be changed.
         * </p>
         * 
         * @param username
         *        The username for the user. Must be unique within the user pool. Must be a UTF-8 string between 1 and
         *        128 characters. After the user is created, the username cannot be changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * An array of name-value pairs that contain user attributes and attribute values to be set for the user to be
         * created. You can create a user without specifying any attributes other than <code>Username</code>. However,
         * any attributes that you specify as required (when creating a user pool or in the <b>Attributes</b> tab of the
         * console) must be supplied either by you (in your call to <code>AdminCreateUser</code>) or by the user (when
         * he or she signs up in response to your welcome message).
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * <p>
         * To send a message inviting the user to sign up, you must specify the user's email address or phone number.
         * This can be done in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito console for
         * managing your user pools.
         * </p>
         * <p>
         * In your call to <code>AdminCreateUser</code>, you can set the <code>email_verified</code> attribute to
         * <code>True</code>, and you can set the <code>phone_number_verified</code> attribute to <code>True</code>.
         * (You can also do this by calling <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html"
         * >AdminUpdateUserAttributes</a>.)
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>email</b>: The email address of the user to whom the message that contains the code and username will be
         * sent. Required if the <code>email_verified</code> attribute is set to <code>True</code>, or if
         * <code>"EMAIL"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>phone_number</b>: The phone number of the user to whom the message that contains the code and username
         * will be sent. Required if the <code>phone_number_verified</code> attribute is set to <code>True</code>, or if
         * <code>"SMS"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </li>
         * </ul>
         * 
         * @param userAttributes
         *        An array of name-value pairs that contain user attributes and attribute values to be set for the user
         *        to be created. You can create a user without specifying any attributes other than
         *        <code>Username</code>. However, any attributes that you specify as required (when creating a user pool
         *        or in the <b>Attributes</b> tab of the console) must be supplied either by you (in your call to
         *        <code>AdminCreateUser</code>) or by the user (when he or she signs up in response to your welcome
         *        message).</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         *        </p>
         *        <p>
         *        To send a message inviting the user to sign up, you must specify the user's email address or phone
         *        number. This can be done in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon
         *        Cognito console for managing your user pools.
         *        </p>
         *        <p>
         *        In your call to <code>AdminCreateUser</code>, you can set the <code>email_verified</code> attribute to
         *        <code>True</code>, and you can set the <code>phone_number_verified</code> attribute to
         *        <code>True</code>. (You can also do this by calling <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html"
         *        >AdminUpdateUserAttributes</a>.)
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>email</b>: The email address of the user to whom the message that contains the code and username
         *        will be sent. Required if the <code>email_verified</code> attribute is set to <code>True</code>, or if
         *        <code>"EMAIL"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>phone_number</b>: The phone number of the user to whom the message that contains the code and
         *        username will be sent. Required if the <code>phone_number_verified</code> attribute is set to
         *        <code>True</code>, or if <code>"SMS"</code> is specified in the <code>DesiredDeliveryMediums</code>
         *        parameter.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(Collection<AttributeType> userAttributes);

        /**
         * <p>
         * An array of name-value pairs that contain user attributes and attribute values to be set for the user to be
         * created. You can create a user without specifying any attributes other than <code>Username</code>. However,
         * any attributes that you specify as required (when creating a user pool or in the <b>Attributes</b> tab of the
         * console) must be supplied either by you (in your call to <code>AdminCreateUser</code>) or by the user (when
         * he or she signs up in response to your welcome message).
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * <p>
         * To send a message inviting the user to sign up, you must specify the user's email address or phone number.
         * This can be done in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito console for
         * managing your user pools.
         * </p>
         * <p>
         * In your call to <code>AdminCreateUser</code>, you can set the <code>email_verified</code> attribute to
         * <code>True</code>, and you can set the <code>phone_number_verified</code> attribute to <code>True</code>.
         * (You can also do this by calling <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html"
         * >AdminUpdateUserAttributes</a>.)
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>email</b>: The email address of the user to whom the message that contains the code and username will be
         * sent. Required if the <code>email_verified</code> attribute is set to <code>True</code>, or if
         * <code>"EMAIL"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>phone_number</b>: The phone number of the user to whom the message that contains the code and username
         * will be sent. Required if the <code>phone_number_verified</code> attribute is set to <code>True</code>, or if
         * <code>"SMS"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </li>
         * </ul>
         * 
         * @param userAttributes
         *        An array of name-value pairs that contain user attributes and attribute values to be set for the user
         *        to be created. You can create a user without specifying any attributes other than
         *        <code>Username</code>. However, any attributes that you specify as required (when creating a user pool
         *        or in the <b>Attributes</b> tab of the console) must be supplied either by you (in your call to
         *        <code>AdminCreateUser</code>) or by the user (when he or she signs up in response to your welcome
         *        message).</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         *        </p>
         *        <p>
         *        To send a message inviting the user to sign up, you must specify the user's email address or phone
         *        number. This can be done in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon
         *        Cognito console for managing your user pools.
         *        </p>
         *        <p>
         *        In your call to <code>AdminCreateUser</code>, you can set the <code>email_verified</code> attribute to
         *        <code>True</code>, and you can set the <code>phone_number_verified</code> attribute to
         *        <code>True</code>. (You can also do this by calling <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html"
         *        >AdminUpdateUserAttributes</a>.)
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>email</b>: The email address of the user to whom the message that contains the code and username
         *        will be sent. Required if the <code>email_verified</code> attribute is set to <code>True</code>, or if
         *        <code>"EMAIL"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>phone_number</b>: The phone number of the user to whom the message that contains the code and
         *        username will be sent. Required if the <code>phone_number_verified</code> attribute is set to
         *        <code>True</code>, or if <code>"SMS"</code> is specified in the <code>DesiredDeliveryMediums</code>
         *        parameter.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(AttributeType... userAttributes);

        /**
         * <p>
         * An array of name-value pairs that contain user attributes and attribute values to be set for the user to be
         * created. You can create a user without specifying any attributes other than <code>Username</code>. However,
         * any attributes that you specify as required (when creating a user pool or in the <b>Attributes</b> tab of the
         * console) must be supplied either by you (in your call to <code>AdminCreateUser</code>) or by the user (when
         * he or she signs up in response to your welcome message).
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * <p>
         * To send a message inviting the user to sign up, you must specify the user's email address or phone number.
         * This can be done in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito console for
         * managing your user pools.
         * </p>
         * <p>
         * In your call to <code>AdminCreateUser</code>, you can set the <code>email_verified</code> attribute to
         * <code>True</code>, and you can set the <code>phone_number_verified</code> attribute to <code>True</code>.
         * (You can also do this by calling <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminUpdateUserAttributes.html"
         * >AdminUpdateUserAttributes</a>.)
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>email</b>: The email address of the user to whom the message that contains the code and username will be
         * sent. Required if the <code>email_verified</code> attribute is set to <code>True</code>, or if
         * <code>"EMAIL"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>phone_number</b>: The phone number of the user to whom the message that contains the code and username
         * will be sent. Required if the <code>phone_number_verified</code> attribute is set to <code>True</code>, or if
         * <code>"SMS"</code> is specified in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link List<AttributeType>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeType>.Builder#build()} is called immediately and
         * its result is passed to {@link #userAttributes(List<AttributeType>)}.
         * 
         * @param userAttributes
         *        a consumer that will call methods on {@link List<AttributeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAttributes(List<AttributeType>)
         */
        Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes);

        /**
         * <p>
         * The user's validation data. This is an array of name-value pairs that contain user attributes and attribute
         * values that you can use for custom validation, such as restricting the types of user accounts that can be
         * registered. For example, you might choose to allow or disallow user sign-up based on the user's domain.
         * </p>
         * <p>
         * To configure custom validation, you must create a Pre Sign-up Lambda trigger for the user pool as described
         * in the Amazon Cognito Developer Guide. The Lambda trigger receives the validation data and uses it in the
         * validation process.
         * </p>
         * <p>
         * The user's validation data is not persisted.
         * </p>
         * 
         * @param validationData
         *        The user's validation data. This is an array of name-value pairs that contain user attributes and
         *        attribute values that you can use for custom validation, such as restricting the types of user
         *        accounts that can be registered. For example, you might choose to allow or disallow user sign-up based
         *        on the user's domain.</p>
         *        <p>
         *        To configure custom validation, you must create a Pre Sign-up Lambda trigger for the user pool as
         *        described in the Amazon Cognito Developer Guide. The Lambda trigger receives the validation data and
         *        uses it in the validation process.
         *        </p>
         *        <p>
         *        The user's validation data is not persisted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(Collection<AttributeType> validationData);

        /**
         * <p>
         * The user's validation data. This is an array of name-value pairs that contain user attributes and attribute
         * values that you can use for custom validation, such as restricting the types of user accounts that can be
         * registered. For example, you might choose to allow or disallow user sign-up based on the user's domain.
         * </p>
         * <p>
         * To configure custom validation, you must create a Pre Sign-up Lambda trigger for the user pool as described
         * in the Amazon Cognito Developer Guide. The Lambda trigger receives the validation data and uses it in the
         * validation process.
         * </p>
         * <p>
         * The user's validation data is not persisted.
         * </p>
         * 
         * @param validationData
         *        The user's validation data. This is an array of name-value pairs that contain user attributes and
         *        attribute values that you can use for custom validation, such as restricting the types of user
         *        accounts that can be registered. For example, you might choose to allow or disallow user sign-up based
         *        on the user's domain.</p>
         *        <p>
         *        To configure custom validation, you must create a Pre Sign-up Lambda trigger for the user pool as
         *        described in the Amazon Cognito Developer Guide. The Lambda trigger receives the validation data and
         *        uses it in the validation process.
         *        </p>
         *        <p>
         *        The user's validation data is not persisted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(AttributeType... validationData);

        /**
         * <p>
         * The user's validation data. This is an array of name-value pairs that contain user attributes and attribute
         * values that you can use for custom validation, such as restricting the types of user accounts that can be
         * registered. For example, you might choose to allow or disallow user sign-up based on the user's domain.
         * </p>
         * <p>
         * To configure custom validation, you must create a Pre Sign-up Lambda trigger for the user pool as described
         * in the Amazon Cognito Developer Guide. The Lambda trigger receives the validation data and uses it in the
         * validation process.
         * </p>
         * <p>
         * The user's validation data is not persisted.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeType>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeType>.Builder#build()} is called immediately and
         * its result is passed to {@link #validationData(List<AttributeType>)}.
         * 
         * @param validationData
         *        a consumer that will call methods on {@link List<AttributeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationData(List<AttributeType>)
         */
        Builder validationData(Consumer<AttributeType.Builder>... validationData);

        /**
         * <p>
         * The user's temporary password. This password must conform to the password policy that you specified when you
         * created the user pool.
         * </p>
         * <p>
         * The temporary password is valid only once. To complete the Admin Create User flow, the user must enter the
         * temporary password in the sign-in page along with a new password to be used in all future sign-ins.
         * </p>
         * <p>
         * This parameter is not required. If you do not specify a value, Amazon Cognito generates one for you.
         * </p>
         * <p>
         * The temporary password can only be used until the user account expiration limit that you specified when you
         * created the user pool. To reset the account after that time limit, you must call <code>AdminCreateUser</code>
         * again, specifying <code>"RESEND"</code> for the <code>MessageAction</code> parameter.
         * </p>
         * 
         * @param temporaryPassword
         *        The user's temporary password. This password must conform to the password policy that you specified
         *        when you created the user pool.</p>
         *        <p>
         *        The temporary password is valid only once. To complete the Admin Create User flow, the user must enter
         *        the temporary password in the sign-in page along with a new password to be used in all future
         *        sign-ins.
         *        </p>
         *        <p>
         *        This parameter is not required. If you do not specify a value, Amazon Cognito generates one for you.
         *        </p>
         *        <p>
         *        The temporary password can only be used until the user account expiration limit that you specified
         *        when you created the user pool. To reset the account after that time limit, you must call
         *        <code>AdminCreateUser</code> again, specifying <code>"RESEND"</code> for the
         *        <code>MessageAction</code> parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder temporaryPassword(String temporaryPassword);

        /**
         * <p>
         * This parameter is only used if the <code>phone_number_verified</code> or <code>email_verified</code>
         * attribute is set to <code>True</code>. Otherwise, it is ignored.
         * </p>
         * <p>
         * If this parameter is set to <code>True</code> and the phone number or email address specified in the
         * UserAttributes parameter already exists as an alias with a different user, the API call will migrate the
         * alias from the previous user to the newly created user. The previous user will no longer be able to log in
         * using that alias.
         * </p>
         * <p>
         * If this parameter is set to <code>False</code>, the API throws an <code>AliasExistsException</code> error if
         * the alias already exists. The default value is <code>False</code>.
         * </p>
         * 
         * @param forceAliasCreation
         *        This parameter is only used if the <code>phone_number_verified</code> or <code>email_verified</code>
         *        attribute is set to <code>True</code>. Otherwise, it is ignored.</p>
         *        <p>
         *        If this parameter is set to <code>True</code> and the phone number or email address specified in the
         *        UserAttributes parameter already exists as an alias with a different user, the API call will migrate
         *        the alias from the previous user to the newly created user. The previous user will no longer be able
         *        to log in using that alias.
         *        </p>
         *        <p>
         *        If this parameter is set to <code>False</code>, the API throws an <code>AliasExistsException</code>
         *        error if the alias already exists. The default value is <code>False</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forceAliasCreation(Boolean forceAliasCreation);

        /**
         * <p>
         * Set to <code>"RESEND"</code> to resend the invitation message to a user that already exists and reset the
         * expiration limit on the user's account. Set to <code>"SUPPRESS"</code> to suppress sending the message. Only
         * one value can be specified.
         * </p>
         * 
         * @param messageAction
         *        Set to <code>"RESEND"</code> to resend the invitation message to a user that already exists and reset
         *        the expiration limit on the user's account. Set to <code>"SUPPRESS"</code> to suppress sending the
         *        message. Only one value can be specified.
         * @see MessageActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageActionType
         */
        Builder messageAction(String messageAction);

        /**
         * <p>
         * Set to <code>"RESEND"</code> to resend the invitation message to a user that already exists and reset the
         * expiration limit on the user's account. Set to <code>"SUPPRESS"</code> to suppress sending the message. Only
         * one value can be specified.
         * </p>
         * 
         * @param messageAction
         *        Set to <code>"RESEND"</code> to resend the invitation message to a user that already exists and reset
         *        the expiration limit on the user's account. Set to <code>"SUPPRESS"</code> to suppress sending the
         *        message. Only one value can be specified.
         * @see MessageActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageActionType
         */
        Builder messageAction(MessageActionType messageAction);

        /**
         * <p>
         * Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify <code>"SMS"</code> if
         * the phone number will be used. The default value is <code>"SMS"</code>. More than one value can be specified.
         * </p>
         * 
         * @param desiredDeliveryMediums
         *        Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify
         *        <code>"SMS"</code> if the phone number will be used. The default value is <code>"SMS"</code>. More
         *        than one value can be specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredDeliveryMediumsWithStrings(Collection<String> desiredDeliveryMediums);

        /**
         * <p>
         * Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify <code>"SMS"</code> if
         * the phone number will be used. The default value is <code>"SMS"</code>. More than one value can be specified.
         * </p>
         * 
         * @param desiredDeliveryMediums
         *        Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify
         *        <code>"SMS"</code> if the phone number will be used. The default value is <code>"SMS"</code>. More
         *        than one value can be specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredDeliveryMediumsWithStrings(String... desiredDeliveryMediums);

        /**
         * <p>
         * Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify <code>"SMS"</code> if
         * the phone number will be used. The default value is <code>"SMS"</code>. More than one value can be specified.
         * </p>
         * 
         * @param desiredDeliveryMediums
         *        Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify
         *        <code>"SMS"</code> if the phone number will be used. The default value is <code>"SMS"</code>. More
         *        than one value can be specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredDeliveryMediums(Collection<DeliveryMediumType> desiredDeliveryMediums);

        /**
         * <p>
         * Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify <code>"SMS"</code> if
         * the phone number will be used. The default value is <code>"SMS"</code>. More than one value can be specified.
         * </p>
         * 
         * @param desiredDeliveryMediums
         *        Specify <code>"EMAIL"</code> if email will be used to send the welcome message. Specify
         *        <code>"SMS"</code> if the phone number will be used. The default value is <code>"SMS"</code>. More
         *        than one value can be specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredDeliveryMediums(DeliveryMediumType... desiredDeliveryMediums);

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for any custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         * AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the <i>pre sign-up</i>
         * trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the function receives as
         * input. This payload contains a <code>clientMetadata</code> attribute, which provides the data that you
         * assigned to the ClientMetadata parameter in your AdminCreateUser request. In your function code in Lambda,
         * you can process the <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * >Customizing User Pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * Take the following limitations into consideration when you use the ClientMetadata parameter:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Cognito does not store the ClientMetadata value. This data is available only to Lambda triggers that
         * are assigned to a user pool to support custom workflows. If your user pool configuration does not include
         * triggers, the ClientMetadata parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Cognito does not validate the ClientMetadata value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Cognito does not encrypt the the ClientMetadata value, so don't use it to provide sensitive
         * information.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for any custom workflows that this
         *        action triggers. </p>
         *        <p>
         *        You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         *        AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the <i>pre
         *        sign-up</i> trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the
         *        function receives as input. This payload contains a <code>clientMetadata</code> attribute, which
         *        provides the data that you assigned to the ClientMetadata parameter in your AdminCreateUser request.
         *        In your function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your
         *        workflow for your specific needs.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        >Customizing User Pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        Take the following limitations into consideration when you use the ClientMetadata parameter:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Cognito does not store the ClientMetadata value. This data is available only to Lambda triggers
         *        that are assigned to a user pool to support custom workflows. If your user pool configuration does not
         *        include triggers, the ClientMetadata parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Cognito does not validate the ClientMetadata value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Cognito does not encrypt the the ClientMetadata value, so don't use it to provide sensitive
         *        information.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String username;

        private List<AttributeType> userAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<AttributeType> validationData = DefaultSdkAutoConstructList.getInstance();

        private String temporaryPassword;

        private Boolean forceAliasCreation;

        private String messageAction;

        private List<String> desiredDeliveryMediums = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdminCreateUserRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            username(model.username);
            userAttributes(model.userAttributes);
            validationData(model.validationData);
            temporaryPassword(model.temporaryPassword);
            forceAliasCreation(model.forceAliasCreation);
            messageAction(model.messageAction);
            desiredDeliveryMediumsWithStrings(model.desiredDeliveryMediums);
            clientMetadata(model.clientMetadata);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        public final String getUsername() {
            return username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        public final List<AttributeType.Builder> getUserAttributes() {
            List<AttributeType.Builder> result = AttributeListTypeCopier.copyToBuilder(this.userAttributes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder userAttributes(Collection<AttributeType> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copy(userAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(AttributeType... userAttributes) {
            userAttributes(Arrays.asList(userAttributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes) {
            userAttributes(Stream.of(userAttributes).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUserAttributes(Collection<AttributeType.BuilderImpl> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copyFromBuilder(userAttributes);
        }

        public final List<AttributeType.Builder> getValidationData() {
            List<AttributeType.Builder> result = AttributeListTypeCopier.copyToBuilder(this.validationData);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder validationData(Collection<AttributeType> validationData) {
            this.validationData = AttributeListTypeCopier.copy(validationData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(AttributeType... validationData) {
            validationData(Arrays.asList(validationData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(Consumer<AttributeType.Builder>... validationData) {
            validationData(Stream.of(validationData).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setValidationData(Collection<AttributeType.BuilderImpl> validationData) {
            this.validationData = AttributeListTypeCopier.copyFromBuilder(validationData);
        }

        public final String getTemporaryPassword() {
            return temporaryPassword;
        }

        @Override
        public final Builder temporaryPassword(String temporaryPassword) {
            this.temporaryPassword = temporaryPassword;
            return this;
        }

        public final void setTemporaryPassword(String temporaryPassword) {
            this.temporaryPassword = temporaryPassword;
        }

        public final Boolean getForceAliasCreation() {
            return forceAliasCreation;
        }

        @Override
        public final Builder forceAliasCreation(Boolean forceAliasCreation) {
            this.forceAliasCreation = forceAliasCreation;
            return this;
        }

        public final void setForceAliasCreation(Boolean forceAliasCreation) {
            this.forceAliasCreation = forceAliasCreation;
        }

        public final String getMessageAction() {
            return messageAction;
        }

        @Override
        public final Builder messageAction(String messageAction) {
            this.messageAction = messageAction;
            return this;
        }

        @Override
        public final Builder messageAction(MessageActionType messageAction) {
            this.messageAction(messageAction == null ? null : messageAction.toString());
            return this;
        }

        public final void setMessageAction(String messageAction) {
            this.messageAction = messageAction;
        }

        public final Collection<String> getDesiredDeliveryMediums() {
            if (desiredDeliveryMediums instanceof SdkAutoConstructList) {
                return null;
            }
            return desiredDeliveryMediums;
        }

        @Override
        public final Builder desiredDeliveryMediumsWithStrings(Collection<String> desiredDeliveryMediums) {
            this.desiredDeliveryMediums = DeliveryMediumListTypeCopier.copy(desiredDeliveryMediums);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder desiredDeliveryMediumsWithStrings(String... desiredDeliveryMediums) {
            desiredDeliveryMediumsWithStrings(Arrays.asList(desiredDeliveryMediums));
            return this;
        }

        @Override
        public final Builder desiredDeliveryMediums(Collection<DeliveryMediumType> desiredDeliveryMediums) {
            this.desiredDeliveryMediums = DeliveryMediumListTypeCopier.copyEnumToString(desiredDeliveryMediums);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder desiredDeliveryMediums(DeliveryMediumType... desiredDeliveryMediums) {
            desiredDeliveryMediums(Arrays.asList(desiredDeliveryMediums));
            return this;
        }

        public final void setDesiredDeliveryMediums(Collection<String> desiredDeliveryMediums) {
            this.desiredDeliveryMediums = DeliveryMediumListTypeCopier.copy(desiredDeliveryMediums);
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AdminCreateUserRequest build() {
            return new AdminCreateUserRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
