/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to create a user pool.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateUserPoolRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<CreateUserPoolRequest.Builder, CreateUserPoolRequest> {
    private static final SdkField<String> POOL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PoolName").getter(getter(CreateUserPoolRequest::poolName)).setter(setter(Builder::poolName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PoolName").build()).build();

    private static final SdkField<UserPoolPolicyType> POLICIES_FIELD = SdkField
            .<UserPoolPolicyType> builder(MarshallingType.SDK_POJO).memberName("Policies")
            .getter(getter(CreateUserPoolRequest::policies)).setter(setter(Builder::policies))
            .constructor(UserPoolPolicyType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policies").build()).build();

    private static final SdkField<LambdaConfigType> LAMBDA_CONFIG_FIELD = SdkField
            .<LambdaConfigType> builder(MarshallingType.SDK_POJO).memberName("LambdaConfig")
            .getter(getter(CreateUserPoolRequest::lambdaConfig)).setter(setter(Builder::lambdaConfig))
            .constructor(LambdaConfigType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LambdaConfig").build()).build();

    private static final SdkField<List<String>> AUTO_VERIFIED_ATTRIBUTES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AutoVerifiedAttributes")
            .getter(getter(CreateUserPoolRequest::autoVerifiedAttributesAsStrings))
            .setter(setter(Builder::autoVerifiedAttributesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoVerifiedAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ALIAS_ATTRIBUTES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AliasAttributes")
            .getter(getter(CreateUserPoolRequest::aliasAttributesAsStrings))
            .setter(setter(Builder::aliasAttributesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AliasAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> USERNAME_ATTRIBUTES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("UsernameAttributes")
            .getter(getter(CreateUserPoolRequest::usernameAttributesAsStrings))
            .setter(setter(Builder::usernameAttributesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsernameAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> SMS_VERIFICATION_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SmsVerificationMessage").getter(getter(CreateUserPoolRequest::smsVerificationMessage))
            .setter(setter(Builder::smsVerificationMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SmsVerificationMessage").build())
            .build();

    private static final SdkField<String> EMAIL_VERIFICATION_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EmailVerificationMessage").getter(getter(CreateUserPoolRequest::emailVerificationMessage))
            .setter(setter(Builder::emailVerificationMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmailVerificationMessage").build())
            .build();

    private static final SdkField<String> EMAIL_VERIFICATION_SUBJECT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EmailVerificationSubject").getter(getter(CreateUserPoolRequest::emailVerificationSubject))
            .setter(setter(Builder::emailVerificationSubject))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmailVerificationSubject").build())
            .build();

    private static final SdkField<VerificationMessageTemplateType> VERIFICATION_MESSAGE_TEMPLATE_FIELD = SdkField
            .<VerificationMessageTemplateType> builder(MarshallingType.SDK_POJO)
            .memberName("VerificationMessageTemplate")
            .getter(getter(CreateUserPoolRequest::verificationMessageTemplate))
            .setter(setter(Builder::verificationMessageTemplate))
            .constructor(VerificationMessageTemplateType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VerificationMessageTemplate")
                    .build()).build();

    private static final SdkField<String> SMS_AUTHENTICATION_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SmsAuthenticationMessage").getter(getter(CreateUserPoolRequest::smsAuthenticationMessage))
            .setter(setter(Builder::smsAuthenticationMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SmsAuthenticationMessage").build())
            .build();

    private static final SdkField<String> MFA_CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MfaConfiguration").getter(getter(CreateUserPoolRequest::mfaConfigurationAsString))
            .setter(setter(Builder::mfaConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MfaConfiguration").build()).build();

    private static final SdkField<UserAttributeUpdateSettingsType> USER_ATTRIBUTE_UPDATE_SETTINGS_FIELD = SdkField
            .<UserAttributeUpdateSettingsType> builder(MarshallingType.SDK_POJO)
            .memberName("UserAttributeUpdateSettings")
            .getter(getter(CreateUserPoolRequest::userAttributeUpdateSettings))
            .setter(setter(Builder::userAttributeUpdateSettings))
            .constructor(UserAttributeUpdateSettingsType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAttributeUpdateSettings")
                    .build()).build();

    private static final SdkField<DeviceConfigurationType> DEVICE_CONFIGURATION_FIELD = SdkField
            .<DeviceConfigurationType> builder(MarshallingType.SDK_POJO).memberName("DeviceConfiguration")
            .getter(getter(CreateUserPoolRequest::deviceConfiguration)).setter(setter(Builder::deviceConfiguration))
            .constructor(DeviceConfigurationType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceConfiguration").build())
            .build();

    private static final SdkField<EmailConfigurationType> EMAIL_CONFIGURATION_FIELD = SdkField
            .<EmailConfigurationType> builder(MarshallingType.SDK_POJO).memberName("EmailConfiguration")
            .getter(getter(CreateUserPoolRequest::emailConfiguration)).setter(setter(Builder::emailConfiguration))
            .constructor(EmailConfigurationType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmailConfiguration").build())
            .build();

    private static final SdkField<SmsConfigurationType> SMS_CONFIGURATION_FIELD = SdkField
            .<SmsConfigurationType> builder(MarshallingType.SDK_POJO).memberName("SmsConfiguration")
            .getter(getter(CreateUserPoolRequest::smsConfiguration)).setter(setter(Builder::smsConfiguration))
            .constructor(SmsConfigurationType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SmsConfiguration").build()).build();

    private static final SdkField<Map<String, String>> USER_POOL_TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("UserPoolTags")
            .getter(getter(CreateUserPoolRequest::userPoolTags))
            .setter(setter(Builder::userPoolTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolTags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<AdminCreateUserConfigType> ADMIN_CREATE_USER_CONFIG_FIELD = SdkField
            .<AdminCreateUserConfigType> builder(MarshallingType.SDK_POJO).memberName("AdminCreateUserConfig")
            .getter(getter(CreateUserPoolRequest::adminCreateUserConfig)).setter(setter(Builder::adminCreateUserConfig))
            .constructor(AdminCreateUserConfigType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdminCreateUserConfig").build())
            .build();

    private static final SdkField<List<SchemaAttributeType>> SCHEMA_FIELD = SdkField
            .<List<SchemaAttributeType>> builder(MarshallingType.LIST)
            .memberName("Schema")
            .getter(getter(CreateUserPoolRequest::schema))
            .setter(setter(Builder::schema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schema").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SchemaAttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(SchemaAttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<UserPoolAddOnsType> USER_POOL_ADD_ONS_FIELD = SdkField
            .<UserPoolAddOnsType> builder(MarshallingType.SDK_POJO).memberName("UserPoolAddOns")
            .getter(getter(CreateUserPoolRequest::userPoolAddOns)).setter(setter(Builder::userPoolAddOns))
            .constructor(UserPoolAddOnsType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolAddOns").build()).build();

    private static final SdkField<UsernameConfigurationType> USERNAME_CONFIGURATION_FIELD = SdkField
            .<UsernameConfigurationType> builder(MarshallingType.SDK_POJO).memberName("UsernameConfiguration")
            .getter(getter(CreateUserPoolRequest::usernameConfiguration)).setter(setter(Builder::usernameConfiguration))
            .constructor(UsernameConfigurationType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsernameConfiguration").build())
            .build();

    private static final SdkField<AccountRecoverySettingType> ACCOUNT_RECOVERY_SETTING_FIELD = SdkField
            .<AccountRecoverySettingType> builder(MarshallingType.SDK_POJO).memberName("AccountRecoverySetting")
            .getter(getter(CreateUserPoolRequest::accountRecoverySetting)).setter(setter(Builder::accountRecoverySetting))
            .constructor(AccountRecoverySettingType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountRecoverySetting").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POOL_NAME_FIELD,
            POLICIES_FIELD, LAMBDA_CONFIG_FIELD, AUTO_VERIFIED_ATTRIBUTES_FIELD, ALIAS_ATTRIBUTES_FIELD,
            USERNAME_ATTRIBUTES_FIELD, SMS_VERIFICATION_MESSAGE_FIELD, EMAIL_VERIFICATION_MESSAGE_FIELD,
            EMAIL_VERIFICATION_SUBJECT_FIELD, VERIFICATION_MESSAGE_TEMPLATE_FIELD, SMS_AUTHENTICATION_MESSAGE_FIELD,
            MFA_CONFIGURATION_FIELD, USER_ATTRIBUTE_UPDATE_SETTINGS_FIELD, DEVICE_CONFIGURATION_FIELD, EMAIL_CONFIGURATION_FIELD,
            SMS_CONFIGURATION_FIELD, USER_POOL_TAGS_FIELD, ADMIN_CREATE_USER_CONFIG_FIELD, SCHEMA_FIELD, USER_POOL_ADD_ONS_FIELD,
            USERNAME_CONFIGURATION_FIELD, ACCOUNT_RECOVERY_SETTING_FIELD));

    private final String poolName;

    private final UserPoolPolicyType policies;

    private final LambdaConfigType lambdaConfig;

    private final List<String> autoVerifiedAttributes;

    private final List<String> aliasAttributes;

    private final List<String> usernameAttributes;

    private final String smsVerificationMessage;

    private final String emailVerificationMessage;

    private final String emailVerificationSubject;

    private final VerificationMessageTemplateType verificationMessageTemplate;

    private final String smsAuthenticationMessage;

    private final String mfaConfiguration;

    private final UserAttributeUpdateSettingsType userAttributeUpdateSettings;

    private final DeviceConfigurationType deviceConfiguration;

    private final EmailConfigurationType emailConfiguration;

    private final SmsConfigurationType smsConfiguration;

    private final Map<String, String> userPoolTags;

    private final AdminCreateUserConfigType adminCreateUserConfig;

    private final List<SchemaAttributeType> schema;

    private final UserPoolAddOnsType userPoolAddOns;

    private final UsernameConfigurationType usernameConfiguration;

    private final AccountRecoverySettingType accountRecoverySetting;

    private CreateUserPoolRequest(BuilderImpl builder) {
        super(builder);
        this.poolName = builder.poolName;
        this.policies = builder.policies;
        this.lambdaConfig = builder.lambdaConfig;
        this.autoVerifiedAttributes = builder.autoVerifiedAttributes;
        this.aliasAttributes = builder.aliasAttributes;
        this.usernameAttributes = builder.usernameAttributes;
        this.smsVerificationMessage = builder.smsVerificationMessage;
        this.emailVerificationMessage = builder.emailVerificationMessage;
        this.emailVerificationSubject = builder.emailVerificationSubject;
        this.verificationMessageTemplate = builder.verificationMessageTemplate;
        this.smsAuthenticationMessage = builder.smsAuthenticationMessage;
        this.mfaConfiguration = builder.mfaConfiguration;
        this.userAttributeUpdateSettings = builder.userAttributeUpdateSettings;
        this.deviceConfiguration = builder.deviceConfiguration;
        this.emailConfiguration = builder.emailConfiguration;
        this.smsConfiguration = builder.smsConfiguration;
        this.userPoolTags = builder.userPoolTags;
        this.adminCreateUserConfig = builder.adminCreateUserConfig;
        this.schema = builder.schema;
        this.userPoolAddOns = builder.userPoolAddOns;
        this.usernameConfiguration = builder.usernameConfiguration;
        this.accountRecoverySetting = builder.accountRecoverySetting;
    }

    /**
     * <p>
     * A string used to name the user pool.
     * </p>
     * 
     * @return A string used to name the user pool.
     */
    public final String poolName() {
        return poolName;
    }

    /**
     * <p>
     * The policies associated with the new user pool.
     * </p>
     * 
     * @return The policies associated with the new user pool.
     */
    public final UserPoolPolicyType policies() {
        return policies;
    }

    /**
     * <p>
     * The Lambda trigger configuration information for the new user pool.
     * </p>
     * <note>
     * <p>
     * In a push model, event sources (such as Amazon S3 and custom applications) need permission to invoke a function.
     * So you must make an extra call to add permission for these event sources to invoke your Lambda function.
     * </p>
     * <p/>
     * <p>
     * For more information on using the Lambda API to add permission, see<a
     * href="https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html"> AddPermission </a>.
     * </p>
     * <p>
     * For adding permission using the CLI, see<a
     * href="https://docs.aws.amazon.com/cli/latest/reference/lambda/add-permission.html"> add-permission </a>.
     * </p>
     * </note>
     * 
     * @return The Lambda trigger configuration information for the new user pool.</p> <note>
     *         <p>
     *         In a push model, event sources (such as Amazon S3 and custom applications) need permission to invoke a
     *         function. So you must make an extra call to add permission for these event sources to invoke your Lambda
     *         function.
     *         </p>
     *         <p/>
     *         <p>
     *         For more information on using the Lambda API to add permission, see<a
     *         href="https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html"> AddPermission </a>.
     *         </p>
     *         <p>
     *         For adding permission using the CLI, see<a
     *         href="https://docs.aws.amazon.com/cli/latest/reference/lambda/add-permission.html"> add-permission </a>.
     *         </p>
     */
    public final LambdaConfigType lambdaConfig() {
        return lambdaConfig;
    }

    /**
     * <p>
     * The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAutoVerifiedAttributes} method.
     * </p>
     * 
     * @return The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
     */
    public final List<VerifiedAttributeType> autoVerifiedAttributes() {
        return VerifiedAttributesListTypeCopier.copyStringToEnum(autoVerifiedAttributes);
    }

    /**
     * For responses, this returns true if the service returned a value for the AutoVerifiedAttributes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAutoVerifiedAttributes() {
        return autoVerifiedAttributes != null && !(autoVerifiedAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAutoVerifiedAttributes} method.
     * </p>
     * 
     * @return The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
     */
    public final List<String> autoVerifiedAttributesAsStrings() {
        return autoVerifiedAttributes;
    }

    /**
     * <p>
     * Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>, <b>email</b>, or
     * <b>preferred_username</b>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAliasAttributes} method.
     * </p>
     * 
     * @return Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>, <b>email</b>,
     *         or <b>preferred_username</b>.
     */
    public final List<AliasAttributeType> aliasAttributes() {
        return AliasAttributesListTypeCopier.copyStringToEnum(aliasAttributes);
    }

    /**
     * For responses, this returns true if the service returned a value for the AliasAttributes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAliasAttributes() {
        return aliasAttributes != null && !(aliasAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>, <b>email</b>, or
     * <b>preferred_username</b>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAliasAttributes} method.
     * </p>
     * 
     * @return Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>, <b>email</b>,
     *         or <b>preferred_username</b>.
     */
    public final List<String> aliasAttributesAsStrings() {
        return aliasAttributes;
    }

    /**
     * <p>
     * Specifies whether a user can use an email address or phone number as a username when they sign up.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUsernameAttributes} method.
     * </p>
     * 
     * @return Specifies whether a user can use an email address or phone number as a username when they sign up.
     */
    public final List<UsernameAttributeType> usernameAttributes() {
        return UsernameAttributesListTypeCopier.copyStringToEnum(usernameAttributes);
    }

    /**
     * For responses, this returns true if the service returned a value for the UsernameAttributes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUsernameAttributes() {
        return usernameAttributes != null && !(usernameAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies whether a user can use an email address or phone number as a username when they sign up.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUsernameAttributes} method.
     * </p>
     * 
     * @return Specifies whether a user can use an email address or phone number as a username when they sign up.
     */
    public final List<String> usernameAttributesAsStrings() {
        return usernameAttributes;
    }

    /**
     * <p>
     * A string representing the SMS verification message.
     * </p>
     * 
     * @return A string representing the SMS verification message.
     */
    public final String smsVerificationMessage() {
        return smsVerificationMessage;
    }

    /**
     * <p>
     * A string representing the email verification message. EmailVerificationMessage is allowed only if <a href=
     * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount"
     * >EmailSendingAccount</a> is DEVELOPER.
     * </p>
     * 
     * @return A string representing the email verification message. EmailVerificationMessage is allowed only if <a
     *         href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount"
     *         >EmailSendingAccount</a> is DEVELOPER.
     */
    public final String emailVerificationMessage() {
        return emailVerificationMessage;
    }

    /**
     * <p>
     * A string representing the email verification subject. EmailVerificationSubject is allowed only if <a href=
     * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount"
     * >EmailSendingAccount</a> is DEVELOPER.
     * </p>
     * 
     * @return A string representing the email verification subject. EmailVerificationSubject is allowed only if <a
     *         href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount"
     *         >EmailSendingAccount</a> is DEVELOPER.
     */
    public final String emailVerificationSubject() {
        return emailVerificationSubject;
    }

    /**
     * <p>
     * The template for the verification message that the user sees when the app requests permission to access the
     * user's information.
     * </p>
     * 
     * @return The template for the verification message that the user sees when the app requests permission to access
     *         the user's information.
     */
    public final VerificationMessageTemplateType verificationMessageTemplate() {
        return verificationMessageTemplate;
    }

    /**
     * <p>
     * A string representing the SMS authentication message.
     * </p>
     * 
     * @return A string representing the SMS authentication message.
     */
    public final String smsAuthenticationMessage() {
        return smsAuthenticationMessage;
    }

    /**
     * <p>
     * Specifies MFA configuration details.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mfaConfiguration}
     * will return {@link UserPoolMfaType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #mfaConfigurationAsString}.
     * </p>
     * 
     * @return Specifies MFA configuration details.
     * @see UserPoolMfaType
     */
    public final UserPoolMfaType mfaConfiguration() {
        return UserPoolMfaType.fromValue(mfaConfiguration);
    }

    /**
     * <p>
     * Specifies MFA configuration details.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mfaConfiguration}
     * will return {@link UserPoolMfaType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #mfaConfigurationAsString}.
     * </p>
     * 
     * @return Specifies MFA configuration details.
     * @see UserPoolMfaType
     */
    public final String mfaConfigurationAsString() {
        return mfaConfiguration;
    }

    /**
     * <p>
     * The settings for updates to user attributes. These settings include the property
     * <code>AttributesRequireVerificationBeforeUpdate</code>, a user-pool setting that tells Amazon Cognito how to
     * handle changes to the value of your users' email address and phone number attributes. For more information, see
     * <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates"
     * > Verifying updates to to email addresses and phone numbers</a>.
     * </p>
     * 
     * @return The settings for updates to user attributes. These settings include the property
     *         <code>AttributesRequireVerificationBeforeUpdate</code>, a user-pool setting that tells Amazon Cognito how
     *         to handle changes to the value of your users' email address and phone number attributes. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates"
     *         > Verifying updates to to email addresses and phone numbers</a>.
     */
    public final UserAttributeUpdateSettingsType userAttributeUpdateSettings() {
        return userAttributeUpdateSettings;
    }

    /**
     * <p>
     * The device configuration.
     * </p>
     * 
     * @return The device configuration.
     */
    public final DeviceConfigurationType deviceConfiguration() {
        return deviceConfiguration;
    }

    /**
     * <p>
     * The email configuration of your user pool. The email configuration type sets your preferred sending method,
     * Amazon Web Services Region, and sender for messages from your user pool.
     * </p>
     * 
     * @return The email configuration of your user pool. The email configuration type sets your preferred sending
     *         method, Amazon Web Services Region, and sender for messages from your user pool.
     */
    public final EmailConfigurationType emailConfiguration() {
        return emailConfiguration;
    }

    /**
     * <p>
     * The SMS configuration with the settings that your Amazon Cognito user pool must use to send an SMS message from
     * your Amazon Web Services account through Amazon Simple Notification Service. To send SMS messages with Amazon SNS
     * in the Amazon Web Services Region that you want, the Amazon Cognito user pool uses an Identity and Access
     * Management (IAM) role in your Amazon Web Services account.
     * </p>
     * 
     * @return The SMS configuration with the settings that your Amazon Cognito user pool must use to send an SMS
     *         message from your Amazon Web Services account through Amazon Simple Notification Service. To send SMS
     *         messages with Amazon SNS in the Amazon Web Services Region that you want, the Amazon Cognito user pool
     *         uses an Identity and Access Management (IAM) role in your Amazon Web Services account.
     */
    public final SmsConfigurationType smsConfiguration() {
        return smsConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the UserPoolTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUserPoolTags() {
        return userPoolTags != null && !(userPoolTags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tag keys and values to assign to the user pool. A tag is a label that you can use to categorize and manage
     * user pools in different ways, such as by purpose, owner, environment, or other criteria.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUserPoolTags} method.
     * </p>
     * 
     * @return The tag keys and values to assign to the user pool. A tag is a label that you can use to categorize and
     *         manage user pools in different ways, such as by purpose, owner, environment, or other criteria.
     */
    public final Map<String, String> userPoolTags() {
        return userPoolTags;
    }

    /**
     * <p>
     * The configuration for <code>AdminCreateUser</code> requests.
     * </p>
     * 
     * @return The configuration for <code>AdminCreateUser</code> requests.
     */
    public final AdminCreateUserConfigType adminCreateUserConfig() {
        return adminCreateUserConfig;
    }

    /**
     * For responses, this returns true if the service returned a value for the Schema property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSchema() {
        return schema != null && !(schema instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of schema attributes for the new user pool. These attributes can be standard or custom attributes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSchema} method.
     * </p>
     * 
     * @return An array of schema attributes for the new user pool. These attributes can be standard or custom
     *         attributes.
     */
    public final List<SchemaAttributeType> schema() {
        return schema;
    }

    /**
     * <p>
     * Enables advanced security risk detection. Set the key <code>AdvancedSecurityMode</code> to the value "AUDIT".
     * </p>
     * 
     * @return Enables advanced security risk detection. Set the key <code>AdvancedSecurityMode</code> to the value
     *         "AUDIT".
     */
    public final UserPoolAddOnsType userPoolAddOns() {
        return userPoolAddOns;
    }

    /**
     * <p>
     * Case sensitivity on the username input for the selected sign-in option. For example, when case sensitivity is set
     * to <code>False</code>, users can sign in using either "username" or "Username". This configuration is immutable
     * once it has been set. For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UsernameConfigurationType.html"
     * >UsernameConfigurationType</a>.
     * </p>
     * 
     * @return Case sensitivity on the username input for the selected sign-in option. For example, when case
     *         sensitivity is set to <code>False</code>, users can sign in using either "username" or "Username". This
     *         configuration is immutable once it has been set. For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UsernameConfigurationType.html"
     *         >UsernameConfigurationType</a>.
     */
    public final UsernameConfigurationType usernameConfiguration() {
        return usernameConfiguration;
    }

    /**
     * <p>
     * The available verified method a user can use to recover their password when they call <code>ForgotPassword</code>
     * . You can use this setting to define a preferred method when a user has more than one method available. With this
     * setting, SMS doesn't qualify for a valid password recovery mechanism if the user also has SMS multi-factor
     * authentication (MFA) activated. In the absence of this setting, Amazon Cognito uses the legacy behavior to
     * determine the recovery method where SMS is preferred through email.
     * </p>
     * 
     * @return The available verified method a user can use to recover their password when they call
     *         <code>ForgotPassword</code>. You can use this setting to define a preferred method when a user has more
     *         than one method available. With this setting, SMS doesn't qualify for a valid password recovery mechanism
     *         if the user also has SMS multi-factor authentication (MFA) activated. In the absence of this setting,
     *         Amazon Cognito uses the legacy behavior to determine the recovery method where SMS is preferred through
     *         email.
     */
    public final AccountRecoverySettingType accountRecoverySetting() {
        return accountRecoverySetting;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(poolName());
        hashCode = 31 * hashCode + Objects.hashCode(policies());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasAutoVerifiedAttributes() ? autoVerifiedAttributesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAliasAttributes() ? aliasAttributesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUsernameAttributes() ? usernameAttributesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(smsVerificationMessage());
        hashCode = 31 * hashCode + Objects.hashCode(emailVerificationMessage());
        hashCode = 31 * hashCode + Objects.hashCode(emailVerificationSubject());
        hashCode = 31 * hashCode + Objects.hashCode(verificationMessageTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(smsAuthenticationMessage());
        hashCode = 31 * hashCode + Objects.hashCode(mfaConfigurationAsString());
        hashCode = 31 * hashCode + Objects.hashCode(userAttributeUpdateSettings());
        hashCode = 31 * hashCode + Objects.hashCode(deviceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(emailConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(smsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasUserPoolTags() ? userPoolTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(adminCreateUserConfig());
        hashCode = 31 * hashCode + Objects.hashCode(hasSchema() ? schema() : null);
        hashCode = 31 * hashCode + Objects.hashCode(userPoolAddOns());
        hashCode = 31 * hashCode + Objects.hashCode(usernameConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(accountRecoverySetting());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateUserPoolRequest)) {
            return false;
        }
        CreateUserPoolRequest other = (CreateUserPoolRequest) obj;
        return Objects.equals(poolName(), other.poolName()) && Objects.equals(policies(), other.policies())
                && Objects.equals(lambdaConfig(), other.lambdaConfig())
                && hasAutoVerifiedAttributes() == other.hasAutoVerifiedAttributes()
                && Objects.equals(autoVerifiedAttributesAsStrings(), other.autoVerifiedAttributesAsStrings())
                && hasAliasAttributes() == other.hasAliasAttributes()
                && Objects.equals(aliasAttributesAsStrings(), other.aliasAttributesAsStrings())
                && hasUsernameAttributes() == other.hasUsernameAttributes()
                && Objects.equals(usernameAttributesAsStrings(), other.usernameAttributesAsStrings())
                && Objects.equals(smsVerificationMessage(), other.smsVerificationMessage())
                && Objects.equals(emailVerificationMessage(), other.emailVerificationMessage())
                && Objects.equals(emailVerificationSubject(), other.emailVerificationSubject())
                && Objects.equals(verificationMessageTemplate(), other.verificationMessageTemplate())
                && Objects.equals(smsAuthenticationMessage(), other.smsAuthenticationMessage())
                && Objects.equals(mfaConfigurationAsString(), other.mfaConfigurationAsString())
                && Objects.equals(userAttributeUpdateSettings(), other.userAttributeUpdateSettings())
                && Objects.equals(deviceConfiguration(), other.deviceConfiguration())
                && Objects.equals(emailConfiguration(), other.emailConfiguration())
                && Objects.equals(smsConfiguration(), other.smsConfiguration()) && hasUserPoolTags() == other.hasUserPoolTags()
                && Objects.equals(userPoolTags(), other.userPoolTags())
                && Objects.equals(adminCreateUserConfig(), other.adminCreateUserConfig()) && hasSchema() == other.hasSchema()
                && Objects.equals(schema(), other.schema()) && Objects.equals(userPoolAddOns(), other.userPoolAddOns())
                && Objects.equals(usernameConfiguration(), other.usernameConfiguration())
                && Objects.equals(accountRecoverySetting(), other.accountRecoverySetting());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateUserPoolRequest").add("PoolName", poolName()).add("Policies", policies())
                .add("LambdaConfig", lambdaConfig())
                .add("AutoVerifiedAttributes", hasAutoVerifiedAttributes() ? autoVerifiedAttributesAsStrings() : null)
                .add("AliasAttributes", hasAliasAttributes() ? aliasAttributesAsStrings() : null)
                .add("UsernameAttributes", hasUsernameAttributes() ? usernameAttributesAsStrings() : null)
                .add("SmsVerificationMessage", smsVerificationMessage())
                .add("EmailVerificationMessage", emailVerificationMessage())
                .add("EmailVerificationSubject", emailVerificationSubject())
                .add("VerificationMessageTemplate", verificationMessageTemplate())
                .add("SmsAuthenticationMessage", smsAuthenticationMessage()).add("MfaConfiguration", mfaConfigurationAsString())
                .add("UserAttributeUpdateSettings", userAttributeUpdateSettings())
                .add("DeviceConfiguration", deviceConfiguration()).add("EmailConfiguration", emailConfiguration())
                .add("SmsConfiguration", smsConfiguration()).add("UserPoolTags", hasUserPoolTags() ? userPoolTags() : null)
                .add("AdminCreateUserConfig", adminCreateUserConfig()).add("Schema", hasSchema() ? schema() : null)
                .add("UserPoolAddOns", userPoolAddOns()).add("UsernameConfiguration", usernameConfiguration())
                .add("AccountRecoverySetting", accountRecoverySetting()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PoolName":
            return Optional.ofNullable(clazz.cast(poolName()));
        case "Policies":
            return Optional.ofNullable(clazz.cast(policies()));
        case "LambdaConfig":
            return Optional.ofNullable(clazz.cast(lambdaConfig()));
        case "AutoVerifiedAttributes":
            return Optional.ofNullable(clazz.cast(autoVerifiedAttributesAsStrings()));
        case "AliasAttributes":
            return Optional.ofNullable(clazz.cast(aliasAttributesAsStrings()));
        case "UsernameAttributes":
            return Optional.ofNullable(clazz.cast(usernameAttributesAsStrings()));
        case "SmsVerificationMessage":
            return Optional.ofNullable(clazz.cast(smsVerificationMessage()));
        case "EmailVerificationMessage":
            return Optional.ofNullable(clazz.cast(emailVerificationMessage()));
        case "EmailVerificationSubject":
            return Optional.ofNullable(clazz.cast(emailVerificationSubject()));
        case "VerificationMessageTemplate":
            return Optional.ofNullable(clazz.cast(verificationMessageTemplate()));
        case "SmsAuthenticationMessage":
            return Optional.ofNullable(clazz.cast(smsAuthenticationMessage()));
        case "MfaConfiguration":
            return Optional.ofNullable(clazz.cast(mfaConfigurationAsString()));
        case "UserAttributeUpdateSettings":
            return Optional.ofNullable(clazz.cast(userAttributeUpdateSettings()));
        case "DeviceConfiguration":
            return Optional.ofNullable(clazz.cast(deviceConfiguration()));
        case "EmailConfiguration":
            return Optional.ofNullable(clazz.cast(emailConfiguration()));
        case "SmsConfiguration":
            return Optional.ofNullable(clazz.cast(smsConfiguration()));
        case "UserPoolTags":
            return Optional.ofNullable(clazz.cast(userPoolTags()));
        case "AdminCreateUserConfig":
            return Optional.ofNullable(clazz.cast(adminCreateUserConfig()));
        case "Schema":
            return Optional.ofNullable(clazz.cast(schema()));
        case "UserPoolAddOns":
            return Optional.ofNullable(clazz.cast(userPoolAddOns()));
        case "UsernameConfiguration":
            return Optional.ofNullable(clazz.cast(usernameConfiguration()));
        case "AccountRecoverySetting":
            return Optional.ofNullable(clazz.cast(accountRecoverySetting()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateUserPoolRequest, T> g) {
        return obj -> g.apply((CreateUserPoolRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateUserPoolRequest> {
        /**
         * <p>
         * A string used to name the user pool.
         * </p>
         * 
         * @param poolName
         *        A string used to name the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder poolName(String poolName);

        /**
         * <p>
         * The policies associated with the new user pool.
         * </p>
         * 
         * @param policies
         *        The policies associated with the new user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policies(UserPoolPolicyType policies);

        /**
         * <p>
         * The policies associated with the new user pool.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserPoolPolicyType.Builder} avoiding the
         * need to create one manually via {@link UserPoolPolicyType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UserPoolPolicyType.Builder#build()} is called immediately and its
         * result is passed to {@link #policies(UserPoolPolicyType)}.
         * 
         * @param policies
         *        a consumer that will call methods on {@link UserPoolPolicyType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #policies(UserPoolPolicyType)
         */
        default Builder policies(Consumer<UserPoolPolicyType.Builder> policies) {
            return policies(UserPoolPolicyType.builder().applyMutation(policies).build());
        }

        /**
         * <p>
         * The Lambda trigger configuration information for the new user pool.
         * </p>
         * <note>
         * <p>
         * In a push model, event sources (such as Amazon S3 and custom applications) need permission to invoke a
         * function. So you must make an extra call to add permission for these event sources to invoke your Lambda
         * function.
         * </p>
         * <p/>
         * <p>
         * For more information on using the Lambda API to add permission, see<a
         * href="https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html"> AddPermission </a>.
         * </p>
         * <p>
         * For adding permission using the CLI, see<a
         * href="https://docs.aws.amazon.com/cli/latest/reference/lambda/add-permission.html"> add-permission </a>.
         * </p>
         * </note>
         * 
         * @param lambdaConfig
         *        The Lambda trigger configuration information for the new user pool.</p> <note>
         *        <p>
         *        In a push model, event sources (such as Amazon S3 and custom applications) need permission to invoke a
         *        function. So you must make an extra call to add permission for these event sources to invoke your
         *        Lambda function.
         *        </p>
         *        <p/>
         *        <p>
         *        For more information on using the Lambda API to add permission, see<a
         *        href="https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html"> AddPermission </a>.
         *        </p>
         *        <p>
         *        For adding permission using the CLI, see<a
         *        href="https://docs.aws.amazon.com/cli/latest/reference/lambda/add-permission.html"> add-permission
         *        </a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaConfig(LambdaConfigType lambdaConfig);

        /**
         * <p>
         * The Lambda trigger configuration information for the new user pool.
         * </p>
         * <note>
         * <p>
         * In a push model, event sources (such as Amazon S3 and custom applications) need permission to invoke a
         * function. So you must make an extra call to add permission for these event sources to invoke your Lambda
         * function.
         * </p>
         * <p/>
         * <p>
         * For more information on using the Lambda API to add permission, see<a
         * href="https://docs.aws.amazon.com/lambda/latest/dg/API_AddPermission.html"> AddPermission </a>.
         * </p>
         * <p>
         * For adding permission using the CLI, see<a
         * href="https://docs.aws.amazon.com/cli/latest/reference/lambda/add-permission.html"> add-permission </a>.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link LambdaConfigType.Builder}
         * avoiding the need to create one manually via {@link LambdaConfigType#builder()}.
         *
         * When the {@link Consumer} completes, {@link LambdaConfigType.Builder#build()} is called immediately and its
         * result is passed to {@link #lambdaConfig(LambdaConfigType)}.
         * 
         * @param lambdaConfig
         *        a consumer that will call methods on {@link LambdaConfigType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lambdaConfig(LambdaConfigType)
         */
        default Builder lambdaConfig(Consumer<LambdaConfigType.Builder> lambdaConfig) {
            return lambdaConfig(LambdaConfigType.builder().applyMutation(lambdaConfig).build());
        }

        /**
         * <p>
         * The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
         * </p>
         * 
         * @param autoVerifiedAttributes
         *        The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoVerifiedAttributesWithStrings(Collection<String> autoVerifiedAttributes);

        /**
         * <p>
         * The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
         * </p>
         * 
         * @param autoVerifiedAttributes
         *        The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoVerifiedAttributesWithStrings(String... autoVerifiedAttributes);

        /**
         * <p>
         * The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
         * </p>
         * 
         * @param autoVerifiedAttributes
         *        The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoVerifiedAttributes(Collection<VerifiedAttributeType> autoVerifiedAttributes);

        /**
         * <p>
         * The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
         * </p>
         * 
         * @param autoVerifiedAttributes
         *        The attributes to be auto-verified. Possible values: <b>email</b>, <b>phone_number</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoVerifiedAttributes(VerifiedAttributeType... autoVerifiedAttributes);

        /**
         * <p>
         * Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>, <b>email</b>, or
         * <b>preferred_username</b>.
         * </p>
         * 
         * @param aliasAttributes
         *        Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>,
         *        <b>email</b>, or <b>preferred_username</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasAttributesWithStrings(Collection<String> aliasAttributes);

        /**
         * <p>
         * Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>, <b>email</b>, or
         * <b>preferred_username</b>.
         * </p>
         * 
         * @param aliasAttributes
         *        Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>,
         *        <b>email</b>, or <b>preferred_username</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasAttributesWithStrings(String... aliasAttributes);

        /**
         * <p>
         * Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>, <b>email</b>, or
         * <b>preferred_username</b>.
         * </p>
         * 
         * @param aliasAttributes
         *        Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>,
         *        <b>email</b>, or <b>preferred_username</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasAttributes(Collection<AliasAttributeType> aliasAttributes);

        /**
         * <p>
         * Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>, <b>email</b>, or
         * <b>preferred_username</b>.
         * </p>
         * 
         * @param aliasAttributes
         *        Attributes supported as an alias for this user pool. Possible values: <b>phone_number</b>,
         *        <b>email</b>, or <b>preferred_username</b>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aliasAttributes(AliasAttributeType... aliasAttributes);

        /**
         * <p>
         * Specifies whether a user can use an email address or phone number as a username when they sign up.
         * </p>
         * 
         * @param usernameAttributes
         *        Specifies whether a user can use an email address or phone number as a username when they sign up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameAttributesWithStrings(Collection<String> usernameAttributes);

        /**
         * <p>
         * Specifies whether a user can use an email address or phone number as a username when they sign up.
         * </p>
         * 
         * @param usernameAttributes
         *        Specifies whether a user can use an email address or phone number as a username when they sign up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameAttributesWithStrings(String... usernameAttributes);

        /**
         * <p>
         * Specifies whether a user can use an email address or phone number as a username when they sign up.
         * </p>
         * 
         * @param usernameAttributes
         *        Specifies whether a user can use an email address or phone number as a username when they sign up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameAttributes(Collection<UsernameAttributeType> usernameAttributes);

        /**
         * <p>
         * Specifies whether a user can use an email address or phone number as a username when they sign up.
         * </p>
         * 
         * @param usernameAttributes
         *        Specifies whether a user can use an email address or phone number as a username when they sign up.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameAttributes(UsernameAttributeType... usernameAttributes);

        /**
         * <p>
         * A string representing the SMS verification message.
         * </p>
         * 
         * @param smsVerificationMessage
         *        A string representing the SMS verification message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smsVerificationMessage(String smsVerificationMessage);

        /**
         * <p>
         * A string representing the email verification message. EmailVerificationMessage is allowed only if <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount"
         * >EmailSendingAccount</a> is DEVELOPER.
         * </p>
         * 
         * @param emailVerificationMessage
         *        A string representing the email verification message. EmailVerificationMessage is allowed only if <a
         *        href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount"
         *        >EmailSendingAccount</a> is DEVELOPER.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailVerificationMessage(String emailVerificationMessage);

        /**
         * <p>
         * A string representing the email verification subject. EmailVerificationSubject is allowed only if <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount"
         * >EmailSendingAccount</a> is DEVELOPER.
         * </p>
         * 
         * @param emailVerificationSubject
         *        A string representing the email verification subject. EmailVerificationSubject is allowed only if <a
         *        href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_EmailConfigurationType.html#CognitoUserPools-Type-EmailConfigurationType-EmailSendingAccount"
         *        >EmailSendingAccount</a> is DEVELOPER.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailVerificationSubject(String emailVerificationSubject);

        /**
         * <p>
         * The template for the verification message that the user sees when the app requests permission to access the
         * user's information.
         * </p>
         * 
         * @param verificationMessageTemplate
         *        The template for the verification message that the user sees when the app requests permission to
         *        access the user's information.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder verificationMessageTemplate(VerificationMessageTemplateType verificationMessageTemplate);

        /**
         * <p>
         * The template for the verification message that the user sees when the app requests permission to access the
         * user's information.
         * </p>
         * This is a convenience method that creates an instance of the {@link VerificationMessageTemplateType.Builder}
         * avoiding the need to create one manually via {@link VerificationMessageTemplateType#builder()}.
         *
         * When the {@link Consumer} completes, {@link VerificationMessageTemplateType.Builder#build()} is called
         * immediately and its result is passed to {@link #verificationMessageTemplate(VerificationMessageTemplateType)}
         * .
         * 
         * @param verificationMessageTemplate
         *        a consumer that will call methods on {@link VerificationMessageTemplateType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #verificationMessageTemplate(VerificationMessageTemplateType)
         */
        default Builder verificationMessageTemplate(Consumer<VerificationMessageTemplateType.Builder> verificationMessageTemplate) {
            return verificationMessageTemplate(VerificationMessageTemplateType.builder()
                    .applyMutation(verificationMessageTemplate).build());
        }

        /**
         * <p>
         * A string representing the SMS authentication message.
         * </p>
         * 
         * @param smsAuthenticationMessage
         *        A string representing the SMS authentication message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smsAuthenticationMessage(String smsAuthenticationMessage);

        /**
         * <p>
         * Specifies MFA configuration details.
         * </p>
         * 
         * @param mfaConfiguration
         *        Specifies MFA configuration details.
         * @see UserPoolMfaType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserPoolMfaType
         */
        Builder mfaConfiguration(String mfaConfiguration);

        /**
         * <p>
         * Specifies MFA configuration details.
         * </p>
         * 
         * @param mfaConfiguration
         *        Specifies MFA configuration details.
         * @see UserPoolMfaType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserPoolMfaType
         */
        Builder mfaConfiguration(UserPoolMfaType mfaConfiguration);

        /**
         * <p>
         * The settings for updates to user attributes. These settings include the property
         * <code>AttributesRequireVerificationBeforeUpdate</code>, a user-pool setting that tells Amazon Cognito how to
         * handle changes to the value of your users' email address and phone number attributes. For more information,
         * see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates"
         * > Verifying updates to to email addresses and phone numbers</a>.
         * </p>
         * 
         * @param userAttributeUpdateSettings
         *        The settings for updates to user attributes. These settings include the property
         *        <code>AttributesRequireVerificationBeforeUpdate</code>, a user-pool setting that tells Amazon Cognito
         *        how to handle changes to the value of your users' email address and phone number attributes. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates"
         *        > Verifying updates to to email addresses and phone numbers</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributeUpdateSettings(UserAttributeUpdateSettingsType userAttributeUpdateSettings);

        /**
         * <p>
         * The settings for updates to user attributes. These settings include the property
         * <code>AttributesRequireVerificationBeforeUpdate</code>, a user-pool setting that tells Amazon Cognito how to
         * handle changes to the value of your users' email address and phone number attributes. For more information,
         * see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-email-phone-verification.html#user-pool-settings-verifications-verify-attribute-updates"
         * > Verifying updates to to email addresses and phone numbers</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserAttributeUpdateSettingsType.Builder}
         * avoiding the need to create one manually via {@link UserAttributeUpdateSettingsType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UserAttributeUpdateSettingsType.Builder#build()} is called
         * immediately and its result is passed to {@link #userAttributeUpdateSettings(UserAttributeUpdateSettingsType)}
         * .
         * 
         * @param userAttributeUpdateSettings
         *        a consumer that will call methods on {@link UserAttributeUpdateSettingsType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAttributeUpdateSettings(UserAttributeUpdateSettingsType)
         */
        default Builder userAttributeUpdateSettings(Consumer<UserAttributeUpdateSettingsType.Builder> userAttributeUpdateSettings) {
            return userAttributeUpdateSettings(UserAttributeUpdateSettingsType.builder()
                    .applyMutation(userAttributeUpdateSettings).build());
        }

        /**
         * <p>
         * The device configuration.
         * </p>
         * 
         * @param deviceConfiguration
         *        The device configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceConfiguration(DeviceConfigurationType deviceConfiguration);

        /**
         * <p>
         * The device configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeviceConfigurationType.Builder} avoiding
         * the need to create one manually via {@link DeviceConfigurationType#builder()}.
         *
         * When the {@link Consumer} completes, {@link DeviceConfigurationType.Builder#build()} is called immediately
         * and its result is passed to {@link #deviceConfiguration(DeviceConfigurationType)}.
         * 
         * @param deviceConfiguration
         *        a consumer that will call methods on {@link DeviceConfigurationType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deviceConfiguration(DeviceConfigurationType)
         */
        default Builder deviceConfiguration(Consumer<DeviceConfigurationType.Builder> deviceConfiguration) {
            return deviceConfiguration(DeviceConfigurationType.builder().applyMutation(deviceConfiguration).build());
        }

        /**
         * <p>
         * The email configuration of your user pool. The email configuration type sets your preferred sending method,
         * Amazon Web Services Region, and sender for messages from your user pool.
         * </p>
         * 
         * @param emailConfiguration
         *        The email configuration of your user pool. The email configuration type sets your preferred sending
         *        method, Amazon Web Services Region, and sender for messages from your user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emailConfiguration(EmailConfigurationType emailConfiguration);

        /**
         * <p>
         * The email configuration of your user pool. The email configuration type sets your preferred sending method,
         * Amazon Web Services Region, and sender for messages from your user pool.
         * </p>
         * This is a convenience method that creates an instance of the {@link EmailConfigurationType.Builder} avoiding
         * the need to create one manually via {@link EmailConfigurationType#builder()}.
         *
         * When the {@link Consumer} completes, {@link EmailConfigurationType.Builder#build()} is called immediately and
         * its result is passed to {@link #emailConfiguration(EmailConfigurationType)}.
         * 
         * @param emailConfiguration
         *        a consumer that will call methods on {@link EmailConfigurationType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #emailConfiguration(EmailConfigurationType)
         */
        default Builder emailConfiguration(Consumer<EmailConfigurationType.Builder> emailConfiguration) {
            return emailConfiguration(EmailConfigurationType.builder().applyMutation(emailConfiguration).build());
        }

        /**
         * <p>
         * The SMS configuration with the settings that your Amazon Cognito user pool must use to send an SMS message
         * from your Amazon Web Services account through Amazon Simple Notification Service. To send SMS messages with
         * Amazon SNS in the Amazon Web Services Region that you want, the Amazon Cognito user pool uses an Identity and
         * Access Management (IAM) role in your Amazon Web Services account.
         * </p>
         * 
         * @param smsConfiguration
         *        The SMS configuration with the settings that your Amazon Cognito user pool must use to send an SMS
         *        message from your Amazon Web Services account through Amazon Simple Notification Service. To send SMS
         *        messages with Amazon SNS in the Amazon Web Services Region that you want, the Amazon Cognito user pool
         *        uses an Identity and Access Management (IAM) role in your Amazon Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder smsConfiguration(SmsConfigurationType smsConfiguration);

        /**
         * <p>
         * The SMS configuration with the settings that your Amazon Cognito user pool must use to send an SMS message
         * from your Amazon Web Services account through Amazon Simple Notification Service. To send SMS messages with
         * Amazon SNS in the Amazon Web Services Region that you want, the Amazon Cognito user pool uses an Identity and
         * Access Management (IAM) role in your Amazon Web Services account.
         * </p>
         * This is a convenience method that creates an instance of the {@link SmsConfigurationType.Builder} avoiding
         * the need to create one manually via {@link SmsConfigurationType#builder()}.
         *
         * When the {@link Consumer} completes, {@link SmsConfigurationType.Builder#build()} is called immediately and
         * its result is passed to {@link #smsConfiguration(SmsConfigurationType)}.
         * 
         * @param smsConfiguration
         *        a consumer that will call methods on {@link SmsConfigurationType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #smsConfiguration(SmsConfigurationType)
         */
        default Builder smsConfiguration(Consumer<SmsConfigurationType.Builder> smsConfiguration) {
            return smsConfiguration(SmsConfigurationType.builder().applyMutation(smsConfiguration).build());
        }

        /**
         * <p>
         * The tag keys and values to assign to the user pool. A tag is a label that you can use to categorize and
         * manage user pools in different ways, such as by purpose, owner, environment, or other criteria.
         * </p>
         * 
         * @param userPoolTags
         *        The tag keys and values to assign to the user pool. A tag is a label that you can use to categorize
         *        and manage user pools in different ways, such as by purpose, owner, environment, or other criteria.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolTags(Map<String, String> userPoolTags);

        /**
         * <p>
         * The configuration for <code>AdminCreateUser</code> requests.
         * </p>
         * 
         * @param adminCreateUserConfig
         *        The configuration for <code>AdminCreateUser</code> requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder adminCreateUserConfig(AdminCreateUserConfigType adminCreateUserConfig);

        /**
         * <p>
         * The configuration for <code>AdminCreateUser</code> requests.
         * </p>
         * This is a convenience method that creates an instance of the {@link AdminCreateUserConfigType.Builder}
         * avoiding the need to create one manually via {@link AdminCreateUserConfigType#builder()}.
         *
         * When the {@link Consumer} completes, {@link AdminCreateUserConfigType.Builder#build()} is called immediately
         * and its result is passed to {@link #adminCreateUserConfig(AdminCreateUserConfigType)}.
         * 
         * @param adminCreateUserConfig
         *        a consumer that will call methods on {@link AdminCreateUserConfigType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #adminCreateUserConfig(AdminCreateUserConfigType)
         */
        default Builder adminCreateUserConfig(Consumer<AdminCreateUserConfigType.Builder> adminCreateUserConfig) {
            return adminCreateUserConfig(AdminCreateUserConfigType.builder().applyMutation(adminCreateUserConfig).build());
        }

        /**
         * <p>
         * An array of schema attributes for the new user pool. These attributes can be standard or custom attributes.
         * </p>
         * 
         * @param schema
         *        An array of schema attributes for the new user pool. These attributes can be standard or custom
         *        attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(Collection<SchemaAttributeType> schema);

        /**
         * <p>
         * An array of schema attributes for the new user pool. These attributes can be standard or custom attributes.
         * </p>
         * 
         * @param schema
         *        An array of schema attributes for the new user pool. These attributes can be standard or custom
         *        attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(SchemaAttributeType... schema);

        /**
         * <p>
         * An array of schema attributes for the new user pool. These attributes can be standard or custom attributes.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<SchemaAttributeType>.Builder}
         * avoiding the need to create one manually via {@link List<SchemaAttributeType>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SchemaAttributeType>.Builder#build()} is called immediately
         * and its result is passed to {@link #schema(List<SchemaAttributeType>)}.
         * 
         * @param schema
         *        a consumer that will call methods on {@link List<SchemaAttributeType>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schema(List<SchemaAttributeType>)
         */
        Builder schema(Consumer<SchemaAttributeType.Builder>... schema);

        /**
         * <p>
         * Enables advanced security risk detection. Set the key <code>AdvancedSecurityMode</code> to the value "AUDIT".
         * </p>
         * 
         * @param userPoolAddOns
         *        Enables advanced security risk detection. Set the key <code>AdvancedSecurityMode</code> to the value
         *        "AUDIT".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolAddOns(UserPoolAddOnsType userPoolAddOns);

        /**
         * <p>
         * Enables advanced security risk detection. Set the key <code>AdvancedSecurityMode</code> to the value "AUDIT".
         * </p>
         * This is a convenience method that creates an instance of the {@link UserPoolAddOnsType.Builder} avoiding the
         * need to create one manually via {@link UserPoolAddOnsType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UserPoolAddOnsType.Builder#build()} is called immediately and its
         * result is passed to {@link #userPoolAddOns(UserPoolAddOnsType)}.
         * 
         * @param userPoolAddOns
         *        a consumer that will call methods on {@link UserPoolAddOnsType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userPoolAddOns(UserPoolAddOnsType)
         */
        default Builder userPoolAddOns(Consumer<UserPoolAddOnsType.Builder> userPoolAddOns) {
            return userPoolAddOns(UserPoolAddOnsType.builder().applyMutation(userPoolAddOns).build());
        }

        /**
         * <p>
         * Case sensitivity on the username input for the selected sign-in option. For example, when case sensitivity is
         * set to <code>False</code>, users can sign in using either "username" or "Username". This configuration is
         * immutable once it has been set. For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UsernameConfigurationType.html"
         * >UsernameConfigurationType</a>.
         * </p>
         * 
         * @param usernameConfiguration
         *        Case sensitivity on the username input for the selected sign-in option. For example, when case
         *        sensitivity is set to <code>False</code>, users can sign in using either "username" or "Username".
         *        This configuration is immutable once it has been set. For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UsernameConfigurationType.html"
         *        >UsernameConfigurationType</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usernameConfiguration(UsernameConfigurationType usernameConfiguration);

        /**
         * <p>
         * Case sensitivity on the username input for the selected sign-in option. For example, when case sensitivity is
         * set to <code>False</code>, users can sign in using either "username" or "Username". This configuration is
         * immutable once it has been set. For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UsernameConfigurationType.html"
         * >UsernameConfigurationType</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link UsernameConfigurationType.Builder}
         * avoiding the need to create one manually via {@link UsernameConfigurationType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UsernameConfigurationType.Builder#build()} is called immediately
         * and its result is passed to {@link #usernameConfiguration(UsernameConfigurationType)}.
         * 
         * @param usernameConfiguration
         *        a consumer that will call methods on {@link UsernameConfigurationType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #usernameConfiguration(UsernameConfigurationType)
         */
        default Builder usernameConfiguration(Consumer<UsernameConfigurationType.Builder> usernameConfiguration) {
            return usernameConfiguration(UsernameConfigurationType.builder().applyMutation(usernameConfiguration).build());
        }

        /**
         * <p>
         * The available verified method a user can use to recover their password when they call
         * <code>ForgotPassword</code>. You can use this setting to define a preferred method when a user has more than
         * one method available. With this setting, SMS doesn't qualify for a valid password recovery mechanism if the
         * user also has SMS multi-factor authentication (MFA) activated. In the absence of this setting, Amazon Cognito
         * uses the legacy behavior to determine the recovery method where SMS is preferred through email.
         * </p>
         * 
         * @param accountRecoverySetting
         *        The available verified method a user can use to recover their password when they call
         *        <code>ForgotPassword</code>. You can use this setting to define a preferred method when a user has
         *        more than one method available. With this setting, SMS doesn't qualify for a valid password recovery
         *        mechanism if the user also has SMS multi-factor authentication (MFA) activated. In the absence of this
         *        setting, Amazon Cognito uses the legacy behavior to determine the recovery method where SMS is
         *        preferred through email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountRecoverySetting(AccountRecoverySettingType accountRecoverySetting);

        /**
         * <p>
         * The available verified method a user can use to recover their password when they call
         * <code>ForgotPassword</code>. You can use this setting to define a preferred method when a user has more than
         * one method available. With this setting, SMS doesn't qualify for a valid password recovery mechanism if the
         * user also has SMS multi-factor authentication (MFA) activated. In the absence of this setting, Amazon Cognito
         * uses the legacy behavior to determine the recovery method where SMS is preferred through email.
         * </p>
         * This is a convenience method that creates an instance of the {@link AccountRecoverySettingType.Builder}
         * avoiding the need to create one manually via {@link AccountRecoverySettingType#builder()}.
         *
         * When the {@link Consumer} completes, {@link AccountRecoverySettingType.Builder#build()} is called immediately
         * and its result is passed to {@link #accountRecoverySetting(AccountRecoverySettingType)}.
         * 
         * @param accountRecoverySetting
         *        a consumer that will call methods on {@link AccountRecoverySettingType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accountRecoverySetting(AccountRecoverySettingType)
         */
        default Builder accountRecoverySetting(Consumer<AccountRecoverySettingType.Builder> accountRecoverySetting) {
            return accountRecoverySetting(AccountRecoverySettingType.builder().applyMutation(accountRecoverySetting).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String poolName;

        private UserPoolPolicyType policies;

        private LambdaConfigType lambdaConfig;

        private List<String> autoVerifiedAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<String> aliasAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<String> usernameAttributes = DefaultSdkAutoConstructList.getInstance();

        private String smsVerificationMessage;

        private String emailVerificationMessage;

        private String emailVerificationSubject;

        private VerificationMessageTemplateType verificationMessageTemplate;

        private String smsAuthenticationMessage;

        private String mfaConfiguration;

        private UserAttributeUpdateSettingsType userAttributeUpdateSettings;

        private DeviceConfigurationType deviceConfiguration;

        private EmailConfigurationType emailConfiguration;

        private SmsConfigurationType smsConfiguration;

        private Map<String, String> userPoolTags = DefaultSdkAutoConstructMap.getInstance();

        private AdminCreateUserConfigType adminCreateUserConfig;

        private List<SchemaAttributeType> schema = DefaultSdkAutoConstructList.getInstance();

        private UserPoolAddOnsType userPoolAddOns;

        private UsernameConfigurationType usernameConfiguration;

        private AccountRecoverySettingType accountRecoverySetting;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateUserPoolRequest model) {
            super(model);
            poolName(model.poolName);
            policies(model.policies);
            lambdaConfig(model.lambdaConfig);
            autoVerifiedAttributesWithStrings(model.autoVerifiedAttributes);
            aliasAttributesWithStrings(model.aliasAttributes);
            usernameAttributesWithStrings(model.usernameAttributes);
            smsVerificationMessage(model.smsVerificationMessage);
            emailVerificationMessage(model.emailVerificationMessage);
            emailVerificationSubject(model.emailVerificationSubject);
            verificationMessageTemplate(model.verificationMessageTemplate);
            smsAuthenticationMessage(model.smsAuthenticationMessage);
            mfaConfiguration(model.mfaConfiguration);
            userAttributeUpdateSettings(model.userAttributeUpdateSettings);
            deviceConfiguration(model.deviceConfiguration);
            emailConfiguration(model.emailConfiguration);
            smsConfiguration(model.smsConfiguration);
            userPoolTags(model.userPoolTags);
            adminCreateUserConfig(model.adminCreateUserConfig);
            schema(model.schema);
            userPoolAddOns(model.userPoolAddOns);
            usernameConfiguration(model.usernameConfiguration);
            accountRecoverySetting(model.accountRecoverySetting);
        }

        public final String getPoolName() {
            return poolName;
        }

        public final void setPoolName(String poolName) {
            this.poolName = poolName;
        }

        @Override
        public final Builder poolName(String poolName) {
            this.poolName = poolName;
            return this;
        }

        public final UserPoolPolicyType.Builder getPolicies() {
            return policies != null ? policies.toBuilder() : null;
        }

        public final void setPolicies(UserPoolPolicyType.BuilderImpl policies) {
            this.policies = policies != null ? policies.build() : null;
        }

        @Override
        public final Builder policies(UserPoolPolicyType policies) {
            this.policies = policies;
            return this;
        }

        public final LambdaConfigType.Builder getLambdaConfig() {
            return lambdaConfig != null ? lambdaConfig.toBuilder() : null;
        }

        public final void setLambdaConfig(LambdaConfigType.BuilderImpl lambdaConfig) {
            this.lambdaConfig = lambdaConfig != null ? lambdaConfig.build() : null;
        }

        @Override
        public final Builder lambdaConfig(LambdaConfigType lambdaConfig) {
            this.lambdaConfig = lambdaConfig;
            return this;
        }

        public final Collection<String> getAutoVerifiedAttributes() {
            if (autoVerifiedAttributes instanceof SdkAutoConstructList) {
                return null;
            }
            return autoVerifiedAttributes;
        }

        public final void setAutoVerifiedAttributes(Collection<String> autoVerifiedAttributes) {
            this.autoVerifiedAttributes = VerifiedAttributesListTypeCopier.copy(autoVerifiedAttributes);
        }

        @Override
        public final Builder autoVerifiedAttributesWithStrings(Collection<String> autoVerifiedAttributes) {
            this.autoVerifiedAttributes = VerifiedAttributesListTypeCopier.copy(autoVerifiedAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autoVerifiedAttributesWithStrings(String... autoVerifiedAttributes) {
            autoVerifiedAttributesWithStrings(Arrays.asList(autoVerifiedAttributes));
            return this;
        }

        @Override
        public final Builder autoVerifiedAttributes(Collection<VerifiedAttributeType> autoVerifiedAttributes) {
            this.autoVerifiedAttributes = VerifiedAttributesListTypeCopier.copyEnumToString(autoVerifiedAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder autoVerifiedAttributes(VerifiedAttributeType... autoVerifiedAttributes) {
            autoVerifiedAttributes(Arrays.asList(autoVerifiedAttributes));
            return this;
        }

        public final Collection<String> getAliasAttributes() {
            if (aliasAttributes instanceof SdkAutoConstructList) {
                return null;
            }
            return aliasAttributes;
        }

        public final void setAliasAttributes(Collection<String> aliasAttributes) {
            this.aliasAttributes = AliasAttributesListTypeCopier.copy(aliasAttributes);
        }

        @Override
        public final Builder aliasAttributesWithStrings(Collection<String> aliasAttributes) {
            this.aliasAttributes = AliasAttributesListTypeCopier.copy(aliasAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aliasAttributesWithStrings(String... aliasAttributes) {
            aliasAttributesWithStrings(Arrays.asList(aliasAttributes));
            return this;
        }

        @Override
        public final Builder aliasAttributes(Collection<AliasAttributeType> aliasAttributes) {
            this.aliasAttributes = AliasAttributesListTypeCopier.copyEnumToString(aliasAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder aliasAttributes(AliasAttributeType... aliasAttributes) {
            aliasAttributes(Arrays.asList(aliasAttributes));
            return this;
        }

        public final Collection<String> getUsernameAttributes() {
            if (usernameAttributes instanceof SdkAutoConstructList) {
                return null;
            }
            return usernameAttributes;
        }

        public final void setUsernameAttributes(Collection<String> usernameAttributes) {
            this.usernameAttributes = UsernameAttributesListTypeCopier.copy(usernameAttributes);
        }

        @Override
        public final Builder usernameAttributesWithStrings(Collection<String> usernameAttributes) {
            this.usernameAttributes = UsernameAttributesListTypeCopier.copy(usernameAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder usernameAttributesWithStrings(String... usernameAttributes) {
            usernameAttributesWithStrings(Arrays.asList(usernameAttributes));
            return this;
        }

        @Override
        public final Builder usernameAttributes(Collection<UsernameAttributeType> usernameAttributes) {
            this.usernameAttributes = UsernameAttributesListTypeCopier.copyEnumToString(usernameAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder usernameAttributes(UsernameAttributeType... usernameAttributes) {
            usernameAttributes(Arrays.asList(usernameAttributes));
            return this;
        }

        public final String getSmsVerificationMessage() {
            return smsVerificationMessage;
        }

        public final void setSmsVerificationMessage(String smsVerificationMessage) {
            this.smsVerificationMessage = smsVerificationMessage;
        }

        @Override
        public final Builder smsVerificationMessage(String smsVerificationMessage) {
            this.smsVerificationMessage = smsVerificationMessage;
            return this;
        }

        public final String getEmailVerificationMessage() {
            return emailVerificationMessage;
        }

        public final void setEmailVerificationMessage(String emailVerificationMessage) {
            this.emailVerificationMessage = emailVerificationMessage;
        }

        @Override
        public final Builder emailVerificationMessage(String emailVerificationMessage) {
            this.emailVerificationMessage = emailVerificationMessage;
            return this;
        }

        public final String getEmailVerificationSubject() {
            return emailVerificationSubject;
        }

        public final void setEmailVerificationSubject(String emailVerificationSubject) {
            this.emailVerificationSubject = emailVerificationSubject;
        }

        @Override
        public final Builder emailVerificationSubject(String emailVerificationSubject) {
            this.emailVerificationSubject = emailVerificationSubject;
            return this;
        }

        public final VerificationMessageTemplateType.Builder getVerificationMessageTemplate() {
            return verificationMessageTemplate != null ? verificationMessageTemplate.toBuilder() : null;
        }

        public final void setVerificationMessageTemplate(VerificationMessageTemplateType.BuilderImpl verificationMessageTemplate) {
            this.verificationMessageTemplate = verificationMessageTemplate != null ? verificationMessageTemplate.build() : null;
        }

        @Override
        public final Builder verificationMessageTemplate(VerificationMessageTemplateType verificationMessageTemplate) {
            this.verificationMessageTemplate = verificationMessageTemplate;
            return this;
        }

        public final String getSmsAuthenticationMessage() {
            return smsAuthenticationMessage;
        }

        public final void setSmsAuthenticationMessage(String smsAuthenticationMessage) {
            this.smsAuthenticationMessage = smsAuthenticationMessage;
        }

        @Override
        public final Builder smsAuthenticationMessage(String smsAuthenticationMessage) {
            this.smsAuthenticationMessage = smsAuthenticationMessage;
            return this;
        }

        public final String getMfaConfiguration() {
            return mfaConfiguration;
        }

        public final void setMfaConfiguration(String mfaConfiguration) {
            this.mfaConfiguration = mfaConfiguration;
        }

        @Override
        public final Builder mfaConfiguration(String mfaConfiguration) {
            this.mfaConfiguration = mfaConfiguration;
            return this;
        }

        @Override
        public final Builder mfaConfiguration(UserPoolMfaType mfaConfiguration) {
            this.mfaConfiguration(mfaConfiguration == null ? null : mfaConfiguration.toString());
            return this;
        }

        public final UserAttributeUpdateSettingsType.Builder getUserAttributeUpdateSettings() {
            return userAttributeUpdateSettings != null ? userAttributeUpdateSettings.toBuilder() : null;
        }

        public final void setUserAttributeUpdateSettings(UserAttributeUpdateSettingsType.BuilderImpl userAttributeUpdateSettings) {
            this.userAttributeUpdateSettings = userAttributeUpdateSettings != null ? userAttributeUpdateSettings.build() : null;
        }

        @Override
        public final Builder userAttributeUpdateSettings(UserAttributeUpdateSettingsType userAttributeUpdateSettings) {
            this.userAttributeUpdateSettings = userAttributeUpdateSettings;
            return this;
        }

        public final DeviceConfigurationType.Builder getDeviceConfiguration() {
            return deviceConfiguration != null ? deviceConfiguration.toBuilder() : null;
        }

        public final void setDeviceConfiguration(DeviceConfigurationType.BuilderImpl deviceConfiguration) {
            this.deviceConfiguration = deviceConfiguration != null ? deviceConfiguration.build() : null;
        }

        @Override
        public final Builder deviceConfiguration(DeviceConfigurationType deviceConfiguration) {
            this.deviceConfiguration = deviceConfiguration;
            return this;
        }

        public final EmailConfigurationType.Builder getEmailConfiguration() {
            return emailConfiguration != null ? emailConfiguration.toBuilder() : null;
        }

        public final void setEmailConfiguration(EmailConfigurationType.BuilderImpl emailConfiguration) {
            this.emailConfiguration = emailConfiguration != null ? emailConfiguration.build() : null;
        }

        @Override
        public final Builder emailConfiguration(EmailConfigurationType emailConfiguration) {
            this.emailConfiguration = emailConfiguration;
            return this;
        }

        public final SmsConfigurationType.Builder getSmsConfiguration() {
            return smsConfiguration != null ? smsConfiguration.toBuilder() : null;
        }

        public final void setSmsConfiguration(SmsConfigurationType.BuilderImpl smsConfiguration) {
            this.smsConfiguration = smsConfiguration != null ? smsConfiguration.build() : null;
        }

        @Override
        public final Builder smsConfiguration(SmsConfigurationType smsConfiguration) {
            this.smsConfiguration = smsConfiguration;
            return this;
        }

        public final Map<String, String> getUserPoolTags() {
            if (userPoolTags instanceof SdkAutoConstructMap) {
                return null;
            }
            return userPoolTags;
        }

        public final void setUserPoolTags(Map<String, String> userPoolTags) {
            this.userPoolTags = UserPoolTagsTypeCopier.copy(userPoolTags);
        }

        @Override
        public final Builder userPoolTags(Map<String, String> userPoolTags) {
            this.userPoolTags = UserPoolTagsTypeCopier.copy(userPoolTags);
            return this;
        }

        public final AdminCreateUserConfigType.Builder getAdminCreateUserConfig() {
            return adminCreateUserConfig != null ? adminCreateUserConfig.toBuilder() : null;
        }

        public final void setAdminCreateUserConfig(AdminCreateUserConfigType.BuilderImpl adminCreateUserConfig) {
            this.adminCreateUserConfig = adminCreateUserConfig != null ? adminCreateUserConfig.build() : null;
        }

        @Override
        public final Builder adminCreateUserConfig(AdminCreateUserConfigType adminCreateUserConfig) {
            this.adminCreateUserConfig = adminCreateUserConfig;
            return this;
        }

        public final List<SchemaAttributeType.Builder> getSchema() {
            List<SchemaAttributeType.Builder> result = SchemaAttributesListTypeCopier.copyToBuilder(this.schema);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSchema(Collection<SchemaAttributeType.BuilderImpl> schema) {
            this.schema = SchemaAttributesListTypeCopier.copyFromBuilder(schema);
        }

        @Override
        public final Builder schema(Collection<SchemaAttributeType> schema) {
            this.schema = SchemaAttributesListTypeCopier.copy(schema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schema(SchemaAttributeType... schema) {
            schema(Arrays.asList(schema));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder schema(Consumer<SchemaAttributeType.Builder>... schema) {
            schema(Stream.of(schema).map(c -> SchemaAttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final UserPoolAddOnsType.Builder getUserPoolAddOns() {
            return userPoolAddOns != null ? userPoolAddOns.toBuilder() : null;
        }

        public final void setUserPoolAddOns(UserPoolAddOnsType.BuilderImpl userPoolAddOns) {
            this.userPoolAddOns = userPoolAddOns != null ? userPoolAddOns.build() : null;
        }

        @Override
        public final Builder userPoolAddOns(UserPoolAddOnsType userPoolAddOns) {
            this.userPoolAddOns = userPoolAddOns;
            return this;
        }

        public final UsernameConfigurationType.Builder getUsernameConfiguration() {
            return usernameConfiguration != null ? usernameConfiguration.toBuilder() : null;
        }

        public final void setUsernameConfiguration(UsernameConfigurationType.BuilderImpl usernameConfiguration) {
            this.usernameConfiguration = usernameConfiguration != null ? usernameConfiguration.build() : null;
        }

        @Override
        public final Builder usernameConfiguration(UsernameConfigurationType usernameConfiguration) {
            this.usernameConfiguration = usernameConfiguration;
            return this;
        }

        public final AccountRecoverySettingType.Builder getAccountRecoverySetting() {
            return accountRecoverySetting != null ? accountRecoverySetting.toBuilder() : null;
        }

        public final void setAccountRecoverySetting(AccountRecoverySettingType.BuilderImpl accountRecoverySetting) {
            this.accountRecoverySetting = accountRecoverySetting != null ? accountRecoverySetting.build() : null;
        }

        @Override
        public final Builder accountRecoverySetting(AccountRecoverySettingType accountRecoverySetting) {
            this.accountRecoverySetting = accountRecoverySetting;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateUserPoolRequest build() {
            return new CreateUserPoolRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
