/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Confirm a user's registration as a user pool administrator.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdminConfirmSignUpRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<AdminConfirmSignUpRequest.Builder, AdminConfirmSignUpRequest> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(AdminConfirmSignUpRequest::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(AdminConfirmSignUpRequest::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(AdminConfirmSignUpRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            USERNAME_FIELD, CLIENT_METADATA_FIELD));

    private final String userPoolId;

    private final String username;

    private final Map<String, String> clientMetadata;

    private AdminConfirmSignUpRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.username = builder.username;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The user pool ID for which you want to confirm user registration.
     * </p>
     * 
     * @return The user pool ID for which you want to confirm user registration.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The username of the user that you want to query or modify. The value of this parameter is typically your user's
     * username, but it can be any of their alias attributes. If <code>username</code> isn't an alias attribute in your
     * user pool, this value must be the <code>sub</code> of a local user or the username of a user from a third-party
     * IdP.
     * </p>
     * 
     * @return The username of the user that you want to query or modify. The value of this parameter is typically your
     *         user's username, but it can be any of their alias attributes. If <code>username</code> isn't an alias
     *         attribute in your user pool, this value must be the <code>sub</code> of a local user or the username of a
     *         user from a third-party IdP.
     */
    public final String username() {
        return username;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.
     * </p>
     * <p>
     * If your user pool configuration includes triggers, the AdminConfirmSignUp API action invokes the Lambda function
     * that is specified for the <i>post confirmation</i> trigger. When Amazon Cognito invokes this function, it passes
     * a JSON payload, which the function receives as input. In this payload, the <code>clientMetadata</code> attribute
     * provides the data that you assigned to the ClientMetadata parameter in your AdminConfirmSignUp request. In your
     * function code in Lambda, you can process the ClientMetadata value to enhance your workflow for your specific
     * needs.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a user pool
     * to support custom workflows. If your user pool configuration doesn't include triggers, the ClientMetadata
     * parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Validate the ClientMetadata value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientMetadata} method.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for any custom workflows that this action
     *         triggers.</p>
     *         <p>
     *         If your user pool configuration includes triggers, the AdminConfirmSignUp API action invokes the Lambda
     *         function that is specified for the <i>post confirmation</i> trigger. When Amazon Cognito invokes this
     *         function, it passes a JSON payload, which the function receives as input. In this payload, the
     *         <code>clientMetadata</code> attribute provides the data that you assigned to the ClientMetadata parameter
     *         in your AdminConfirmSignUp request. In your function code in Lambda, you can process the ClientMetadata
     *         value to enhance your workflow for your specific needs.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a
     *         user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
     *         ClientMetadata parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Validate the ClientMetadata value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdminConfirmSignUpRequest)) {
            return false;
        }
        AdminConfirmSignUpRequest other = (AdminConfirmSignUpRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(username(), other.username())
                && hasClientMetadata() == other.hasClientMetadata() && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdminConfirmSignUpRequest").add("UserPoolId", userPoolId())
                .add("Username", username() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AdminConfirmSignUpRequest, T> g) {
        return obj -> g.apply((AdminConfirmSignUpRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, AdminConfirmSignUpRequest> {
        /**
         * <p>
         * The user pool ID for which you want to confirm user registration.
         * </p>
         * 
         * @param userPoolId
         *        The user pool ID for which you want to confirm user registration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The username of the user that you want to query or modify. The value of this parameter is typically your
         * user's username, but it can be any of their alias attributes. If <code>username</code> isn't an alias
         * attribute in your user pool, this value must be the <code>sub</code> of a local user or the username of a
         * user from a third-party IdP.
         * </p>
         * 
         * @param username
         *        The username of the user that you want to query or modify. The value of this parameter is typically
         *        your user's username, but it can be any of their alias attributes. If <code>username</code> isn't an
         *        alias attribute in your user pool, this value must be the <code>sub</code> of a local user or the
         *        username of a user from a third-party IdP.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for any custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * If your user pool configuration includes triggers, the AdminConfirmSignUp API action invokes the Lambda
         * function that is specified for the <i>post confirmation</i> trigger. When Amazon Cognito invokes this
         * function, it passes a JSON payload, which the function receives as input. In this payload, the
         * <code>clientMetadata</code> attribute provides the data that you assigned to the ClientMetadata parameter in
         * your AdminConfirmSignUp request. In your function code in Lambda, you can process the ClientMetadata value to
         * enhance your workflow for your specific needs.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a user
         * pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         * ClientMetadata parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Validate the ClientMetadata value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for any custom workflows that this
         *        action triggers.</p>
         *        <p>
         *        If your user pool configuration includes triggers, the AdminConfirmSignUp API action invokes the
         *        Lambda function that is specified for the <i>post confirmation</i> trigger. When Amazon Cognito
         *        invokes this function, it passes a JSON payload, which the function receives as input. In this
         *        payload, the <code>clientMetadata</code> attribute provides the data that you assigned to the
         *        ClientMetadata parameter in your AdminConfirmSignUp request. In your function code in Lambda, you can
         *        process the ClientMetadata value to enhance your workflow for your specific needs.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a
         *        user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         *        ClientMetadata parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Validate the ClientMetadata value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String username;

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdminConfirmSignUpRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            username(model.username);
            clientMetadata(model.clientMetadata);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final String getUsername() {
            return username;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AdminConfirmSignUpRequest build() {
            return new AdminConfirmSignUpRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
