/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to list users.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ListUsersRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<ListUsersRequest.Builder, ListUsersRequest> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(ListUsersRequest::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<List<String>> ATTRIBUTES_TO_GET_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AttributesToGet")
            .getter(getter(ListUsersRequest::attributesToGet))
            .setter(setter(Builder::attributesToGet))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributesToGet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> LIMIT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Limit")
            .getter(getter(ListUsersRequest::limit)).setter(setter(Builder::limit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Limit").build()).build();

    private static final SdkField<String> PAGINATION_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PaginationToken").getter(getter(ListUsersRequest::paginationToken))
            .setter(setter(Builder::paginationToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PaginationToken").build()).build();

    private static final SdkField<String> FILTER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Filter")
            .getter(getter(ListUsersRequest::filter)).setter(setter(Builder::filter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filter").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            ATTRIBUTES_TO_GET_FIELD, LIMIT_FIELD, PAGINATION_TOKEN_FIELD, FILTER_FIELD));

    private final String userPoolId;

    private final List<String> attributesToGet;

    private final Integer limit;

    private final String paginationToken;

    private final String filter;

    private ListUsersRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.attributesToGet = builder.attributesToGet;
        this.limit = builder.limit;
        this.paginationToken = builder.paginationToken;
        this.filter = builder.filter;
    }

    /**
     * <p>
     * The user pool ID for the user pool on which the search should be performed.
     * </p>
     * 
     * @return The user pool ID for the user pool on which the search should be performed.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * For responses, this returns true if the service returned a value for the AttributesToGet property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAttributesToGet() {
        return attributesToGet != null && !(attributesToGet instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A JSON array of user attribute names, for example <code>given_name</code>, that you want Amazon Cognito to
     * include in the response for each user. When you don't provide an <code>AttributesToGet</code> parameter, Amazon
     * Cognito returns all attributes for each user.
     * </p>
     * <p>
     * Use <code>AttributesToGet</code> with required attributes in your user pool, or in conjunction with
     * <code>Filter</code>. Amazon Cognito returns an error if not all users in the results have set a value for the
     * attribute you request. Attributes that you can't filter on, including custom attributes, must have a value set in
     * every user profile before an <code>AttributesToGet</code> parameter returns results.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributesToGet} method.
     * </p>
     * 
     * @return A JSON array of user attribute names, for example <code>given_name</code>, that you want Amazon Cognito
     *         to include in the response for each user. When you don't provide an <code>AttributesToGet</code>
     *         parameter, Amazon Cognito returns all attributes for each user.</p>
     *         <p>
     *         Use <code>AttributesToGet</code> with required attributes in your user pool, or in conjunction with
     *         <code>Filter</code>. Amazon Cognito returns an error if not all users in the results have set a value for
     *         the attribute you request. Attributes that you can't filter on, including custom attributes, must have a
     *         value set in every user profile before an <code>AttributesToGet</code> parameter returns results.
     */
    public final List<String> attributesToGet() {
        return attributesToGet;
    }

    /**
     * <p>
     * Maximum number of users to be returned.
     * </p>
     * 
     * @return Maximum number of users to be returned.
     */
    public final Integer limit() {
        return limit;
    }

    /**
     * <p>
     * This API operation returns a limited number of results. The pagination token is an identifier that you can
     * present in an additional API request with the same parameters. When you include the pagination token, Amazon
     * Cognito returns the next set of items after the current list. Subsequent requests return a new pagination token.
     * By use of this token, you can paginate through the full list of items.
     * </p>
     * 
     * @return This API operation returns a limited number of results. The pagination token is an identifier that you
     *         can present in an additional API request with the same parameters. When you include the pagination token,
     *         Amazon Cognito returns the next set of items after the current list. Subsequent requests return a new
     *         pagination token. By use of this token, you can paginate through the full list of items.
     */
    public final String paginationToken() {
        return paginationToken;
    }

    /**
     * <p>
     * A filter string of the form "<i>AttributeName</i> <i>Filter-Type</i> "<i>AttributeValue</i>"". Quotation marks
     * within the filter string must be escaped using the backslash (<code>\</code>) character. For example,
     * <code>"family_name = \"Reddy\""</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <i>AttributeName</i>: The name of the attribute to search for. You can only search for one attribute at a time.
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>Filter-Type</i>: For an exact match, use <code>=</code>, for example, "<code>given_name = \"Jon\"</code>
     * ". For a prefix ("starts with") match, use <code>^=</code>, for example, "<code>given_name ^= \"Jon\"</code>".
     * </p>
     * </li>
     * <li>
     * <p>
     * <i>AttributeValue</i>: The attribute value that must be matched for each user.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the filter string is empty, <code>ListUsers</code> returns all users in the user pool.
     * </p>
     * <p>
     * You can only search for the following standard attributes:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>username</code> (case-sensitive)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>email</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>phone_number</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>name</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>given_name</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>family_name</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>preferred_username</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cognito:user_status</code> (called <b>Status</b> in the Console) (case-insensitive)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>status (called <b>Enabled</b> in the Console) (case-sensitive)</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sub</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * Custom attributes aren't searchable.
     * </p>
     * <note>
     * <p>
     * You can also list users with a client-side filter. The server-side filter matches no more than one attribute. For
     * an advanced search, use a client-side filter with the <code>--query</code> parameter of the
     * <code>list-users</code> action in the CLI. When you use a client-side filter, ListUsers returns a paginated list
     * of zero or more users. You can receive multiple pages in a row with zero results. Repeat the query with each
     * pagination token that is returned until you receive a null pagination token value, and then review the combined
     * result.
     * </p>
     * <p>
     * For more information about server-side and client-side filtering, see <a
     * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-filter.html">FilteringCLI output</a> in the <a
     * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-filter.html">Command Line Interface User
     * Guide</a>.
     * </p>
     * </note>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-manage-user-accounts.html#cognito-user-pools-searching-for-users-using-listusers-api"
     * >Searching for Users Using the ListUsers API</a> and <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-manage-user-accounts.html#cognito-user-pools-searching-for-users-listusers-api-examples"
     * >Examples of Using the ListUsers API</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * 
     * @return A filter string of the form "<i>AttributeName</i> <i>Filter-Type</i> "<i>AttributeValue</i>"". Quotation
     *         marks within the filter string must be escaped using the backslash (<code>\</code>) character. For
     *         example, <code>"family_name = \"Reddy\""</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <i>AttributeName</i>: The name of the attribute to search for. You can only search for one attribute at a
     *         time.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>Filter-Type</i>: For an exact match, use <code>=</code>, for example, "
     *         <code>given_name = \"Jon\"</code>". For a prefix ("starts with") match, use <code>^=</code>, for example,
     *         "<code>given_name ^= \"Jon\"</code>".
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <i>AttributeValue</i>: The attribute value that must be matched for each user.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the filter string is empty, <code>ListUsers</code> returns all users in the user pool.
     *         </p>
     *         <p>
     *         You can only search for the following standard attributes:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>username</code> (case-sensitive)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>email</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>phone_number</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>name</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>given_name</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>family_name</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>preferred_username</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cognito:user_status</code> (called <b>Status</b> in the Console) (case-insensitive)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>status (called <b>Enabled</b> in the Console) (case-sensitive)</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sub</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Custom attributes aren't searchable.
     *         </p>
     *         <note>
     *         <p>
     *         You can also list users with a client-side filter. The server-side filter matches no more than one
     *         attribute. For an advanced search, use a client-side filter with the <code>--query</code> parameter of
     *         the <code>list-users</code> action in the CLI. When you use a client-side filter, ListUsers returns a
     *         paginated list of zero or more users. You can receive multiple pages in a row with zero results. Repeat
     *         the query with each pagination token that is returned until you receive a null pagination token value,
     *         and then review the combined result.
     *         </p>
     *         <p>
     *         For more information about server-side and client-side filtering, see <a
     *         href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-filter.html">FilteringCLI output</a> in
     *         the <a href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-filter.html">Command Line
     *         Interface User Guide</a>.
     *         </p>
     *         </note>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-manage-user-accounts.html#cognito-user-pools-searching-for-users-using-listusers-api"
     *         >Searching for Users Using the ListUsers API</a> and <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-manage-user-accounts.html#cognito-user-pools-searching-for-users-listusers-api-examples"
     *         >Examples of Using the ListUsers API</a> in the <i>Amazon Cognito Developer Guide</i>.
     */
    public final String filter() {
        return filter;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributesToGet() ? attributesToGet() : null);
        hashCode = 31 * hashCode + Objects.hashCode(limit());
        hashCode = 31 * hashCode + Objects.hashCode(paginationToken());
        hashCode = 31 * hashCode + Objects.hashCode(filter());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ListUsersRequest)) {
            return false;
        }
        ListUsersRequest other = (ListUsersRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && hasAttributesToGet() == other.hasAttributesToGet()
                && Objects.equals(attributesToGet(), other.attributesToGet()) && Objects.equals(limit(), other.limit())
                && Objects.equals(paginationToken(), other.paginationToken()) && Objects.equals(filter(), other.filter());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ListUsersRequest").add("UserPoolId", userPoolId())
                .add("AttributesToGet", hasAttributesToGet() ? attributesToGet() : null).add("Limit", limit())
                .add("PaginationToken", paginationToken()).add("Filter", filter()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "AttributesToGet":
            return Optional.ofNullable(clazz.cast(attributesToGet()));
        case "Limit":
            return Optional.ofNullable(clazz.cast(limit()));
        case "PaginationToken":
            return Optional.ofNullable(clazz.cast(paginationToken()));
        case "Filter":
            return Optional.ofNullable(clazz.cast(filter()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ListUsersRequest, T> g) {
        return obj -> g.apply((ListUsersRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo, CopyableBuilder<Builder, ListUsersRequest> {
        /**
         * <p>
         * The user pool ID for the user pool on which the search should be performed.
         * </p>
         * 
         * @param userPoolId
         *        The user pool ID for the user pool on which the search should be performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * A JSON array of user attribute names, for example <code>given_name</code>, that you want Amazon Cognito to
         * include in the response for each user. When you don't provide an <code>AttributesToGet</code> parameter,
         * Amazon Cognito returns all attributes for each user.
         * </p>
         * <p>
         * Use <code>AttributesToGet</code> with required attributes in your user pool, or in conjunction with
         * <code>Filter</code>. Amazon Cognito returns an error if not all users in the results have set a value for the
         * attribute you request. Attributes that you can't filter on, including custom attributes, must have a value
         * set in every user profile before an <code>AttributesToGet</code> parameter returns results.
         * </p>
         * 
         * @param attributesToGet
         *        A JSON array of user attribute names, for example <code>given_name</code>, that you want Amazon
         *        Cognito to include in the response for each user. When you don't provide an
         *        <code>AttributesToGet</code> parameter, Amazon Cognito returns all attributes for each user.</p>
         *        <p>
         *        Use <code>AttributesToGet</code> with required attributes in your user pool, or in conjunction with
         *        <code>Filter</code>. Amazon Cognito returns an error if not all users in the results have set a value
         *        for the attribute you request. Attributes that you can't filter on, including custom attributes, must
         *        have a value set in every user profile before an <code>AttributesToGet</code> parameter returns
         *        results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributesToGet(Collection<String> attributesToGet);

        /**
         * <p>
         * A JSON array of user attribute names, for example <code>given_name</code>, that you want Amazon Cognito to
         * include in the response for each user. When you don't provide an <code>AttributesToGet</code> parameter,
         * Amazon Cognito returns all attributes for each user.
         * </p>
         * <p>
         * Use <code>AttributesToGet</code> with required attributes in your user pool, or in conjunction with
         * <code>Filter</code>. Amazon Cognito returns an error if not all users in the results have set a value for the
         * attribute you request. Attributes that you can't filter on, including custom attributes, must have a value
         * set in every user profile before an <code>AttributesToGet</code> parameter returns results.
         * </p>
         * 
         * @param attributesToGet
         *        A JSON array of user attribute names, for example <code>given_name</code>, that you want Amazon
         *        Cognito to include in the response for each user. When you don't provide an
         *        <code>AttributesToGet</code> parameter, Amazon Cognito returns all attributes for each user.</p>
         *        <p>
         *        Use <code>AttributesToGet</code> with required attributes in your user pool, or in conjunction with
         *        <code>Filter</code>. Amazon Cognito returns an error if not all users in the results have set a value
         *        for the attribute you request. Attributes that you can't filter on, including custom attributes, must
         *        have a value set in every user profile before an <code>AttributesToGet</code> parameter returns
         *        results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributesToGet(String... attributesToGet);

        /**
         * <p>
         * Maximum number of users to be returned.
         * </p>
         * 
         * @param limit
         *        Maximum number of users to be returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder limit(Integer limit);

        /**
         * <p>
         * This API operation returns a limited number of results. The pagination token is an identifier that you can
         * present in an additional API request with the same parameters. When you include the pagination token, Amazon
         * Cognito returns the next set of items after the current list. Subsequent requests return a new pagination
         * token. By use of this token, you can paginate through the full list of items.
         * </p>
         * 
         * @param paginationToken
         *        This API operation returns a limited number of results. The pagination token is an identifier that you
         *        can present in an additional API request with the same parameters. When you include the pagination
         *        token, Amazon Cognito returns the next set of items after the current list. Subsequent requests return
         *        a new pagination token. By use of this token, you can paginate through the full list of items.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder paginationToken(String paginationToken);

        /**
         * <p>
         * A filter string of the form "<i>AttributeName</i> <i>Filter-Type</i> "<i>AttributeValue</i>"". Quotation
         * marks within the filter string must be escaped using the backslash (<code>\</code>) character. For example,
         * <code>"family_name = \"Reddy\""</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <i>AttributeName</i>: The name of the attribute to search for. You can only search for one attribute at a
         * time.
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>Filter-Type</i>: For an exact match, use <code>=</code>, for example, "<code>given_name = \"Jon\"</code>
         * ". For a prefix ("starts with") match, use <code>^=</code>, for example, "<code>given_name ^= \"Jon\"</code>
         * ".
         * </p>
         * </li>
         * <li>
         * <p>
         * <i>AttributeValue</i>: The attribute value that must be matched for each user.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the filter string is empty, <code>ListUsers</code> returns all users in the user pool.
         * </p>
         * <p>
         * You can only search for the following standard attributes:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>username</code> (case-sensitive)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>email</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>phone_number</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>name</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>given_name</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>family_name</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>preferred_username</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cognito:user_status</code> (called <b>Status</b> in the Console) (case-insensitive)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>status (called <b>Enabled</b> in the Console) (case-sensitive)</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sub</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * Custom attributes aren't searchable.
         * </p>
         * <note>
         * <p>
         * You can also list users with a client-side filter. The server-side filter matches no more than one attribute.
         * For an advanced search, use a client-side filter with the <code>--query</code> parameter of the
         * <code>list-users</code> action in the CLI. When you use a client-side filter, ListUsers returns a paginated
         * list of zero or more users. You can receive multiple pages in a row with zero results. Repeat the query with
         * each pagination token that is returned until you receive a null pagination token value, and then review the
         * combined result.
         * </p>
         * <p>
         * For more information about server-side and client-side filtering, see <a
         * href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-filter.html">FilteringCLI output</a> in the
         * <a href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-filter.html">Command Line Interface User
         * Guide</a>.
         * </p>
         * </note>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-manage-user-accounts.html#cognito-user-pools-searching-for-users-using-listusers-api"
         * >Searching for Users Using the ListUsers API</a> and <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-manage-user-accounts.html#cognito-user-pools-searching-for-users-listusers-api-examples"
         * >Examples of Using the ListUsers API</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * 
         * @param filter
         *        A filter string of the form "<i>AttributeName</i> <i>Filter-Type</i> "<i>AttributeValue</i>"".
         *        Quotation marks within the filter string must be escaped using the backslash (<code>\</code>)
         *        character. For example, <code>"family_name = \"Reddy\""</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <i>AttributeName</i>: The name of the attribute to search for. You can only search for one attribute
         *        at a time.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>Filter-Type</i>: For an exact match, use <code>=</code>, for example, "
         *        <code>given_name = \"Jon\"</code>". For a prefix ("starts with") match, use <code>^=</code>, for
         *        example, "<code>given_name ^= \"Jon\"</code>".
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <i>AttributeValue</i>: The attribute value that must be matched for each user.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the filter string is empty, <code>ListUsers</code> returns all users in the user pool.
         *        </p>
         *        <p>
         *        You can only search for the following standard attributes:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>username</code> (case-sensitive)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>email</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>phone_number</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>name</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>given_name</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>family_name</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>preferred_username</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cognito:user_status</code> (called <b>Status</b> in the Console) (case-insensitive)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>status (called <b>Enabled</b> in the Console) (case-sensitive)</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sub</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Custom attributes aren't searchable.
         *        </p>
         *        <note>
         *        <p>
         *        You can also list users with a client-side filter. The server-side filter matches no more than one
         *        attribute. For an advanced search, use a client-side filter with the <code>--query</code> parameter of
         *        the <code>list-users</code> action in the CLI. When you use a client-side filter, ListUsers returns a
         *        paginated list of zero or more users. You can receive multiple pages in a row with zero results.
         *        Repeat the query with each pagination token that is returned until you receive a null pagination token
         *        value, and then review the combined result.
         *        </p>
         *        <p>
         *        For more information about server-side and client-side filtering, see <a
         *        href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-filter.html">FilteringCLI output</a>
         *        in the <a href="https://docs.aws.amazon.com/cli/latest/userguide/cli-usage-filter.html">Command Line
         *        Interface User Guide</a>.
         *        </p>
         *        </note>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-manage-user-accounts.html#cognito-user-pools-searching-for-users-using-listusers-api"
         *        >Searching for Users Using the ListUsers API</a> and <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-manage-user-accounts.html#cognito-user-pools-searching-for-users-listusers-api-examples"
         *        >Examples of Using the ListUsers API</a> in the <i>Amazon Cognito Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filter(String filter);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private List<String> attributesToGet = DefaultSdkAutoConstructList.getInstance();

        private Integer limit;

        private String paginationToken;

        private String filter;

        private BuilderImpl() {
        }

        private BuilderImpl(ListUsersRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            attributesToGet(model.attributesToGet);
            limit(model.limit);
            paginationToken(model.paginationToken);
            filter(model.filter);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final Collection<String> getAttributesToGet() {
            if (attributesToGet instanceof SdkAutoConstructList) {
                return null;
            }
            return attributesToGet;
        }

        public final void setAttributesToGet(Collection<String> attributesToGet) {
            this.attributesToGet = SearchedAttributeNamesListTypeCopier.copy(attributesToGet);
        }

        @Override
        public final Builder attributesToGet(Collection<String> attributesToGet) {
            this.attributesToGet = SearchedAttributeNamesListTypeCopier.copy(attributesToGet);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder attributesToGet(String... attributesToGet) {
            attributesToGet(Arrays.asList(attributesToGet));
            return this;
        }

        public final Integer getLimit() {
            return limit;
        }

        public final void setLimit(Integer limit) {
            this.limit = limit;
        }

        @Override
        public final Builder limit(Integer limit) {
            this.limit = limit;
            return this;
        }

        public final String getPaginationToken() {
            return paginationToken;
        }

        public final void setPaginationToken(String paginationToken) {
            this.paginationToken = paginationToken;
        }

        @Override
        public final Builder paginationToken(String paginationToken) {
            this.paginationToken = paginationToken;
            return this;
        }

        public final String getFilter() {
            return filter;
        }

        public final void setFilter(String filter) {
            this.filter = filter;
        }

        @Override
        public final Builder filter(String filter) {
            this.filter = filter;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ListUsersRequest build() {
            return new ListUsersRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
