/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to confirm registration of a user.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfirmSignUpRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<ConfirmSignUpRequest.Builder, ConfirmSignUpRequest> {
    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientId").getter(getter(ConfirmSignUpRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<String> SECRET_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretHash").getter(getter(ConfirmSignUpRequest::secretHash)).setter(setter(Builder::secretHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretHash").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(ConfirmSignUpRequest::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<String> CONFIRMATION_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfirmationCode").getter(getter(ConfirmSignUpRequest::confirmationCode))
            .setter(setter(Builder::confirmationCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfirmationCode").build()).build();

    private static final SdkField<Boolean> FORCE_ALIAS_CREATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ForceAliasCreation").getter(getter(ConfirmSignUpRequest::forceAliasCreation))
            .setter(setter(Builder::forceAliasCreation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForceAliasCreation").build())
            .build();

    private static final SdkField<AnalyticsMetadataType> ANALYTICS_METADATA_FIELD = SdkField
            .<AnalyticsMetadataType> builder(MarshallingType.SDK_POJO).memberName("AnalyticsMetadata")
            .getter(getter(ConfirmSignUpRequest::analyticsMetadata)).setter(setter(Builder::analyticsMetadata))
            .constructor(AnalyticsMetadataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalyticsMetadata").build()).build();

    private static final SdkField<UserContextDataType> USER_CONTEXT_DATA_FIELD = SdkField
            .<UserContextDataType> builder(MarshallingType.SDK_POJO).memberName("UserContextData")
            .getter(getter(ConfirmSignUpRequest::userContextData)).setter(setter(Builder::userContextData))
            .constructor(UserContextDataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserContextData").build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(ConfirmSignUpRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_ID_FIELD,
            SECRET_HASH_FIELD, USERNAME_FIELD, CONFIRMATION_CODE_FIELD, FORCE_ALIAS_CREATION_FIELD, ANALYTICS_METADATA_FIELD,
            USER_CONTEXT_DATA_FIELD, CLIENT_METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("ClientId", CLIENT_ID_FIELD);
                    put("SecretHash", SECRET_HASH_FIELD);
                    put("Username", USERNAME_FIELD);
                    put("ConfirmationCode", CONFIRMATION_CODE_FIELD);
                    put("ForceAliasCreation", FORCE_ALIAS_CREATION_FIELD);
                    put("AnalyticsMetadata", ANALYTICS_METADATA_FIELD);
                    put("UserContextData", USER_CONTEXT_DATA_FIELD);
                    put("ClientMetadata", CLIENT_METADATA_FIELD);
                }
            });

    private final String clientId;

    private final String secretHash;

    private final String username;

    private final String confirmationCode;

    private final Boolean forceAliasCreation;

    private final AnalyticsMetadataType analyticsMetadata;

    private final UserContextDataType userContextData;

    private final Map<String, String> clientMetadata;

    private ConfirmSignUpRequest(BuilderImpl builder) {
        super(builder);
        this.clientId = builder.clientId;
        this.secretHash = builder.secretHash;
        this.username = builder.username;
        this.confirmationCode = builder.confirmationCode;
        this.forceAliasCreation = builder.forceAliasCreation;
        this.analyticsMetadata = builder.analyticsMetadata;
        this.userContextData = builder.userContextData;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The ID of the app client associated with the user pool.
     * </p>
     * 
     * @return The ID of the app client associated with the user pool.
     */
    public final String clientId() {
        return clientId;
    }

    /**
     * <p>
     * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     * username plus the client ID in the message.
     * </p>
     * 
     * @return A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     *         username plus the client ID in the message.
     */
    public final String secretHash() {
        return secretHash;
    }

    /**
     * <p>
     * The username of the user that you want to query or modify. The value of this parameter is typically your user's
     * username, but it can be any of their alias attributes. If <code>username</code> isn't an alias attribute in your
     * user pool, this value must be the <code>sub</code> of a local user or the username of a user from a third-party
     * IdP.
     * </p>
     * 
     * @return The username of the user that you want to query or modify. The value of this parameter is typically your
     *         user's username, but it can be any of their alias attributes. If <code>username</code> isn't an alias
     *         attribute in your user pool, this value must be the <code>sub</code> of a local user or the username of a
     *         user from a third-party IdP.
     */
    public final String username() {
        return username;
    }

    /**
     * <p>
     * The confirmation code sent by a user's request to confirm registration.
     * </p>
     * 
     * @return The confirmation code sent by a user's request to confirm registration.
     */
    public final String confirmationCode() {
        return confirmationCode;
    }

    /**
     * <p>
     * Boolean to be specified to force user confirmation irrespective of existing alias. By default set to
     * <code>False</code>. If this parameter is set to <code>True</code> and the phone number/email used for sign up
     * confirmation already exists as an alias with a different user, the API call will migrate the alias from the
     * previous user to the newly created user being confirmed. If set to <code>False</code>, the API will throw an
     * <b>AliasExistsException</b> error.
     * </p>
     * 
     * @return Boolean to be specified to force user confirmation irrespective of existing alias. By default set to
     *         <code>False</code>. If this parameter is set to <code>True</code> and the phone number/email used for
     *         sign up confirmation already exists as an alias with a different user, the API call will migrate the
     *         alias from the previous user to the newly created user being confirmed. If set to <code>False</code>, the
     *         API will throw an <b>AliasExistsException</b> error.
     */
    public final Boolean forceAliasCreation() {
        return forceAliasCreation;
    }

    /**
     * <p>
     * The Amazon Pinpoint analytics metadata for collecting metrics for <code>ConfirmSignUp</code> calls.
     * </p>
     * 
     * @return The Amazon Pinpoint analytics metadata for collecting metrics for <code>ConfirmSignUp</code> calls.
     */
    public final AnalyticsMetadataType analyticsMetadata() {
        return analyticsMetadata;
    }

    /**
     * <p>
     * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon Cognito
     * advanced security evaluates the risk of an authentication event based on the context that your app generates and
     * passes to Amazon Cognito when it makes API requests.
     * </p>
     * 
     * @return Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
     *         Cognito advanced security evaluates the risk of an authentication event based on the context that your
     *         app generates and passes to Amazon Cognito when it makes API requests.
     */
    public final UserContextDataType userContextData() {
        return userContextData;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the ConfirmSignUp
     * API action, Amazon Cognito invokes the function that is assigned to the <i>post confirmation</i> trigger. When
     * Amazon Cognito invokes this function, it passes a JSON payload, which the function receives as input. This
     * payload contains a <code>clientMetadata</code> attribute, which provides the data that you assigned to the
     * ClientMetadata parameter in your ConfirmSignUp request. In your function code in Lambda, you can process the
     * <code>clientMetadata</code> value to enhance your workflow for your specific needs.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a user pool
     * to support custom workflows. If your user pool configuration doesn't include triggers, the ClientMetadata
     * parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Validate the ClientMetadata value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientMetadata} method.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for any custom workflows that this action
     *         triggers.</p>
     *         <p>
     *         You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     *         ConfirmSignUp API action, Amazon Cognito invokes the function that is assigned to the <i>post
     *         confirmation</i> trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the
     *         function receives as input. This payload contains a <code>clientMetadata</code> attribute, which provides
     *         the data that you assigned to the ClientMetadata parameter in your ConfirmSignUp request. In your
     *         function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your workflow
     *         for your specific needs.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a
     *         user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
     *         ClientMetadata parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Validate the ClientMetadata value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(secretHash());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationCode());
        hashCode = 31 * hashCode + Objects.hashCode(forceAliasCreation());
        hashCode = 31 * hashCode + Objects.hashCode(analyticsMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(userContextData());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfirmSignUpRequest)) {
            return false;
        }
        ConfirmSignUpRequest other = (ConfirmSignUpRequest) obj;
        return Objects.equals(clientId(), other.clientId()) && Objects.equals(secretHash(), other.secretHash())
                && Objects.equals(username(), other.username()) && Objects.equals(confirmationCode(), other.confirmationCode())
                && Objects.equals(forceAliasCreation(), other.forceAliasCreation())
                && Objects.equals(analyticsMetadata(), other.analyticsMetadata())
                && Objects.equals(userContextData(), other.userContextData()) && hasClientMetadata() == other.hasClientMetadata()
                && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConfirmSignUpRequest")
                .add("ClientId", clientId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretHash", secretHash() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Username", username() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ConfirmationCode", confirmationCode()).add("ForceAliasCreation", forceAliasCreation())
                .add("AnalyticsMetadata", analyticsMetadata())
                .add("UserContextData", userContextData() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "SecretHash":
            return Optional.ofNullable(clazz.cast(secretHash()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "ConfirmationCode":
            return Optional.ofNullable(clazz.cast(confirmationCode()));
        case "ForceAliasCreation":
            return Optional.ofNullable(clazz.cast(forceAliasCreation()));
        case "AnalyticsMetadata":
            return Optional.ofNullable(clazz.cast(analyticsMetadata()));
        case "UserContextData":
            return Optional.ofNullable(clazz.cast(userContextData()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ConfirmSignUpRequest, T> g) {
        return obj -> g.apply((ConfirmSignUpRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, ConfirmSignUpRequest> {
        /**
         * <p>
         * The ID of the app client associated with the user pool.
         * </p>
         * 
         * @param clientId
         *        The ID of the app client associated with the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
         * username plus the client ID in the message.
         * </p>
         * 
         * @param secretHash
         *        A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client
         *        and username plus the client ID in the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretHash(String secretHash);

        /**
         * <p>
         * The username of the user that you want to query or modify. The value of this parameter is typically your
         * user's username, but it can be any of their alias attributes. If <code>username</code> isn't an alias
         * attribute in your user pool, this value must be the <code>sub</code> of a local user or the username of a
         * user from a third-party IdP.
         * </p>
         * 
         * @param username
         *        The username of the user that you want to query or modify. The value of this parameter is typically
         *        your user's username, but it can be any of their alias attributes. If <code>username</code> isn't an
         *        alias attribute in your user pool, this value must be the <code>sub</code> of a local user or the
         *        username of a user from a third-party IdP.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * The confirmation code sent by a user's request to confirm registration.
         * </p>
         * 
         * @param confirmationCode
         *        The confirmation code sent by a user's request to confirm registration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder confirmationCode(String confirmationCode);

        /**
         * <p>
         * Boolean to be specified to force user confirmation irrespective of existing alias. By default set to
         * <code>False</code>. If this parameter is set to <code>True</code> and the phone number/email used for sign up
         * confirmation already exists as an alias with a different user, the API call will migrate the alias from the
         * previous user to the newly created user being confirmed. If set to <code>False</code>, the API will throw an
         * <b>AliasExistsException</b> error.
         * </p>
         * 
         * @param forceAliasCreation
         *        Boolean to be specified to force user confirmation irrespective of existing alias. By default set to
         *        <code>False</code>. If this parameter is set to <code>True</code> and the phone number/email used for
         *        sign up confirmation already exists as an alias with a different user, the API call will migrate the
         *        alias from the previous user to the newly created user being confirmed. If set to <code>False</code>,
         *        the API will throw an <b>AliasExistsException</b> error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forceAliasCreation(Boolean forceAliasCreation);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>ConfirmSignUp</code> calls.
         * </p>
         * 
         * @param analyticsMetadata
         *        The Amazon Pinpoint analytics metadata for collecting metrics for <code>ConfirmSignUp</code> calls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>ConfirmSignUp</code> calls.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalyticsMetadataType.Builder} avoiding
         * the need to create one manually via {@link AnalyticsMetadataType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnalyticsMetadataType.Builder#build()} is called immediately and
         * its result is passed to {@link #analyticsMetadata(AnalyticsMetadataType)}.
         * 
         * @param analyticsMetadata
         *        a consumer that will call methods on {@link AnalyticsMetadataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #analyticsMetadata(AnalyticsMetadataType)
         */
        default Builder analyticsMetadata(Consumer<AnalyticsMetadataType.Builder> analyticsMetadata) {
            return analyticsMetadata(AnalyticsMetadataType.builder().applyMutation(analyticsMetadata).build());
        }

        /**
         * <p>
         * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
         * Cognito advanced security evaluates the risk of an authentication event based on the context that your app
         * generates and passes to Amazon Cognito when it makes API requests.
         * </p>
         * 
         * @param userContextData
         *        Contextual data about your user session, such as the device fingerprint, IP address, or location.
         *        Amazon Cognito advanced security evaluates the risk of an authentication event based on the context
         *        that your app generates and passes to Amazon Cognito when it makes API requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userContextData(UserContextDataType userContextData);

        /**
         * <p>
         * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
         * Cognito advanced security evaluates the risk of an authentication event based on the context that your app
         * generates and passes to Amazon Cognito when it makes API requests.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserContextDataType.Builder} avoiding the
         * need to create one manually via {@link UserContextDataType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UserContextDataType.Builder#build()} is called immediately and
         * its result is passed to {@link #userContextData(UserContextDataType)}.
         * 
         * @param userContextData
         *        a consumer that will call methods on {@link UserContextDataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userContextData(UserContextDataType)
         */
        default Builder userContextData(Consumer<UserContextDataType.Builder> userContextData) {
            return userContextData(UserContextDataType.builder().applyMutation(userContextData).build());
        }

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for any custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         * ConfirmSignUp API action, Amazon Cognito invokes the function that is assigned to the <i>post
         * confirmation</i> trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the
         * function receives as input. This payload contains a <code>clientMetadata</code> attribute, which provides the
         * data that you assigned to the ClientMetadata parameter in your ConfirmSignUp request. In your function code
         * in Lambda, you can process the <code>clientMetadata</code> value to enhance your workflow for your specific
         * needs.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a user
         * pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         * ClientMetadata parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Validate the ClientMetadata value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for any custom workflows that this
         *        action triggers.</p>
         *        <p>
         *        You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         *        ConfirmSignUp API action, Amazon Cognito invokes the function that is assigned to the <i>post
         *        confirmation</i> trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which
         *        the function receives as input. This payload contains a <code>clientMetadata</code> attribute, which
         *        provides the data that you assigned to the ClientMetadata parameter in your ConfirmSignUp request. In
         *        your function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your
         *        workflow for your specific needs.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        When you use the ClientMetadata parameter, remember that Amazon Cognito won't do the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Store the ClientMetadata value. This data is available only to Lambda triggers that are assigned to a
         *        user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         *        ClientMetadata parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Validate the ClientMetadata value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Encrypt the ClientMetadata value. Don't use Amazon Cognito to provide sensitive information.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String clientId;

        private String secretHash;

        private String username;

        private String confirmationCode;

        private Boolean forceAliasCreation;

        private AnalyticsMetadataType analyticsMetadata;

        private UserContextDataType userContextData;

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConfirmSignUpRequest model) {
            super(model);
            clientId(model.clientId);
            secretHash(model.secretHash);
            username(model.username);
            confirmationCode(model.confirmationCode);
            forceAliasCreation(model.forceAliasCreation);
            analyticsMetadata(model.analyticsMetadata);
            userContextData(model.userContextData);
            clientMetadata(model.clientMetadata);
        }

        public final String getClientId() {
            return clientId;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final String getSecretHash() {
            return secretHash;
        }

        public final void setSecretHash(String secretHash) {
            this.secretHash = secretHash;
        }

        @Override
        public final Builder secretHash(String secretHash) {
            this.secretHash = secretHash;
            return this;
        }

        public final String getUsername() {
            return username;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final String getConfirmationCode() {
            return confirmationCode;
        }

        public final void setConfirmationCode(String confirmationCode) {
            this.confirmationCode = confirmationCode;
        }

        @Override
        public final Builder confirmationCode(String confirmationCode) {
            this.confirmationCode = confirmationCode;
            return this;
        }

        public final Boolean getForceAliasCreation() {
            return forceAliasCreation;
        }

        public final void setForceAliasCreation(Boolean forceAliasCreation) {
            this.forceAliasCreation = forceAliasCreation;
        }

        @Override
        public final Builder forceAliasCreation(Boolean forceAliasCreation) {
            this.forceAliasCreation = forceAliasCreation;
            return this;
        }

        public final AnalyticsMetadataType.Builder getAnalyticsMetadata() {
            return analyticsMetadata != null ? analyticsMetadata.toBuilder() : null;
        }

        public final void setAnalyticsMetadata(AnalyticsMetadataType.BuilderImpl analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata != null ? analyticsMetadata.build() : null;
        }

        @Override
        public final Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata;
            return this;
        }

        public final UserContextDataType.Builder getUserContextData() {
            return userContextData != null ? userContextData.toBuilder() : null;
        }

        public final void setUserContextData(UserContextDataType.BuilderImpl userContextData) {
            this.userContextData = userContextData != null ? userContextData.build() : null;
        }

        @Override
        public final Builder userContextData(UserContextDataType userContextData) {
            this.userContextData = userContextData;
            return this;
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ConfirmSignUpRequest build() {
            return new ConfirmSignUpRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
