/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A recovery option for a user. The <code>AccountRecoverySettingType</code> data type is an array of this object. Each
 * <code>RecoveryOptionType</code> has a priority property that determines whether it is a primary or secondary option.
 * </p>
 * <p>
 * For example, if <code>verified_email</code> has a priority of <code>1</code> and <code>verified_phone_number</code>
 * has a priority of <code>2</code>, your user pool sends account-recovery messages to a verified email address but
 * falls back to an SMS message if the user has a verified phone number. The <code>admin_only</code> option prevents
 * self-service account recovery.
 * </p>
 * <p>
 * This data type is a request and response parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html"
 * >CreateUserPool</a> and <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html"
 * >UpdateUserPool</a>, and a response parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPool.html"
 * >DescribeUserPool</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecoveryOptionType implements SdkPojo, Serializable,
        ToCopyableBuilder<RecoveryOptionType.Builder, RecoveryOptionType> {
    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority").getter(getter(RecoveryOptionType::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(RecoveryOptionType::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PRIORITY_FIELD, NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Priority", PRIORITY_FIELD);
                    put("Name", NAME_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final Integer priority;

    private final String name;

    private RecoveryOptionType(BuilderImpl builder) {
        this.priority = builder.priority;
        this.name = builder.name;
    }

    /**
     * <p>
     * Your priority preference for using the specified attribute in account recovery. The highest priority is
     * <code>1</code>.
     * </p>
     * 
     * @return Your priority preference for using the specified attribute in account recovery. The highest priority is
     *         <code>1</code>.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * <p>
     * The recovery method that this object sets a recovery option for.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link RecoveryOptionNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The recovery method that this object sets a recovery option for.
     * @see RecoveryOptionNameType
     */
    public final RecoveryOptionNameType name() {
        return RecoveryOptionNameType.fromValue(name);
    }

    /**
     * <p>
     * The recovery method that this object sets a recovery option for.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link RecoveryOptionNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The recovery method that this object sets a recovery option for.
     * @see RecoveryOptionNameType
     */
    public final String nameAsString() {
        return name;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecoveryOptionType)) {
            return false;
        }
        RecoveryOptionType other = (RecoveryOptionType) obj;
        return Objects.equals(priority(), other.priority()) && Objects.equals(nameAsString(), other.nameAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecoveryOptionType").add("Priority", priority()).add("Name", nameAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "Name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<RecoveryOptionType, T> g) {
        return obj -> g.apply((RecoveryOptionType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecoveryOptionType> {
        /**
         * <p>
         * Your priority preference for using the specified attribute in account recovery. The highest priority is
         * <code>1</code>.
         * </p>
         * 
         * @param priority
         *        Your priority preference for using the specified attribute in account recovery. The highest priority
         *        is <code>1</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * The recovery method that this object sets a recovery option for.
         * </p>
         * 
         * @param name
         *        The recovery method that this object sets a recovery option for.
         * @see RecoveryOptionNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecoveryOptionNameType
         */
        Builder name(String name);

        /**
         * <p>
         * The recovery method that this object sets a recovery option for.
         * </p>
         * 
         * @param name
         *        The recovery method that this object sets a recovery option for.
         * @see RecoveryOptionNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecoveryOptionNameType
         */
        Builder name(RecoveryOptionNameType name);
    }

    static final class BuilderImpl implements Builder {
        private Integer priority;

        private String name;

        private BuilderImpl() {
        }

        private BuilderImpl(RecoveryOptionType model) {
            priority(model.priority);
            name(model.name);
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(RecoveryOptionNameType name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        @Override
        public RecoveryOptionType build() {
            return new RecoveryOptionType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
