/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list of the user attributes and their properties in your user pool. The attribute schema contains standard
 * attributes, custom attributes with a <code>custom:</code> prefix, and developer attributes with a <code>dev:</code>
 * prefix. For more information, see <a
 * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html">User pool
 * attributes</a>.
 * </p>
 * <p>
 * Developer-only <code>dev:</code> attributes are a legacy feature of user pools, and are read-only to all app clients.
 * You can create and update developer-only attributes only with IAM-authenticated API operations. Use app client
 * read/write permissions instead.
 * </p>
 * <p>
 * This data type is a request and response parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html"
 * >CreateUserPool</a> and <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html"
 * >UpdateUserPool</a>, and a response parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPool.html"
 * >DescribeUserPool</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SchemaAttributeType implements SdkPojo, Serializable,
        ToCopyableBuilder<SchemaAttributeType.Builder, SchemaAttributeType> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(SchemaAttributeType::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ATTRIBUTE_DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AttributeDataType").getter(getter(SchemaAttributeType::attributeDataTypeAsString))
            .setter(setter(Builder::attributeDataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeDataType").build()).build();

    private static final SdkField<Boolean> DEVELOPER_ONLY_ATTRIBUTE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeveloperOnlyAttribute").getter(getter(SchemaAttributeType::developerOnlyAttribute))
            .setter(setter(Builder::developerOnlyAttribute))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeveloperOnlyAttribute").build())
            .build();

    private static final SdkField<Boolean> MUTABLE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Mutable").getter(getter(SchemaAttributeType::mutable)).setter(setter(Builder::mutable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mutable").build()).build();

    private static final SdkField<Boolean> REQUIRED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Required").getter(getter(SchemaAttributeType::required)).setter(setter(Builder::required))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Required").build()).build();

    private static final SdkField<NumberAttributeConstraintsType> NUMBER_ATTRIBUTE_CONSTRAINTS_FIELD = SdkField
            .<NumberAttributeConstraintsType> builder(MarshallingType.SDK_POJO)
            .memberName("NumberAttributeConstraints")
            .getter(getter(SchemaAttributeType::numberAttributeConstraints))
            .setter(setter(Builder::numberAttributeConstraints))
            .constructor(NumberAttributeConstraintsType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberAttributeConstraints").build())
            .build();

    private static final SdkField<StringAttributeConstraintsType> STRING_ATTRIBUTE_CONSTRAINTS_FIELD = SdkField
            .<StringAttributeConstraintsType> builder(MarshallingType.SDK_POJO)
            .memberName("StringAttributeConstraints")
            .getter(getter(SchemaAttributeType::stringAttributeConstraints))
            .setter(setter(Builder::stringAttributeConstraints))
            .constructor(StringAttributeConstraintsType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StringAttributeConstraints").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            ATTRIBUTE_DATA_TYPE_FIELD, DEVELOPER_ONLY_ATTRIBUTE_FIELD, MUTABLE_FIELD, REQUIRED_FIELD,
            NUMBER_ATTRIBUTE_CONSTRAINTS_FIELD, STRING_ATTRIBUTE_CONSTRAINTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Name", NAME_FIELD);
                    put("AttributeDataType", ATTRIBUTE_DATA_TYPE_FIELD);
                    put("DeveloperOnlyAttribute", DEVELOPER_ONLY_ATTRIBUTE_FIELD);
                    put("Mutable", MUTABLE_FIELD);
                    put("Required", REQUIRED_FIELD);
                    put("NumberAttributeConstraints", NUMBER_ATTRIBUTE_CONSTRAINTS_FIELD);
                    put("StringAttributeConstraints", STRING_ATTRIBUTE_CONSTRAINTS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String attributeDataType;

    private final Boolean developerOnlyAttribute;

    private final Boolean mutable;

    private final Boolean required;

    private final NumberAttributeConstraintsType numberAttributeConstraints;

    private final StringAttributeConstraintsType stringAttributeConstraints;

    private SchemaAttributeType(BuilderImpl builder) {
        this.name = builder.name;
        this.attributeDataType = builder.attributeDataType;
        this.developerOnlyAttribute = builder.developerOnlyAttribute;
        this.mutable = builder.mutable;
        this.required = builder.required;
        this.numberAttributeConstraints = builder.numberAttributeConstraints;
        this.stringAttributeConstraints = builder.stringAttributeConstraints;
    }

    /**
     * <p>
     * The name of your user pool attribute. When you create or update a user pool, adding a schema attribute creates a
     * custom or developer-only attribute. When you add an attribute with a <code>Name</code> value of
     * <code>MyAttribute</code>, Amazon Cognito creates the custom attribute <code>custom:MyAttribute</code>. When
     * <code>DeveloperOnlyAttribute</code> is <code>true</code>, Amazon Cognito creates your attribute as
     * <code>dev:MyAttribute</code>. In an operation that describes a user pool, Amazon Cognito returns this value as
     * <code>value</code> for standard attributes, <code>custom:value</code> for custom attributes, and
     * <code>dev:value</code> for developer-only attributes..
     * </p>
     * 
     * @return The name of your user pool attribute. When you create or update a user pool, adding a schema attribute
     *         creates a custom or developer-only attribute. When you add an attribute with a <code>Name</code> value of
     *         <code>MyAttribute</code>, Amazon Cognito creates the custom attribute <code>custom:MyAttribute</code>.
     *         When <code>DeveloperOnlyAttribute</code> is <code>true</code>, Amazon Cognito creates your attribute as
     *         <code>dev:MyAttribute</code>. In an operation that describes a user pool, Amazon Cognito returns this
     *         value as <code>value</code> for standard attributes, <code>custom:value</code> for custom attributes, and
     *         <code>dev:value</code> for developer-only attributes..
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The data format of the values for your attribute. When you choose an <code>AttributeDataType</code>, Amazon
     * Cognito validates the input against the data type. A custom attribute value in your user's ID token is always a
     * string, for example <code>"custom:isMember" : "true"</code> or <code>"custom:YearsAsMember" : "12"</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #attributeDataType}
     * will return {@link AttributeDataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #attributeDataTypeAsString}.
     * </p>
     * 
     * @return The data format of the values for your attribute. When you choose an <code>AttributeDataType</code>,
     *         Amazon Cognito validates the input against the data type. A custom attribute value in your user's ID
     *         token is always a string, for example <code>"custom:isMember" : "true"</code> or
     *         <code>"custom:YearsAsMember" : "12"</code>.
     * @see AttributeDataType
     */
    public final AttributeDataType attributeDataType() {
        return AttributeDataType.fromValue(attributeDataType);
    }

    /**
     * <p>
     * The data format of the values for your attribute. When you choose an <code>AttributeDataType</code>, Amazon
     * Cognito validates the input against the data type. A custom attribute value in your user's ID token is always a
     * string, for example <code>"custom:isMember" : "true"</code> or <code>"custom:YearsAsMember" : "12"</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #attributeDataType}
     * will return {@link AttributeDataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #attributeDataTypeAsString}.
     * </p>
     * 
     * @return The data format of the values for your attribute. When you choose an <code>AttributeDataType</code>,
     *         Amazon Cognito validates the input against the data type. A custom attribute value in your user's ID
     *         token is always a string, for example <code>"custom:isMember" : "true"</code> or
     *         <code>"custom:YearsAsMember" : "12"</code>.
     * @see AttributeDataType
     */
    public final String attributeDataTypeAsString() {
        return attributeDataType;
    }

    /**
     * <note>
     * <p>
     * You should use <a href=
     * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UserPoolClientType.html#CognitoUserPools-Type-UserPoolClientType-WriteAttributes"
     * >WriteAttributes</a> in the user pool client to control how attributes can be mutated for new use cases instead
     * of using <code>DeveloperOnlyAttribute</code>.
     * </p>
     * </note>
     * <p>
     * Specifies whether the attribute type is developer only. This attribute can only be modified by an administrator.
     * Users won't be able to modify this attribute using their access token. For example,
     * <code>DeveloperOnlyAttribute</code> can be modified using AdminUpdateUserAttributes but can't be updated using
     * UpdateUserAttributes.
     * </p>
     * 
     * @return <p>
     *         You should use <a href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UserPoolClientType.html#CognitoUserPools-Type-UserPoolClientType-WriteAttributes"
     *         >WriteAttributes</a> in the user pool client to control how attributes can be mutated for new use cases
     *         instead of using <code>DeveloperOnlyAttribute</code>.
     *         </p>
     *         </note>
     *         <p>
     *         Specifies whether the attribute type is developer only. This attribute can only be modified by an
     *         administrator. Users won't be able to modify this attribute using their access token. For example,
     *         <code>DeveloperOnlyAttribute</code> can be modified using AdminUpdateUserAttributes but can't be updated
     *         using UpdateUserAttributes.
     */
    public final Boolean developerOnlyAttribute() {
        return developerOnlyAttribute;
    }

    /**
     * <p>
     * Specifies whether the value of the attribute can be changed.
     * </p>
     * <p>
     * Any user pool attribute whose value you map from an IdP attribute must be mutable, with a parameter value of
     * <code>true</code>. Amazon Cognito updates mapped attributes when users sign in to your application through an
     * IdP. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html"
     * >Specifying Identity Provider Attribute Mappings for Your User Pool</a>.
     * </p>
     * 
     * @return Specifies whether the value of the attribute can be changed.</p>
     *         <p>
     *         Any user pool attribute whose value you map from an IdP attribute must be mutable, with a parameter value
     *         of <code>true</code>. Amazon Cognito updates mapped attributes when users sign in to your application
     *         through an IdP. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update
     *         the attribute. For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html"
     *         >Specifying Identity Provider Attribute Mappings for Your User Pool</a>.
     */
    public final Boolean mutable() {
        return mutable;
    }

    /**
     * <p>
     * Specifies whether a user pool attribute is required. If the attribute is required and the user doesn't provide a
     * value, registration or sign-in will fail.
     * </p>
     * 
     * @return Specifies whether a user pool attribute is required. If the attribute is required and the user doesn't
     *         provide a value, registration or sign-in will fail.
     */
    public final Boolean required() {
        return required;
    }

    /**
     * <p>
     * Specifies the constraints for an attribute of the number type.
     * </p>
     * 
     * @return Specifies the constraints for an attribute of the number type.
     */
    public final NumberAttributeConstraintsType numberAttributeConstraints() {
        return numberAttributeConstraints;
    }

    /**
     * <p>
     * Specifies the constraints for an attribute of the string type.
     * </p>
     * 
     * @return Specifies the constraints for an attribute of the string type.
     */
    public final StringAttributeConstraintsType stringAttributeConstraints() {
        return stringAttributeConstraints;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(attributeDataTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(developerOnlyAttribute());
        hashCode = 31 * hashCode + Objects.hashCode(mutable());
        hashCode = 31 * hashCode + Objects.hashCode(required());
        hashCode = 31 * hashCode + Objects.hashCode(numberAttributeConstraints());
        hashCode = 31 * hashCode + Objects.hashCode(stringAttributeConstraints());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SchemaAttributeType)) {
            return false;
        }
        SchemaAttributeType other = (SchemaAttributeType) obj;
        return Objects.equals(name(), other.name())
                && Objects.equals(attributeDataTypeAsString(), other.attributeDataTypeAsString())
                && Objects.equals(developerOnlyAttribute(), other.developerOnlyAttribute())
                && Objects.equals(mutable(), other.mutable()) && Objects.equals(required(), other.required())
                && Objects.equals(numberAttributeConstraints(), other.numberAttributeConstraints())
                && Objects.equals(stringAttributeConstraints(), other.stringAttributeConstraints());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SchemaAttributeType").add("Name", name()).add("AttributeDataType", attributeDataTypeAsString())
                .add("DeveloperOnlyAttribute", developerOnlyAttribute()).add("Mutable", mutable()).add("Required", required())
                .add("NumberAttributeConstraints", numberAttributeConstraints())
                .add("StringAttributeConstraints", stringAttributeConstraints()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "AttributeDataType":
            return Optional.ofNullable(clazz.cast(attributeDataTypeAsString()));
        case "DeveloperOnlyAttribute":
            return Optional.ofNullable(clazz.cast(developerOnlyAttribute()));
        case "Mutable":
            return Optional.ofNullable(clazz.cast(mutable()));
        case "Required":
            return Optional.ofNullable(clazz.cast(required()));
        case "NumberAttributeConstraints":
            return Optional.ofNullable(clazz.cast(numberAttributeConstraints()));
        case "StringAttributeConstraints":
            return Optional.ofNullable(clazz.cast(stringAttributeConstraints()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<SchemaAttributeType, T> g) {
        return obj -> g.apply((SchemaAttributeType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SchemaAttributeType> {
        /**
         * <p>
         * The name of your user pool attribute. When you create or update a user pool, adding a schema attribute
         * creates a custom or developer-only attribute. When you add an attribute with a <code>Name</code> value of
         * <code>MyAttribute</code>, Amazon Cognito creates the custom attribute <code>custom:MyAttribute</code>. When
         * <code>DeveloperOnlyAttribute</code> is <code>true</code>, Amazon Cognito creates your attribute as
         * <code>dev:MyAttribute</code>. In an operation that describes a user pool, Amazon Cognito returns this value
         * as <code>value</code> for standard attributes, <code>custom:value</code> for custom attributes, and
         * <code>dev:value</code> for developer-only attributes..
         * </p>
         * 
         * @param name
         *        The name of your user pool attribute. When you create or update a user pool, adding a schema attribute
         *        creates a custom or developer-only attribute. When you add an attribute with a <code>Name</code> value
         *        of <code>MyAttribute</code>, Amazon Cognito creates the custom attribute
         *        <code>custom:MyAttribute</code>. When <code>DeveloperOnlyAttribute</code> is <code>true</code>, Amazon
         *        Cognito creates your attribute as <code>dev:MyAttribute</code>. In an operation that describes a user
         *        pool, Amazon Cognito returns this value as <code>value</code> for standard attributes,
         *        <code>custom:value</code> for custom attributes, and <code>dev:value</code> for developer-only
         *        attributes..
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The data format of the values for your attribute. When you choose an <code>AttributeDataType</code>, Amazon
         * Cognito validates the input against the data type. A custom attribute value in your user's ID token is always
         * a string, for example <code>"custom:isMember" : "true"</code> or <code>"custom:YearsAsMember" : "12"</code>.
         * </p>
         * 
         * @param attributeDataType
         *        The data format of the values for your attribute. When you choose an <code>AttributeDataType</code>,
         *        Amazon Cognito validates the input against the data type. A custom attribute value in your user's ID
         *        token is always a string, for example <code>"custom:isMember" : "true"</code> or
         *        <code>"custom:YearsAsMember" : "12"</code>.
         * @see AttributeDataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AttributeDataType
         */
        Builder attributeDataType(String attributeDataType);

        /**
         * <p>
         * The data format of the values for your attribute. When you choose an <code>AttributeDataType</code>, Amazon
         * Cognito validates the input against the data type. A custom attribute value in your user's ID token is always
         * a string, for example <code>"custom:isMember" : "true"</code> or <code>"custom:YearsAsMember" : "12"</code>.
         * </p>
         * 
         * @param attributeDataType
         *        The data format of the values for your attribute. When you choose an <code>AttributeDataType</code>,
         *        Amazon Cognito validates the input against the data type. A custom attribute value in your user's ID
         *        token is always a string, for example <code>"custom:isMember" : "true"</code> or
         *        <code>"custom:YearsAsMember" : "12"</code>.
         * @see AttributeDataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AttributeDataType
         */
        Builder attributeDataType(AttributeDataType attributeDataType);

        /**
         * <note>
         * <p>
         * You should use <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UserPoolClientType.html#CognitoUserPools-Type-UserPoolClientType-WriteAttributes"
         * >WriteAttributes</a> in the user pool client to control how attributes can be mutated for new use cases
         * instead of using <code>DeveloperOnlyAttribute</code>.
         * </p>
         * </note>
         * <p>
         * Specifies whether the attribute type is developer only. This attribute can only be modified by an
         * administrator. Users won't be able to modify this attribute using their access token. For example,
         * <code>DeveloperOnlyAttribute</code> can be modified using AdminUpdateUserAttributes but can't be updated
         * using UpdateUserAttributes.
         * </p>
         * 
         * @param developerOnlyAttribute
         *        <p>
         *        You should use <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UserPoolClientType.html#CognitoUserPools-Type-UserPoolClientType-WriteAttributes"
         *        >WriteAttributes</a> in the user pool client to control how attributes can be mutated for new use
         *        cases instead of using <code>DeveloperOnlyAttribute</code>.
         *        </p>
         *        </note>
         *        <p>
         *        Specifies whether the attribute type is developer only. This attribute can only be modified by an
         *        administrator. Users won't be able to modify this attribute using their access token. For example,
         *        <code>DeveloperOnlyAttribute</code> can be modified using AdminUpdateUserAttributes but can't be
         *        updated using UpdateUserAttributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder developerOnlyAttribute(Boolean developerOnlyAttribute);

        /**
         * <p>
         * Specifies whether the value of the attribute can be changed.
         * </p>
         * <p>
         * Any user pool attribute whose value you map from an IdP attribute must be mutable, with a parameter value of
         * <code>true</code>. Amazon Cognito updates mapped attributes when users sign in to your application through an
         * IdP. If an attribute is immutable, Amazon Cognito throws an error when it attempts to update the attribute.
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html"
         * >Specifying Identity Provider Attribute Mappings for Your User Pool</a>.
         * </p>
         * 
         * @param mutable
         *        Specifies whether the value of the attribute can be changed.</p>
         *        <p>
         *        Any user pool attribute whose value you map from an IdP attribute must be mutable, with a parameter
         *        value of <code>true</code>. Amazon Cognito updates mapped attributes when users sign in to your
         *        application through an IdP. If an attribute is immutable, Amazon Cognito throws an error when it
         *        attempts to update the attribute. For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-specifying-attribute-mapping.html"
         *        >Specifying Identity Provider Attribute Mappings for Your User Pool</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mutable(Boolean mutable);

        /**
         * <p>
         * Specifies whether a user pool attribute is required. If the attribute is required and the user doesn't
         * provide a value, registration or sign-in will fail.
         * </p>
         * 
         * @param required
         *        Specifies whether a user pool attribute is required. If the attribute is required and the user doesn't
         *        provide a value, registration or sign-in will fail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder required(Boolean required);

        /**
         * <p>
         * Specifies the constraints for an attribute of the number type.
         * </p>
         * 
         * @param numberAttributeConstraints
         *        Specifies the constraints for an attribute of the number type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberAttributeConstraints(NumberAttributeConstraintsType numberAttributeConstraints);

        /**
         * <p>
         * Specifies the constraints for an attribute of the number type.
         * </p>
         * This is a convenience method that creates an instance of the {@link NumberAttributeConstraintsType.Builder}
         * avoiding the need to create one manually via {@link NumberAttributeConstraintsType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NumberAttributeConstraintsType.Builder#build()} is called
         * immediately and its result is passed to {@link #numberAttributeConstraints(NumberAttributeConstraintsType)}.
         * 
         * @param numberAttributeConstraints
         *        a consumer that will call methods on {@link NumberAttributeConstraintsType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #numberAttributeConstraints(NumberAttributeConstraintsType)
         */
        default Builder numberAttributeConstraints(Consumer<NumberAttributeConstraintsType.Builder> numberAttributeConstraints) {
            return numberAttributeConstraints(NumberAttributeConstraintsType.builder().applyMutation(numberAttributeConstraints)
                    .build());
        }

        /**
         * <p>
         * Specifies the constraints for an attribute of the string type.
         * </p>
         * 
         * @param stringAttributeConstraints
         *        Specifies the constraints for an attribute of the string type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stringAttributeConstraints(StringAttributeConstraintsType stringAttributeConstraints);

        /**
         * <p>
         * Specifies the constraints for an attribute of the string type.
         * </p>
         * This is a convenience method that creates an instance of the {@link StringAttributeConstraintsType.Builder}
         * avoiding the need to create one manually via {@link StringAttributeConstraintsType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link StringAttributeConstraintsType.Builder#build()} is called
         * immediately and its result is passed to {@link #stringAttributeConstraints(StringAttributeConstraintsType)}.
         * 
         * @param stringAttributeConstraints
         *        a consumer that will call methods on {@link StringAttributeConstraintsType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stringAttributeConstraints(StringAttributeConstraintsType)
         */
        default Builder stringAttributeConstraints(Consumer<StringAttributeConstraintsType.Builder> stringAttributeConstraints) {
            return stringAttributeConstraints(StringAttributeConstraintsType.builder().applyMutation(stringAttributeConstraints)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String attributeDataType;

        private Boolean developerOnlyAttribute;

        private Boolean mutable;

        private Boolean required;

        private NumberAttributeConstraintsType numberAttributeConstraints;

        private StringAttributeConstraintsType stringAttributeConstraints;

        private BuilderImpl() {
        }

        private BuilderImpl(SchemaAttributeType model) {
            name(model.name);
            attributeDataType(model.attributeDataType);
            developerOnlyAttribute(model.developerOnlyAttribute);
            mutable(model.mutable);
            required(model.required);
            numberAttributeConstraints(model.numberAttributeConstraints);
            stringAttributeConstraints(model.stringAttributeConstraints);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getAttributeDataType() {
            return attributeDataType;
        }

        public final void setAttributeDataType(String attributeDataType) {
            this.attributeDataType = attributeDataType;
        }

        @Override
        public final Builder attributeDataType(String attributeDataType) {
            this.attributeDataType = attributeDataType;
            return this;
        }

        @Override
        public final Builder attributeDataType(AttributeDataType attributeDataType) {
            this.attributeDataType(attributeDataType == null ? null : attributeDataType.toString());
            return this;
        }

        public final Boolean getDeveloperOnlyAttribute() {
            return developerOnlyAttribute;
        }

        public final void setDeveloperOnlyAttribute(Boolean developerOnlyAttribute) {
            this.developerOnlyAttribute = developerOnlyAttribute;
        }

        @Override
        public final Builder developerOnlyAttribute(Boolean developerOnlyAttribute) {
            this.developerOnlyAttribute = developerOnlyAttribute;
            return this;
        }

        public final Boolean getMutable() {
            return mutable;
        }

        public final void setMutable(Boolean mutable) {
            this.mutable = mutable;
        }

        @Override
        public final Builder mutable(Boolean mutable) {
            this.mutable = mutable;
            return this;
        }

        public final Boolean getRequired() {
            return required;
        }

        public final void setRequired(Boolean required) {
            this.required = required;
        }

        @Override
        public final Builder required(Boolean required) {
            this.required = required;
            return this;
        }

        public final NumberAttributeConstraintsType.Builder getNumberAttributeConstraints() {
            return numberAttributeConstraints != null ? numberAttributeConstraints.toBuilder() : null;
        }

        public final void setNumberAttributeConstraints(NumberAttributeConstraintsType.BuilderImpl numberAttributeConstraints) {
            this.numberAttributeConstraints = numberAttributeConstraints != null ? numberAttributeConstraints.build() : null;
        }

        @Override
        public final Builder numberAttributeConstraints(NumberAttributeConstraintsType numberAttributeConstraints) {
            this.numberAttributeConstraints = numberAttributeConstraints;
            return this;
        }

        public final StringAttributeConstraintsType.Builder getStringAttributeConstraints() {
            return stringAttributeConstraints != null ? stringAttributeConstraints.toBuilder() : null;
        }

        public final void setStringAttributeConstraints(StringAttributeConstraintsType.BuilderImpl stringAttributeConstraints) {
            this.stringAttributeConstraints = stringAttributeConstraints != null ? stringAttributeConstraints.build() : null;
        }

        @Override
        public final Builder stringAttributeConstraints(StringAttributeConstraintsType stringAttributeConstraints) {
            this.stringAttributeConstraints = stringAttributeConstraints;
            return this;
        }

        @Override
        public SchemaAttributeType build() {
            return new SchemaAttributeType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
