/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Settings for multi-factor authentication (MFA) with passkey, or webauthN, biometric and security-key devices in a
 * user pool. Configures the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Configuration at the user-pool level for whether you want to require passkey configuration as an MFA factor, or
 * include it as a choice.
 * </p>
 * </li>
 * <li>
 * <p>
 * The user pool relying-party ID. This is the user pool domain that user's passkey providers should trust as a receiver
 * of passkey authentication.
 * </p>
 * </li>
 * <li>
 * <p>
 * The providers that you want to allow as origins for passkey authentication.
 * </p>
 * </li>
 * </ul>
 * <p>
 * This data type is a request parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SetUserPoolMfaConfig.html"
 * >SetUserPoolMfaConfig</a> and a response parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetUserPoolMfaConfig.html"
 * >GetUserPoolMfaConfig</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class WebAuthnConfigurationType implements SdkPojo, Serializable,
        ToCopyableBuilder<WebAuthnConfigurationType.Builder, WebAuthnConfigurationType> {
    private static final SdkField<String> RELYING_PARTY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RelyingPartyId").getter(getter(WebAuthnConfigurationType::relyingPartyId))
            .setter(setter(Builder::relyingPartyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelyingPartyId").build()).build();

    private static final SdkField<String> USER_VERIFICATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserVerification").getter(getter(WebAuthnConfigurationType::userVerificationAsString))
            .setter(setter(Builder::userVerification))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserVerification").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RELYING_PARTY_ID_FIELD,
            USER_VERIFICATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("RelyingPartyId", RELYING_PARTY_ID_FIELD);
                    put("UserVerification", USER_VERIFICATION_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String relyingPartyId;

    private final String userVerification;

    private WebAuthnConfigurationType(BuilderImpl builder) {
        this.relyingPartyId = builder.relyingPartyId;
        this.userVerification = builder.userVerification;
    }

    /**
     * <p>
     * Sets or displays the authentication domain, typically your user pool domain, that passkey providers must use as a
     * relying party (RP) in their configuration.
     * </p>
     * <p>
     * Under the following conditions, the passkey relying party ID must be the fully-qualified domain name of your
     * custom domain:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The user pool is configured for passkey authentication.
     * </p>
     * </li>
     * <li>
     * <p>
     * The user pool has a custom domain, whether or not it also has a prefix domain.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your application performs authentication with managed login or the classic hosted UI.
     * </p>
     * </li>
     * </ul>
     * 
     * @return Sets or displays the authentication domain, typically your user pool domain, that passkey providers must
     *         use as a relying party (RP) in their configuration.</p>
     *         <p>
     *         Under the following conditions, the passkey relying party ID must be the fully-qualified domain name of
     *         your custom domain:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The user pool is configured for passkey authentication.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The user pool has a custom domain, whether or not it also has a prefix domain.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your application performs authentication with managed login or the classic hosted UI.
     *         </p>
     *         </li>
     */
    public final String relyingPartyId() {
        return relyingPartyId;
    }

    /**
     * <p>
     * Sets or displays your user-pool treatment for MFA with a passkey. You can override other MFA options and require
     * passkey MFA, or you can set it as preferred. When passkey MFA is preferred, the hosted UI encourages users to
     * register a passkey at sign-in.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #userVerification}
     * will return {@link UserVerificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #userVerificationAsString}.
     * </p>
     * 
     * @return Sets or displays your user-pool treatment for MFA with a passkey. You can override other MFA options and
     *         require passkey MFA, or you can set it as preferred. When passkey MFA is preferred, the hosted UI
     *         encourages users to register a passkey at sign-in.
     * @see UserVerificationType
     */
    public final UserVerificationType userVerification() {
        return UserVerificationType.fromValue(userVerification);
    }

    /**
     * <p>
     * Sets or displays your user-pool treatment for MFA with a passkey. You can override other MFA options and require
     * passkey MFA, or you can set it as preferred. When passkey MFA is preferred, the hosted UI encourages users to
     * register a passkey at sign-in.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #userVerification}
     * will return {@link UserVerificationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #userVerificationAsString}.
     * </p>
     * 
     * @return Sets or displays your user-pool treatment for MFA with a passkey. You can override other MFA options and
     *         require passkey MFA, or you can set it as preferred. When passkey MFA is preferred, the hosted UI
     *         encourages users to register a passkey at sign-in.
     * @see UserVerificationType
     */
    public final String userVerificationAsString() {
        return userVerification;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(relyingPartyId());
        hashCode = 31 * hashCode + Objects.hashCode(userVerificationAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof WebAuthnConfigurationType)) {
            return false;
        }
        WebAuthnConfigurationType other = (WebAuthnConfigurationType) obj;
        return Objects.equals(relyingPartyId(), other.relyingPartyId())
                && Objects.equals(userVerificationAsString(), other.userVerificationAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("WebAuthnConfigurationType").add("RelyingPartyId", relyingPartyId())
                .add("UserVerification", userVerificationAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RelyingPartyId":
            return Optional.ofNullable(clazz.cast(relyingPartyId()));
        case "UserVerification":
            return Optional.ofNullable(clazz.cast(userVerificationAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<WebAuthnConfigurationType, T> g) {
        return obj -> g.apply((WebAuthnConfigurationType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, WebAuthnConfigurationType> {
        /**
         * <p>
         * Sets or displays the authentication domain, typically your user pool domain, that passkey providers must use
         * as a relying party (RP) in their configuration.
         * </p>
         * <p>
         * Under the following conditions, the passkey relying party ID must be the fully-qualified domain name of your
         * custom domain:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The user pool is configured for passkey authentication.
         * </p>
         * </li>
         * <li>
         * <p>
         * The user pool has a custom domain, whether or not it also has a prefix domain.
         * </p>
         * </li>
         * <li>
         * <p>
         * Your application performs authentication with managed login or the classic hosted UI.
         * </p>
         * </li>
         * </ul>
         * 
         * @param relyingPartyId
         *        Sets or displays the authentication domain, typically your user pool domain, that passkey providers
         *        must use as a relying party (RP) in their configuration.</p>
         *        <p>
         *        Under the following conditions, the passkey relying party ID must be the fully-qualified domain name
         *        of your custom domain:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The user pool is configured for passkey authentication.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The user pool has a custom domain, whether or not it also has a prefix domain.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Your application performs authentication with managed login or the classic hosted UI.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relyingPartyId(String relyingPartyId);

        /**
         * <p>
         * Sets or displays your user-pool treatment for MFA with a passkey. You can override other MFA options and
         * require passkey MFA, or you can set it as preferred. When passkey MFA is preferred, the hosted UI encourages
         * users to register a passkey at sign-in.
         * </p>
         * 
         * @param userVerification
         *        Sets or displays your user-pool treatment for MFA with a passkey. You can override other MFA options
         *        and require passkey MFA, or you can set it as preferred. When passkey MFA is preferred, the hosted UI
         *        encourages users to register a passkey at sign-in.
         * @see UserVerificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserVerificationType
         */
        Builder userVerification(String userVerification);

        /**
         * <p>
         * Sets or displays your user-pool treatment for MFA with a passkey. You can override other MFA options and
         * require passkey MFA, or you can set it as preferred. When passkey MFA is preferred, the hosted UI encourages
         * users to register a passkey at sign-in.
         * </p>
         * 
         * @param userVerification
         *        Sets or displays your user-pool treatment for MFA with a passkey. You can override other MFA options
         *        and require passkey MFA, or you can set it as preferred. When passkey MFA is preferred, the hosted UI
         *        encourages users to register a passkey at sign-in.
         * @see UserVerificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserVerificationType
         */
        Builder userVerification(UserVerificationType userVerification);
    }

    static final class BuilderImpl implements Builder {
        private String relyingPartyId;

        private String userVerification;

        private BuilderImpl() {
        }

        private BuilderImpl(WebAuthnConfigurationType model) {
            relyingPartyId(model.relyingPartyId);
            userVerification(model.userVerification);
        }

        public final String getRelyingPartyId() {
            return relyingPartyId;
        }

        public final void setRelyingPartyId(String relyingPartyId) {
            this.relyingPartyId = relyingPartyId;
        }

        @Override
        public final Builder relyingPartyId(String relyingPartyId) {
            this.relyingPartyId = relyingPartyId;
            return this;
        }

        public final String getUserVerification() {
            return userVerification;
        }

        public final void setUserVerification(String userVerification) {
            this.userVerification = userVerification;
        }

        @Override
        public final Builder userVerification(String userVerification) {
            this.userVerification = userVerification;
            return this;
        }

        @Override
        public final Builder userVerification(UserVerificationType userVerification) {
            this.userVerification(userVerification == null ? null : userVerification.toString());
            return this;
        }

        @Override
        public WebAuthnConfigurationType build() {
            return new WebAuthnConfigurationType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
