/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A user import job in a user pool. Describes the status of user import with a CSV file. For more information, see <a
 * href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-using-import-tool.html">Importing
 * users into user pools from a CSV file</a>.
 * </p>
 * <p>
 * This data type is a request parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserImportJob.html"
 * >CreateUserImportJob</a>, <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserImportJob.html"
 * >DescribeUserImportJob</a>, <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_ListUserImportJobs.html"
 * >ListUserImportJobs</a>, <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_StartUserImportJob.html"
 * >StartUserImportJob</a>, and <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_StopUserImportJob.html"
 * >StopUserImportJob</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UserImportJobType implements SdkPojo, Serializable,
        ToCopyableBuilder<UserImportJobType.Builder, UserImportJobType> {
    private static final SdkField<String> JOB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("JobName").getter(getter(UserImportJobType::jobName)).setter(setter(Builder::jobName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobName").build()).build();

    private static final SdkField<String> JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("JobId")
            .getter(getter(UserImportJobType::jobId)).setter(setter(Builder::jobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("JobId").build()).build();

    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(UserImportJobType::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> PRE_SIGNED_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreSignedUrl").getter(getter(UserImportJobType::preSignedUrl)).setter(setter(Builder::preSignedUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreSignedUrl").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate").getter(getter(UserImportJobType::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final SdkField<Instant> START_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartDate").getter(getter(UserImportJobType::startDate)).setter(setter(Builder::startDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartDate").build()).build();

    private static final SdkField<Instant> COMPLETION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionDate").getter(getter(UserImportJobType::completionDate))
            .setter(setter(Builder::completionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionDate").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(UserImportJobType::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> CLOUD_WATCH_LOGS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CloudWatchLogsRoleArn").getter(getter(UserImportJobType::cloudWatchLogsRoleArn))
            .setter(setter(Builder::cloudWatchLogsRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudWatchLogsRoleArn").build())
            .build();

    private static final SdkField<Long> IMPORTED_USERS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("ImportedUsers").getter(getter(UserImportJobType::importedUsers)).setter(setter(Builder::importedUsers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImportedUsers").build()).build();

    private static final SdkField<Long> SKIPPED_USERS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("SkippedUsers").getter(getter(UserImportJobType::skippedUsers)).setter(setter(Builder::skippedUsers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SkippedUsers").build()).build();

    private static final SdkField<Long> FAILED_USERS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("FailedUsers").getter(getter(UserImportJobType::failedUsers)).setter(setter(Builder::failedUsers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailedUsers").build()).build();

    private static final SdkField<String> COMPLETION_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CompletionMessage").getter(getter(UserImportJobType::completionMessage))
            .setter(setter(Builder::completionMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionMessage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(JOB_NAME_FIELD, JOB_ID_FIELD,
            USER_POOL_ID_FIELD, PRE_SIGNED_URL_FIELD, CREATION_DATE_FIELD, START_DATE_FIELD, COMPLETION_DATE_FIELD, STATUS_FIELD,
            CLOUD_WATCH_LOGS_ROLE_ARN_FIELD, IMPORTED_USERS_FIELD, SKIPPED_USERS_FIELD, FAILED_USERS_FIELD,
            COMPLETION_MESSAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String jobName;

    private final String jobId;

    private final String userPoolId;

    private final String preSignedUrl;

    private final Instant creationDate;

    private final Instant startDate;

    private final Instant completionDate;

    private final String status;

    private final String cloudWatchLogsRoleArn;

    private final Long importedUsers;

    private final Long skippedUsers;

    private final Long failedUsers;

    private final String completionMessage;

    private UserImportJobType(BuilderImpl builder) {
        this.jobName = builder.jobName;
        this.jobId = builder.jobId;
        this.userPoolId = builder.userPoolId;
        this.preSignedUrl = builder.preSignedUrl;
        this.creationDate = builder.creationDate;
        this.startDate = builder.startDate;
        this.completionDate = builder.completionDate;
        this.status = builder.status;
        this.cloudWatchLogsRoleArn = builder.cloudWatchLogsRoleArn;
        this.importedUsers = builder.importedUsers;
        this.skippedUsers = builder.skippedUsers;
        this.failedUsers = builder.failedUsers;
        this.completionMessage = builder.completionMessage;
    }

    /**
     * <p>
     * The friendly name of the user import job.
     * </p>
     * 
     * @return The friendly name of the user import job.
     */
    public final String jobName() {
        return jobName;
    }

    /**
     * <p>
     * The ID of the user import job.
     * </p>
     * 
     * @return The ID of the user import job.
     */
    public final String jobId() {
        return jobId;
    }

    /**
     * <p>
     * The ID of the user pool that the users are being imported into.
     * </p>
     * 
     * @return The ID of the user pool that the users are being imported into.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The pre-signed URL target for uploading the CSV file.
     * </p>
     * 
     * @return The pre-signed URL target for uploading the CSV file.
     */
    public final String preSignedUrl() {
        return preSignedUrl;
    }

    /**
     * <p>
     * The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time format.
     * Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code> object.
     * </p>
     * 
     * @return The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time
     *         format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
     *         <code>Date</code> object.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The date when the user import job was started.
     * </p>
     * 
     * @return The date when the user import job was started.
     */
    public final Instant startDate() {
        return startDate;
    }

    /**
     * <p>
     * The date when the user import job was completed.
     * </p>
     * 
     * @return The date when the user import job was completed.
     */
    public final Instant completionDate() {
        return completionDate;
    }

    /**
     * <p>
     * The status of the user import job. One of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Created</code> - The job was created but not started.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Pending</code> - A transition state. You have started the job, but it has not begun importing users yet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InProgress</code> - The job has started, and users are being imported.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Stopping</code> - You have stopped the job, but the job has not stopped importing users yet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Stopped</code> - You have stopped the job, and the job has stopped importing users.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Succeeded</code> - The job has completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - The job has stopped due to an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Expired</code> - You created a job, but did not start the job within 24-48 hours. All data associated with
     * the job was deleted, and the job can't be started.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link UserImportJobStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the user import job. One of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Created</code> - The job was created but not started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Pending</code> - A transition state. You have started the job, but it has not begun importing users
     *         yet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>InProgress</code> - The job has started, and users are being imported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Stopping</code> - You have stopped the job, but the job has not stopped importing users yet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Stopped</code> - You have stopped the job, and the job has stopped importing users.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Succeeded</code> - The job has completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The job has stopped due to an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Expired</code> - You created a job, but did not start the job within 24-48 hours. All data
     *         associated with the job was deleted, and the job can't be started.
     *         </p>
     *         </li>
     * @see UserImportJobStatusType
     */
    public final UserImportJobStatusType status() {
        return UserImportJobStatusType.fromValue(status);
    }

    /**
     * <p>
     * The status of the user import job. One of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Created</code> - The job was created but not started.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Pending</code> - A transition state. You have started the job, but it has not begun importing users yet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>InProgress</code> - The job has started, and users are being imported.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Stopping</code> - You have stopped the job, but the job has not stopped importing users yet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Stopped</code> - You have stopped the job, and the job has stopped importing users.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Succeeded</code> - The job has completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - The job has stopped due to an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Expired</code> - You created a job, but did not start the job within 24-48 hours. All data associated with
     * the job was deleted, and the job can't be started.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link UserImportJobStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the user import job. One of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Created</code> - The job was created but not started.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Pending</code> - A transition state. You have started the job, but it has not begun importing users
     *         yet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>InProgress</code> - The job has started, and users are being imported.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Stopping</code> - You have stopped the job, but the job has not stopped importing users yet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Stopped</code> - You have stopped the job, and the job has stopped importing users.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Succeeded</code> - The job has completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The job has stopped due to an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Expired</code> - You created a job, but did not start the job within 24-48 hours. All data
     *         associated with the job was deleted, and the job can't be started.
     *         </p>
     *         </li>
     * @see UserImportJobStatusType
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The role Amazon Resource Name (ARN) for the Amazon CloudWatch Logging role for the user import job. For more
     * information, see "Creating the CloudWatch Logs IAM Role" in the Amazon Cognito Developer Guide.
     * </p>
     * 
     * @return The role Amazon Resource Name (ARN) for the Amazon CloudWatch Logging role for the user import job. For
     *         more information, see "Creating the CloudWatch Logs IAM Role" in the Amazon Cognito Developer Guide.
     */
    public final String cloudWatchLogsRoleArn() {
        return cloudWatchLogsRoleArn;
    }

    /**
     * <p>
     * The number of users that were successfully imported.
     * </p>
     * 
     * @return The number of users that were successfully imported.
     */
    public final Long importedUsers() {
        return importedUsers;
    }

    /**
     * <p>
     * The number of users that were skipped.
     * </p>
     * 
     * @return The number of users that were skipped.
     */
    public final Long skippedUsers() {
        return skippedUsers;
    }

    /**
     * <p>
     * The number of users that couldn't be imported.
     * </p>
     * 
     * @return The number of users that couldn't be imported.
     */
    public final Long failedUsers() {
        return failedUsers;
    }

    /**
     * <p>
     * The message returned when the user import job is completed.
     * </p>
     * 
     * @return The message returned when the user import job is completed.
     */
    public final String completionMessage() {
        return completionMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(jobName());
        hashCode = 31 * hashCode + Objects.hashCode(jobId());
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(preSignedUrl());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(startDate());
        hashCode = 31 * hashCode + Objects.hashCode(completionDate());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(cloudWatchLogsRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(importedUsers());
        hashCode = 31 * hashCode + Objects.hashCode(skippedUsers());
        hashCode = 31 * hashCode + Objects.hashCode(failedUsers());
        hashCode = 31 * hashCode + Objects.hashCode(completionMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UserImportJobType)) {
            return false;
        }
        UserImportJobType other = (UserImportJobType) obj;
        return Objects.equals(jobName(), other.jobName()) && Objects.equals(jobId(), other.jobId())
                && Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(preSignedUrl(), other.preSignedUrl())
                && Objects.equals(creationDate(), other.creationDate()) && Objects.equals(startDate(), other.startDate())
                && Objects.equals(completionDate(), other.completionDate())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(cloudWatchLogsRoleArn(), other.cloudWatchLogsRoleArn())
                && Objects.equals(importedUsers(), other.importedUsers()) && Objects.equals(skippedUsers(), other.skippedUsers())
                && Objects.equals(failedUsers(), other.failedUsers())
                && Objects.equals(completionMessage(), other.completionMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UserImportJobType").add("JobName", jobName()).add("JobId", jobId())
                .add("UserPoolId", userPoolId()).add("PreSignedUrl", preSignedUrl()).add("CreationDate", creationDate())
                .add("StartDate", startDate()).add("CompletionDate", completionDate()).add("Status", statusAsString())
                .add("CloudWatchLogsRoleArn", cloudWatchLogsRoleArn()).add("ImportedUsers", importedUsers())
                .add("SkippedUsers", skippedUsers()).add("FailedUsers", failedUsers())
                .add("CompletionMessage", completionMessage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "JobName":
            return Optional.ofNullable(clazz.cast(jobName()));
        case "JobId":
            return Optional.ofNullable(clazz.cast(jobId()));
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "PreSignedUrl":
            return Optional.ofNullable(clazz.cast(preSignedUrl()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "StartDate":
            return Optional.ofNullable(clazz.cast(startDate()));
        case "CompletionDate":
            return Optional.ofNullable(clazz.cast(completionDate()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CloudWatchLogsRoleArn":
            return Optional.ofNullable(clazz.cast(cloudWatchLogsRoleArn()));
        case "ImportedUsers":
            return Optional.ofNullable(clazz.cast(importedUsers()));
        case "SkippedUsers":
            return Optional.ofNullable(clazz.cast(skippedUsers()));
        case "FailedUsers":
            return Optional.ofNullable(clazz.cast(failedUsers()));
        case "CompletionMessage":
            return Optional.ofNullable(clazz.cast(completionMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("JobName", JOB_NAME_FIELD);
        map.put("JobId", JOB_ID_FIELD);
        map.put("UserPoolId", USER_POOL_ID_FIELD);
        map.put("PreSignedUrl", PRE_SIGNED_URL_FIELD);
        map.put("CreationDate", CREATION_DATE_FIELD);
        map.put("StartDate", START_DATE_FIELD);
        map.put("CompletionDate", COMPLETION_DATE_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("CloudWatchLogsRoleArn", CLOUD_WATCH_LOGS_ROLE_ARN_FIELD);
        map.put("ImportedUsers", IMPORTED_USERS_FIELD);
        map.put("SkippedUsers", SKIPPED_USERS_FIELD);
        map.put("FailedUsers", FAILED_USERS_FIELD);
        map.put("CompletionMessage", COMPLETION_MESSAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UserImportJobType, T> g) {
        return obj -> g.apply((UserImportJobType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UserImportJobType> {
        /**
         * <p>
         * The friendly name of the user import job.
         * </p>
         * 
         * @param jobName
         *        The friendly name of the user import job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobName(String jobName);

        /**
         * <p>
         * The ID of the user import job.
         * </p>
         * 
         * @param jobId
         *        The ID of the user import job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobId(String jobId);

        /**
         * <p>
         * The ID of the user pool that the users are being imported into.
         * </p>
         * 
         * @param userPoolId
         *        The ID of the user pool that the users are being imported into.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The pre-signed URL target for uploading the CSV file.
         * </p>
         * 
         * @param preSignedUrl
         *        The pre-signed URL target for uploading the CSV file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preSignedUrl(String preSignedUrl);

        /**
         * <p>
         * The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time format.
         * Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code> object.
         * </p>
         * 
         * @param creationDate
         *        The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time
         *        format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
         *        <code>Date</code> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The date when the user import job was started.
         * </p>
         * 
         * @param startDate
         *        The date when the user import job was started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startDate(Instant startDate);

        /**
         * <p>
         * The date when the user import job was completed.
         * </p>
         * 
         * @param completionDate
         *        The date when the user import job was completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionDate(Instant completionDate);

        /**
         * <p>
         * The status of the user import job. One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Created</code> - The job was created but not started.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Pending</code> - A transition state. You have started the job, but it has not begun importing users
         * yet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InProgress</code> - The job has started, and users are being imported.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Stopping</code> - You have stopped the job, but the job has not stopped importing users yet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Stopped</code> - You have stopped the job, and the job has stopped importing users.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Succeeded</code> - The job has completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - The job has stopped due to an error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Expired</code> - You created a job, but did not start the job within 24-48 hours. All data associated
         * with the job was deleted, and the job can't be started.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the user import job. One of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Created</code> - The job was created but not started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Pending</code> - A transition state. You have started the job, but it has not begun importing
         *        users yet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InProgress</code> - The job has started, and users are being imported.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Stopping</code> - You have stopped the job, but the job has not stopped importing users yet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Stopped</code> - You have stopped the job, and the job has stopped importing users.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Succeeded</code> - The job has completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The job has stopped due to an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Expired</code> - You created a job, but did not start the job within 24-48 hours. All data
         *        associated with the job was deleted, and the job can't be started.
         *        </p>
         *        </li>
         * @see UserImportJobStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserImportJobStatusType
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the user import job. One of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Created</code> - The job was created but not started.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Pending</code> - A transition state. You have started the job, but it has not begun importing users
         * yet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>InProgress</code> - The job has started, and users are being imported.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Stopping</code> - You have stopped the job, but the job has not stopped importing users yet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Stopped</code> - You have stopped the job, and the job has stopped importing users.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Succeeded</code> - The job has completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - The job has stopped due to an error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Expired</code> - You created a job, but did not start the job within 24-48 hours. All data associated
         * with the job was deleted, and the job can't be started.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the user import job. One of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Created</code> - The job was created but not started.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Pending</code> - A transition state. You have started the job, but it has not begun importing
         *        users yet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>InProgress</code> - The job has started, and users are being imported.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Stopping</code> - You have stopped the job, but the job has not stopped importing users yet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Stopped</code> - You have stopped the job, and the job has stopped importing users.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Succeeded</code> - The job has completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The job has stopped due to an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Expired</code> - You created a job, but did not start the job within 24-48 hours. All data
         *        associated with the job was deleted, and the job can't be started.
         *        </p>
         *        </li>
         * @see UserImportJobStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserImportJobStatusType
         */
        Builder status(UserImportJobStatusType status);

        /**
         * <p>
         * The role Amazon Resource Name (ARN) for the Amazon CloudWatch Logging role for the user import job. For more
         * information, see "Creating the CloudWatch Logs IAM Role" in the Amazon Cognito Developer Guide.
         * </p>
         * 
         * @param cloudWatchLogsRoleArn
         *        The role Amazon Resource Name (ARN) for the Amazon CloudWatch Logging role for the user import job.
         *        For more information, see "Creating the CloudWatch Logs IAM Role" in the Amazon Cognito Developer
         *        Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudWatchLogsRoleArn(String cloudWatchLogsRoleArn);

        /**
         * <p>
         * The number of users that were successfully imported.
         * </p>
         * 
         * @param importedUsers
         *        The number of users that were successfully imported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder importedUsers(Long importedUsers);

        /**
         * <p>
         * The number of users that were skipped.
         * </p>
         * 
         * @param skippedUsers
         *        The number of users that were skipped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder skippedUsers(Long skippedUsers);

        /**
         * <p>
         * The number of users that couldn't be imported.
         * </p>
         * 
         * @param failedUsers
         *        The number of users that couldn't be imported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failedUsers(Long failedUsers);

        /**
         * <p>
         * The message returned when the user import job is completed.
         * </p>
         * 
         * @param completionMessage
         *        The message returned when the user import job is completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionMessage(String completionMessage);
    }

    static final class BuilderImpl implements Builder {
        private String jobName;

        private String jobId;

        private String userPoolId;

        private String preSignedUrl;

        private Instant creationDate;

        private Instant startDate;

        private Instant completionDate;

        private String status;

        private String cloudWatchLogsRoleArn;

        private Long importedUsers;

        private Long skippedUsers;

        private Long failedUsers;

        private String completionMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(UserImportJobType model) {
            jobName(model.jobName);
            jobId(model.jobId);
            userPoolId(model.userPoolId);
            preSignedUrl(model.preSignedUrl);
            creationDate(model.creationDate);
            startDate(model.startDate);
            completionDate(model.completionDate);
            status(model.status);
            cloudWatchLogsRoleArn(model.cloudWatchLogsRoleArn);
            importedUsers(model.importedUsers);
            skippedUsers(model.skippedUsers);
            failedUsers(model.failedUsers);
            completionMessage(model.completionMessage);
        }

        public final String getJobName() {
            return jobName;
        }

        public final void setJobName(String jobName) {
            this.jobName = jobName;
        }

        @Override
        public final Builder jobName(String jobName) {
            this.jobName = jobName;
            return this;
        }

        public final String getJobId() {
            return jobId;
        }

        public final void setJobId(String jobId) {
            this.jobId = jobId;
        }

        @Override
        public final Builder jobId(String jobId) {
            this.jobId = jobId;
            return this;
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final String getPreSignedUrl() {
            return preSignedUrl;
        }

        public final void setPreSignedUrl(String preSignedUrl) {
            this.preSignedUrl = preSignedUrl;
        }

        @Override
        public final Builder preSignedUrl(String preSignedUrl) {
            this.preSignedUrl = preSignedUrl;
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final Instant getStartDate() {
            return startDate;
        }

        public final void setStartDate(Instant startDate) {
            this.startDate = startDate;
        }

        @Override
        public final Builder startDate(Instant startDate) {
            this.startDate = startDate;
            return this;
        }

        public final Instant getCompletionDate() {
            return completionDate;
        }

        public final void setCompletionDate(Instant completionDate) {
            this.completionDate = completionDate;
        }

        @Override
        public final Builder completionDate(Instant completionDate) {
            this.completionDate = completionDate;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(UserImportJobStatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getCloudWatchLogsRoleArn() {
            return cloudWatchLogsRoleArn;
        }

        public final void setCloudWatchLogsRoleArn(String cloudWatchLogsRoleArn) {
            this.cloudWatchLogsRoleArn = cloudWatchLogsRoleArn;
        }

        @Override
        public final Builder cloudWatchLogsRoleArn(String cloudWatchLogsRoleArn) {
            this.cloudWatchLogsRoleArn = cloudWatchLogsRoleArn;
            return this;
        }

        public final Long getImportedUsers() {
            return importedUsers;
        }

        public final void setImportedUsers(Long importedUsers) {
            this.importedUsers = importedUsers;
        }

        @Override
        public final Builder importedUsers(Long importedUsers) {
            this.importedUsers = importedUsers;
            return this;
        }

        public final Long getSkippedUsers() {
            return skippedUsers;
        }

        public final void setSkippedUsers(Long skippedUsers) {
            this.skippedUsers = skippedUsers;
        }

        @Override
        public final Builder skippedUsers(Long skippedUsers) {
            this.skippedUsers = skippedUsers;
            return this;
        }

        public final Long getFailedUsers() {
            return failedUsers;
        }

        public final void setFailedUsers(Long failedUsers) {
            this.failedUsers = failedUsers;
        }

        @Override
        public final Builder failedUsers(Long failedUsers) {
            this.failedUsers = failedUsers;
            return this;
        }

        public final String getCompletionMessage() {
            return completionMessage;
        }

        public final void setCompletionMessage(String completionMessage) {
            this.completionMessage = completionMessage;
        }

        @Override
        public final Builder completionMessage(String completionMessage) {
            this.completionMessage = completionMessage;
            return this;
        }

        @Override
        public UserImportJobType build() {
            return new UserImportJobType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
