/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A user pool identity provider (IdP). Contains information about a third-party IdP to a user pool, the attributes that
 * it populates to user profiles, and the trust relationship between the IdP and your user pool.
 * </p>
 * <p>
 * This data type is a response parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateIdentityProvider.html"
 * >CreateIdentityProvider</a>, <a href=
 * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeIdentityProvider.html"
 * >DescribeIdentityProvider</a>, <a href=
 * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_GetIdentityProviderByIdentifier.html"
 * >GetIdentityProviderByIdentifier</a>, and <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateIdentityProvider.html"
 * >UpdateIdentityProvider</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IdentityProviderType implements SdkPojo, Serializable,
        ToCopyableBuilder<IdentityProviderType.Builder, IdentityProviderType> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(IdentityProviderType::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> PROVIDER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProviderName").getter(getter(IdentityProviderType::providerName)).setter(setter(Builder::providerName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProviderName").build()).build();

    private static final SdkField<String> PROVIDER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProviderType").getter(getter(IdentityProviderType::providerTypeAsString))
            .setter(setter(Builder::providerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProviderType").build()).build();

    private static final SdkField<Map<String, String>> PROVIDER_DETAILS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ProviderDetails")
            .getter(getter(IdentityProviderType::providerDetails))
            .setter(setter(Builder::providerDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProviderDetails").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> ATTRIBUTE_MAPPING_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("AttributeMapping")
            .getter(getter(IdentityProviderType::attributeMapping))
            .setter(setter(Builder::attributeMapping))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeMapping").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> IDP_IDENTIFIERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IdpIdentifiers")
            .getter(getter(IdentityProviderType::idpIdentifiers))
            .setter(setter(Builder::idpIdentifiers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdpIdentifiers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedDate").getter(getter(IdentityProviderType::lastModifiedDate))
            .setter(setter(Builder::lastModifiedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedDate").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate").getter(getter(IdentityProviderType::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            PROVIDER_NAME_FIELD, PROVIDER_TYPE_FIELD, PROVIDER_DETAILS_FIELD, ATTRIBUTE_MAPPING_FIELD, IDP_IDENTIFIERS_FIELD,
            LAST_MODIFIED_DATE_FIELD, CREATION_DATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String userPoolId;

    private final String providerName;

    private final String providerType;

    private final Map<String, String> providerDetails;

    private final Map<String, String> attributeMapping;

    private final List<String> idpIdentifiers;

    private final Instant lastModifiedDate;

    private final Instant creationDate;

    private IdentityProviderType(BuilderImpl builder) {
        this.userPoolId = builder.userPoolId;
        this.providerName = builder.providerName;
        this.providerType = builder.providerType;
        this.providerDetails = builder.providerDetails;
        this.attributeMapping = builder.attributeMapping;
        this.idpIdentifiers = builder.idpIdentifiers;
        this.lastModifiedDate = builder.lastModifiedDate;
        this.creationDate = builder.creationDate;
    }

    /**
     * <p>
     * The ID of the user pool associated with the IdP.
     * </p>
     * 
     * @return The ID of the user pool associated with the IdP.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * A friendly name for the IdP.
     * </p>
     * 
     * @return A friendly name for the IdP.
     */
    public final String providerName() {
        return providerName;
    }

    /**
     * <p>
     * The type of IdP. Either SAML, OIDC, or a named social identity provider.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #providerType} will
     * return {@link IdentityProviderTypeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #providerTypeAsString}.
     * </p>
     * 
     * @return The type of IdP. Either SAML, OIDC, or a named social identity provider.
     * @see IdentityProviderTypeType
     */
    public final IdentityProviderTypeType providerType() {
        return IdentityProviderTypeType.fromValue(providerType);
    }

    /**
     * <p>
     * The type of IdP. Either SAML, OIDC, or a named social identity provider.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #providerType} will
     * return {@link IdentityProviderTypeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #providerTypeAsString}.
     * </p>
     * 
     * @return The type of IdP. Either SAML, OIDC, or a named social identity provider.
     * @see IdentityProviderTypeType
     */
    public final String providerTypeAsString() {
        return providerType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ProviderDetails property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasProviderDetails() {
        return providerDetails != null && !(providerDetails instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The scopes, URLs, and identifiers for your external identity provider. The following examples describe the
     * provider detail keys for each IdP type. These values and their schema are subject to change. Social IdP
     * <code>authorize_scopes</code> values must match the values listed here.
     * </p>
     * <dl>
     * <dt>OpenID Connect (OIDC)</dt>
     * <dd>
     * <p>
     * Amazon Cognito accepts the following elements when it can't discover endpoint URLs from <code>oidc_issuer</code>:
     * <code>attributes_url</code>, <code>authorize_url</code>, <code>jwks_uri</code>, <code>token_url</code>.
     * </p>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
     * </p>
     * </dd>
     * <dt>SAML</dt>
     * <dd>
     * <p>
     * Create or update request with Metadata URL:
     * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }</code>
     * </p>
     * <p>
     * Create or update request with Metadata file:
     * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }</code>
     * </p>
     * <p>
     * The value of <code>MetadataFile</code> must be the plaintext metadata document with all quote (") characters
     * escaped by backslashes.
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }</code>
     * </p>
     * </dd>
     * <dt>LoginWithAmazon</dt>
     * <dd>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }</code>
     * </p>
     * </dd>
     * <dt>Google</dt>
     * <dd>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }</code>
     * </p>
     * </dd>
     * <dt>SignInWithApple</dt>
     * <dd>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }</code>
     * </p>
     * </dd>
     * <dt>Facebook</dt>
     * <dd>
     * <p>
     * Create or update request:
     * <code>"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }</code>
     * </p>
     * <p>
     * Describe response:
     * <code>"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }</code>
     * </p>
     * </dd>
     * </dl>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasProviderDetails} method.
     * </p>
     * 
     * @return The scopes, URLs, and identifiers for your external identity provider. The following examples describe
     *         the provider detail keys for each IdP type. These values and their schema are subject to change. Social
     *         IdP <code>authorize_scopes</code> values must match the values listed here.</p>
     *         <dl>
     *         <dt>OpenID Connect (OIDC)</dt>
     *         <dd>
     *         <p>
     *         Amazon Cognito accepts the following elements when it can't discover endpoint URLs from
     *         <code>oidc_issuer</code>: <code>attributes_url</code>, <code>authorize_url</code>, <code>jwks_uri</code>,
     *         <code>token_url</code>.
     *         </p>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
     *         </p>
     *         </dd>
     *         <dt>SAML</dt>
     *         <dd>
     *         <p>
     *         Create or update request with Metadata URL:
     *         <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }</code>
     *         </p>
     *         <p>
     *         Create or update request with Metadata file:
     *         <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }</code>
     *         </p>
     *         <p>
     *         The value of <code>MetadataFile</code> must be the plaintext metadata document with all quote (")
     *         characters escaped by backslashes.
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }</code>
     *         </p>
     *         </dd>
     *         <dt>LoginWithAmazon</dt>
     *         <dd>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }</code>
     *         </p>
     *         </dd>
     *         <dt>Google</dt>
     *         <dd>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }</code>
     *         </p>
     *         </dd>
     *         <dt>SignInWithApple</dt>
     *         <dd>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }</code>
     *         </p>
     *         </dd>
     *         <dt>Facebook</dt>
     *         <dd>
     *         <p>
     *         Create or update request:
     *         <code>"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }</code>
     *         </p>
     *         <p>
     *         Describe response:
     *         <code>"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }</code>
     *         </p>
     *         </dd>
     */
    public final Map<String, String> providerDetails() {
        return providerDetails;
    }

    /**
     * For responses, this returns true if the service returned a value for the AttributeMapping property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAttributeMapping() {
        return attributeMapping != null && !(attributeMapping instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A mapping of IdP attributes to standard and custom user pool attributes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAttributeMapping} method.
     * </p>
     * 
     * @return A mapping of IdP attributes to standard and custom user pool attributes.
     */
    public final Map<String, String> attributeMapping() {
        return attributeMapping;
    }

    /**
     * For responses, this returns true if the service returned a value for the IdpIdentifiers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIdpIdentifiers() {
        return idpIdentifiers != null && !(idpIdentifiers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IdP identifiers. IdP identifiers are strings that represent friendly names or domain names of IdPs, for
     * example <code>MyIdP</code> or <code>auth.example.com</code>. You can choose to route user authorization requests
     * to the right IdP with either IdP identifiers or IdP names. For more information, see
     * <code>identity_provider</code> and <code>idp_identifier</code> at <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html#get-authorize-request-parameters"
     * >Authorize endpoint</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIdpIdentifiers} method.
     * </p>
     * 
     * @return A list of IdP identifiers. IdP identifiers are strings that represent friendly names or domain names of
     *         IdPs, for example <code>MyIdP</code> or <code>auth.example.com</code>. You can choose to route user
     *         authorization requests to the right IdP with either IdP identifiers or IdP names. For more information,
     *         see <code>identity_provider</code> and <code>idp_identifier</code> at <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html#get-authorize-request-parameters"
     *         >Authorize endpoint</a>.
     */
    public final List<String> idpIdentifiers() {
        return idpIdentifiers;
    }

    /**
     * <p>
     * The date and time when the item was modified. Amazon Cognito returns this timestamp in UNIX epoch time format.
     * Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code> object.
     * </p>
     * 
     * @return The date and time when the item was modified. Amazon Cognito returns this timestamp in UNIX epoch time
     *         format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
     *         <code>Date</code> object.
     */
    public final Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    /**
     * <p>
     * The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time format.
     * Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code> object.
     * </p>
     * 
     * @return The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time
     *         format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
     *         <code>Date</code> object.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(providerName());
        hashCode = 31 * hashCode + Objects.hashCode(providerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasProviderDetails() ? providerDetails() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributeMapping() ? attributeMapping() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasIdpIdentifiers() ? idpIdentifiers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IdentityProviderType)) {
            return false;
        }
        IdentityProviderType other = (IdentityProviderType) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(providerName(), other.providerName())
                && Objects.equals(providerTypeAsString(), other.providerTypeAsString())
                && hasProviderDetails() == other.hasProviderDetails()
                && Objects.equals(providerDetails(), other.providerDetails())
                && hasAttributeMapping() == other.hasAttributeMapping()
                && Objects.equals(attributeMapping(), other.attributeMapping())
                && hasIdpIdentifiers() == other.hasIdpIdentifiers() && Objects.equals(idpIdentifiers(), other.idpIdentifiers())
                && Objects.equals(lastModifiedDate(), other.lastModifiedDate())
                && Objects.equals(creationDate(), other.creationDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IdentityProviderType").add("UserPoolId", userPoolId()).add("ProviderName", providerName())
                .add("ProviderType", providerTypeAsString())
                .add("ProviderDetails", hasProviderDetails() ? providerDetails() : null)
                .add("AttributeMapping", hasAttributeMapping() ? attributeMapping() : null)
                .add("IdpIdentifiers", hasIdpIdentifiers() ? idpIdentifiers() : null).add("LastModifiedDate", lastModifiedDate())
                .add("CreationDate", creationDate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "ProviderName":
            return Optional.ofNullable(clazz.cast(providerName()));
        case "ProviderType":
            return Optional.ofNullable(clazz.cast(providerTypeAsString()));
        case "ProviderDetails":
            return Optional.ofNullable(clazz.cast(providerDetails()));
        case "AttributeMapping":
            return Optional.ofNullable(clazz.cast(attributeMapping()));
        case "IdpIdentifiers":
            return Optional.ofNullable(clazz.cast(idpIdentifiers()));
        case "LastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("UserPoolId", USER_POOL_ID_FIELD);
        map.put("ProviderName", PROVIDER_NAME_FIELD);
        map.put("ProviderType", PROVIDER_TYPE_FIELD);
        map.put("ProviderDetails", PROVIDER_DETAILS_FIELD);
        map.put("AttributeMapping", ATTRIBUTE_MAPPING_FIELD);
        map.put("IdpIdentifiers", IDP_IDENTIFIERS_FIELD);
        map.put("LastModifiedDate", LAST_MODIFIED_DATE_FIELD);
        map.put("CreationDate", CREATION_DATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IdentityProviderType, T> g) {
        return obj -> g.apply((IdentityProviderType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IdentityProviderType> {
        /**
         * <p>
         * The ID of the user pool associated with the IdP.
         * </p>
         * 
         * @param userPoolId
         *        The ID of the user pool associated with the IdP.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * A friendly name for the IdP.
         * </p>
         * 
         * @param providerName
         *        A friendly name for the IdP.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerName(String providerName);

        /**
         * <p>
         * The type of IdP. Either SAML, OIDC, or a named social identity provider.
         * </p>
         * 
         * @param providerType
         *        The type of IdP. Either SAML, OIDC, or a named social identity provider.
         * @see IdentityProviderTypeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentityProviderTypeType
         */
        Builder providerType(String providerType);

        /**
         * <p>
         * The type of IdP. Either SAML, OIDC, or a named social identity provider.
         * </p>
         * 
         * @param providerType
         *        The type of IdP. Either SAML, OIDC, or a named social identity provider.
         * @see IdentityProviderTypeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IdentityProviderTypeType
         */
        Builder providerType(IdentityProviderTypeType providerType);

        /**
         * <p>
         * The scopes, URLs, and identifiers for your external identity provider. The following examples describe the
         * provider detail keys for each IdP type. These values and their schema are subject to change. Social IdP
         * <code>authorize_scopes</code> values must match the values listed here.
         * </p>
         * <dl>
         * <dt>OpenID Connect (OIDC)</dt>
         * <dd>
         * <p>
         * Amazon Cognito accepts the following elements when it can't discover endpoint URLs from
         * <code>oidc_issuer</code>: <code>attributes_url</code>, <code>authorize_url</code>, <code>jwks_uri</code>,
         * <code>token_url</code>.
         * </p>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
         * </p>
         * </dd>
         * <dt>SAML</dt>
         * <dd>
         * <p>
         * Create or update request with Metadata URL:
         * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }</code>
         * </p>
         * <p>
         * Create or update request with Metadata file:
         * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }</code>
         * </p>
         * <p>
         * The value of <code>MetadataFile</code> must be the plaintext metadata document with all quote (") characters
         * escaped by backslashes.
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }</code>
         * </p>
         * </dd>
         * <dt>LoginWithAmazon</dt>
         * <dd>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }</code>
         * </p>
         * </dd>
         * <dt>Google</dt>
         * <dd>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }</code>
         * </p>
         * </dd>
         * <dt>SignInWithApple</dt>
         * <dd>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }</code>
         * </p>
         * </dd>
         * <dt>Facebook</dt>
         * <dd>
         * <p>
         * Create or update request:
         * <code>"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }</code>
         * </p>
         * <p>
         * Describe response:
         * <code>"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }</code>
         * </p>
         * </dd>
         * </dl>
         * 
         * @param providerDetails
         *        The scopes, URLs, and identifiers for your external identity provider. The following examples describe
         *        the provider detail keys for each IdP type. These values and their schema are subject to change.
         *        Social IdP <code>authorize_scopes</code> values must match the values listed here.</p>
         *        <dl>
         *        <dt>OpenID Connect (OIDC)</dt>
         *        <dd>
         *        <p>
         *        Amazon Cognito accepts the following elements when it can't discover endpoint URLs from
         *        <code>oidc_issuer</code>: <code>attributes_url</code>, <code>authorize_url</code>,
         *        <code>jwks_uri</code>, <code>token_url</code>.
         *        </p>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "attributes_request_method": "GET", "attributes_url": "https://auth.example.com/userInfo", "attributes_url_add_attributes": "false", "authorize_scopes": "openid profile email", "authorize_url": "https://auth.example.com/authorize", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "jwks_uri": "https://auth.example.com/.well-known/jwks.json", "oidc_issuer": "https://auth.example.com", "token_url": "https://example.com/token" }</code>
         *        </p>
         *        </dd>
         *        <dt>SAML</dt>
         *        <dd>
         *        <p>
         *        Create or update request with Metadata URL:
         *        <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256" }</code>
         *        </p>
         *        <p>
         *        Create or update request with Metadata file:
         *        <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "MetadataFile": "[metadata XML]", "RequestSigningAlgorithm": "rsa-sha256" }</code>
         *        </p>
         *        <p>
         *        The value of <code>MetadataFile</code> must be the plaintext metadata document with all quote (")
         *        characters escaped by backslashes.
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "IDPInit": "true", "IDPSignout": "true", "EncryptedResponses" : "true", "ActiveEncryptionCertificate": "[certificate]", "MetadataURL": "https://auth.example.com/sso/saml/metadata", "RequestSigningAlgorithm": "rsa-sha256", "SLORedirectBindingURI": "https://auth.example.com/slo/saml", "SSORedirectBindingURI": "https://auth.example.com/sso/saml" }</code>
         *        </p>
         *        </dd>
         *        <dt>LoginWithAmazon</dt>
         *        <dd>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "authorize_scopes": "profile postal_code", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret"</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "attributes_url": "https://api.amazon.com/user/profile", "attributes_url_add_attributes": "false", "authorize_scopes": "profile postal_code", "authorize_url": "https://www.amazon.com/ap/oa", "client_id": "amzn1.application-oa2-client.1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "POST", "token_url": "https://api.amazon.com/auth/o2/token" }</code>
         *        </p>
         *        </dd>
         *        <dt>Google</dt>
         *        <dd>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "authorize_scopes": "email profile openid", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret" }</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "attributes_url": "https://people.googleapis.com/v1/people/me?personFields=", "attributes_url_add_attributes": "true", "authorize_scopes": "email profile openid", "authorize_url": "https://accounts.google.com/o/oauth2/v2/auth", "client_id": "1example23456789.apps.googleusercontent.com", "client_secret": "provider-app-client-secret", "oidc_issuer": "https://accounts.google.com", "token_request_method": "POST", "token_url": "https://www.googleapis.com/oauth2/v4/token" }</code>
         *        </p>
         *        </dd>
         *        <dt>SignInWithApple</dt>
         *        <dd>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "authorize_scopes": "email name", "client_id": "com.example.cognito", "private_key": "1EXAMPLE", "key_id": "2EXAMPLE", "team_id": "3EXAMPLE" }</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "attributes_url_add_attributes": "false", "authorize_scopes": "email name", "authorize_url": "https://appleid.apple.com/auth/authorize", "client_id": "com.example.cognito", "key_id": "1EXAMPLE", "oidc_issuer": "https://appleid.apple.com", "team_id": "2EXAMPLE", "token_request_method": "POST", "token_url": "https://appleid.apple.com/auth/token" }</code>
         *        </p>
         *        </dd>
         *        <dt>Facebook</dt>
         *        <dd>
         *        <p>
         *        Create or update request:
         *        <code>"ProviderDetails": { "api_version": "v17.0", "authorize_scopes": "public_profile, email", "client_id": "1example23456789", "client_secret": "provider-app-client-secret" }</code>
         *        </p>
         *        <p>
         *        Describe response:
         *        <code>"ProviderDetails": { "api_version": "v17.0", "attributes_url": "https://graph.facebook.com/v17.0/me?fields=", "attributes_url_add_attributes": "true", "authorize_scopes": "public_profile, email", "authorize_url": "https://www.facebook.com/v17.0/dialog/oauth", "client_id": "1example23456789", "client_secret": "provider-app-client-secret", "token_request_method": "GET", "token_url": "https://graph.facebook.com/v17.0/oauth/access_token" }</code>
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder providerDetails(Map<String, String> providerDetails);

        /**
         * <p>
         * A mapping of IdP attributes to standard and custom user pool attributes.
         * </p>
         * 
         * @param attributeMapping
         *        A mapping of IdP attributes to standard and custom user pool attributes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeMapping(Map<String, String> attributeMapping);

        /**
         * <p>
         * A list of IdP identifiers. IdP identifiers are strings that represent friendly names or domain names of IdPs,
         * for example <code>MyIdP</code> or <code>auth.example.com</code>. You can choose to route user authorization
         * requests to the right IdP with either IdP identifiers or IdP names. For more information, see
         * <code>identity_provider</code> and <code>idp_identifier</code> at <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html#get-authorize-request-parameters"
         * >Authorize endpoint</a>.
         * </p>
         * 
         * @param idpIdentifiers
         *        A list of IdP identifiers. IdP identifiers are strings that represent friendly names or domain names
         *        of IdPs, for example <code>MyIdP</code> or <code>auth.example.com</code>. You can choose to route user
         *        authorization requests to the right IdP with either IdP identifiers or IdP names. For more
         *        information, see <code>identity_provider</code> and <code>idp_identifier</code> at <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html#get-authorize-request-parameters"
         *        >Authorize endpoint</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idpIdentifiers(Collection<String> idpIdentifiers);

        /**
         * <p>
         * A list of IdP identifiers. IdP identifiers are strings that represent friendly names or domain names of IdPs,
         * for example <code>MyIdP</code> or <code>auth.example.com</code>. You can choose to route user authorization
         * requests to the right IdP with either IdP identifiers or IdP names. For more information, see
         * <code>identity_provider</code> and <code>idp_identifier</code> at <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html#get-authorize-request-parameters"
         * >Authorize endpoint</a>.
         * </p>
         * 
         * @param idpIdentifiers
         *        A list of IdP identifiers. IdP identifiers are strings that represent friendly names or domain names
         *        of IdPs, for example <code>MyIdP</code> or <code>auth.example.com</code>. You can choose to route user
         *        authorization requests to the right IdP with either IdP identifiers or IdP names. For more
         *        information, see <code>identity_provider</code> and <code>idp_identifier</code> at <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/authorization-endpoint.html#get-authorize-request-parameters"
         *        >Authorize endpoint</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idpIdentifiers(String... idpIdentifiers);

        /**
         * <p>
         * The date and time when the item was modified. Amazon Cognito returns this timestamp in UNIX epoch time
         * format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code>
         * object.
         * </p>
         * 
         * @param lastModifiedDate
         *        The date and time when the item was modified. Amazon Cognito returns this timestamp in UNIX epoch time
         *        format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
         *        <code>Date</code> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);

        /**
         * <p>
         * The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time format.
         * Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code> object.
         * </p>
         * 
         * @param creationDate
         *        The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time
         *        format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
         *        <code>Date</code> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);
    }

    static final class BuilderImpl implements Builder {
        private String userPoolId;

        private String providerName;

        private String providerType;

        private Map<String, String> providerDetails = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, String> attributeMapping = DefaultSdkAutoConstructMap.getInstance();

        private List<String> idpIdentifiers = DefaultSdkAutoConstructList.getInstance();

        private Instant lastModifiedDate;

        private Instant creationDate;

        private BuilderImpl() {
        }

        private BuilderImpl(IdentityProviderType model) {
            userPoolId(model.userPoolId);
            providerName(model.providerName);
            providerType(model.providerType);
            providerDetails(model.providerDetails);
            attributeMapping(model.attributeMapping);
            idpIdentifiers(model.idpIdentifiers);
            lastModifiedDate(model.lastModifiedDate);
            creationDate(model.creationDate);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final String getProviderName() {
            return providerName;
        }

        public final void setProviderName(String providerName) {
            this.providerName = providerName;
        }

        @Override
        public final Builder providerName(String providerName) {
            this.providerName = providerName;
            return this;
        }

        public final String getProviderType() {
            return providerType;
        }

        public final void setProviderType(String providerType) {
            this.providerType = providerType;
        }

        @Override
        public final Builder providerType(String providerType) {
            this.providerType = providerType;
            return this;
        }

        @Override
        public final Builder providerType(IdentityProviderTypeType providerType) {
            this.providerType(providerType == null ? null : providerType.toString());
            return this;
        }

        public final Map<String, String> getProviderDetails() {
            if (providerDetails instanceof SdkAutoConstructMap) {
                return null;
            }
            return providerDetails;
        }

        public final void setProviderDetails(Map<String, String> providerDetails) {
            this.providerDetails = ProviderDetailsTypeCopier.copy(providerDetails);
        }

        @Override
        public final Builder providerDetails(Map<String, String> providerDetails) {
            this.providerDetails = ProviderDetailsTypeCopier.copy(providerDetails);
            return this;
        }

        public final Map<String, String> getAttributeMapping() {
            if (attributeMapping instanceof SdkAutoConstructMap) {
                return null;
            }
            return attributeMapping;
        }

        public final void setAttributeMapping(Map<String, String> attributeMapping) {
            this.attributeMapping = AttributeMappingTypeCopier.copy(attributeMapping);
        }

        @Override
        public final Builder attributeMapping(Map<String, String> attributeMapping) {
            this.attributeMapping = AttributeMappingTypeCopier.copy(attributeMapping);
            return this;
        }

        public final Collection<String> getIdpIdentifiers() {
            if (idpIdentifiers instanceof SdkAutoConstructList) {
                return null;
            }
            return idpIdentifiers;
        }

        public final void setIdpIdentifiers(Collection<String> idpIdentifiers) {
            this.idpIdentifiers = IdpIdentifiersListTypeCopier.copy(idpIdentifiers);
        }

        @Override
        public final Builder idpIdentifiers(Collection<String> idpIdentifiers) {
            this.idpIdentifiers = IdpIdentifiersListTypeCopier.copy(idpIdentifiers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder idpIdentifiers(String... idpIdentifiers) {
            idpIdentifiers(Arrays.asList(idpIdentifiers));
            return this;
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        @Override
        public IdentityProviderType build() {
            return new IdentityProviderType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
