/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The request to respond to the authentication challenge, as an administrator.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdminRespondToAuthChallengeRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<AdminRespondToAuthChallengeRequest.Builder, AdminRespondToAuthChallengeRequest> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(AdminRespondToAuthChallengeRequest::userPoolId))
            .setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientId").getter(getter(AdminRespondToAuthChallengeRequest::clientId))
            .setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<String> CHALLENGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChallengeName").getter(getter(AdminRespondToAuthChallengeRequest::challengeNameAsString))
            .setter(setter(Builder::challengeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChallengeName").build()).build();

    private static final SdkField<Map<String, String>> CHALLENGE_RESPONSES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ChallengeResponses")
            .getter(getter(AdminRespondToAuthChallengeRequest::challengeResponses))
            .setter(setter(Builder::challengeResponses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChallengeResponses").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> SESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Session")
            .getter(getter(AdminRespondToAuthChallengeRequest::session)).setter(setter(Builder::session))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Session").build()).build();

    private static final SdkField<AnalyticsMetadataType> ANALYTICS_METADATA_FIELD = SdkField
            .<AnalyticsMetadataType> builder(MarshallingType.SDK_POJO).memberName("AnalyticsMetadata")
            .getter(getter(AdminRespondToAuthChallengeRequest::analyticsMetadata)).setter(setter(Builder::analyticsMetadata))
            .constructor(AnalyticsMetadataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalyticsMetadata").build()).build();

    private static final SdkField<ContextDataType> CONTEXT_DATA_FIELD = SdkField
            .<ContextDataType> builder(MarshallingType.SDK_POJO).memberName("ContextData")
            .getter(getter(AdminRespondToAuthChallengeRequest::contextData)).setter(setter(Builder::contextData))
            .constructor(ContextDataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContextData").build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(AdminRespondToAuthChallengeRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            CLIENT_ID_FIELD, CHALLENGE_NAME_FIELD, CHALLENGE_RESPONSES_FIELD, SESSION_FIELD, ANALYTICS_METADATA_FIELD,
            CONTEXT_DATA_FIELD, CLIENT_METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String userPoolId;

    private final String clientId;

    private final String challengeName;

    private final Map<String, String> challengeResponses;

    private final String session;

    private final AnalyticsMetadataType analyticsMetadata;

    private final ContextDataType contextData;

    private final Map<String, String> clientMetadata;

    private AdminRespondToAuthChallengeRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.clientId = builder.clientId;
        this.challengeName = builder.challengeName;
        this.challengeResponses = builder.challengeResponses;
        this.session = builder.session;
        this.analyticsMetadata = builder.analyticsMetadata;
        this.contextData = builder.contextData;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The ID of the user pool where you want to respond to an authentication challenge.
     * </p>
     * 
     * @return The ID of the user pool where you want to respond to an authentication challenge.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The ID of the app client where you initiated sign-in.
     * </p>
     * 
     * @return The ID of the app client where you initiated sign-in.
     */
    public final String clientId() {
        return clientId;
    }

    /**
     * <p>
     * The name of the challenge that you are responding to. You can find more information about values for
     * <code>ChallengeName</code> in the response parameters of <a href=
     * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html#CognitoUserPools-AdminInitiateAuth-response-ChallengeName"
     * >AdminInitiateAuth</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #challengeName}
     * will return {@link ChallengeNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #challengeNameAsString}.
     * </p>
     * 
     * @return The name of the challenge that you are responding to. You can find more information about values for
     *         <code>ChallengeName</code> in the response parameters of <a href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html#CognitoUserPools-AdminInitiateAuth-response-ChallengeName"
     *         >AdminInitiateAuth</a>.
     * @see ChallengeNameType
     */
    public final ChallengeNameType challengeName() {
        return ChallengeNameType.fromValue(challengeName);
    }

    /**
     * <p>
     * The name of the challenge that you are responding to. You can find more information about values for
     * <code>ChallengeName</code> in the response parameters of <a href=
     * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html#CognitoUserPools-AdminInitiateAuth-response-ChallengeName"
     * >AdminInitiateAuth</a>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #challengeName}
     * will return {@link ChallengeNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #challengeNameAsString}.
     * </p>
     * 
     * @return The name of the challenge that you are responding to. You can find more information about values for
     *         <code>ChallengeName</code> in the response parameters of <a href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html#CognitoUserPools-AdminInitiateAuth-response-ChallengeName"
     *         >AdminInitiateAuth</a>.
     * @see ChallengeNameType
     */
    public final String challengeNameAsString() {
        return challengeName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ChallengeResponses property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasChallengeResponses() {
        return challengeResponses != null && !(challengeResponses instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The responses to the challenge that you received in the previous request. Each challenge has its own required
     * response parameters. The following examples are partial JSON request bodies that highlight challenge-response
     * parameters.
     * </p>
     * <important>
     * <p>
     * You must provide a SECRET_HASH parameter in all challenge responses to an app client that has a client secret.
     * Include a <code>DEVICE_KEY</code> for device authentication.
     * </p>
     * </important>
     * <dl>
     * <dt>SELECT_CHALLENGE</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "USERNAME": "[username]", "ANSWER": "[Challenge name]"}</code>
     * </p>
     * <p>
     * Available challenges are <code>PASSWORD</code>, <code>PASSWORD_SRP</code>, <code>EMAIL_OTP</code>,
     * <code>SMS_OTP</code>, and <code>WEB_AUTHN</code>.
     * </p>
     * <p>
     * Complete authentication in the <code>SELECT_CHALLENGE</code> response for <code>PASSWORD</code>,
     * <code>PASSWORD_SRP</code>, and <code>WEB_AUTHN</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "WEB_AUTHN", "USERNAME": "[username]", "CREDENTIAL": "[AuthenticationResponseJSON]"}</code>
     * </p>
     * <p>
     * See <a href="https://www.w3.org/TR/webauthn-3/#dictdef-authenticationresponsejson">
     * AuthenticationResponseJSON</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "PASSWORD", "USERNAME": "[username]", "PASSWORD": "[password]"}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "PASSWORD_SRP", "USERNAME": "[username]", "SRP_A": "[SRP_A]"}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For <code>SMS_OTP</code> and <code>EMAIL_OTP</code>, respond with the username and answer. Your user pool will
     * send a code for the user to submit in the next challenge response.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "SMS_OTP", "USERNAME": "[username]"}</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "EMAIL_OTP", "USERNAME": "[username]"}</code>
     * </p>
     * </li>
     * </ul>
     * </dd>
     * <dt>SMS_OTP</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "SMS_OTP", "ChallengeResponses": {"SMS_OTP_CODE": "[code]", "USERNAME": "[username]"}</code>
     * </p>
     * </dd>
     * <dt>EMAIL_OTP</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "EMAIL_OTP", "ChallengeResponses": {"EMAIL_OTP_CODE": "[code]", "USERNAME": "[username]"}</code>
     * </p>
     * </dd>
     * <dt>SMS_MFA</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "SMS_MFA", "ChallengeResponses": {"SMS_MFA_CODE": "[code]", "USERNAME": "[username]"}</code>
     * </p>
     * </dd>
     * <dt>PASSWORD_VERIFIER</dt>
     * <dd>
     * <p>
     * This challenge response is part of the SRP flow. Amazon Cognito requires that your application respond to this
     * challenge within a few seconds. When the response time exceeds this period, your user pool returns a
     * <code>NotAuthorizedException</code> error.
     * </p>
     * <p>
     * <code>"ChallengeName": "PASSWORD_VERIFIER", "ChallengeResponses": {"PASSWORD_CLAIM_SIGNATURE": "[claim_signature]", "PASSWORD_CLAIM_SECRET_BLOCK": "[secret_block]", "TIMESTAMP": [timestamp], "USERNAME": "[username]"}</code>
     * </p>
     * <p>
     * Add <code>"DEVICE_KEY"</code> when you sign in with a remembered device.
     * </p>
     * </dd>
     * <dt>CUSTOM_CHALLENGE</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "CUSTOM_CHALLENGE", "ChallengeResponses": {"USERNAME": "[username]", "ANSWER": "[challenge_answer]"}</code>
     * </p>
     * <p>
     * Add <code>"DEVICE_KEY"</code> when you sign in with a remembered device.
     * </p>
     * </dd>
     * <dt>NEW_PASSWORD_REQUIRED</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "NEW_PASSWORD_REQUIRED", "ChallengeResponses": {"NEW_PASSWORD": "[new_password]", "USERNAME": "[username]"}</code>
     * </p>
     * <p>
     * To set any required attributes that <code>InitiateAuth</code> returned in an <code>requiredAttributes</code>
     * parameter, add <code>"userAttributes.[attribute_name]": "[attribute_value]"</code>. This parameter can also set
     * values for writable attributes that aren't required by your user pool.
     * </p>
     * <note>
     * <p>
     * In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that already
     * has a value. In <code>RespondToAuthChallenge</code>, set a value for any keys that Amazon Cognito returned in the
     * <code>requiredAttributes</code> parameter, then use the <code>UpdateUserAttributes</code> API operation to modify
     * the value of any additional attributes.
     * </p>
     * </note></dd>
     * <dt>SOFTWARE_TOKEN_MFA</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "SOFTWARE_TOKEN_MFA", "ChallengeResponses": {"USERNAME": "[username]", "SOFTWARE_TOKEN_MFA_CODE": [authenticator_code]}</code>
     * </p>
     * </dd>
     * <dt>DEVICE_SRP_AUTH</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "DEVICE_SRP_AUTH", "ChallengeResponses": {"USERNAME": "[username]", "DEVICE_KEY": "[device_key]", "SRP_A": "[srp_a]"}</code>
     * </p>
     * </dd>
     * <dt>DEVICE_PASSWORD_VERIFIER</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "DEVICE_PASSWORD_VERIFIER", "ChallengeResponses": {"DEVICE_KEY": "[device_key]", "PASSWORD_CLAIM_SIGNATURE": "[claim_signature]", "PASSWORD_CLAIM_SECRET_BLOCK": "[secret_block]", "TIMESTAMP": [timestamp], "USERNAME": "[username]"}</code>
     * </p>
     * </dd>
     * <dt>MFA_SETUP</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "MFA_SETUP", "ChallengeResponses": {"USERNAME": "[username]"}, "SESSION": "[Session ID from VerifySoftwareToken]"</code>
     * </p>
     * </dd>
     * <dt>SELECT_MFA_TYPE</dt>
     * <dd>
     * <p>
     * <code>"ChallengeName": "SELECT_MFA_TYPE", "ChallengeResponses": {"USERNAME": "[username]", "ANSWER": "[SMS_MFA or SOFTWARE_TOKEN_MFA]"}</code>
     * </p>
     * </dd>
     * </dl>
     * <p>
     * For more information about <code>SECRET_HASH</code>, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#cognito-user-pools-computing-secret-hash"
     * >Computing secret hash values</a>. For information about <code>DEVICE_KEY</code>, see <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html"
     * >Working with user devices in your user pool</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChallengeResponses} method.
     * </p>
     * 
     * @return The responses to the challenge that you received in the previous request. Each challenge has its own
     *         required response parameters. The following examples are partial JSON request bodies that highlight
     *         challenge-response parameters.</p> <important>
     *         <p>
     *         You must provide a SECRET_HASH parameter in all challenge responses to an app client that has a client
     *         secret. Include a <code>DEVICE_KEY</code> for device authentication.
     *         </p>
     *         </important>
     *         <dl>
     *         <dt>SELECT_CHALLENGE</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "USERNAME": "[username]", "ANSWER": "[Challenge name]"}</code>
     *         </p>
     *         <p>
     *         Available challenges are <code>PASSWORD</code>, <code>PASSWORD_SRP</code>, <code>EMAIL_OTP</code>,
     *         <code>SMS_OTP</code>, and <code>WEB_AUTHN</code>.
     *         </p>
     *         <p>
     *         Complete authentication in the <code>SELECT_CHALLENGE</code> response for <code>PASSWORD</code>,
     *         <code>PASSWORD_SRP</code>, and <code>WEB_AUTHN</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "WEB_AUTHN", "USERNAME": "[username]", "CREDENTIAL": "[AuthenticationResponseJSON]"}</code>
     *         </p>
     *         <p>
     *         See <a href="https://www.w3.org/TR/webauthn-3/#dictdef-authenticationresponsejson">
     *         AuthenticationResponseJSON</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "PASSWORD", "USERNAME": "[username]", "PASSWORD": "[password]"}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "PASSWORD_SRP", "USERNAME": "[username]", "SRP_A": "[SRP_A]"}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For <code>SMS_OTP</code> and <code>EMAIL_OTP</code>, respond with the username and answer. Your user pool
     *         will send a code for the user to submit in the next challenge response.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "SMS_OTP", "USERNAME": "[username]"}</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "EMAIL_OTP", "USERNAME": "[username]"}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </dd>
     *         <dt>SMS_OTP</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "SMS_OTP", "ChallengeResponses": {"SMS_OTP_CODE": "[code]", "USERNAME": "[username]"}</code>
     *         </p>
     *         </dd>
     *         <dt>EMAIL_OTP</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "EMAIL_OTP", "ChallengeResponses": {"EMAIL_OTP_CODE": "[code]", "USERNAME": "[username]"}</code>
     *         </p>
     *         </dd>
     *         <dt>SMS_MFA</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "SMS_MFA", "ChallengeResponses": {"SMS_MFA_CODE": "[code]", "USERNAME": "[username]"}</code>
     *         </p>
     *         </dd>
     *         <dt>PASSWORD_VERIFIER</dt>
     *         <dd>
     *         <p>
     *         This challenge response is part of the SRP flow. Amazon Cognito requires that your application respond to
     *         this challenge within a few seconds. When the response time exceeds this period, your user pool returns a
     *         <code>NotAuthorizedException</code> error.
     *         </p>
     *         <p>
     *         <code>"ChallengeName": "PASSWORD_VERIFIER", "ChallengeResponses": {"PASSWORD_CLAIM_SIGNATURE": "[claim_signature]", "PASSWORD_CLAIM_SECRET_BLOCK": "[secret_block]", "TIMESTAMP": [timestamp], "USERNAME": "[username]"}</code>
     *         </p>
     *         <p>
     *         Add <code>"DEVICE_KEY"</code> when you sign in with a remembered device.
     *         </p>
     *         </dd>
     *         <dt>CUSTOM_CHALLENGE</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "CUSTOM_CHALLENGE", "ChallengeResponses": {"USERNAME": "[username]", "ANSWER": "[challenge_answer]"}</code>
     *         </p>
     *         <p>
     *         Add <code>"DEVICE_KEY"</code> when you sign in with a remembered device.
     *         </p>
     *         </dd>
     *         <dt>NEW_PASSWORD_REQUIRED</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "NEW_PASSWORD_REQUIRED", "ChallengeResponses": {"NEW_PASSWORD": "[new_password]", "USERNAME": "[username]"}</code>
     *         </p>
     *         <p>
     *         To set any required attributes that <code>InitiateAuth</code> returned in an
     *         <code>requiredAttributes</code> parameter, add
     *         <code>"userAttributes.[attribute_name]": "[attribute_value]"</code>. This parameter can also set values
     *         for writable attributes that aren't required by your user pool.
     *         </p>
     *         <note>
     *         <p>
     *         In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
     *         already has a value. In <code>RespondToAuthChallenge</code>, set a value for any keys that Amazon Cognito
     *         returned in the <code>requiredAttributes</code> parameter, then use the <code>UpdateUserAttributes</code>
     *         API operation to modify the value of any additional attributes.
     *         </p>
     *         </note></dd>
     *         <dt>SOFTWARE_TOKEN_MFA</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "SOFTWARE_TOKEN_MFA", "ChallengeResponses": {"USERNAME": "[username]", "SOFTWARE_TOKEN_MFA_CODE": [authenticator_code]}</code>
     *         </p>
     *         </dd>
     *         <dt>DEVICE_SRP_AUTH</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "DEVICE_SRP_AUTH", "ChallengeResponses": {"USERNAME": "[username]", "DEVICE_KEY": "[device_key]", "SRP_A": "[srp_a]"}</code>
     *         </p>
     *         </dd>
     *         <dt>DEVICE_PASSWORD_VERIFIER</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "DEVICE_PASSWORD_VERIFIER", "ChallengeResponses": {"DEVICE_KEY": "[device_key]", "PASSWORD_CLAIM_SIGNATURE": "[claim_signature]", "PASSWORD_CLAIM_SECRET_BLOCK": "[secret_block]", "TIMESTAMP": [timestamp], "USERNAME": "[username]"}</code>
     *         </p>
     *         </dd>
     *         <dt>MFA_SETUP</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "MFA_SETUP", "ChallengeResponses": {"USERNAME": "[username]"}, "SESSION": "[Session ID from VerifySoftwareToken]"</code>
     *         </p>
     *         </dd>
     *         <dt>SELECT_MFA_TYPE</dt>
     *         <dd>
     *         <p>
     *         <code>"ChallengeName": "SELECT_MFA_TYPE", "ChallengeResponses": {"USERNAME": "[username]", "ANSWER": "[SMS_MFA or SOFTWARE_TOKEN_MFA]"}</code>
     *         </p>
     *         </dd>
     *         </dl>
     *         <p>
     *         For more information about <code>SECRET_HASH</code>, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#cognito-user-pools-computing-secret-hash"
     *         >Computing secret hash values</a>. For information about <code>DEVICE_KEY</code>, see <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html"
     *         >Working with user devices in your user pool</a>.
     */
    public final Map<String, String> challengeResponses() {
        return challengeResponses;
    }

    /**
     * <p>
     * The session identifier that maintains the state of authentication requests and challenge responses. If an
     * <code>AdminInitiateAuth</code> or <code>AdminRespondToAuthChallenge</code> API request results in a determination
     * that your application must pass another challenge, Amazon Cognito returns a session with other challenge
     * parameters. Send this session identifier, unmodified, to the next <code>AdminRespondToAuthChallenge</code>
     * request.
     * </p>
     * 
     * @return The session identifier that maintains the state of authentication requests and challenge responses. If an
     *         <code>AdminInitiateAuth</code> or <code>AdminRespondToAuthChallenge</code> API request results in a
     *         determination that your application must pass another challenge, Amazon Cognito returns a session with
     *         other challenge parameters. Send this session identifier, unmodified, to the next
     *         <code>AdminRespondToAuthChallenge</code> request.
     */
    public final String session() {
        return session;
    }

    /**
     * <p>
     * The analytics metadata for collecting Amazon Pinpoint metrics for <code>AdminRespondToAuthChallenge</code> calls.
     * </p>
     * 
     * @return The analytics metadata for collecting Amazon Pinpoint metrics for
     *         <code>AdminRespondToAuthChallenge</code> calls.
     */
    public final AnalyticsMetadataType analyticsMetadata() {
        return analyticsMetadata;
    }

    /**
     * <p>
     * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon Cognito
     * advanced security evaluates the risk of an authentication event based on the context that your app generates and
     * passes to Amazon Cognito when it makes API requests.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
     * >Collecting data for threat protection in applications</a>.
     * </p>
     * 
     * @return Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
     *         Cognito advanced security evaluates the risk of an authentication event based on the context that your
     *         app generates and passes to Amazon Cognito when it makes API requests.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
     *         >Collecting data for threat protection in applications</a>.
     */
    public final ContextDataType contextData() {
        return contextData;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     * AdminRespondToAuthChallenge API action, Amazon Cognito invokes any functions that you have assigned to the
     * following triggers:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Pre sign-up
     * </p>
     * </li>
     * <li>
     * <p>
     * custom message
     * </p>
     * </li>
     * <li>
     * <p>
     * Post authentication
     * </p>
     * </li>
     * <li>
     * <p>
     * User migration
     * </p>
     * </li>
     * <li>
     * <p>
     * Pre token generation
     * </p>
     * </li>
     * <li>
     * <p>
     * Define auth challenge
     * </p>
     * </li>
     * <li>
     * <p>
     * Create auth challenge
     * </p>
     * </li>
     * <li>
     * <p>
     * Verify auth challenge response
     * </p>
     * </li>
     * </ul>
     * <p>
     * When Amazon Cognito invokes any of these functions, it passes a JSON payload, which the function receives as
     * input. This payload contains a <code>clientMetadata</code> attribute that provides the data that you assigned to
     * the ClientMetadata parameter in your AdminRespondToAuthChallenge request. In your function code in Lambda, you
     * can process the <code>clientMetadata</code> value to enhance your workflow for your specific needs.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are assigned to
     * a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
     * <code>ClientMetadata</code> parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Validate the <code>ClientMetadata</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientMetadata} method.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for any custom workflows that this action
     *         triggers.</p>
     *         <p>
     *         You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     *         AdminRespondToAuthChallenge API action, Amazon Cognito invokes any functions that you have assigned to
     *         the following triggers:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Pre sign-up
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         custom message
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Post authentication
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         User migration
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Pre token generation
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Define auth challenge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Create auth challenge
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Verify auth challenge response
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When Amazon Cognito invokes any of these functions, it passes a JSON payload, which the function receives
     *         as input. This payload contains a <code>clientMetadata</code> attribute that provides the data that you
     *         assigned to the ClientMetadata parameter in your AdminRespondToAuthChallenge request. In your function
     *         code in Lambda, you can process the <code>clientMetadata</code> value to enhance your workflow for your
     *         specific needs.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are
     *         assigned to a user pool to support custom workflows. If your user pool configuration doesn't include
     *         triggers, the <code>ClientMetadata</code> parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Validate the <code>ClientMetadata</code> value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(challengeNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasChallengeResponses() ? challengeResponses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(session());
        hashCode = 31 * hashCode + Objects.hashCode(analyticsMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(contextData());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdminRespondToAuthChallengeRequest)) {
            return false;
        }
        AdminRespondToAuthChallengeRequest other = (AdminRespondToAuthChallengeRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(clientId(), other.clientId())
                && Objects.equals(challengeNameAsString(), other.challengeNameAsString())
                && hasChallengeResponses() == other.hasChallengeResponses()
                && Objects.equals(challengeResponses(), other.challengeResponses()) && Objects.equals(session(), other.session())
                && Objects.equals(analyticsMetadata(), other.analyticsMetadata())
                && Objects.equals(contextData(), other.contextData()) && hasClientMetadata() == other.hasClientMetadata()
                && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdminRespondToAuthChallengeRequest").add("UserPoolId", userPoolId())
                .add("ClientId", clientId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ChallengeName", challengeNameAsString())
                .add("ChallengeResponses", challengeResponses() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Session", session() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AnalyticsMetadata", analyticsMetadata()).add("ContextData", contextData())
                .add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "ChallengeName":
            return Optional.ofNullable(clazz.cast(challengeNameAsString()));
        case "ChallengeResponses":
            return Optional.ofNullable(clazz.cast(challengeResponses()));
        case "Session":
            return Optional.ofNullable(clazz.cast(session()));
        case "AnalyticsMetadata":
            return Optional.ofNullable(clazz.cast(analyticsMetadata()));
        case "ContextData":
            return Optional.ofNullable(clazz.cast(contextData()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("UserPoolId", USER_POOL_ID_FIELD);
        map.put("ClientId", CLIENT_ID_FIELD);
        map.put("ChallengeName", CHALLENGE_NAME_FIELD);
        map.put("ChallengeResponses", CHALLENGE_RESPONSES_FIELD);
        map.put("Session", SESSION_FIELD);
        map.put("AnalyticsMetadata", ANALYTICS_METADATA_FIELD);
        map.put("ContextData", CONTEXT_DATA_FIELD);
        map.put("ClientMetadata", CLIENT_METADATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AdminRespondToAuthChallengeRequest, T> g) {
        return obj -> g.apply((AdminRespondToAuthChallengeRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, AdminRespondToAuthChallengeRequest> {
        /**
         * <p>
         * The ID of the user pool where you want to respond to an authentication challenge.
         * </p>
         * 
         * @param userPoolId
         *        The ID of the user pool where you want to respond to an authentication challenge.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The ID of the app client where you initiated sign-in.
         * </p>
         * 
         * @param clientId
         *        The ID of the app client where you initiated sign-in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * The name of the challenge that you are responding to. You can find more information about values for
         * <code>ChallengeName</code> in the response parameters of <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html#CognitoUserPools-AdminInitiateAuth-response-ChallengeName"
         * >AdminInitiateAuth</a>.
         * </p>
         * 
         * @param challengeName
         *        The name of the challenge that you are responding to. You can find more information about values for
         *        <code>ChallengeName</code> in the response parameters of <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html#CognitoUserPools-AdminInitiateAuth-response-ChallengeName"
         *        >AdminInitiateAuth</a>.
         * @see ChallengeNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChallengeNameType
         */
        Builder challengeName(String challengeName);

        /**
         * <p>
         * The name of the challenge that you are responding to. You can find more information about values for
         * <code>ChallengeName</code> in the response parameters of <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html#CognitoUserPools-AdminInitiateAuth-response-ChallengeName"
         * >AdminInitiateAuth</a>.
         * </p>
         * 
         * @param challengeName
         *        The name of the challenge that you are responding to. You can find more information about values for
         *        <code>ChallengeName</code> in the response parameters of <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_AdminInitiateAuth.html#CognitoUserPools-AdminInitiateAuth-response-ChallengeName"
         *        >AdminInitiateAuth</a>.
         * @see ChallengeNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChallengeNameType
         */
        Builder challengeName(ChallengeNameType challengeName);

        /**
         * <p>
         * The responses to the challenge that you received in the previous request. Each challenge has its own required
         * response parameters. The following examples are partial JSON request bodies that highlight challenge-response
         * parameters.
         * </p>
         * <important>
         * <p>
         * You must provide a SECRET_HASH parameter in all challenge responses to an app client that has a client
         * secret. Include a <code>DEVICE_KEY</code> for device authentication.
         * </p>
         * </important>
         * <dl>
         * <dt>SELECT_CHALLENGE</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "USERNAME": "[username]", "ANSWER": "[Challenge name]"}</code>
         * </p>
         * <p>
         * Available challenges are <code>PASSWORD</code>, <code>PASSWORD_SRP</code>, <code>EMAIL_OTP</code>,
         * <code>SMS_OTP</code>, and <code>WEB_AUTHN</code>.
         * </p>
         * <p>
         * Complete authentication in the <code>SELECT_CHALLENGE</code> response for <code>PASSWORD</code>,
         * <code>PASSWORD_SRP</code>, and <code>WEB_AUTHN</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "WEB_AUTHN", "USERNAME": "[username]", "CREDENTIAL": "[AuthenticationResponseJSON]"}</code>
         * </p>
         * <p>
         * See <a href="https://www.w3.org/TR/webauthn-3/#dictdef-authenticationresponsejson">
         * AuthenticationResponseJSON</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "PASSWORD", "USERNAME": "[username]", "PASSWORD": "[password]"}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "PASSWORD_SRP", "USERNAME": "[username]", "SRP_A": "[SRP_A]"}</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For <code>SMS_OTP</code> and <code>EMAIL_OTP</code>, respond with the username and answer. Your user pool
         * will send a code for the user to submit in the next challenge response.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "SMS_OTP", "USERNAME": "[username]"}</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "EMAIL_OTP", "USERNAME": "[username]"}</code>
         * </p>
         * </li>
         * </ul>
         * </dd>
         * <dt>SMS_OTP</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "SMS_OTP", "ChallengeResponses": {"SMS_OTP_CODE": "[code]", "USERNAME": "[username]"}</code>
         * </p>
         * </dd>
         * <dt>EMAIL_OTP</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "EMAIL_OTP", "ChallengeResponses": {"EMAIL_OTP_CODE": "[code]", "USERNAME": "[username]"}</code>
         * </p>
         * </dd>
         * <dt>SMS_MFA</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "SMS_MFA", "ChallengeResponses": {"SMS_MFA_CODE": "[code]", "USERNAME": "[username]"}</code>
         * </p>
         * </dd>
         * <dt>PASSWORD_VERIFIER</dt>
         * <dd>
         * <p>
         * This challenge response is part of the SRP flow. Amazon Cognito requires that your application respond to
         * this challenge within a few seconds. When the response time exceeds this period, your user pool returns a
         * <code>NotAuthorizedException</code> error.
         * </p>
         * <p>
         * <code>"ChallengeName": "PASSWORD_VERIFIER", "ChallengeResponses": {"PASSWORD_CLAIM_SIGNATURE": "[claim_signature]", "PASSWORD_CLAIM_SECRET_BLOCK": "[secret_block]", "TIMESTAMP": [timestamp], "USERNAME": "[username]"}</code>
         * </p>
         * <p>
         * Add <code>"DEVICE_KEY"</code> when you sign in with a remembered device.
         * </p>
         * </dd>
         * <dt>CUSTOM_CHALLENGE</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "CUSTOM_CHALLENGE", "ChallengeResponses": {"USERNAME": "[username]", "ANSWER": "[challenge_answer]"}</code>
         * </p>
         * <p>
         * Add <code>"DEVICE_KEY"</code> when you sign in with a remembered device.
         * </p>
         * </dd>
         * <dt>NEW_PASSWORD_REQUIRED</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "NEW_PASSWORD_REQUIRED", "ChallengeResponses": {"NEW_PASSWORD": "[new_password]", "USERNAME": "[username]"}</code>
         * </p>
         * <p>
         * To set any required attributes that <code>InitiateAuth</code> returned in an <code>requiredAttributes</code>
         * parameter, add <code>"userAttributes.[attribute_name]": "[attribute_value]"</code>. This parameter can also
         * set values for writable attributes that aren't required by your user pool.
         * </p>
         * <note>
         * <p>
         * In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
         * already has a value. In <code>RespondToAuthChallenge</code>, set a value for any keys that Amazon Cognito
         * returned in the <code>requiredAttributes</code> parameter, then use the <code>UpdateUserAttributes</code> API
         * operation to modify the value of any additional attributes.
         * </p>
         * </note></dd>
         * <dt>SOFTWARE_TOKEN_MFA</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "SOFTWARE_TOKEN_MFA", "ChallengeResponses": {"USERNAME": "[username]", "SOFTWARE_TOKEN_MFA_CODE": [authenticator_code]}</code>
         * </p>
         * </dd>
         * <dt>DEVICE_SRP_AUTH</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "DEVICE_SRP_AUTH", "ChallengeResponses": {"USERNAME": "[username]", "DEVICE_KEY": "[device_key]", "SRP_A": "[srp_a]"}</code>
         * </p>
         * </dd>
         * <dt>DEVICE_PASSWORD_VERIFIER</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "DEVICE_PASSWORD_VERIFIER", "ChallengeResponses": {"DEVICE_KEY": "[device_key]", "PASSWORD_CLAIM_SIGNATURE": "[claim_signature]", "PASSWORD_CLAIM_SECRET_BLOCK": "[secret_block]", "TIMESTAMP": [timestamp], "USERNAME": "[username]"}</code>
         * </p>
         * </dd>
         * <dt>MFA_SETUP</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "MFA_SETUP", "ChallengeResponses": {"USERNAME": "[username]"}, "SESSION": "[Session ID from VerifySoftwareToken]"</code>
         * </p>
         * </dd>
         * <dt>SELECT_MFA_TYPE</dt>
         * <dd>
         * <p>
         * <code>"ChallengeName": "SELECT_MFA_TYPE", "ChallengeResponses": {"USERNAME": "[username]", "ANSWER": "[SMS_MFA or SOFTWARE_TOKEN_MFA]"}</code>
         * </p>
         * </dd>
         * </dl>
         * <p>
         * For more information about <code>SECRET_HASH</code>, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#cognito-user-pools-computing-secret-hash"
         * >Computing secret hash values</a>. For information about <code>DEVICE_KEY</code>, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html"
         * >Working with user devices in your user pool</a>.
         * </p>
         * 
         * @param challengeResponses
         *        The responses to the challenge that you received in the previous request. Each challenge has its own
         *        required response parameters. The following examples are partial JSON request bodies that highlight
         *        challenge-response parameters.</p> <important>
         *        <p>
         *        You must provide a SECRET_HASH parameter in all challenge responses to an app client that has a client
         *        secret. Include a <code>DEVICE_KEY</code> for device authentication.
         *        </p>
         *        </important>
         *        <dl>
         *        <dt>SELECT_CHALLENGE</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "USERNAME": "[username]", "ANSWER": "[Challenge name]"}</code>
         *        </p>
         *        <p>
         *        Available challenges are <code>PASSWORD</code>, <code>PASSWORD_SRP</code>, <code>EMAIL_OTP</code>,
         *        <code>SMS_OTP</code>, and <code>WEB_AUTHN</code>.
         *        </p>
         *        <p>
         *        Complete authentication in the <code>SELECT_CHALLENGE</code> response for <code>PASSWORD</code>,
         *        <code>PASSWORD_SRP</code>, and <code>WEB_AUTHN</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "WEB_AUTHN", "USERNAME": "[username]", "CREDENTIAL": "[AuthenticationResponseJSON]"}</code>
         *        </p>
         *        <p>
         *        See <a href="https://www.w3.org/TR/webauthn-3/#dictdef-authenticationresponsejson">
         *        AuthenticationResponseJSON</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "PASSWORD", "USERNAME": "[username]", "PASSWORD": "[password]"}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "PASSWORD_SRP", "USERNAME": "[username]", "SRP_A": "[SRP_A]"}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For <code>SMS_OTP</code> and <code>EMAIL_OTP</code>, respond with the username and answer. Your user
         *        pool will send a code for the user to submit in the next challenge response.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "SMS_OTP", "USERNAME": "[username]"}</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"ChallengeName": "SELECT_CHALLENGE", "ChallengeResponses": { "ANSWER": "EMAIL_OTP", "USERNAME": "[username]"}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </dd>
         *        <dt>SMS_OTP</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "SMS_OTP", "ChallengeResponses": {"SMS_OTP_CODE": "[code]", "USERNAME": "[username]"}</code>
         *        </p>
         *        </dd>
         *        <dt>EMAIL_OTP</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "EMAIL_OTP", "ChallengeResponses": {"EMAIL_OTP_CODE": "[code]", "USERNAME": "[username]"}</code>
         *        </p>
         *        </dd>
         *        <dt>SMS_MFA</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "SMS_MFA", "ChallengeResponses": {"SMS_MFA_CODE": "[code]", "USERNAME": "[username]"}</code>
         *        </p>
         *        </dd>
         *        <dt>PASSWORD_VERIFIER</dt>
         *        <dd>
         *        <p>
         *        This challenge response is part of the SRP flow. Amazon Cognito requires that your application respond
         *        to this challenge within a few seconds. When the response time exceeds this period, your user pool
         *        returns a <code>NotAuthorizedException</code> error.
         *        </p>
         *        <p>
         *        <code>"ChallengeName": "PASSWORD_VERIFIER", "ChallengeResponses": {"PASSWORD_CLAIM_SIGNATURE": "[claim_signature]", "PASSWORD_CLAIM_SECRET_BLOCK": "[secret_block]", "TIMESTAMP": [timestamp], "USERNAME": "[username]"}</code>
         *        </p>
         *        <p>
         *        Add <code>"DEVICE_KEY"</code> when you sign in with a remembered device.
         *        </p>
         *        </dd>
         *        <dt>CUSTOM_CHALLENGE</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "CUSTOM_CHALLENGE", "ChallengeResponses": {"USERNAME": "[username]", "ANSWER": "[challenge_answer]"}</code>
         *        </p>
         *        <p>
         *        Add <code>"DEVICE_KEY"</code> when you sign in with a remembered device.
         *        </p>
         *        </dd>
         *        <dt>NEW_PASSWORD_REQUIRED</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "NEW_PASSWORD_REQUIRED", "ChallengeResponses": {"NEW_PASSWORD": "[new_password]", "USERNAME": "[username]"}</code>
         *        </p>
         *        <p>
         *        To set any required attributes that <code>InitiateAuth</code> returned in an
         *        <code>requiredAttributes</code> parameter, add
         *        <code>"userAttributes.[attribute_name]": "[attribute_value]"</code>. This parameter can also set
         *        values for writable attributes that aren't required by your user pool.
         *        </p>
         *        <note>
         *        <p>
         *        In a <code>NEW_PASSWORD_REQUIRED</code> challenge response, you can't modify a required attribute that
         *        already has a value. In <code>RespondToAuthChallenge</code>, set a value for any keys that Amazon
         *        Cognito returned in the <code>requiredAttributes</code> parameter, then use the
         *        <code>UpdateUserAttributes</code> API operation to modify the value of any additional attributes.
         *        </p>
         *        </note></dd>
         *        <dt>SOFTWARE_TOKEN_MFA</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "SOFTWARE_TOKEN_MFA", "ChallengeResponses": {"USERNAME": "[username]", "SOFTWARE_TOKEN_MFA_CODE": [authenticator_code]}</code>
         *        </p>
         *        </dd>
         *        <dt>DEVICE_SRP_AUTH</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "DEVICE_SRP_AUTH", "ChallengeResponses": {"USERNAME": "[username]", "DEVICE_KEY": "[device_key]", "SRP_A": "[srp_a]"}</code>
         *        </p>
         *        </dd>
         *        <dt>DEVICE_PASSWORD_VERIFIER</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "DEVICE_PASSWORD_VERIFIER", "ChallengeResponses": {"DEVICE_KEY": "[device_key]", "PASSWORD_CLAIM_SIGNATURE": "[claim_signature]", "PASSWORD_CLAIM_SECRET_BLOCK": "[secret_block]", "TIMESTAMP": [timestamp], "USERNAME": "[username]"}</code>
         *        </p>
         *        </dd>
         *        <dt>MFA_SETUP</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "MFA_SETUP", "ChallengeResponses": {"USERNAME": "[username]"}, "SESSION": "[Session ID from VerifySoftwareToken]"</code>
         *        </p>
         *        </dd>
         *        <dt>SELECT_MFA_TYPE</dt>
         *        <dd>
         *        <p>
         *        <code>"ChallengeName": "SELECT_MFA_TYPE", "ChallengeResponses": {"USERNAME": "[username]", "ANSWER": "[SMS_MFA or SOFTWARE_TOKEN_MFA]"}</code>
         *        </p>
         *        </dd>
         *        </dl>
         *        <p>
         *        For more information about <code>SECRET_HASH</code>, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#cognito-user-pools-computing-secret-hash"
         *        >Computing secret hash values</a>. For information about <code>DEVICE_KEY</code>, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-device-tracking.html"
         *        >Working with user devices in your user pool</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder challengeResponses(Map<String, String> challengeResponses);

        /**
         * <p>
         * The session identifier that maintains the state of authentication requests and challenge responses. If an
         * <code>AdminInitiateAuth</code> or <code>AdminRespondToAuthChallenge</code> API request results in a
         * determination that your application must pass another challenge, Amazon Cognito returns a session with other
         * challenge parameters. Send this session identifier, unmodified, to the next
         * <code>AdminRespondToAuthChallenge</code> request.
         * </p>
         * 
         * @param session
         *        The session identifier that maintains the state of authentication requests and challenge responses. If
         *        an <code>AdminInitiateAuth</code> or <code>AdminRespondToAuthChallenge</code> API request results in a
         *        determination that your application must pass another challenge, Amazon Cognito returns a session with
         *        other challenge parameters. Send this session identifier, unmodified, to the next
         *        <code>AdminRespondToAuthChallenge</code> request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder session(String session);

        /**
         * <p>
         * The analytics metadata for collecting Amazon Pinpoint metrics for <code>AdminRespondToAuthChallenge</code>
         * calls.
         * </p>
         * 
         * @param analyticsMetadata
         *        The analytics metadata for collecting Amazon Pinpoint metrics for
         *        <code>AdminRespondToAuthChallenge</code> calls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata);

        /**
         * <p>
         * The analytics metadata for collecting Amazon Pinpoint metrics for <code>AdminRespondToAuthChallenge</code>
         * calls.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalyticsMetadataType.Builder} avoiding
         * the need to create one manually via {@link AnalyticsMetadataType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnalyticsMetadataType.Builder#build()} is called immediately and
         * its result is passed to {@link #analyticsMetadata(AnalyticsMetadataType)}.
         * 
         * @param analyticsMetadata
         *        a consumer that will call methods on {@link AnalyticsMetadataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #analyticsMetadata(AnalyticsMetadataType)
         */
        default Builder analyticsMetadata(Consumer<AnalyticsMetadataType.Builder> analyticsMetadata) {
            return analyticsMetadata(AnalyticsMetadataType.builder().applyMutation(analyticsMetadata).build());
        }

        /**
         * <p>
         * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
         * Cognito advanced security evaluates the risk of an authentication event based on the context that your app
         * generates and passes to Amazon Cognito when it makes API requests.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
         * >Collecting data for threat protection in applications</a>.
         * </p>
         * 
         * @param contextData
         *        Contextual data about your user session, such as the device fingerprint, IP address, or location.
         *        Amazon Cognito advanced security evaluates the risk of an authentication event based on the context
         *        that your app generates and passes to Amazon Cognito when it makes API requests.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
         *        >Collecting data for threat protection in applications</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contextData(ContextDataType contextData);

        /**
         * <p>
         * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
         * Cognito advanced security evaluates the risk of an authentication event based on the context that your app
         * generates and passes to Amazon Cognito when it makes API requests.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
         * >Collecting data for threat protection in applications</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ContextDataType.Builder} avoiding the
         * need to create one manually via {@link ContextDataType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ContextDataType.Builder#build()} is called immediately and its
         * result is passed to {@link #contextData(ContextDataType)}.
         * 
         * @param contextData
         *        a consumer that will call methods on {@link ContextDataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #contextData(ContextDataType)
         */
        default Builder contextData(Consumer<ContextDataType.Builder> contextData) {
            return contextData(ContextDataType.builder().applyMutation(contextData).build());
        }

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for any custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         * AdminRespondToAuthChallenge API action, Amazon Cognito invokes any functions that you have assigned to the
         * following triggers:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Pre sign-up
         * </p>
         * </li>
         * <li>
         * <p>
         * custom message
         * </p>
         * </li>
         * <li>
         * <p>
         * Post authentication
         * </p>
         * </li>
         * <li>
         * <p>
         * User migration
         * </p>
         * </li>
         * <li>
         * <p>
         * Pre token generation
         * </p>
         * </li>
         * <li>
         * <p>
         * Define auth challenge
         * </p>
         * </li>
         * <li>
         * <p>
         * Create auth challenge
         * </p>
         * </li>
         * <li>
         * <p>
         * Verify auth challenge response
         * </p>
         * </li>
         * </ul>
         * <p>
         * When Amazon Cognito invokes any of these functions, it passes a JSON payload, which the function receives as
         * input. This payload contains a <code>clientMetadata</code> attribute that provides the data that you assigned
         * to the ClientMetadata parameter in your AdminRespondToAuthChallenge request. In your function code in Lambda,
         * you can process the <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are assigned
         * to a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         * <code>ClientMetadata</code> parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Validate the <code>ClientMetadata</code> value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for any custom workflows that this
         *        action triggers.</p>
         *        <p>
         *        You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         *        AdminRespondToAuthChallenge API action, Amazon Cognito invokes any functions that you have assigned to
         *        the following triggers:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Pre sign-up
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        custom message
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Post authentication
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        User migration
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Pre token generation
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Define auth challenge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Create auth challenge
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Verify auth challenge response
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When Amazon Cognito invokes any of these functions, it passes a JSON payload, which the function
         *        receives as input. This payload contains a <code>clientMetadata</code> attribute that provides the
         *        data that you assigned to the ClientMetadata parameter in your AdminRespondToAuthChallenge request. In
         *        your function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your
         *        workflow for your specific needs.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the
         *        following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are
         *        assigned to a user pool to support custom workflows. If your user pool configuration doesn't include
         *        triggers, the <code>ClientMetadata</code> parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Validate the <code>ClientMetadata</code> value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String clientId;

        private String challengeName;

        private Map<String, String> challengeResponses = DefaultSdkAutoConstructMap.getInstance();

        private String session;

        private AnalyticsMetadataType analyticsMetadata;

        private ContextDataType contextData;

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdminRespondToAuthChallengeRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            clientId(model.clientId);
            challengeName(model.challengeName);
            challengeResponses(model.challengeResponses);
            session(model.session);
            analyticsMetadata(model.analyticsMetadata);
            contextData(model.contextData);
            clientMetadata(model.clientMetadata);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final String getClientId() {
            return clientId;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final String getChallengeName() {
            return challengeName;
        }

        public final void setChallengeName(String challengeName) {
            this.challengeName = challengeName;
        }

        @Override
        public final Builder challengeName(String challengeName) {
            this.challengeName = challengeName;
            return this;
        }

        @Override
        public final Builder challengeName(ChallengeNameType challengeName) {
            this.challengeName(challengeName == null ? null : challengeName.toString());
            return this;
        }

        public final Map<String, String> getChallengeResponses() {
            if (challengeResponses instanceof SdkAutoConstructMap) {
                return null;
            }
            return challengeResponses;
        }

        public final void setChallengeResponses(Map<String, String> challengeResponses) {
            this.challengeResponses = ChallengeResponsesTypeCopier.copy(challengeResponses);
        }

        @Override
        public final Builder challengeResponses(Map<String, String> challengeResponses) {
            this.challengeResponses = ChallengeResponsesTypeCopier.copy(challengeResponses);
            return this;
        }

        public final String getSession() {
            return session;
        }

        public final void setSession(String session) {
            this.session = session;
        }

        @Override
        public final Builder session(String session) {
            this.session = session;
            return this;
        }

        public final AnalyticsMetadataType.Builder getAnalyticsMetadata() {
            return analyticsMetadata != null ? analyticsMetadata.toBuilder() : null;
        }

        public final void setAnalyticsMetadata(AnalyticsMetadataType.BuilderImpl analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata != null ? analyticsMetadata.build() : null;
        }

        @Override
        public final Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata;
            return this;
        }

        public final ContextDataType.Builder getContextData() {
            return contextData != null ? contextData.toBuilder() : null;
        }

        public final void setContextData(ContextDataType.BuilderImpl contextData) {
            this.contextData = contextData != null ? contextData.build() : null;
        }

        @Override
        public final Builder contextData(ContextDataType contextData) {
            this.contextData = contextData;
            return this;
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AdminRespondToAuthChallengeRequest build() {
            return new AdminRespondToAuthChallengeRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
