/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the request to register a user.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SignUpRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<SignUpRequest.Builder, SignUpRequest> {
    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClientId").getter(getter(SignUpRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<String> SECRET_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretHash").getter(getter(SignUpRequest::secretHash)).setter(setter(Builder::secretHash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretHash").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(SignUpRequest::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<String> PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Password").getter(getter(SignUpRequest::password)).setter(setter(Builder::password))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Password").build()).build();

    private static final SdkField<List<AttributeType>> USER_ATTRIBUTES_FIELD = SdkField
            .<List<AttributeType>> builder(MarshallingType.LIST)
            .memberName("UserAttributes")
            .getter(getter(SignUpRequest::userAttributes))
            .setter(setter(Builder::userAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AttributeType>> VALIDATION_DATA_FIELD = SdkField
            .<List<AttributeType>> builder(MarshallingType.LIST)
            .memberName("ValidationData")
            .getter(getter(SignUpRequest::validationData))
            .setter(setter(Builder::validationData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidationData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AnalyticsMetadataType> ANALYTICS_METADATA_FIELD = SdkField
            .<AnalyticsMetadataType> builder(MarshallingType.SDK_POJO).memberName("AnalyticsMetadata")
            .getter(getter(SignUpRequest::analyticsMetadata)).setter(setter(Builder::analyticsMetadata))
            .constructor(AnalyticsMetadataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalyticsMetadata").build()).build();

    private static final SdkField<UserContextDataType> USER_CONTEXT_DATA_FIELD = SdkField
            .<UserContextDataType> builder(MarshallingType.SDK_POJO).memberName("UserContextData")
            .getter(getter(SignUpRequest::userContextData)).setter(setter(Builder::userContextData))
            .constructor(UserContextDataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserContextData").build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(SignUpRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_ID_FIELD,
            SECRET_HASH_FIELD, USERNAME_FIELD, PASSWORD_FIELD, USER_ATTRIBUTES_FIELD, VALIDATION_DATA_FIELD,
            ANALYTICS_METADATA_FIELD, USER_CONTEXT_DATA_FIELD, CLIENT_METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String clientId;

    private final String secretHash;

    private final String username;

    private final String password;

    private final List<AttributeType> userAttributes;

    private final List<AttributeType> validationData;

    private final AnalyticsMetadataType analyticsMetadata;

    private final UserContextDataType userContextData;

    private final Map<String, String> clientMetadata;

    private SignUpRequest(BuilderImpl builder) {
        super(builder);
        this.clientId = builder.clientId;
        this.secretHash = builder.secretHash;
        this.username = builder.username;
        this.password = builder.password;
        this.userAttributes = builder.userAttributes;
        this.validationData = builder.validationData;
        this.analyticsMetadata = builder.analyticsMetadata;
        this.userContextData = builder.userContextData;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The ID of the client associated with the user pool.
     * </p>
     * 
     * @return The ID of the client associated with the user pool.
     */
    public final String clientId() {
        return clientId;
    }

    /**
     * <p>
     * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     * username plus the client ID in the message. For more information about <code>SecretHash</code>, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#cognito-user-pools-computing-secret-hash"
     * >Computing secret hash values</a>.
     * </p>
     * 
     * @return A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
     *         username plus the client ID in the message. For more information about <code>SecretHash</code>, see <a
     *         href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#cognito-user-pools-computing-secret-hash"
     *         >Computing secret hash values</a>.
     */
    public final String secretHash() {
        return secretHash;
    }

    /**
     * <p>
     * The username of the user that you want to sign up. The value of this parameter is typically a username, but can
     * be any alias attribute in your user pool.
     * </p>
     * 
     * @return The username of the user that you want to sign up. The value of this parameter is typically a username,
     *         but can be any alias attribute in your user pool.
     */
    public final String username() {
        return username;
    }

    /**
     * <p>
     * The password of the user you want to register.
     * </p>
     * <p>
     * Users can sign up without a password when your user pool supports passwordless sign-in with email or SMS OTPs. To
     * create a user with no password, omit this parameter or submit a blank value. You can only create a passwordless
     * user when passwordless sign-in is available. See <a
     * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SignInPolicyType.html">the
     * SignInPolicyType</a> property of <a
     * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html"
     * >CreateUserPool</a> and <a
     * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html"
     * >UpdateUserPool</a>.
     * </p>
     * 
     * @return The password of the user you want to register.</p>
     *         <p>
     *         Users can sign up without a password when your user pool supports passwordless sign-in with email or SMS
     *         OTPs. To create a user with no password, omit this parameter or submit a blank value. You can only create
     *         a passwordless user when passwordless sign-in is available. See <a href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SignInPolicyType.html"
     *         >the SignInPolicyType</a> property of <a href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html"
     *         >CreateUserPool</a> and <a href=
     *         "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html"
     *         >UpdateUserPool</a>.
     */
    public final String password() {
        return password;
    }

    /**
     * For responses, this returns true if the service returned a value for the UserAttributes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUserAttributes() {
        return userAttributes != null && !(userAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of name-value pairs representing user attributes.
     * </p>
     * <p>
     * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUserAttributes} method.
     * </p>
     * 
     * @return An array of name-value pairs representing user attributes.</p>
     *         <p>
     *         For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     */
    public final List<AttributeType> userAttributes() {
        return userAttributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the ValidationData property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasValidationData() {
        return validationData != null && !(validationData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
     * key-value pairs are for custom validation of information that you collect from your users but don't need to
     * retain.
     * </p>
     * <p>
     * Your Lambda function can analyze this additional data and act on it. Your function might perform external API
     * operations like logging user attributes and validation data to Amazon CloudWatch Logs. Validation data might also
     * affect the response that your function returns to Amazon Cognito, like automatically confirming the user if they
     * sign up from within your network.
     * </p>
     * <p>
     * For more information about the pre sign-up Lambda trigger, see <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre sign-up
     * Lambda trigger</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValidationData} method.
     * </p>
     * 
     * @return Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
     *         key-value pairs are for custom validation of information that you collect from your users but don't need
     *         to retain.</p>
     *         <p>
     *         Your Lambda function can analyze this additional data and act on it. Your function might perform external
     *         API operations like logging user attributes and validation data to Amazon CloudWatch Logs. Validation
     *         data might also affect the response that your function returns to Amazon Cognito, like automatically
     *         confirming the user if they sign up from within your network.
     *         </p>
     *         <p>
     *         For more information about the pre sign-up Lambda trigger, see <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
     *         sign-up Lambda trigger</a>.
     */
    public final List<AttributeType> validationData() {
        return validationData;
    }

    /**
     * <p>
     * The Amazon Pinpoint analytics metadata that contributes to your metrics for <code>SignUp</code> calls.
     * </p>
     * 
     * @return The Amazon Pinpoint analytics metadata that contributes to your metrics for <code>SignUp</code> calls.
     */
    public final AnalyticsMetadataType analyticsMetadata() {
        return analyticsMetadata;
    }

    /**
     * <p>
     * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon Cognito
     * advanced security evaluates the risk of an authentication event based on the context that your app generates and
     * passes to Amazon Cognito when it makes API requests.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
     * >Collecting data for threat protection in applications</a>.
     * </p>
     * 
     * @return Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
     *         Cognito advanced security evaluates the risk of an authentication event based on the context that your
     *         app generates and passes to Amazon Cognito when it makes API requests.</p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
     *         >Collecting data for threat protection in applications</a>.
     */
    public final UserContextDataType userContextData() {
        return userContextData;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the SignUp API
     * action, Amazon Cognito invokes any functions that are assigned to the following triggers: <i>pre sign-up</i>,
     * <i>custom message</i>, and <i>post confirmation</i>. When Amazon Cognito invokes any of these functions, it
     * passes a JSON payload, which the function receives as input. This payload contains a <code>clientMetadata</code>
     * attribute, which provides the data that you assigned to the ClientMetadata parameter in your SignUp request. In
     * your function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your workflow for
     * your specific needs.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are assigned to
     * a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
     * <code>ClientMetadata</code> parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Validate the <code>ClientMetadata</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientMetadata} method.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for any custom workflows that this action
     *         triggers.</p>
     *         <p>
     *         You create custom workflows by assigning Lambda functions to user pool triggers. When you use the SignUp
     *         API action, Amazon Cognito invokes any functions that are assigned to the following triggers: <i>pre
     *         sign-up</i>, <i>custom message</i>, and <i>post confirmation</i>. When Amazon Cognito invokes any of
     *         these functions, it passes a JSON payload, which the function receives as input. This payload contains a
     *         <code>clientMetadata</code> attribute, which provides the data that you assigned to the ClientMetadata
     *         parameter in your SignUp request. In your function code in Lambda, you can process the
     *         <code>clientMetadata</code> value to enhance your workflow for your specific needs.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are
     *         assigned to a user pool to support custom workflows. If your user pool configuration doesn't include
     *         triggers, the <code>ClientMetadata</code> parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Validate the <code>ClientMetadata</code> value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(secretHash());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(password());
        hashCode = 31 * hashCode + Objects.hashCode(hasUserAttributes() ? userAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasValidationData() ? validationData() : null);
        hashCode = 31 * hashCode + Objects.hashCode(analyticsMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(userContextData());
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SignUpRequest)) {
            return false;
        }
        SignUpRequest other = (SignUpRequest) obj;
        return Objects.equals(clientId(), other.clientId()) && Objects.equals(secretHash(), other.secretHash())
                && Objects.equals(username(), other.username()) && Objects.equals(password(), other.password())
                && hasUserAttributes() == other.hasUserAttributes() && Objects.equals(userAttributes(), other.userAttributes())
                && hasValidationData() == other.hasValidationData() && Objects.equals(validationData(), other.validationData())
                && Objects.equals(analyticsMetadata(), other.analyticsMetadata())
                && Objects.equals(userContextData(), other.userContextData()) && hasClientMetadata() == other.hasClientMetadata()
                && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SignUpRequest").add("ClientId", clientId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SecretHash", secretHash() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Username", username() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Password", password() == null ? null : "*** Sensitive Data Redacted ***")
                .add("UserAttributes", hasUserAttributes() ? userAttributes() : null)
                .add("ValidationData", hasValidationData() ? validationData() : null)
                .add("AnalyticsMetadata", analyticsMetadata())
                .add("UserContextData", userContextData() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "SecretHash":
            return Optional.ofNullable(clazz.cast(secretHash()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "Password":
            return Optional.ofNullable(clazz.cast(password()));
        case "UserAttributes":
            return Optional.ofNullable(clazz.cast(userAttributes()));
        case "ValidationData":
            return Optional.ofNullable(clazz.cast(validationData()));
        case "AnalyticsMetadata":
            return Optional.ofNullable(clazz.cast(analyticsMetadata()));
        case "UserContextData":
            return Optional.ofNullable(clazz.cast(userContextData()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ClientId", CLIENT_ID_FIELD);
        map.put("SecretHash", SECRET_HASH_FIELD);
        map.put("Username", USERNAME_FIELD);
        map.put("Password", PASSWORD_FIELD);
        map.put("UserAttributes", USER_ATTRIBUTES_FIELD);
        map.put("ValidationData", VALIDATION_DATA_FIELD);
        map.put("AnalyticsMetadata", ANALYTICS_METADATA_FIELD);
        map.put("UserContextData", USER_CONTEXT_DATA_FIELD);
        map.put("ClientMetadata", CLIENT_METADATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SignUpRequest, T> g) {
        return obj -> g.apply((SignUpRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo, CopyableBuilder<Builder, SignUpRequest> {
        /**
         * <p>
         * The ID of the client associated with the user pool.
         * </p>
         * 
         * @param clientId
         *        The ID of the client associated with the user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client and
         * username plus the client ID in the message. For more information about <code>SecretHash</code>, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#cognito-user-pools-computing-secret-hash"
         * >Computing secret hash values</a>.
         * </p>
         * 
         * @param secretHash
         *        A keyed-hash message authentication code (HMAC) calculated using the secret key of a user pool client
         *        and username plus the client ID in the message. For more information about <code>SecretHash</code>,
         *        see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/signing-up-users-in-your-app.html#cognito-user-pools-computing-secret-hash"
         *        >Computing secret hash values</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretHash(String secretHash);

        /**
         * <p>
         * The username of the user that you want to sign up. The value of this parameter is typically a username, but
         * can be any alias attribute in your user pool.
         * </p>
         * 
         * @param username
         *        The username of the user that you want to sign up. The value of this parameter is typically a
         *        username, but can be any alias attribute in your user pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * The password of the user you want to register.
         * </p>
         * <p>
         * Users can sign up without a password when your user pool supports passwordless sign-in with email or SMS
         * OTPs. To create a user with no password, omit this parameter or submit a blank value. You can only create a
         * passwordless user when passwordless sign-in is available. See <a href=
         * "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SignInPolicyType.html">the
         * SignInPolicyType</a> property of <a
         * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html"
         * >CreateUserPool</a> and <a
         * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html"
         * >UpdateUserPool</a>.
         * </p>
         * 
         * @param password
         *        The password of the user you want to register.</p>
         *        <p>
         *        Users can sign up without a password when your user pool supports passwordless sign-in with email or
         *        SMS OTPs. To create a user with no password, omit this parameter or submit a blank value. You can only
         *        create a passwordless user when passwordless sign-in is available. See <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_SignInPolicyType.html"
         *        >the SignInPolicyType</a> property of <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html"
         *        >CreateUserPool</a> and <a href=
         *        "https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html"
         *        >UpdateUserPool</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder password(String password);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * 
         * @param userAttributes
         *        An array of name-value pairs representing user attributes.</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(Collection<AttributeType> userAttributes);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * 
         * @param userAttributes
         *        An array of name-value pairs representing user attributes.</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(AttributeType... userAttributes);

        /**
         * <p>
         * An array of name-value pairs representing user attributes.
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder#build()} is called
         * immediately and its result is passed to {@link #userAttributes(List<AttributeType>)}.
         * 
         * @param userAttributes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAttributes(java.util.Collection<AttributeType>)
         */
        Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes);

        /**
         * <p>
         * Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
         * key-value pairs are for custom validation of information that you collect from your users but don't need to
         * retain.
         * </p>
         * <p>
         * Your Lambda function can analyze this additional data and act on it. Your function might perform external API
         * operations like logging user attributes and validation data to Amazon CloudWatch Logs. Validation data might
         * also affect the response that your function returns to Amazon Cognito, like automatically confirming the user
         * if they sign up from within your network.
         * </p>
         * <p>
         * For more information about the pre sign-up Lambda trigger, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         * sign-up Lambda trigger</a>.
         * </p>
         * 
         * @param validationData
         *        Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set
         *        of key-value pairs are for custom validation of information that you collect from your users but don't
         *        need to retain.</p>
         *        <p>
         *        Your Lambda function can analyze this additional data and act on it. Your function might perform
         *        external API operations like logging user attributes and validation data to Amazon CloudWatch Logs.
         *        Validation data might also affect the response that your function returns to Amazon Cognito, like
         *        automatically confirming the user if they sign up from within your network.
         *        </p>
         *        <p>
         *        For more information about the pre sign-up Lambda trigger, see <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         *        sign-up Lambda trigger</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(Collection<AttributeType> validationData);

        /**
         * <p>
         * Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
         * key-value pairs are for custom validation of information that you collect from your users but don't need to
         * retain.
         * </p>
         * <p>
         * Your Lambda function can analyze this additional data and act on it. Your function might perform external API
         * operations like logging user attributes and validation data to Amazon CloudWatch Logs. Validation data might
         * also affect the response that your function returns to Amazon Cognito, like automatically confirming the user
         * if they sign up from within your network.
         * </p>
         * <p>
         * For more information about the pre sign-up Lambda trigger, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         * sign-up Lambda trigger</a>.
         * </p>
         * 
         * @param validationData
         *        Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set
         *        of key-value pairs are for custom validation of information that you collect from your users but don't
         *        need to retain.</p>
         *        <p>
         *        Your Lambda function can analyze this additional data and act on it. Your function might perform
         *        external API operations like logging user attributes and validation data to Amazon CloudWatch Logs.
         *        Validation data might also affect the response that your function returns to Amazon Cognito, like
         *        automatically confirming the user if they sign up from within your network.
         *        </p>
         *        <p>
         *        For more information about the pre sign-up Lambda trigger, see <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         *        sign-up Lambda trigger</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(AttributeType... validationData);

        /**
         * <p>
         * Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
         * key-value pairs are for custom validation of information that you collect from your users but don't need to
         * retain.
         * </p>
         * <p>
         * Your Lambda function can analyze this additional data and act on it. Your function might perform external API
         * operations like logging user attributes and validation data to Amazon CloudWatch Logs. Validation data might
         * also affect the response that your function returns to Amazon Cognito, like automatically confirming the user
         * if they sign up from within your network.
         * </p>
         * <p>
         * For more information about the pre sign-up Lambda trigger, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         * sign-up Lambda trigger</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder#build()} is called
         * immediately and its result is passed to {@link #validationData(List<AttributeType>)}.
         * 
         * @param validationData
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationData(java.util.Collection<AttributeType>)
         */
        Builder validationData(Consumer<AttributeType.Builder>... validationData);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata that contributes to your metrics for <code>SignUp</code> calls.
         * </p>
         * 
         * @param analyticsMetadata
         *        The Amazon Pinpoint analytics metadata that contributes to your metrics for <code>SignUp</code> calls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata that contributes to your metrics for <code>SignUp</code> calls.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalyticsMetadataType.Builder} avoiding
         * the need to create one manually via {@link AnalyticsMetadataType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnalyticsMetadataType.Builder#build()} is called immediately and
         * its result is passed to {@link #analyticsMetadata(AnalyticsMetadataType)}.
         * 
         * @param analyticsMetadata
         *        a consumer that will call methods on {@link AnalyticsMetadataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #analyticsMetadata(AnalyticsMetadataType)
         */
        default Builder analyticsMetadata(Consumer<AnalyticsMetadataType.Builder> analyticsMetadata) {
            return analyticsMetadata(AnalyticsMetadataType.builder().applyMutation(analyticsMetadata).build());
        }

        /**
         * <p>
         * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
         * Cognito advanced security evaluates the risk of an authentication event based on the context that your app
         * generates and passes to Amazon Cognito when it makes API requests.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
         * >Collecting data for threat protection in applications</a>.
         * </p>
         * 
         * @param userContextData
         *        Contextual data about your user session, such as the device fingerprint, IP address, or location.
         *        Amazon Cognito advanced security evaluates the risk of an authentication event based on the context
         *        that your app generates and passes to Amazon Cognito when it makes API requests.</p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
         *        >Collecting data for threat protection in applications</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userContextData(UserContextDataType userContextData);

        /**
         * <p>
         * Contextual data about your user session, such as the device fingerprint, IP address, or location. Amazon
         * Cognito advanced security evaluates the risk of an authentication event based on the context that your app
         * generates and passes to Amazon Cognito when it makes API requests.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-viewing-threat-protection-app.html"
         * >Collecting data for threat protection in applications</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserContextDataType.Builder} avoiding the
         * need to create one manually via {@link UserContextDataType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UserContextDataType.Builder#build()} is called immediately and
         * its result is passed to {@link #userContextData(UserContextDataType)}.
         * 
         * @param userContextData
         *        a consumer that will call methods on {@link UserContextDataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userContextData(UserContextDataType)
         */
        default Builder userContextData(Consumer<UserContextDataType.Builder> userContextData) {
            return userContextData(UserContextDataType.builder().applyMutation(userContextData).build());
        }

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for any custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the SignUp API
         * action, Amazon Cognito invokes any functions that are assigned to the following triggers: <i>pre sign-up</i>,
         * <i>custom message</i>, and <i>post confirmation</i>. When Amazon Cognito invokes any of these functions, it
         * passes a JSON payload, which the function receives as input. This payload contains a
         * <code>clientMetadata</code> attribute, which provides the data that you assigned to the ClientMetadata
         * parameter in your SignUp request. In your function code in Lambda, you can process the
         * <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are assigned
         * to a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         * <code>ClientMetadata</code> parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Validate the <code>ClientMetadata</code> value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for any custom workflows that this
         *        action triggers.</p>
         *        <p>
         *        You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         *        SignUp API action, Amazon Cognito invokes any functions that are assigned to the following triggers:
         *        <i>pre sign-up</i>, <i>custom message</i>, and <i>post confirmation</i>. When Amazon Cognito invokes
         *        any of these functions, it passes a JSON payload, which the function receives as input. This payload
         *        contains a <code>clientMetadata</code> attribute, which provides the data that you assigned to the
         *        ClientMetadata parameter in your SignUp request. In your function code in Lambda, you can process the
         *        <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        > Customizing user pool Workflows with Lambda Triggers</a> in the <i>Amazon Cognito Developer
         *        Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the
         *        following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are
         *        assigned to a user pool to support custom workflows. If your user pool configuration doesn't include
         *        triggers, the <code>ClientMetadata</code> parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Validate the <code>ClientMetadata</code> value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String clientId;

        private String secretHash;

        private String username;

        private String password;

        private List<AttributeType> userAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<AttributeType> validationData = DefaultSdkAutoConstructList.getInstance();

        private AnalyticsMetadataType analyticsMetadata;

        private UserContextDataType userContextData;

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SignUpRequest model) {
            super(model);
            clientId(model.clientId);
            secretHash(model.secretHash);
            username(model.username);
            password(model.password);
            userAttributes(model.userAttributes);
            validationData(model.validationData);
            analyticsMetadata(model.analyticsMetadata);
            userContextData(model.userContextData);
            clientMetadata(model.clientMetadata);
        }

        public final String getClientId() {
            return clientId;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final String getSecretHash() {
            return secretHash;
        }

        public final void setSecretHash(String secretHash) {
            this.secretHash = secretHash;
        }

        @Override
        public final Builder secretHash(String secretHash) {
            this.secretHash = secretHash;
            return this;
        }

        public final String getUsername() {
            return username;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final String getPassword() {
            return password;
        }

        public final void setPassword(String password) {
            this.password = password;
        }

        @Override
        public final Builder password(String password) {
            this.password = password;
            return this;
        }

        public final List<AttributeType.Builder> getUserAttributes() {
            List<AttributeType.Builder> result = AttributeListTypeCopier.copyToBuilder(this.userAttributes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUserAttributes(Collection<AttributeType.BuilderImpl> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copyFromBuilder(userAttributes);
        }

        @Override
        public final Builder userAttributes(Collection<AttributeType> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copy(userAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(AttributeType... userAttributes) {
            userAttributes(Arrays.asList(userAttributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes) {
            userAttributes(Stream.of(userAttributes).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<AttributeType.Builder> getValidationData() {
            List<AttributeType.Builder> result = AttributeListTypeCopier.copyToBuilder(this.validationData);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setValidationData(Collection<AttributeType.BuilderImpl> validationData) {
            this.validationData = AttributeListTypeCopier.copyFromBuilder(validationData);
        }

        @Override
        public final Builder validationData(Collection<AttributeType> validationData) {
            this.validationData = AttributeListTypeCopier.copy(validationData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(AttributeType... validationData) {
            validationData(Arrays.asList(validationData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(Consumer<AttributeType.Builder>... validationData) {
            validationData(Stream.of(validationData).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final AnalyticsMetadataType.Builder getAnalyticsMetadata() {
            return analyticsMetadata != null ? analyticsMetadata.toBuilder() : null;
        }

        public final void setAnalyticsMetadata(AnalyticsMetadataType.BuilderImpl analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata != null ? analyticsMetadata.build() : null;
        }

        @Override
        public final Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata;
            return this;
        }

        public final UserContextDataType.Builder getUserContextData() {
            return userContextData != null ? userContextData.toBuilder() : null;
        }

        public final void setUserContextData(UserContextDataType.BuilderImpl userContextData) {
            this.userContextData = userContextData != null ? userContextData.build() : null;
        }

        @Override
        public final Builder userContextData(UserContextDataType userContextData) {
            this.userContextData = userContextData;
            return this;
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public SignUpRequest build() {
            return new SignUpRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
