/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * User pool add-ons. Contains settings for activation of advanced security features. To log user security information
 * but take no action, set to <code>AUDIT</code>. To configure automatic security responses to risky traffic to your
 * user pool, set to <code>ENFORCED</code>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-settings-advanced-security.html"
 * >Adding advanced security to a user pool</a>.
 * </p>
 * <p>
 * This data type is a request and response parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_CreateUserPool.html"
 * >CreateUserPool</a> and <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_UpdateUserPool.html"
 * >UpdateUserPool</a>, and a response parameter of <a
 * href="https://docs.aws.amazon.com/cognito-user-identity-pools/latest/APIReference/API_DescribeUserPool.html"
 * >DescribeUserPool</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UserPoolAddOnsType implements SdkPojo, Serializable,
        ToCopyableBuilder<UserPoolAddOnsType.Builder, UserPoolAddOnsType> {
    private static final SdkField<String> ADVANCED_SECURITY_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AdvancedSecurityMode").getter(getter(UserPoolAddOnsType::advancedSecurityModeAsString))
            .setter(setter(Builder::advancedSecurityMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedSecurityMode").build())
            .build();

    private static final SdkField<AdvancedSecurityAdditionalFlowsType> ADVANCED_SECURITY_ADDITIONAL_FLOWS_FIELD = SdkField
            .<AdvancedSecurityAdditionalFlowsType> builder(MarshallingType.SDK_POJO)
            .memberName("AdvancedSecurityAdditionalFlows")
            .getter(getter(UserPoolAddOnsType::advancedSecurityAdditionalFlows))
            .setter(setter(Builder::advancedSecurityAdditionalFlows))
            .constructor(AdvancedSecurityAdditionalFlowsType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdvancedSecurityAdditionalFlows")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ADVANCED_SECURITY_MODE_FIELD,
            ADVANCED_SECURITY_ADDITIONAL_FLOWS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String advancedSecurityMode;

    private final AdvancedSecurityAdditionalFlowsType advancedSecurityAdditionalFlows;

    private UserPoolAddOnsType(BuilderImpl builder) {
        this.advancedSecurityMode = builder.advancedSecurityMode;
        this.advancedSecurityAdditionalFlows = builder.advancedSecurityAdditionalFlows;
    }

    /**
     * <p>
     * The operating mode of advanced security features for standard authentication types in your user pool, including
     * username-password and secure remote password (SRP) authentication.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #advancedSecurityMode} will return {@link AdvancedSecurityModeType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #advancedSecurityModeAsString}.
     * </p>
     * 
     * @return The operating mode of advanced security features for standard authentication types in your user pool,
     *         including username-password and secure remote password (SRP) authentication.
     * @see AdvancedSecurityModeType
     */
    public final AdvancedSecurityModeType advancedSecurityMode() {
        return AdvancedSecurityModeType.fromValue(advancedSecurityMode);
    }

    /**
     * <p>
     * The operating mode of advanced security features for standard authentication types in your user pool, including
     * username-password and secure remote password (SRP) authentication.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #advancedSecurityMode} will return {@link AdvancedSecurityModeType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #advancedSecurityModeAsString}.
     * </p>
     * 
     * @return The operating mode of advanced security features for standard authentication types in your user pool,
     *         including username-password and secure remote password (SRP) authentication.
     * @see AdvancedSecurityModeType
     */
    public final String advancedSecurityModeAsString() {
        return advancedSecurityMode;
    }

    /**
     * <p>
     * Advanced security configuration options for additional authentication types in your user pool, including custom
     * authentication.
     * </p>
     * 
     * @return Advanced security configuration options for additional authentication types in your user pool, including
     *         custom authentication.
     */
    public final AdvancedSecurityAdditionalFlowsType advancedSecurityAdditionalFlows() {
        return advancedSecurityAdditionalFlows;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(advancedSecurityModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(advancedSecurityAdditionalFlows());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UserPoolAddOnsType)) {
            return false;
        }
        UserPoolAddOnsType other = (UserPoolAddOnsType) obj;
        return Objects.equals(advancedSecurityModeAsString(), other.advancedSecurityModeAsString())
                && Objects.equals(advancedSecurityAdditionalFlows(), other.advancedSecurityAdditionalFlows());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UserPoolAddOnsType").add("AdvancedSecurityMode", advancedSecurityModeAsString())
                .add("AdvancedSecurityAdditionalFlows", advancedSecurityAdditionalFlows()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AdvancedSecurityMode":
            return Optional.ofNullable(clazz.cast(advancedSecurityModeAsString()));
        case "AdvancedSecurityAdditionalFlows":
            return Optional.ofNullable(clazz.cast(advancedSecurityAdditionalFlows()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AdvancedSecurityMode", ADVANCED_SECURITY_MODE_FIELD);
        map.put("AdvancedSecurityAdditionalFlows", ADVANCED_SECURITY_ADDITIONAL_FLOWS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UserPoolAddOnsType, T> g) {
        return obj -> g.apply((UserPoolAddOnsType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UserPoolAddOnsType> {
        /**
         * <p>
         * The operating mode of advanced security features for standard authentication types in your user pool,
         * including username-password and secure remote password (SRP) authentication.
         * </p>
         * 
         * @param advancedSecurityMode
         *        The operating mode of advanced security features for standard authentication types in your user pool,
         *        including username-password and secure remote password (SRP) authentication.
         * @see AdvancedSecurityModeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AdvancedSecurityModeType
         */
        Builder advancedSecurityMode(String advancedSecurityMode);

        /**
         * <p>
         * The operating mode of advanced security features for standard authentication types in your user pool,
         * including username-password and secure remote password (SRP) authentication.
         * </p>
         * 
         * @param advancedSecurityMode
         *        The operating mode of advanced security features for standard authentication types in your user pool,
         *        including username-password and secure remote password (SRP) authentication.
         * @see AdvancedSecurityModeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AdvancedSecurityModeType
         */
        Builder advancedSecurityMode(AdvancedSecurityModeType advancedSecurityMode);

        /**
         * <p>
         * Advanced security configuration options for additional authentication types in your user pool, including
         * custom authentication.
         * </p>
         * 
         * @param advancedSecurityAdditionalFlows
         *        Advanced security configuration options for additional authentication types in your user pool,
         *        including custom authentication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder advancedSecurityAdditionalFlows(AdvancedSecurityAdditionalFlowsType advancedSecurityAdditionalFlows);

        /**
         * <p>
         * Advanced security configuration options for additional authentication types in your user pool, including
         * custom authentication.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link AdvancedSecurityAdditionalFlowsType.Builder} avoiding the need to create one manually via
         * {@link AdvancedSecurityAdditionalFlowsType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AdvancedSecurityAdditionalFlowsType.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #advancedSecurityAdditionalFlows(AdvancedSecurityAdditionalFlowsType)}.
         * 
         * @param advancedSecurityAdditionalFlows
         *        a consumer that will call methods on {@link AdvancedSecurityAdditionalFlowsType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #advancedSecurityAdditionalFlows(AdvancedSecurityAdditionalFlowsType)
         */
        default Builder advancedSecurityAdditionalFlows(
                Consumer<AdvancedSecurityAdditionalFlowsType.Builder> advancedSecurityAdditionalFlows) {
            return advancedSecurityAdditionalFlows(AdvancedSecurityAdditionalFlowsType.builder()
                    .applyMutation(advancedSecurityAdditionalFlows).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String advancedSecurityMode;

        private AdvancedSecurityAdditionalFlowsType advancedSecurityAdditionalFlows;

        private BuilderImpl() {
        }

        private BuilderImpl(UserPoolAddOnsType model) {
            advancedSecurityMode(model.advancedSecurityMode);
            advancedSecurityAdditionalFlows(model.advancedSecurityAdditionalFlows);
        }

        public final String getAdvancedSecurityMode() {
            return advancedSecurityMode;
        }

        public final void setAdvancedSecurityMode(String advancedSecurityMode) {
            this.advancedSecurityMode = advancedSecurityMode;
        }

        @Override
        public final Builder advancedSecurityMode(String advancedSecurityMode) {
            this.advancedSecurityMode = advancedSecurityMode;
            return this;
        }

        @Override
        public final Builder advancedSecurityMode(AdvancedSecurityModeType advancedSecurityMode) {
            this.advancedSecurityMode(advancedSecurityMode == null ? null : advancedSecurityMode.toString());
            return this;
        }

        public final AdvancedSecurityAdditionalFlowsType.Builder getAdvancedSecurityAdditionalFlows() {
            return advancedSecurityAdditionalFlows != null ? advancedSecurityAdditionalFlows.toBuilder() : null;
        }

        public final void setAdvancedSecurityAdditionalFlows(
                AdvancedSecurityAdditionalFlowsType.BuilderImpl advancedSecurityAdditionalFlows) {
            this.advancedSecurityAdditionalFlows = advancedSecurityAdditionalFlows != null ? advancedSecurityAdditionalFlows
                    .build() : null;
        }

        @Override
        public final Builder advancedSecurityAdditionalFlows(AdvancedSecurityAdditionalFlowsType advancedSecurityAdditionalFlows) {
            this.advancedSecurityAdditionalFlows = advancedSecurityAdditionalFlows;
            return this;
        }

        @Override
        public UserPoolAddOnsType build() {
            return new UserPoolAddOnsType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
