/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Creates a new user in the specified user pool.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdminCreateUserRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<AdminCreateUserRequest.Builder, AdminCreateUserRequest> {
    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(AdminCreateUserRequest::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Username").getter(getter(AdminCreateUserRequest::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<List<AttributeType>> USER_ATTRIBUTES_FIELD = SdkField
            .<List<AttributeType>> builder(MarshallingType.LIST)
            .memberName("UserAttributes")
            .getter(getter(AdminCreateUserRequest::userAttributes))
            .setter(setter(Builder::userAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AttributeType>> VALIDATION_DATA_FIELD = SdkField
            .<List<AttributeType>> builder(MarshallingType.LIST)
            .memberName("ValidationData")
            .getter(getter(AdminCreateUserRequest::validationData))
            .setter(setter(Builder::validationData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidationData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> TEMPORARY_PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TemporaryPassword").getter(getter(AdminCreateUserRequest::temporaryPassword))
            .setter(setter(Builder::temporaryPassword))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TemporaryPassword").build()).build();

    private static final SdkField<Boolean> FORCE_ALIAS_CREATION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ForceAliasCreation").getter(getter(AdminCreateUserRequest::forceAliasCreation))
            .setter(setter(Builder::forceAliasCreation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ForceAliasCreation").build())
            .build();

    private static final SdkField<String> MESSAGE_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageAction").getter(getter(AdminCreateUserRequest::messageActionAsString))
            .setter(setter(Builder::messageAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageAction").build()).build();

    private static final SdkField<List<String>> DESIRED_DELIVERY_MEDIUMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DesiredDeliveryMediums")
            .getter(getter(AdminCreateUserRequest::desiredDeliveryMediumsAsStrings))
            .setter(setter(Builder::desiredDeliveryMediumsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DesiredDeliveryMediums").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> CLIENT_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ClientMetadata")
            .getter(getter(AdminCreateUserRequest::clientMetadata))
            .setter(setter(Builder::clientMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientMetadata").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(USER_POOL_ID_FIELD,
            USERNAME_FIELD, USER_ATTRIBUTES_FIELD, VALIDATION_DATA_FIELD, TEMPORARY_PASSWORD_FIELD, FORCE_ALIAS_CREATION_FIELD,
            MESSAGE_ACTION_FIELD, DESIRED_DELIVERY_MEDIUMS_FIELD, CLIENT_METADATA_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String userPoolId;

    private final String username;

    private final List<AttributeType> userAttributes;

    private final List<AttributeType> validationData;

    private final String temporaryPassword;

    private final Boolean forceAliasCreation;

    private final String messageAction;

    private final List<String> desiredDeliveryMediums;

    private final Map<String, String> clientMetadata;

    private AdminCreateUserRequest(BuilderImpl builder) {
        super(builder);
        this.userPoolId = builder.userPoolId;
        this.username = builder.username;
        this.userAttributes = builder.userAttributes;
        this.validationData = builder.validationData;
        this.temporaryPassword = builder.temporaryPassword;
        this.forceAliasCreation = builder.forceAliasCreation;
        this.messageAction = builder.messageAction;
        this.desiredDeliveryMediums = builder.desiredDeliveryMediums;
        this.clientMetadata = builder.clientMetadata;
    }

    /**
     * <p>
     * The ID of the user pool where you want to create a user.
     * </p>
     * 
     * @return The ID of the user pool where you want to create a user.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * The value that you want to set as the username sign-in attribute. The following conditions apply to the username
     * parameter.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The username can't be a duplicate of another username in the same user pool.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can't change the value of a username after you create it.
     * </p>
     * </li>
     * <li>
     * <p>
     * You can only provide a value if usernames are a valid sign-in attribute for your user pool. If your user pool
     * only supports phone numbers or email addresses as sign-in attributes, Amazon Cognito automatically generates a
     * username value. For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases"
     * >Customizing sign-in attributes</a>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The value that you want to set as the username sign-in attribute. The following conditions apply to the
     *         username parameter.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The username can't be a duplicate of another username in the same user pool.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can't change the value of a username after you create it.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can only provide a value if usernames are a valid sign-in attribute for your user pool. If your user
     *         pool only supports phone numbers or email addresses as sign-in attributes, Amazon Cognito automatically
     *         generates a username value. For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases"
     *         >Customizing sign-in attributes</a>.
     *         </p>
     *         </li>
     */
    public final String username() {
        return username;
    }

    /**
     * For responses, this returns true if the service returned a value for the UserAttributes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUserAttributes() {
        return userAttributes != null && !(userAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of name-value pairs that contain user attributes and attribute values to be set for the user to be
     * created. You can create a user without specifying any attributes other than <code>Username</code>. However, any
     * attributes that you specify as required (when creating a user pool or in the <b>Attributes</b> tab of the
     * console) either you should supply (in your call to <code>AdminCreateUser</code>) or the user should supply (when
     * they sign up in response to your welcome message).
     * </p>
     * <p>
     * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     * </p>
     * <p>
     * To send a message inviting the user to sign up, you must specify the user's email address or phone number. You
     * can do this in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito console for managing
     * your user pools.
     * </p>
     * <p>
     * You must also provide an email address or phone number when you expect the user to do passwordless sign-in with
     * an email or SMS OTP. These attributes must be provided when passwordless options are the only available, or when
     * you don't submit a <code>TemporaryPassword</code>.
     * </p>
     * <p>
     * In your <code>AdminCreateUser</code> request, you can set the <code>email_verified</code> and
     * <code>phone_number_verified</code> attributes to <code>true</code>. The following conditions apply:
     * </p>
     * <dl>
     * <dt>email</dt>
     * <dd>
     * <p>
     * The email address where you want the user to receive their confirmation code and username. You must provide a
     * value for the <code>email</code> when you want to set <code>email_verified</code> to <code>true</code>, or if you
     * set <code>EMAIL</code> in the <code>DesiredDeliveryMediums</code> parameter.
     * </p>
     * </dd>
     * <dt>phone_number</dt>
     * <dd>
     * <p>
     * The phone number where you want the user to receive their confirmation code and username. You must provide a
     * value for the <code>email</code> when you want to set <code>phone_number</code> to <code>true</code>, or if you
     * set <code>SMS</code> in the <code>DesiredDeliveryMediums</code> parameter.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUserAttributes} method.
     * </p>
     * 
     * @return An array of name-value pairs that contain user attributes and attribute values to be set for the user to
     *         be created. You can create a user without specifying any attributes other than <code>Username</code>.
     *         However, any attributes that you specify as required (when creating a user pool or in the
     *         <b>Attributes</b> tab of the console) either you should supply (in your call to
     *         <code>AdminCreateUser</code>) or the user should supply (when they sign up in response to your welcome
     *         message).</p>
     *         <p>
     *         For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
     *         </p>
     *         <p>
     *         To send a message inviting the user to sign up, you must specify the user's email address or phone
     *         number. You can do this in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito
     *         console for managing your user pools.
     *         </p>
     *         <p>
     *         You must also provide an email address or phone number when you expect the user to do passwordless
     *         sign-in with an email or SMS OTP. These attributes must be provided when passwordless options are the
     *         only available, or when you don't submit a <code>TemporaryPassword</code>.
     *         </p>
     *         <p>
     *         In your <code>AdminCreateUser</code> request, you can set the <code>email_verified</code> and
     *         <code>phone_number_verified</code> attributes to <code>true</code>. The following conditions apply:
     *         </p>
     *         <dl>
     *         <dt>email</dt>
     *         <dd>
     *         <p>
     *         The email address where you want the user to receive their confirmation code and username. You must
     *         provide a value for the <code>email</code> when you want to set <code>email_verified</code> to
     *         <code>true</code>, or if you set <code>EMAIL</code> in the <code>DesiredDeliveryMediums</code> parameter.
     *         </p>
     *         </dd>
     *         <dt>phone_number</dt>
     *         <dd>
     *         <p>
     *         The phone number where you want the user to receive their confirmation code and username. You must
     *         provide a value for the <code>email</code> when you want to set <code>phone_number</code> to
     *         <code>true</code>, or if you set <code>SMS</code> in the <code>DesiredDeliveryMediums</code> parameter.
     *         </p>
     *         </dd>
     */
    public final List<AttributeType> userAttributes() {
        return userAttributes;
    }

    /**
     * For responses, this returns true if the service returned a value for the ValidationData property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasValidationData() {
        return validationData != null && !(validationData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
     * key-value pairs are for custom validation of information that you collect from your users but don't need to
     * retain.
     * </p>
     * <p>
     * Your Lambda function can analyze this additional data and act on it. Your function can automatically confirm and
     * verify select users or perform external API operations like logging user attributes and validation data to Amazon
     * CloudWatch Logs.
     * </p>
     * <p>
     * For more information about the pre sign-up Lambda trigger, see <a
     * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre sign-up
     * Lambda trigger</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValidationData} method.
     * </p>
     * 
     * @return Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
     *         key-value pairs are for custom validation of information that you collect from your users but don't need
     *         to retain.</p>
     *         <p>
     *         Your Lambda function can analyze this additional data and act on it. Your function can automatically
     *         confirm and verify select users or perform external API operations like logging user attributes and
     *         validation data to Amazon CloudWatch Logs.
     *         </p>
     *         <p>
     *         For more information about the pre sign-up Lambda trigger, see <a
     *         href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
     *         sign-up Lambda trigger</a>.
     */
    public final List<AttributeType> validationData() {
        return validationData;
    }

    /**
     * <p>
     * The user's temporary password. This password must conform to the password policy that you specified when you
     * created the user pool.
     * </p>
     * <p>
     * The exception to the requirement for a password is when your user pool supports passwordless sign-in with email
     * or SMS OTPs. To create a user with no password, omit this parameter or submit a blank value. You can only create
     * a passwordless user when passwordless sign-in is available.
     * </p>
     * <p>
     * The temporary password is valid only once. To complete the Admin Create User flow, the user must enter the
     * temporary password in the sign-in page, along with a new password to be used in all future sign-ins.
     * </p>
     * <p>
     * If you don't specify a value, Amazon Cognito generates one for you unless you have passwordless options active
     * for your user pool.
     * </p>
     * <p>
     * The temporary password can only be used until the user account expiration limit that you set for your user pool.
     * To reset the account after that time limit, you must call <code>AdminCreateUser</code> again and specify
     * <code>RESEND</code> for the <code>MessageAction</code> parameter.
     * </p>
     * 
     * @return The user's temporary password. This password must conform to the password policy that you specified when
     *         you created the user pool.</p>
     *         <p>
     *         The exception to the requirement for a password is when your user pool supports passwordless sign-in with
     *         email or SMS OTPs. To create a user with no password, omit this parameter or submit a blank value. You
     *         can only create a passwordless user when passwordless sign-in is available.
     *         </p>
     *         <p>
     *         The temporary password is valid only once. To complete the Admin Create User flow, the user must enter
     *         the temporary password in the sign-in page, along with a new password to be used in all future sign-ins.
     *         </p>
     *         <p>
     *         If you don't specify a value, Amazon Cognito generates one for you unless you have passwordless options
     *         active for your user pool.
     *         </p>
     *         <p>
     *         The temporary password can only be used until the user account expiration limit that you set for your
     *         user pool. To reset the account after that time limit, you must call <code>AdminCreateUser</code> again
     *         and specify <code>RESEND</code> for the <code>MessageAction</code> parameter.
     */
    public final String temporaryPassword() {
        return temporaryPassword;
    }

    /**
     * <p>
     * This parameter is used only if the <code>phone_number_verified</code> or <code>email_verified</code> attribute is
     * set to <code>True</code>. Otherwise, it is ignored.
     * </p>
     * <p>
     * If this parameter is set to <code>True</code> and the phone number or email address specified in the
     * <code>UserAttributes</code> parameter already exists as an alias with a different user, this request migrates the
     * alias from the previous user to the newly-created user. The previous user will no longer be able to log in using
     * that alias.
     * </p>
     * <p>
     * If this parameter is set to <code>False</code>, the API throws an <code>AliasExistsException</code> error if the
     * alias already exists. The default value is <code>False</code>.
     * </p>
     * 
     * @return This parameter is used only if the <code>phone_number_verified</code> or <code>email_verified</code>
     *         attribute is set to <code>True</code>. Otherwise, it is ignored.</p>
     *         <p>
     *         If this parameter is set to <code>True</code> and the phone number or email address specified in the
     *         <code>UserAttributes</code> parameter already exists as an alias with a different user, this request
     *         migrates the alias from the previous user to the newly-created user. The previous user will no longer be
     *         able to log in using that alias.
     *         </p>
     *         <p>
     *         If this parameter is set to <code>False</code>, the API throws an <code>AliasExistsException</code> error
     *         if the alias already exists. The default value is <code>False</code>.
     */
    public final Boolean forceAliasCreation() {
        return forceAliasCreation;
    }

    /**
     * <p>
     * Set to <code>RESEND</code> to resend the invitation message to a user that already exists, and to reset the
     * temporary-password duration with a new temporary password. Set to <code>SUPPRESS</code> to suppress sending the
     * message. You can specify only one value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageAction}
     * will return {@link MessageActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageActionAsString}.
     * </p>
     * 
     * @return Set to <code>RESEND</code> to resend the invitation message to a user that already exists, and to reset
     *         the temporary-password duration with a new temporary password. Set to <code>SUPPRESS</code> to suppress
     *         sending the message. You can specify only one value.
     * @see MessageActionType
     */
    public final MessageActionType messageAction() {
        return MessageActionType.fromValue(messageAction);
    }

    /**
     * <p>
     * Set to <code>RESEND</code> to resend the invitation message to a user that already exists, and to reset the
     * temporary-password duration with a new temporary password. Set to <code>SUPPRESS</code> to suppress sending the
     * message. You can specify only one value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #messageAction}
     * will return {@link MessageActionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #messageActionAsString}.
     * </p>
     * 
     * @return Set to <code>RESEND</code> to resend the invitation message to a user that already exists, and to reset
     *         the temporary-password duration with a new temporary password. Set to <code>SUPPRESS</code> to suppress
     *         sending the message. You can specify only one value.
     * @see MessageActionType
     */
    public final String messageActionAsString() {
        return messageAction;
    }

    /**
     * <p>
     * Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code> if the
     * phone number will be used. The default value is <code>SMS</code>. You can specify more than one value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDesiredDeliveryMediums} method.
     * </p>
     * 
     * @return Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code> if
     *         the phone number will be used. The default value is <code>SMS</code>. You can specify more than one
     *         value.
     */
    public final List<DeliveryMediumType> desiredDeliveryMediums() {
        return DeliveryMediumListTypeCopier.copyStringToEnum(desiredDeliveryMediums);
    }

    /**
     * For responses, this returns true if the service returned a value for the DesiredDeliveryMediums property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDesiredDeliveryMediums() {
        return desiredDeliveryMediums != null && !(desiredDeliveryMediums instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code> if the
     * phone number will be used. The default value is <code>SMS</code>. You can specify more than one value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDesiredDeliveryMediums} method.
     * </p>
     * 
     * @return Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code> if
     *         the phone number will be used. The default value is <code>SMS</code>. You can specify more than one
     *         value.
     */
    public final List<String> desiredDeliveryMediumsAsStrings() {
        return desiredDeliveryMediums;
    }

    /**
     * For responses, this returns true if the service returned a value for the ClientMetadata property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasClientMetadata() {
        return clientMetadata != null && !(clientMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of custom key-value pairs that you can provide as input for any custom workflows that this action triggers.
     * </p>
     * <p>
     * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the AdminCreateUser
     * API action, Amazon Cognito invokes the function that is assigned to the <i>pre sign-up</i> trigger. When Amazon
     * Cognito invokes this function, it passes a JSON payload, which the function receives as input. This payload
     * contains a <code>ClientMetadata</code> attribute, which provides the data that you assigned to the ClientMetadata
     * parameter in your AdminCreateUser request. In your function code in Lambda, you can process the
     * <code>clientMetadata</code> value to enhance your workflow for your specific needs.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     * > Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     * </p>
     * <note>
     * <p>
     * When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are assigned to
     * a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
     * <code>ClientMetadata</code> parameter serves no purpose.
     * </p>
     * </li>
     * <li>
     * <p>
     * Validate the <code>ClientMetadata</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
     * </p>
     * </li>
     * </ul>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasClientMetadata} method.
     * </p>
     * 
     * @return A map of custom key-value pairs that you can provide as input for any custom workflows that this action
     *         triggers.</p>
     *         <p>
     *         You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
     *         AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the <i>pre
     *         sign-up</i> trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the
     *         function receives as input. This payload contains a <code>ClientMetadata</code> attribute, which provides
     *         the data that you assigned to the ClientMetadata parameter in your AdminCreateUser request. In your
     *         function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your workflow
     *         for your specific needs.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
     *         > Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are
     *         assigned to a user pool to support custom workflows. If your user pool configuration doesn't include
     *         triggers, the <code>ClientMetadata</code> parameter serves no purpose.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Validate the <code>ClientMetadata</code> value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
     *         </p>
     *         </li>
     *         </ul>
     */
    public final Map<String, String> clientMetadata() {
        return clientMetadata;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(hasUserAttributes() ? userAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasValidationData() ? validationData() : null);
        hashCode = 31 * hashCode + Objects.hashCode(temporaryPassword());
        hashCode = 31 * hashCode + Objects.hashCode(forceAliasCreation());
        hashCode = 31 * hashCode + Objects.hashCode(messageActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasDesiredDeliveryMediums() ? desiredDeliveryMediumsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasClientMetadata() ? clientMetadata() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdminCreateUserRequest)) {
            return false;
        }
        AdminCreateUserRequest other = (AdminCreateUserRequest) obj;
        return Objects.equals(userPoolId(), other.userPoolId()) && Objects.equals(username(), other.username())
                && hasUserAttributes() == other.hasUserAttributes() && Objects.equals(userAttributes(), other.userAttributes())
                && hasValidationData() == other.hasValidationData() && Objects.equals(validationData(), other.validationData())
                && Objects.equals(temporaryPassword(), other.temporaryPassword())
                && Objects.equals(forceAliasCreation(), other.forceAliasCreation())
                && Objects.equals(messageActionAsString(), other.messageActionAsString())
                && hasDesiredDeliveryMediums() == other.hasDesiredDeliveryMediums()
                && Objects.equals(desiredDeliveryMediumsAsStrings(), other.desiredDeliveryMediumsAsStrings())
                && hasClientMetadata() == other.hasClientMetadata() && Objects.equals(clientMetadata(), other.clientMetadata());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdminCreateUserRequest").add("UserPoolId", userPoolId())
                .add("Username", username() == null ? null : "*** Sensitive Data Redacted ***")
                .add("UserAttributes", hasUserAttributes() ? userAttributes() : null)
                .add("ValidationData", hasValidationData() ? validationData() : null)
                .add("TemporaryPassword", temporaryPassword() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ForceAliasCreation", forceAliasCreation()).add("MessageAction", messageActionAsString())
                .add("DesiredDeliveryMediums", hasDesiredDeliveryMediums() ? desiredDeliveryMediumsAsStrings() : null)
                .add("ClientMetadata", hasClientMetadata() ? clientMetadata() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "UserAttributes":
            return Optional.ofNullable(clazz.cast(userAttributes()));
        case "ValidationData":
            return Optional.ofNullable(clazz.cast(validationData()));
        case "TemporaryPassword":
            return Optional.ofNullable(clazz.cast(temporaryPassword()));
        case "ForceAliasCreation":
            return Optional.ofNullable(clazz.cast(forceAliasCreation()));
        case "MessageAction":
            return Optional.ofNullable(clazz.cast(messageActionAsString()));
        case "DesiredDeliveryMediums":
            return Optional.ofNullable(clazz.cast(desiredDeliveryMediumsAsStrings()));
        case "ClientMetadata":
            return Optional.ofNullable(clazz.cast(clientMetadata()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("UserPoolId", USER_POOL_ID_FIELD);
        map.put("Username", USERNAME_FIELD);
        map.put("UserAttributes", USER_ATTRIBUTES_FIELD);
        map.put("ValidationData", VALIDATION_DATA_FIELD);
        map.put("TemporaryPassword", TEMPORARY_PASSWORD_FIELD);
        map.put("ForceAliasCreation", FORCE_ALIAS_CREATION_FIELD);
        map.put("MessageAction", MESSAGE_ACTION_FIELD);
        map.put("DesiredDeliveryMediums", DESIRED_DELIVERY_MEDIUMS_FIELD);
        map.put("ClientMetadata", CLIENT_METADATA_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AdminCreateUserRequest, T> g) {
        return obj -> g.apply((AdminCreateUserRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, AdminCreateUserRequest> {
        /**
         * <p>
         * The ID of the user pool where you want to create a user.
         * </p>
         * 
         * @param userPoolId
         *        The ID of the user pool where you want to create a user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * The value that you want to set as the username sign-in attribute. The following conditions apply to the
         * username parameter.
         * </p>
         * <ul>
         * <li>
         * <p>
         * The username can't be a duplicate of another username in the same user pool.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can't change the value of a username after you create it.
         * </p>
         * </li>
         * <li>
         * <p>
         * You can only provide a value if usernames are a valid sign-in attribute for your user pool. If your user pool
         * only supports phone numbers or email addresses as sign-in attributes, Amazon Cognito automatically generates
         * a username value. For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases"
         * >Customizing sign-in attributes</a>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param username
         *        The value that you want to set as the username sign-in attribute. The following conditions apply to
         *        the username parameter.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The username can't be a duplicate of another username in the same user pool.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can't change the value of a username after you create it.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can only provide a value if usernames are a valid sign-in attribute for your user pool. If your
         *        user pool only supports phone numbers or email addresses as sign-in attributes, Amazon Cognito
         *        automatically generates a username value. For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-aliases"
         *        >Customizing sign-in attributes</a>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * An array of name-value pairs that contain user attributes and attribute values to be set for the user to be
         * created. You can create a user without specifying any attributes other than <code>Username</code>. However,
         * any attributes that you specify as required (when creating a user pool or in the <b>Attributes</b> tab of the
         * console) either you should supply (in your call to <code>AdminCreateUser</code>) or the user should supply
         * (when they sign up in response to your welcome message).
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * <p>
         * To send a message inviting the user to sign up, you must specify the user's email address or phone number.
         * You can do this in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito console for
         * managing your user pools.
         * </p>
         * <p>
         * You must also provide an email address or phone number when you expect the user to do passwordless sign-in
         * with an email or SMS OTP. These attributes must be provided when passwordless options are the only available,
         * or when you don't submit a <code>TemporaryPassword</code>.
         * </p>
         * <p>
         * In your <code>AdminCreateUser</code> request, you can set the <code>email_verified</code> and
         * <code>phone_number_verified</code> attributes to <code>true</code>. The following conditions apply:
         * </p>
         * <dl>
         * <dt>email</dt>
         * <dd>
         * <p>
         * The email address where you want the user to receive their confirmation code and username. You must provide a
         * value for the <code>email</code> when you want to set <code>email_verified</code> to <code>true</code>, or if
         * you set <code>EMAIL</code> in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </dd>
         * <dt>phone_number</dt>
         * <dd>
         * <p>
         * The phone number where you want the user to receive their confirmation code and username. You must provide a
         * value for the <code>email</code> when you want to set <code>phone_number</code> to <code>true</code>, or if
         * you set <code>SMS</code> in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param userAttributes
         *        An array of name-value pairs that contain user attributes and attribute values to be set for the user
         *        to be created. You can create a user without specifying any attributes other than
         *        <code>Username</code>. However, any attributes that you specify as required (when creating a user pool
         *        or in the <b>Attributes</b> tab of the console) either you should supply (in your call to
         *        <code>AdminCreateUser</code>) or the user should supply (when they sign up in response to your welcome
         *        message).</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         *        </p>
         *        <p>
         *        To send a message inviting the user to sign up, you must specify the user's email address or phone
         *        number. You can do this in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon
         *        Cognito console for managing your user pools.
         *        </p>
         *        <p>
         *        You must also provide an email address or phone number when you expect the user to do passwordless
         *        sign-in with an email or SMS OTP. These attributes must be provided when passwordless options are the
         *        only available, or when you don't submit a <code>TemporaryPassword</code>.
         *        </p>
         *        <p>
         *        In your <code>AdminCreateUser</code> request, you can set the <code>email_verified</code> and
         *        <code>phone_number_verified</code> attributes to <code>true</code>. The following conditions apply:
         *        </p>
         *        <dl>
         *        <dt>email</dt>
         *        <dd>
         *        <p>
         *        The email address where you want the user to receive their confirmation code and username. You must
         *        provide a value for the <code>email</code> when you want to set <code>email_verified</code> to
         *        <code>true</code>, or if you set <code>EMAIL</code> in the <code>DesiredDeliveryMediums</code>
         *        parameter.
         *        </p>
         *        </dd>
         *        <dt>phone_number</dt>
         *        <dd>
         *        <p>
         *        The phone number where you want the user to receive their confirmation code and username. You must
         *        provide a value for the <code>email</code> when you want to set <code>phone_number</code> to
         *        <code>true</code>, or if you set <code>SMS</code> in the <code>DesiredDeliveryMediums</code>
         *        parameter.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(Collection<AttributeType> userAttributes);

        /**
         * <p>
         * An array of name-value pairs that contain user attributes and attribute values to be set for the user to be
         * created. You can create a user without specifying any attributes other than <code>Username</code>. However,
         * any attributes that you specify as required (when creating a user pool or in the <b>Attributes</b> tab of the
         * console) either you should supply (in your call to <code>AdminCreateUser</code>) or the user should supply
         * (when they sign up in response to your welcome message).
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * <p>
         * To send a message inviting the user to sign up, you must specify the user's email address or phone number.
         * You can do this in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito console for
         * managing your user pools.
         * </p>
         * <p>
         * You must also provide an email address or phone number when you expect the user to do passwordless sign-in
         * with an email or SMS OTP. These attributes must be provided when passwordless options are the only available,
         * or when you don't submit a <code>TemporaryPassword</code>.
         * </p>
         * <p>
         * In your <code>AdminCreateUser</code> request, you can set the <code>email_verified</code> and
         * <code>phone_number_verified</code> attributes to <code>true</code>. The following conditions apply:
         * </p>
         * <dl>
         * <dt>email</dt>
         * <dd>
         * <p>
         * The email address where you want the user to receive their confirmation code and username. You must provide a
         * value for the <code>email</code> when you want to set <code>email_verified</code> to <code>true</code>, or if
         * you set <code>EMAIL</code> in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </dd>
         * <dt>phone_number</dt>
         * <dd>
         * <p>
         * The phone number where you want the user to receive their confirmation code and username. You must provide a
         * value for the <code>email</code> when you want to set <code>phone_number</code> to <code>true</code>, or if
         * you set <code>SMS</code> in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param userAttributes
         *        An array of name-value pairs that contain user attributes and attribute values to be set for the user
         *        to be created. You can create a user without specifying any attributes other than
         *        <code>Username</code>. However, any attributes that you specify as required (when creating a user pool
         *        or in the <b>Attributes</b> tab of the console) either you should supply (in your call to
         *        <code>AdminCreateUser</code>) or the user should supply (when they sign up in response to your welcome
         *        message).</p>
         *        <p>
         *        For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         *        </p>
         *        <p>
         *        To send a message inviting the user to sign up, you must specify the user's email address or phone
         *        number. You can do this in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon
         *        Cognito console for managing your user pools.
         *        </p>
         *        <p>
         *        You must also provide an email address or phone number when you expect the user to do passwordless
         *        sign-in with an email or SMS OTP. These attributes must be provided when passwordless options are the
         *        only available, or when you don't submit a <code>TemporaryPassword</code>.
         *        </p>
         *        <p>
         *        In your <code>AdminCreateUser</code> request, you can set the <code>email_verified</code> and
         *        <code>phone_number_verified</code> attributes to <code>true</code>. The following conditions apply:
         *        </p>
         *        <dl>
         *        <dt>email</dt>
         *        <dd>
         *        <p>
         *        The email address where you want the user to receive their confirmation code and username. You must
         *        provide a value for the <code>email</code> when you want to set <code>email_verified</code> to
         *        <code>true</code>, or if you set <code>EMAIL</code> in the <code>DesiredDeliveryMediums</code>
         *        parameter.
         *        </p>
         *        </dd>
         *        <dt>phone_number</dt>
         *        <dd>
         *        <p>
         *        The phone number where you want the user to receive their confirmation code and username. You must
         *        provide a value for the <code>email</code> when you want to set <code>phone_number</code> to
         *        <code>true</code>, or if you set <code>SMS</code> in the <code>DesiredDeliveryMediums</code>
         *        parameter.
         *        </p>
         *        </dd>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAttributes(AttributeType... userAttributes);

        /**
         * <p>
         * An array of name-value pairs that contain user attributes and attribute values to be set for the user to be
         * created. You can create a user without specifying any attributes other than <code>Username</code>. However,
         * any attributes that you specify as required (when creating a user pool or in the <b>Attributes</b> tab of the
         * console) either you should supply (in your call to <code>AdminCreateUser</code>) or the user should supply
         * (when they sign up in response to your welcome message).
         * </p>
         * <p>
         * For custom attributes, you must prepend the <code>custom:</code> prefix to the attribute name.
         * </p>
         * <p>
         * To send a message inviting the user to sign up, you must specify the user's email address or phone number.
         * You can do this in your call to AdminCreateUser or in the <b>Users</b> tab of the Amazon Cognito console for
         * managing your user pools.
         * </p>
         * <p>
         * You must also provide an email address or phone number when you expect the user to do passwordless sign-in
         * with an email or SMS OTP. These attributes must be provided when passwordless options are the only available,
         * or when you don't submit a <code>TemporaryPassword</code>.
         * </p>
         * <p>
         * In your <code>AdminCreateUser</code> request, you can set the <code>email_verified</code> and
         * <code>phone_number_verified</code> attributes to <code>true</code>. The following conditions apply:
         * </p>
         * <dl>
         * <dt>email</dt>
         * <dd>
         * <p>
         * The email address where you want the user to receive their confirmation code and username. You must provide a
         * value for the <code>email</code> when you want to set <code>email_verified</code> to <code>true</code>, or if
         * you set <code>EMAIL</code> in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </dd>
         * <dt>phone_number</dt>
         * <dd>
         * <p>
         * The phone number where you want the user to receive their confirmation code and username. You must provide a
         * value for the <code>email</code> when you want to set <code>phone_number</code> to <code>true</code>, or if
         * you set <code>SMS</code> in the <code>DesiredDeliveryMediums</code> parameter.
         * </p>
         * </dd>
         * </dl>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder#build()} is called
         * immediately and its result is passed to {@link #userAttributes(List<AttributeType>)}.
         * 
         * @param userAttributes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAttributes(java.util.Collection<AttributeType>)
         */
        Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes);

        /**
         * <p>
         * Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
         * key-value pairs are for custom validation of information that you collect from your users but don't need to
         * retain.
         * </p>
         * <p>
         * Your Lambda function can analyze this additional data and act on it. Your function can automatically confirm
         * and verify select users or perform external API operations like logging user attributes and validation data
         * to Amazon CloudWatch Logs.
         * </p>
         * <p>
         * For more information about the pre sign-up Lambda trigger, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         * sign-up Lambda trigger</a>.
         * </p>
         * 
         * @param validationData
         *        Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set
         *        of key-value pairs are for custom validation of information that you collect from your users but don't
         *        need to retain.</p>
         *        <p>
         *        Your Lambda function can analyze this additional data and act on it. Your function can automatically
         *        confirm and verify select users or perform external API operations like logging user attributes and
         *        validation data to Amazon CloudWatch Logs.
         *        </p>
         *        <p>
         *        For more information about the pre sign-up Lambda trigger, see <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         *        sign-up Lambda trigger</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(Collection<AttributeType> validationData);

        /**
         * <p>
         * Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
         * key-value pairs are for custom validation of information that you collect from your users but don't need to
         * retain.
         * </p>
         * <p>
         * Your Lambda function can analyze this additional data and act on it. Your function can automatically confirm
         * and verify select users or perform external API operations like logging user attributes and validation data
         * to Amazon CloudWatch Logs.
         * </p>
         * <p>
         * For more information about the pre sign-up Lambda trigger, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         * sign-up Lambda trigger</a>.
         * </p>
         * 
         * @param validationData
         *        Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set
         *        of key-value pairs are for custom validation of information that you collect from your users but don't
         *        need to retain.</p>
         *        <p>
         *        Your Lambda function can analyze this additional data and act on it. Your function can automatically
         *        confirm and verify select users or perform external API operations like logging user attributes and
         *        validation data to Amazon CloudWatch Logs.
         *        </p>
         *        <p>
         *        For more information about the pre sign-up Lambda trigger, see <a
         *        href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         *        sign-up Lambda trigger</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validationData(AttributeType... validationData);

        /**
         * <p>
         * Temporary user attributes that contribute to the outcomes of your pre sign-up Lambda trigger. This set of
         * key-value pairs are for custom validation of information that you collect from your users but don't need to
         * retain.
         * </p>
         * <p>
         * Your Lambda function can analyze this additional data and act on it. Your function can automatically confirm
         * and verify select users or perform external API operations like logging user attributes and validation data
         * to Amazon CloudWatch Logs.
         * </p>
         * <p>
         * For more information about the pre sign-up Lambda trigger, see <a
         * href="https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html">Pre
         * sign-up Lambda trigger</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder#build()} is called
         * immediately and its result is passed to {@link #validationData(List<AttributeType>)}.
         * 
         * @param validationData
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AttributeType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validationData(java.util.Collection<AttributeType>)
         */
        Builder validationData(Consumer<AttributeType.Builder>... validationData);

        /**
         * <p>
         * The user's temporary password. This password must conform to the password policy that you specified when you
         * created the user pool.
         * </p>
         * <p>
         * The exception to the requirement for a password is when your user pool supports passwordless sign-in with
         * email or SMS OTPs. To create a user with no password, omit this parameter or submit a blank value. You can
         * only create a passwordless user when passwordless sign-in is available.
         * </p>
         * <p>
         * The temporary password is valid only once. To complete the Admin Create User flow, the user must enter the
         * temporary password in the sign-in page, along with a new password to be used in all future sign-ins.
         * </p>
         * <p>
         * If you don't specify a value, Amazon Cognito generates one for you unless you have passwordless options
         * active for your user pool.
         * </p>
         * <p>
         * The temporary password can only be used until the user account expiration limit that you set for your user
         * pool. To reset the account after that time limit, you must call <code>AdminCreateUser</code> again and
         * specify <code>RESEND</code> for the <code>MessageAction</code> parameter.
         * </p>
         * 
         * @param temporaryPassword
         *        The user's temporary password. This password must conform to the password policy that you specified
         *        when you created the user pool.</p>
         *        <p>
         *        The exception to the requirement for a password is when your user pool supports passwordless sign-in
         *        with email or SMS OTPs. To create a user with no password, omit this parameter or submit a blank
         *        value. You can only create a passwordless user when passwordless sign-in is available.
         *        </p>
         *        <p>
         *        The temporary password is valid only once. To complete the Admin Create User flow, the user must enter
         *        the temporary password in the sign-in page, along with a new password to be used in all future
         *        sign-ins.
         *        </p>
         *        <p>
         *        If you don't specify a value, Amazon Cognito generates one for you unless you have passwordless
         *        options active for your user pool.
         *        </p>
         *        <p>
         *        The temporary password can only be used until the user account expiration limit that you set for your
         *        user pool. To reset the account after that time limit, you must call <code>AdminCreateUser</code>
         *        again and specify <code>RESEND</code> for the <code>MessageAction</code> parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder temporaryPassword(String temporaryPassword);

        /**
         * <p>
         * This parameter is used only if the <code>phone_number_verified</code> or <code>email_verified</code>
         * attribute is set to <code>True</code>. Otherwise, it is ignored.
         * </p>
         * <p>
         * If this parameter is set to <code>True</code> and the phone number or email address specified in the
         * <code>UserAttributes</code> parameter already exists as an alias with a different user, this request migrates
         * the alias from the previous user to the newly-created user. The previous user will no longer be able to log
         * in using that alias.
         * </p>
         * <p>
         * If this parameter is set to <code>False</code>, the API throws an <code>AliasExistsException</code> error if
         * the alias already exists. The default value is <code>False</code>.
         * </p>
         * 
         * @param forceAliasCreation
         *        This parameter is used only if the <code>phone_number_verified</code> or <code>email_verified</code>
         *        attribute is set to <code>True</code>. Otherwise, it is ignored.</p>
         *        <p>
         *        If this parameter is set to <code>True</code> and the phone number or email address specified in the
         *        <code>UserAttributes</code> parameter already exists as an alias with a different user, this request
         *        migrates the alias from the previous user to the newly-created user. The previous user will no longer
         *        be able to log in using that alias.
         *        </p>
         *        <p>
         *        If this parameter is set to <code>False</code>, the API throws an <code>AliasExistsException</code>
         *        error if the alias already exists. The default value is <code>False</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder forceAliasCreation(Boolean forceAliasCreation);

        /**
         * <p>
         * Set to <code>RESEND</code> to resend the invitation message to a user that already exists, and to reset the
         * temporary-password duration with a new temporary password. Set to <code>SUPPRESS</code> to suppress sending
         * the message. You can specify only one value.
         * </p>
         * 
         * @param messageAction
         *        Set to <code>RESEND</code> to resend the invitation message to a user that already exists, and to
         *        reset the temporary-password duration with a new temporary password. Set to <code>SUPPRESS</code> to
         *        suppress sending the message. You can specify only one value.
         * @see MessageActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageActionType
         */
        Builder messageAction(String messageAction);

        /**
         * <p>
         * Set to <code>RESEND</code> to resend the invitation message to a user that already exists, and to reset the
         * temporary-password duration with a new temporary password. Set to <code>SUPPRESS</code> to suppress sending
         * the message. You can specify only one value.
         * </p>
         * 
         * @param messageAction
         *        Set to <code>RESEND</code> to resend the invitation message to a user that already exists, and to
         *        reset the temporary-password duration with a new temporary password. Set to <code>SUPPRESS</code> to
         *        suppress sending the message. You can specify only one value.
         * @see MessageActionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MessageActionType
         */
        Builder messageAction(MessageActionType messageAction);

        /**
         * <p>
         * Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code> if the
         * phone number will be used. The default value is <code>SMS</code>. You can specify more than one value.
         * </p>
         * 
         * @param desiredDeliveryMediums
         *        Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code>
         *        if the phone number will be used. The default value is <code>SMS</code>. You can specify more than one
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredDeliveryMediumsWithStrings(Collection<String> desiredDeliveryMediums);

        /**
         * <p>
         * Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code> if the
         * phone number will be used. The default value is <code>SMS</code>. You can specify more than one value.
         * </p>
         * 
         * @param desiredDeliveryMediums
         *        Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code>
         *        if the phone number will be used. The default value is <code>SMS</code>. You can specify more than one
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredDeliveryMediumsWithStrings(String... desiredDeliveryMediums);

        /**
         * <p>
         * Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code> if the
         * phone number will be used. The default value is <code>SMS</code>. You can specify more than one value.
         * </p>
         * 
         * @param desiredDeliveryMediums
         *        Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code>
         *        if the phone number will be used. The default value is <code>SMS</code>. You can specify more than one
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredDeliveryMediums(Collection<DeliveryMediumType> desiredDeliveryMediums);

        /**
         * <p>
         * Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code> if the
         * phone number will be used. The default value is <code>SMS</code>. You can specify more than one value.
         * </p>
         * 
         * @param desiredDeliveryMediums
         *        Specify <code>EMAIL</code> if email will be used to send the welcome message. Specify <code>SMS</code>
         *        if the phone number will be used. The default value is <code>SMS</code>. You can specify more than one
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder desiredDeliveryMediums(DeliveryMediumType... desiredDeliveryMediums);

        /**
         * <p>
         * A map of custom key-value pairs that you can provide as input for any custom workflows that this action
         * triggers.
         * </p>
         * <p>
         * You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         * AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the <i>pre sign-up</i>
         * trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the function receives as
         * input. This payload contains a <code>ClientMetadata</code> attribute, which provides the data that you
         * assigned to the ClientMetadata parameter in your AdminCreateUser request. In your function code in Lambda,
         * you can process the <code>clientMetadata</code> value to enhance your workflow for your specific needs.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         * > Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         * </p>
         * <note>
         * <p>
         * When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are assigned
         * to a user pool to support custom workflows. If your user pool configuration doesn't include triggers, the
         * <code>ClientMetadata</code> parameter serves no purpose.
         * </p>
         * </li>
         * <li>
         * <p>
         * Validate the <code>ClientMetadata</code> value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
         * </p>
         * </li>
         * </ul>
         * </note>
         * 
         * @param clientMetadata
         *        A map of custom key-value pairs that you can provide as input for any custom workflows that this
         *        action triggers.</p>
         *        <p>
         *        You create custom workflows by assigning Lambda functions to user pool triggers. When you use the
         *        AdminCreateUser API action, Amazon Cognito invokes the function that is assigned to the <i>pre
         *        sign-up</i> trigger. When Amazon Cognito invokes this function, it passes a JSON payload, which the
         *        function receives as input. This payload contains a <code>ClientMetadata</code> attribute, which
         *        provides the data that you assigned to the ClientMetadata parameter in your AdminCreateUser request.
         *        In your function code in Lambda, you can process the <code>clientMetadata</code> value to enhance your
         *        workflow for your specific needs.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html"
         *        > Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        When you use the <code>ClientMetadata</code> parameter, note that Amazon Cognito won't do the
         *        following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Store the <code>ClientMetadata</code> value. This data is available only to Lambda triggers that are
         *        assigned to a user pool to support custom workflows. If your user pool configuration doesn't include
         *        triggers, the <code>ClientMetadata</code> parameter serves no purpose.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Validate the <code>ClientMetadata</code> value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Encrypt the <code>ClientMetadata</code> value. Don't send sensitive information in this parameter.
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientMetadata(Map<String, String> clientMetadata);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String userPoolId;

        private String username;

        private List<AttributeType> userAttributes = DefaultSdkAutoConstructList.getInstance();

        private List<AttributeType> validationData = DefaultSdkAutoConstructList.getInstance();

        private String temporaryPassword;

        private Boolean forceAliasCreation;

        private String messageAction;

        private List<String> desiredDeliveryMediums = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> clientMetadata = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AdminCreateUserRequest model) {
            super(model);
            userPoolId(model.userPoolId);
            username(model.username);
            userAttributes(model.userAttributes);
            validationData(model.validationData);
            temporaryPassword(model.temporaryPassword);
            forceAliasCreation(model.forceAliasCreation);
            messageAction(model.messageAction);
            desiredDeliveryMediumsWithStrings(model.desiredDeliveryMediums);
            clientMetadata(model.clientMetadata);
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final String getUsername() {
            return username;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final List<AttributeType.Builder> getUserAttributes() {
            List<AttributeType.Builder> result = AttributeListTypeCopier.copyToBuilder(this.userAttributes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUserAttributes(Collection<AttributeType.BuilderImpl> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copyFromBuilder(userAttributes);
        }

        @Override
        public final Builder userAttributes(Collection<AttributeType> userAttributes) {
            this.userAttributes = AttributeListTypeCopier.copy(userAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(AttributeType... userAttributes) {
            userAttributes(Arrays.asList(userAttributes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAttributes(Consumer<AttributeType.Builder>... userAttributes) {
            userAttributes(Stream.of(userAttributes).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<AttributeType.Builder> getValidationData() {
            List<AttributeType.Builder> result = AttributeListTypeCopier.copyToBuilder(this.validationData);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setValidationData(Collection<AttributeType.BuilderImpl> validationData) {
            this.validationData = AttributeListTypeCopier.copyFromBuilder(validationData);
        }

        @Override
        public final Builder validationData(Collection<AttributeType> validationData) {
            this.validationData = AttributeListTypeCopier.copy(validationData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(AttributeType... validationData) {
            validationData(Arrays.asList(validationData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validationData(Consumer<AttributeType.Builder>... validationData) {
            validationData(Stream.of(validationData).map(c -> AttributeType.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getTemporaryPassword() {
            return temporaryPassword;
        }

        public final void setTemporaryPassword(String temporaryPassword) {
            this.temporaryPassword = temporaryPassword;
        }

        @Override
        public final Builder temporaryPassword(String temporaryPassword) {
            this.temporaryPassword = temporaryPassword;
            return this;
        }

        public final Boolean getForceAliasCreation() {
            return forceAliasCreation;
        }

        public final void setForceAliasCreation(Boolean forceAliasCreation) {
            this.forceAliasCreation = forceAliasCreation;
        }

        @Override
        public final Builder forceAliasCreation(Boolean forceAliasCreation) {
            this.forceAliasCreation = forceAliasCreation;
            return this;
        }

        public final String getMessageAction() {
            return messageAction;
        }

        public final void setMessageAction(String messageAction) {
            this.messageAction = messageAction;
        }

        @Override
        public final Builder messageAction(String messageAction) {
            this.messageAction = messageAction;
            return this;
        }

        @Override
        public final Builder messageAction(MessageActionType messageAction) {
            this.messageAction(messageAction == null ? null : messageAction.toString());
            return this;
        }

        public final Collection<String> getDesiredDeliveryMediums() {
            if (desiredDeliveryMediums instanceof SdkAutoConstructList) {
                return null;
            }
            return desiredDeliveryMediums;
        }

        public final void setDesiredDeliveryMediums(Collection<String> desiredDeliveryMediums) {
            this.desiredDeliveryMediums = DeliveryMediumListTypeCopier.copy(desiredDeliveryMediums);
        }

        @Override
        public final Builder desiredDeliveryMediumsWithStrings(Collection<String> desiredDeliveryMediums) {
            this.desiredDeliveryMediums = DeliveryMediumListTypeCopier.copy(desiredDeliveryMediums);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder desiredDeliveryMediumsWithStrings(String... desiredDeliveryMediums) {
            desiredDeliveryMediumsWithStrings(Arrays.asList(desiredDeliveryMediums));
            return this;
        }

        @Override
        public final Builder desiredDeliveryMediums(Collection<DeliveryMediumType> desiredDeliveryMediums) {
            this.desiredDeliveryMediums = DeliveryMediumListTypeCopier.copyEnumToString(desiredDeliveryMediums);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder desiredDeliveryMediums(DeliveryMediumType... desiredDeliveryMediums) {
            desiredDeliveryMediums(Arrays.asList(desiredDeliveryMediums));
            return this;
        }

        public final Map<String, String> getClientMetadata() {
            if (clientMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return clientMetadata;
        }

        public final void setClientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
        }

        @Override
        public final Builder clientMetadata(Map<String, String> clientMetadata) {
            this.clientMetadata = ClientMetadataTypeCopier.copy(clientMetadata);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public AdminCreateUserRequest build() {
            return new AdminCreateUserRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
