/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.document.Document;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A managed login branding style that's assigned to a user pool app client.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ManagedLoginBrandingType implements SdkPojo, Serializable,
        ToCopyableBuilder<ManagedLoginBrandingType.Builder, ManagedLoginBrandingType> {
    private static final SdkField<String> MANAGED_LOGIN_BRANDING_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ManagedLoginBrandingId").getter(getter(ManagedLoginBrandingType::managedLoginBrandingId))
            .setter(setter(Builder::managedLoginBrandingId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedLoginBrandingId").build())
            .build();

    private static final SdkField<String> USER_POOL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UserPoolId").getter(getter(ManagedLoginBrandingType::userPoolId)).setter(setter(Builder::userPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserPoolId").build()).build();

    private static final SdkField<Boolean> USE_COGNITO_PROVIDED_VALUES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("UseCognitoProvidedValues")
            .getter(getter(ManagedLoginBrandingType::useCognitoProvidedValues)).setter(setter(Builder::useCognitoProvidedValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UseCognitoProvidedValues").build())
            .build();

    private static final SdkField<Document> SETTINGS_FIELD = SdkField.<Document> builder(MarshallingType.DOCUMENT)
            .memberName("Settings").getter(getter(ManagedLoginBrandingType::settings)).setter(setter(Builder::settings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Settings").build()).build();

    private static final SdkField<List<AssetType>> ASSETS_FIELD = SdkField
            .<List<AssetType>> builder(MarshallingType.LIST)
            .memberName("Assets")
            .getter(getter(ManagedLoginBrandingType::assets))
            .setter(setter(Builder::assets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Assets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AssetType> builder(MarshallingType.SDK_POJO)
                                            .constructor(AssetType::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate").getter(getter(ManagedLoginBrandingType::creationDate))
            .setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedDate").getter(getter(ManagedLoginBrandingType::lastModifiedDate))
            .setter(setter(Builder::lastModifiedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedDate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            MANAGED_LOGIN_BRANDING_ID_FIELD, USER_POOL_ID_FIELD, USE_COGNITO_PROVIDED_VALUES_FIELD, SETTINGS_FIELD, ASSETS_FIELD,
            CREATION_DATE_FIELD, LAST_MODIFIED_DATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String managedLoginBrandingId;

    private final String userPoolId;

    private final Boolean useCognitoProvidedValues;

    private final Document settings;

    private final List<AssetType> assets;

    private final Instant creationDate;

    private final Instant lastModifiedDate;

    private ManagedLoginBrandingType(BuilderImpl builder) {
        this.managedLoginBrandingId = builder.managedLoginBrandingId;
        this.userPoolId = builder.userPoolId;
        this.useCognitoProvidedValues = builder.useCognitoProvidedValues;
        this.settings = builder.settings;
        this.assets = builder.assets;
        this.creationDate = builder.creationDate;
        this.lastModifiedDate = builder.lastModifiedDate;
    }

    /**
     * <p>
     * The ID of the managed login branding style.
     * </p>
     * 
     * @return The ID of the managed login branding style.
     */
    public final String managedLoginBrandingId() {
        return managedLoginBrandingId;
    }

    /**
     * <p>
     * The user pool where the branding style is assigned.
     * </p>
     * 
     * @return The user pool where the branding style is assigned.
     */
    public final String userPoolId() {
        return userPoolId;
    }

    /**
     * <p>
     * When true, applies the default branding style options. This option reverts to default style options that are
     * managed by Amazon Cognito. You can modify them later in the branding designer.
     * </p>
     * <p>
     * When you specify <code>true</code> for this option, you must also omit values for <code>Settings</code> and
     * <code>Assets</code> in the request.
     * </p>
     * 
     * @return When true, applies the default branding style options. This option reverts to default style options that
     *         are managed by Amazon Cognito. You can modify them later in the branding designer.</p>
     *         <p>
     *         When you specify <code>true</code> for this option, you must also omit values for <code>Settings</code>
     *         and <code>Assets</code> in the request.
     */
    public final Boolean useCognitoProvidedValues() {
        return useCognitoProvidedValues;
    }

    /**
     * <p>
     * A JSON file, encoded as a <code>Document</code> type, with the the settings that you want to apply to your style.
     * </p>
     * 
     * @return A JSON file, encoded as a <code>Document</code> type, with the the settings that you want to apply to
     *         your style.
     */
    public final Document settings() {
        return settings;
    }

    /**
     * For responses, this returns true if the service returned a value for the Assets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAssets() {
        return assets != null && !(assets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each object must also
     * indicate whether it is for dark mode, light mode, or browser-adaptive mode.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAssets} method.
     * </p>
     * 
     * @return An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each object
     *         must also indicate whether it is for dark mode, light mode, or browser-adaptive mode.
     */
    public final List<AssetType> assets() {
        return assets;
    }

    /**
     * <p>
     * The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time format.
     * Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code> object.
     * </p>
     * 
     * @return The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time
     *         format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
     *         <code>Date</code> object.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The date and time when the item was modified. Amazon Cognito returns this timestamp in UNIX epoch time format.
     * Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code> object.
     * </p>
     * 
     * @return The date and time when the item was modified. Amazon Cognito returns this timestamp in UNIX epoch time
     *         format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
     *         <code>Date</code> object.
     */
    public final Instant lastModifiedDate() {
        return lastModifiedDate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(managedLoginBrandingId());
        hashCode = 31 * hashCode + Objects.hashCode(userPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(useCognitoProvidedValues());
        hashCode = 31 * hashCode + Objects.hashCode(settings());
        hashCode = 31 * hashCode + Objects.hashCode(hasAssets() ? assets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedDate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ManagedLoginBrandingType)) {
            return false;
        }
        ManagedLoginBrandingType other = (ManagedLoginBrandingType) obj;
        return Objects.equals(managedLoginBrandingId(), other.managedLoginBrandingId())
                && Objects.equals(userPoolId(), other.userPoolId())
                && Objects.equals(useCognitoProvidedValues(), other.useCognitoProvidedValues())
                && Objects.equals(settings(), other.settings()) && hasAssets() == other.hasAssets()
                && Objects.equals(assets(), other.assets()) && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(lastModifiedDate(), other.lastModifiedDate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ManagedLoginBrandingType").add("ManagedLoginBrandingId", managedLoginBrandingId())
                .add("UserPoolId", userPoolId()).add("UseCognitoProvidedValues", useCognitoProvidedValues())
                .add("Settings", settings()).add("Assets", hasAssets() ? assets() : null).add("CreationDate", creationDate())
                .add("LastModifiedDate", lastModifiedDate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ManagedLoginBrandingId":
            return Optional.ofNullable(clazz.cast(managedLoginBrandingId()));
        case "UserPoolId":
            return Optional.ofNullable(clazz.cast(userPoolId()));
        case "UseCognitoProvidedValues":
            return Optional.ofNullable(clazz.cast(useCognitoProvidedValues()));
        case "Settings":
            return Optional.ofNullable(clazz.cast(settings()));
        case "Assets":
            return Optional.ofNullable(clazz.cast(assets()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "LastModifiedDate":
            return Optional.ofNullable(clazz.cast(lastModifiedDate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ManagedLoginBrandingId", MANAGED_LOGIN_BRANDING_ID_FIELD);
        map.put("UserPoolId", USER_POOL_ID_FIELD);
        map.put("UseCognitoProvidedValues", USE_COGNITO_PROVIDED_VALUES_FIELD);
        map.put("Settings", SETTINGS_FIELD);
        map.put("Assets", ASSETS_FIELD);
        map.put("CreationDate", CREATION_DATE_FIELD);
        map.put("LastModifiedDate", LAST_MODIFIED_DATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ManagedLoginBrandingType, T> g) {
        return obj -> g.apply((ManagedLoginBrandingType) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ManagedLoginBrandingType> {
        /**
         * <p>
         * The ID of the managed login branding style.
         * </p>
         * 
         * @param managedLoginBrandingId
         *        The ID of the managed login branding style.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedLoginBrandingId(String managedLoginBrandingId);

        /**
         * <p>
         * The user pool where the branding style is assigned.
         * </p>
         * 
         * @param userPoolId
         *        The user pool where the branding style is assigned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userPoolId(String userPoolId);

        /**
         * <p>
         * When true, applies the default branding style options. This option reverts to default style options that are
         * managed by Amazon Cognito. You can modify them later in the branding designer.
         * </p>
         * <p>
         * When you specify <code>true</code> for this option, you must also omit values for <code>Settings</code> and
         * <code>Assets</code> in the request.
         * </p>
         * 
         * @param useCognitoProvidedValues
         *        When true, applies the default branding style options. This option reverts to default style options
         *        that are managed by Amazon Cognito. You can modify them later in the branding designer.</p>
         *        <p>
         *        When you specify <code>true</code> for this option, you must also omit values for
         *        <code>Settings</code> and <code>Assets</code> in the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder useCognitoProvidedValues(Boolean useCognitoProvidedValues);

        /**
         * <p>
         * A JSON file, encoded as a <code>Document</code> type, with the the settings that you want to apply to your
         * style.
         * </p>
         * 
         * @param settings
         *        A JSON file, encoded as a <code>Document</code> type, with the the settings that you want to apply to
         *        your style.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder settings(Document settings);

        /**
         * <p>
         * An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each object must
         * also indicate whether it is for dark mode, light mode, or browser-adaptive mode.
         * </p>
         * 
         * @param assets
         *        An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each
         *        object must also indicate whether it is for dark mode, light mode, or browser-adaptive mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assets(Collection<AssetType> assets);

        /**
         * <p>
         * An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each object must
         * also indicate whether it is for dark mode, light mode, or browser-adaptive mode.
         * </p>
         * 
         * @param assets
         *        An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each
         *        object must also indicate whether it is for dark mode, light mode, or browser-adaptive mode.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder assets(AssetType... assets);

        /**
         * <p>
         * An array of image files that you want to apply to roles like backgrounds, logos, and icons. Each object must
         * also indicate whether it is for dark mode, light mode, or browser-adaptive mode.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AssetType.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AssetType#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AssetType.Builder#build()} is called
         * immediately and its result is passed to {@link #assets(List<AssetType>)}.
         * 
         * @param assets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cognitoidentityprovider.model.AssetType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #assets(java.util.Collection<AssetType>)
         */
        Builder assets(Consumer<AssetType.Builder>... assets);

        /**
         * <p>
         * The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time format.
         * Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code> object.
         * </p>
         * 
         * @param creationDate
         *        The date and time when the item was created. Amazon Cognito returns this timestamp in UNIX epoch time
         *        format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
         *        <code>Date</code> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The date and time when the item was modified. Amazon Cognito returns this timestamp in UNIX epoch time
         * format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java <code>Date</code>
         * object.
         * </p>
         * 
         * @param lastModifiedDate
         *        The date and time when the item was modified. Amazon Cognito returns this timestamp in UNIX epoch time
         *        format. Your SDK might render the output in a human-readable format like ISO 8601 or a Java
         *        <code>Date</code> object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedDate(Instant lastModifiedDate);
    }

    static final class BuilderImpl implements Builder {
        private String managedLoginBrandingId;

        private String userPoolId;

        private Boolean useCognitoProvidedValues;

        private Document settings;

        private List<AssetType> assets = DefaultSdkAutoConstructList.getInstance();

        private Instant creationDate;

        private Instant lastModifiedDate;

        private BuilderImpl() {
        }

        private BuilderImpl(ManagedLoginBrandingType model) {
            managedLoginBrandingId(model.managedLoginBrandingId);
            userPoolId(model.userPoolId);
            useCognitoProvidedValues(model.useCognitoProvidedValues);
            settings(model.settings);
            assets(model.assets);
            creationDate(model.creationDate);
            lastModifiedDate(model.lastModifiedDate);
        }

        public final String getManagedLoginBrandingId() {
            return managedLoginBrandingId;
        }

        public final void setManagedLoginBrandingId(String managedLoginBrandingId) {
            this.managedLoginBrandingId = managedLoginBrandingId;
        }

        @Override
        public final Builder managedLoginBrandingId(String managedLoginBrandingId) {
            this.managedLoginBrandingId = managedLoginBrandingId;
            return this;
        }

        public final String getUserPoolId() {
            return userPoolId;
        }

        public final void setUserPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
        }

        @Override
        public final Builder userPoolId(String userPoolId) {
            this.userPoolId = userPoolId;
            return this;
        }

        public final Boolean getUseCognitoProvidedValues() {
            return useCognitoProvidedValues;
        }

        public final void setUseCognitoProvidedValues(Boolean useCognitoProvidedValues) {
            this.useCognitoProvidedValues = useCognitoProvidedValues;
        }

        @Override
        public final Builder useCognitoProvidedValues(Boolean useCognitoProvidedValues) {
            this.useCognitoProvidedValues = useCognitoProvidedValues;
            return this;
        }

        public final Document getSettings() {
            return settings;
        }

        public final void setSettings(Document settings) {
            this.settings = settings;
        }

        @Override
        @Transient
        public final Builder settings(Document settings) {
            this.settings = settings;
            return this;
        }

        public final List<AssetType.Builder> getAssets() {
            List<AssetType.Builder> result = AssetListTypeCopier.copyToBuilder(this.assets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAssets(Collection<AssetType.BuilderImpl> assets) {
            this.assets = AssetListTypeCopier.copyFromBuilder(assets);
        }

        @Override
        public final Builder assets(Collection<AssetType> assets) {
            this.assets = AssetListTypeCopier.copy(assets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder assets(AssetType... assets) {
            assets(Arrays.asList(assets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder assets(Consumer<AssetType.Builder>... assets) {
            assets(Stream.of(assets).map(c -> AssetType.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final Instant getLastModifiedDate() {
            return lastModifiedDate;
        }

        public final void setLastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
        }

        @Override
        public final Builder lastModifiedDate(Instant lastModifiedDate) {
            this.lastModifiedDate = lastModifiedDate;
            return this;
        }

        @Override
        public ManagedLoginBrandingType build() {
            return new ManagedLoginBrandingType(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
