/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cognitoidentityprovider.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The request to respond to an authentication challenge.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RespondToAuthChallengeRequest extends CognitoIdentityProviderRequest implements
        ToCopyableBuilder<RespondToAuthChallengeRequest.Builder, RespondToAuthChallengeRequest> {
    private static final SdkField<String> CLIENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RespondToAuthChallengeRequest::clientId)).setter(setter(Builder::clientId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientId").build()).build();

    private static final SdkField<String> CHALLENGE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RespondToAuthChallengeRequest::challengeNameAsString)).setter(setter(Builder::challengeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChallengeName").build()).build();

    private static final SdkField<String> SESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(RespondToAuthChallengeRequest::session)).setter(setter(Builder::session))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Session").build()).build();

    private static final SdkField<Map<String, String>> CHALLENGE_RESPONSES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(RespondToAuthChallengeRequest::challengeResponses))
            .setter(setter(Builder::challengeResponses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChallengeResponses").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<AnalyticsMetadataType> ANALYTICS_METADATA_FIELD = SdkField
            .<AnalyticsMetadataType> builder(MarshallingType.SDK_POJO)
            .getter(getter(RespondToAuthChallengeRequest::analyticsMetadata)).setter(setter(Builder::analyticsMetadata))
            .constructor(AnalyticsMetadataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AnalyticsMetadata").build()).build();

    private static final SdkField<UserContextDataType> USER_CONTEXT_DATA_FIELD = SdkField
            .<UserContextDataType> builder(MarshallingType.SDK_POJO)
            .getter(getter(RespondToAuthChallengeRequest::userContextData)).setter(setter(Builder::userContextData))
            .constructor(UserContextDataType::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserContextData").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_ID_FIELD,
            CHALLENGE_NAME_FIELD, SESSION_FIELD, CHALLENGE_RESPONSES_FIELD, ANALYTICS_METADATA_FIELD, USER_CONTEXT_DATA_FIELD));

    private final String clientId;

    private final String challengeName;

    private final String session;

    private final Map<String, String> challengeResponses;

    private final AnalyticsMetadataType analyticsMetadata;

    private final UserContextDataType userContextData;

    private RespondToAuthChallengeRequest(BuilderImpl builder) {
        super(builder);
        this.clientId = builder.clientId;
        this.challengeName = builder.challengeName;
        this.session = builder.session;
        this.challengeResponses = builder.challengeResponses;
        this.analyticsMetadata = builder.analyticsMetadata;
        this.userContextData = builder.userContextData;
    }

    /**
     * <p>
     * The app client ID.
     * </p>
     * 
     * @return The app client ID.
     */
    public String clientId() {
        return clientId;
    }

    /**
     * <p>
     * The challenge name. For more information, see .
     * </p>
     * <p>
     * <code>ADMIN_NO_SRP_AUTH</code> is not a valid value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #challengeName}
     * will return {@link ChallengeNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #challengeNameAsString}.
     * </p>
     * 
     * @return The challenge name. For more information, see .</p>
     *         <p>
     *         <code>ADMIN_NO_SRP_AUTH</code> is not a valid value.
     * @see ChallengeNameType
     */
    public ChallengeNameType challengeName() {
        return ChallengeNameType.fromValue(challengeName);
    }

    /**
     * <p>
     * The challenge name. For more information, see .
     * </p>
     * <p>
     * <code>ADMIN_NO_SRP_AUTH</code> is not a valid value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #challengeName}
     * will return {@link ChallengeNameType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #challengeNameAsString}.
     * </p>
     * 
     * @return The challenge name. For more information, see .</p>
     *         <p>
     *         <code>ADMIN_NO_SRP_AUTH</code> is not a valid value.
     * @see ChallengeNameType
     */
    public String challengeNameAsString() {
        return challengeName;
    }

    /**
     * <p>
     * The session which should be passed both ways in challenge-response calls to the service. If
     * <code>InitiateAuth</code> or <code>RespondToAuthChallenge</code> API call determines that the caller needs to go
     * through another challenge, they return a session with other challenge parameters. This session should be passed
     * as it is to the next <code>RespondToAuthChallenge</code> API call.
     * </p>
     * 
     * @return The session which should be passed both ways in challenge-response calls to the service. If
     *         <code>InitiateAuth</code> or <code>RespondToAuthChallenge</code> API call determines that the caller
     *         needs to go through another challenge, they return a session with other challenge parameters. This
     *         session should be passed as it is to the next <code>RespondToAuthChallenge</code> API call.
     */
    public String session() {
        return session;
    }

    /**
     * <p>
     * The challenge responses. These are inputs corresponding to the value of <code>ChallengeName</code>, for example:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SMS_MFA</code>: <code>SMS_MFA_CODE</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client
     * is configured with client secret).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PASSWORD_VERIFIER</code>: <code>PASSWORD_CLAIM_SIGNATURE</code>, <code>PASSWORD_CLAIM_SECRET_BLOCK</code>,
     * <code>TIMESTAMP</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client
     * secret).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_PASSWORD_REQUIRED</code>: <code>NEW_PASSWORD</code>, any other required attributes,
     * <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client secret).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The challenge responses. These are inputs corresponding to the value of <code>ChallengeName</code>, for
     *         example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SMS_MFA</code>: <code>SMS_MFA_CODE</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app
     *         client is configured with client secret).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PASSWORD_VERIFIER</code>: <code>PASSWORD_CLAIM_SIGNATURE</code>,
     *         <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, <code>TIMESTAMP</code>, <code>USERNAME</code>,
     *         <code>SECRET_HASH</code> (if app client is configured with client secret).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_PASSWORD_REQUIRED</code>: <code>NEW_PASSWORD</code>, any other required attributes,
     *         <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client secret).
     *         </p>
     *         </li>
     */
    public Map<String, String> challengeResponses() {
        return challengeResponses;
    }

    /**
     * <p>
     * The Amazon Pinpoint analytics metadata for collecting metrics for <code>RespondToAuthChallenge</code> calls.
     * </p>
     * 
     * @return The Amazon Pinpoint analytics metadata for collecting metrics for <code>RespondToAuthChallenge</code>
     *         calls.
     */
    public AnalyticsMetadataType analyticsMetadata() {
        return analyticsMetadata;
    }

    /**
     * <p>
     * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk of an
     * unexpected event by Amazon Cognito advanced security.
     * </p>
     * 
     * @return Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the
     *         risk of an unexpected event by Amazon Cognito advanced security.
     */
    public UserContextDataType userContextData() {
        return userContextData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clientId());
        hashCode = 31 * hashCode + Objects.hashCode(challengeNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(session());
        hashCode = 31 * hashCode + Objects.hashCode(challengeResponses());
        hashCode = 31 * hashCode + Objects.hashCode(analyticsMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(userContextData());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RespondToAuthChallengeRequest)) {
            return false;
        }
        RespondToAuthChallengeRequest other = (RespondToAuthChallengeRequest) obj;
        return Objects.equals(clientId(), other.clientId())
                && Objects.equals(challengeNameAsString(), other.challengeNameAsString())
                && Objects.equals(session(), other.session()) && Objects.equals(challengeResponses(), other.challengeResponses())
                && Objects.equals(analyticsMetadata(), other.analyticsMetadata())
                && Objects.equals(userContextData(), other.userContextData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RespondToAuthChallengeRequest")
                .add("ClientId", clientId() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ChallengeName", challengeNameAsString()).add("Session", session())
                .add("ChallengeResponses", challengeResponses()).add("AnalyticsMetadata", analyticsMetadata())
                .add("UserContextData", userContextData()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientId":
            return Optional.ofNullable(clazz.cast(clientId()));
        case "ChallengeName":
            return Optional.ofNullable(clazz.cast(challengeNameAsString()));
        case "Session":
            return Optional.ofNullable(clazz.cast(session()));
        case "ChallengeResponses":
            return Optional.ofNullable(clazz.cast(challengeResponses()));
        case "AnalyticsMetadata":
            return Optional.ofNullable(clazz.cast(analyticsMetadata()));
        case "UserContextData":
            return Optional.ofNullable(clazz.cast(userContextData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RespondToAuthChallengeRequest, T> g) {
        return obj -> g.apply((RespondToAuthChallengeRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CognitoIdentityProviderRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, RespondToAuthChallengeRequest> {
        /**
         * <p>
         * The app client ID.
         * </p>
         * 
         * @param clientId
         *        The app client ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientId(String clientId);

        /**
         * <p>
         * The challenge name. For more information, see .
         * </p>
         * <p>
         * <code>ADMIN_NO_SRP_AUTH</code> is not a valid value.
         * </p>
         * 
         * @param challengeName
         *        The challenge name. For more information, see .</p>
         *        <p>
         *        <code>ADMIN_NO_SRP_AUTH</code> is not a valid value.
         * @see ChallengeNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChallengeNameType
         */
        Builder challengeName(String challengeName);

        /**
         * <p>
         * The challenge name. For more information, see .
         * </p>
         * <p>
         * <code>ADMIN_NO_SRP_AUTH</code> is not a valid value.
         * </p>
         * 
         * @param challengeName
         *        The challenge name. For more information, see .</p>
         *        <p>
         *        <code>ADMIN_NO_SRP_AUTH</code> is not a valid value.
         * @see ChallengeNameType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChallengeNameType
         */
        Builder challengeName(ChallengeNameType challengeName);

        /**
         * <p>
         * The session which should be passed both ways in challenge-response calls to the service. If
         * <code>InitiateAuth</code> or <code>RespondToAuthChallenge</code> API call determines that the caller needs to
         * go through another challenge, they return a session with other challenge parameters. This session should be
         * passed as it is to the next <code>RespondToAuthChallenge</code> API call.
         * </p>
         * 
         * @param session
         *        The session which should be passed both ways in challenge-response calls to the service. If
         *        <code>InitiateAuth</code> or <code>RespondToAuthChallenge</code> API call determines that the caller
         *        needs to go through another challenge, they return a session with other challenge parameters. This
         *        session should be passed as it is to the next <code>RespondToAuthChallenge</code> API call.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder session(String session);

        /**
         * <p>
         * The challenge responses. These are inputs corresponding to the value of <code>ChallengeName</code>, for
         * example:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SMS_MFA</code>: <code>SMS_MFA_CODE</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if app
         * client is configured with client secret).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PASSWORD_VERIFIER</code>: <code>PASSWORD_CLAIM_SIGNATURE</code>,
         * <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, <code>TIMESTAMP</code>, <code>USERNAME</code>,
         * <code>SECRET_HASH</code> (if app client is configured with client secret).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_PASSWORD_REQUIRED</code>: <code>NEW_PASSWORD</code>, any other required attributes,
         * <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client secret).
         * </p>
         * </li>
         * </ul>
         * 
         * @param challengeResponses
         *        The challenge responses. These are inputs corresponding to the value of <code>ChallengeName</code>,
         *        for example:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SMS_MFA</code>: <code>SMS_MFA_CODE</code>, <code>USERNAME</code>, <code>SECRET_HASH</code> (if
         *        app client is configured with client secret).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PASSWORD_VERIFIER</code>: <code>PASSWORD_CLAIM_SIGNATURE</code>,
         *        <code>PASSWORD_CLAIM_SECRET_BLOCK</code>, <code>TIMESTAMP</code>, <code>USERNAME</code>,
         *        <code>SECRET_HASH</code> (if app client is configured with client secret).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_PASSWORD_REQUIRED</code>: <code>NEW_PASSWORD</code>, any other required attributes,
         *        <code>USERNAME</code>, <code>SECRET_HASH</code> (if app client is configured with client secret).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder challengeResponses(Map<String, String> challengeResponses);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>RespondToAuthChallenge</code> calls.
         * </p>
         * 
         * @param analyticsMetadata
         *        The Amazon Pinpoint analytics metadata for collecting metrics for <code>RespondToAuthChallenge</code>
         *        calls.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata);

        /**
         * <p>
         * The Amazon Pinpoint analytics metadata for collecting metrics for <code>RespondToAuthChallenge</code> calls.
         * </p>
         * This is a convenience that creates an instance of the {@link AnalyticsMetadataType.Builder} avoiding the need
         * to create one manually via {@link AnalyticsMetadataType#builder()}.
         *
         * When the {@link Consumer} completes, {@link AnalyticsMetadataType.Builder#build()} is called immediately and
         * its result is passed to {@link #analyticsMetadata(AnalyticsMetadataType)}.
         * 
         * @param analyticsMetadata
         *        a consumer that will call methods on {@link AnalyticsMetadataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #analyticsMetadata(AnalyticsMetadataType)
         */
        default Builder analyticsMetadata(Consumer<AnalyticsMetadataType.Builder> analyticsMetadata) {
            return analyticsMetadata(AnalyticsMetadataType.builder().applyMutation(analyticsMetadata).build());
        }

        /**
         * <p>
         * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk
         * of an unexpected event by Amazon Cognito advanced security.
         * </p>
         * 
         * @param userContextData
         *        Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the
         *        risk of an unexpected event by Amazon Cognito advanced security.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userContextData(UserContextDataType userContextData);

        /**
         * <p>
         * Contextual data such as the user's device fingerprint, IP address, or location used for evaluating the risk
         * of an unexpected event by Amazon Cognito advanced security.
         * </p>
         * This is a convenience that creates an instance of the {@link UserContextDataType.Builder} avoiding the need
         * to create one manually via {@link UserContextDataType#builder()}.
         *
         * When the {@link Consumer} completes, {@link UserContextDataType.Builder#build()} is called immediately and
         * its result is passed to {@link #userContextData(UserContextDataType)}.
         * 
         * @param userContextData
         *        a consumer that will call methods on {@link UserContextDataType.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userContextData(UserContextDataType)
         */
        default Builder userContextData(Consumer<UserContextDataType.Builder> userContextData) {
            return userContextData(UserContextDataType.builder().applyMutation(userContextData).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CognitoIdentityProviderRequest.BuilderImpl implements Builder {
        private String clientId;

        private String challengeName;

        private String session;

        private Map<String, String> challengeResponses = DefaultSdkAutoConstructMap.getInstance();

        private AnalyticsMetadataType analyticsMetadata;

        private UserContextDataType userContextData;

        private BuilderImpl() {
        }

        private BuilderImpl(RespondToAuthChallengeRequest model) {
            super(model);
            clientId(model.clientId);
            challengeName(model.challengeName);
            session(model.session);
            challengeResponses(model.challengeResponses);
            analyticsMetadata(model.analyticsMetadata);
            userContextData(model.userContextData);
        }

        public final String getClientId() {
            return clientId;
        }

        @Override
        public final Builder clientId(String clientId) {
            this.clientId = clientId;
            return this;
        }

        public final void setClientId(String clientId) {
            this.clientId = clientId;
        }

        public final String getChallengeNameAsString() {
            return challengeName;
        }

        @Override
        public final Builder challengeName(String challengeName) {
            this.challengeName = challengeName;
            return this;
        }

        @Override
        public final Builder challengeName(ChallengeNameType challengeName) {
            this.challengeName(challengeName.toString());
            return this;
        }

        public final void setChallengeName(String challengeName) {
            this.challengeName = challengeName;
        }

        public final String getSession() {
            return session;
        }

        @Override
        public final Builder session(String session) {
            this.session = session;
            return this;
        }

        public final void setSession(String session) {
            this.session = session;
        }

        public final Map<String, String> getChallengeResponses() {
            return challengeResponses;
        }

        @Override
        public final Builder challengeResponses(Map<String, String> challengeResponses) {
            this.challengeResponses = ChallengeResponsesTypeCopier.copy(challengeResponses);
            return this;
        }

        public final void setChallengeResponses(Map<String, String> challengeResponses) {
            this.challengeResponses = ChallengeResponsesTypeCopier.copy(challengeResponses);
        }

        public final AnalyticsMetadataType.Builder getAnalyticsMetadata() {
            return analyticsMetadata != null ? analyticsMetadata.toBuilder() : null;
        }

        @Override
        public final Builder analyticsMetadata(AnalyticsMetadataType analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata;
            return this;
        }

        public final void setAnalyticsMetadata(AnalyticsMetadataType.BuilderImpl analyticsMetadata) {
            this.analyticsMetadata = analyticsMetadata != null ? analyticsMetadata.build() : null;
        }

        public final UserContextDataType.Builder getUserContextData() {
            return userContextData != null ? userContextData.toBuilder() : null;
        }

        @Override
        public final Builder userContextData(UserContextDataType userContextData) {
            this.userContextData = userContextData;
            return this;
        }

        public final void setUserContextData(UserContextDataType.BuilderImpl userContextData) {
            this.userContextData = userContextData != null ? userContextData.build() : null;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RespondToAuthChallengeRequest build() {
            return new RespondToAuthChallengeRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
