/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the format and location of the input data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EntityRecognizerInputDataConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<EntityRecognizerInputDataConfig.Builder, EntityRecognizerInputDataConfig> {
    private static final SdkField<String> DATA_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataFormat").getter(getter(EntityRecognizerInputDataConfig::dataFormatAsString))
            .setter(setter(Builder::dataFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataFormat").build()).build();

    private static final SdkField<List<EntityTypesListItem>> ENTITY_TYPES_FIELD = SdkField
            .<List<EntityTypesListItem>> builder(MarshallingType.LIST)
            .memberName("EntityTypes")
            .getter(getter(EntityRecognizerInputDataConfig::entityTypes))
            .setter(setter(Builder::entityTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntityTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EntityTypesListItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(EntityTypesListItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EntityRecognizerDocuments> DOCUMENTS_FIELD = SdkField
            .<EntityRecognizerDocuments> builder(MarshallingType.SDK_POJO).memberName("Documents")
            .getter(getter(EntityRecognizerInputDataConfig::documents)).setter(setter(Builder::documents))
            .constructor(EntityRecognizerDocuments::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Documents").build()).build();

    private static final SdkField<EntityRecognizerAnnotations> ANNOTATIONS_FIELD = SdkField
            .<EntityRecognizerAnnotations> builder(MarshallingType.SDK_POJO).memberName("Annotations")
            .getter(getter(EntityRecognizerInputDataConfig::annotations)).setter(setter(Builder::annotations))
            .constructor(EntityRecognizerAnnotations::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Annotations").build()).build();

    private static final SdkField<EntityRecognizerEntityList> ENTITY_LIST_FIELD = SdkField
            .<EntityRecognizerEntityList> builder(MarshallingType.SDK_POJO).memberName("EntityList")
            .getter(getter(EntityRecognizerInputDataConfig::entityList)).setter(setter(Builder::entityList))
            .constructor(EntityRecognizerEntityList::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EntityList").build()).build();

    private static final SdkField<List<AugmentedManifestsListItem>> AUGMENTED_MANIFESTS_FIELD = SdkField
            .<List<AugmentedManifestsListItem>> builder(MarshallingType.LIST)
            .memberName("AugmentedManifests")
            .getter(getter(EntityRecognizerInputDataConfig::augmentedManifests))
            .setter(setter(Builder::augmentedManifests))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AugmentedManifests").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AugmentedManifestsListItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(AugmentedManifestsListItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_FORMAT_FIELD,
            ENTITY_TYPES_FIELD, DOCUMENTS_FIELD, ANNOTATIONS_FIELD, ENTITY_LIST_FIELD, AUGMENTED_MANIFESTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dataFormat;

    private final List<EntityTypesListItem> entityTypes;

    private final EntityRecognizerDocuments documents;

    private final EntityRecognizerAnnotations annotations;

    private final EntityRecognizerEntityList entityList;

    private final List<AugmentedManifestsListItem> augmentedManifests;

    private EntityRecognizerInputDataConfig(BuilderImpl builder) {
        this.dataFormat = builder.dataFormat;
        this.entityTypes = builder.entityTypes;
        this.documents = builder.documents;
        this.annotations = builder.annotations;
        this.entityList = builder.entityList;
        this.augmentedManifests = builder.augmentedManifests;
    }

    /**
     * <p>
     * The format of your training data:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPREHEND_CSV</code>: A CSV file that supplements your training documents. The CSV file contains
     * information about the custom entities that your trained model will detect. The required format of the file
     * depends on whether you are providing annotations or an entity list.
     * </p>
     * <p>
     * If you use this value, you must provide your CSV file by using either the <code>Annotations</code> or
     * <code>EntityList</code> parameters. You must provide your training documents by using the <code>Documents</code>
     * parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth. This file
     * is in JSON lines format. Each line is a complete JSON object that contains a training document and its labels.
     * Each label annotates a named entity in the training document.
     * </p>
     * <p>
     * If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataFormat} will
     * return {@link EntityRecognizerDataFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #dataFormatAsString}.
     * </p>
     * 
     * @return The format of your training data:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPREHEND_CSV</code>: A CSV file that supplements your training documents. The CSV file contains
     *         information about the custom entities that your trained model will detect. The required format of the
     *         file depends on whether you are providing annotations or an entity list.
     *         </p>
     *         <p>
     *         If you use this value, you must provide your CSV file by using either the <code>Annotations</code> or
     *         <code>EntityList</code> parameters. You must provide your training documents by using the
     *         <code>Documents</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth.
     *         This file is in JSON lines format. Each line is a complete JSON object that contains a training document
     *         and its labels. Each label annotates a named entity in the training document.
     *         </p>
     *         <p>
     *         If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
     * @see EntityRecognizerDataFormat
     */
    public final EntityRecognizerDataFormat dataFormat() {
        return EntityRecognizerDataFormat.fromValue(dataFormat);
    }

    /**
     * <p>
     * The format of your training data:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPREHEND_CSV</code>: A CSV file that supplements your training documents. The CSV file contains
     * information about the custom entities that your trained model will detect. The required format of the file
     * depends on whether you are providing annotations or an entity list.
     * </p>
     * <p>
     * If you use this value, you must provide your CSV file by using either the <code>Annotations</code> or
     * <code>EntityList</code> parameters. You must provide your training documents by using the <code>Documents</code>
     * parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth. This file
     * is in JSON lines format. Each line is a complete JSON object that contains a training document and its labels.
     * Each label annotates a named entity in the training document.
     * </p>
     * <p>
     * If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataFormat} will
     * return {@link EntityRecognizerDataFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #dataFormatAsString}.
     * </p>
     * 
     * @return The format of your training data:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPREHEND_CSV</code>: A CSV file that supplements your training documents. The CSV file contains
     *         information about the custom entities that your trained model will detect. The required format of the
     *         file depends on whether you are providing annotations or an entity list.
     *         </p>
     *         <p>
     *         If you use this value, you must provide your CSV file by using either the <code>Annotations</code> or
     *         <code>EntityList</code> parameters. You must provide your training documents by using the
     *         <code>Documents</code> parameter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth.
     *         This file is in JSON lines format. Each line is a complete JSON object that contains a training document
     *         and its labels. Each label annotates a named entity in the training document.
     *         </p>
     *         <p>
     *         If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
     * @see EntityRecognizerDataFormat
     */
    public final String dataFormatAsString() {
        return dataFormat;
    }

    /**
     * Returns true if the EntityTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasEntityTypes() {
        return entityTypes != null && !(entityTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The entity types in the labeled training data that Amazon Comprehend uses to train the custom entity recognizer.
     * Any entity types that you don't specify are ignored.
     * </p>
     * <p>
     * A maximum of 25 entity types can be used at one time to train an entity recognizer. Entity types must not contain
     * the following invalid characters: \n (line break), \\n (escaped line break), \r (carriage return), \\r (escaped
     * carriage return), \t (tab), \\t (escaped tab), space, and , (comma).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEntityTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The entity types in the labeled training data that Amazon Comprehend uses to train the custom entity
     *         recognizer. Any entity types that you don't specify are ignored.</p>
     *         <p>
     *         A maximum of 25 entity types can be used at one time to train an entity recognizer. Entity types must not
     *         contain the following invalid characters: \n (line break), \\n (escaped line break), \r (carriage
     *         return), \\r (escaped carriage return), \t (tab), \\t (escaped tab), space, and , (comma).
     */
    public final List<EntityTypesListItem> entityTypes() {
        return entityTypes;
    }

    /**
     * <p>
     * The S3 location of the folder that contains the training documents for your custom entity recognizer.
     * </p>
     * <p>
     * This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
     * </p>
     * 
     * @return The S3 location of the folder that contains the training documents for your custom entity recognizer.</p>
     *         <p>
     *         This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
     */
    public final EntityRecognizerDocuments documents() {
        return documents;
    }

    /**
     * <p>
     * The S3 location of the CSV file that annotates your training documents.
     * </p>
     * 
     * @return The S3 location of the CSV file that annotates your training documents.
     */
    public final EntityRecognizerAnnotations annotations() {
        return annotations;
    }

    /**
     * <p>
     * The S3 location of the CSV file that has the entity list for your custom entity recognizer.
     * </p>
     * 
     * @return The S3 location of the CSV file that has the entity list for your custom entity recognizer.
     */
    public final EntityRecognizerEntityList entityList() {
        return entityList;
    }

    /**
     * Returns true if the AugmentedManifests property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasAugmentedManifests() {
        return augmentedManifests != null && !(augmentedManifests instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of augmented manifest files that provide training data for your custom model. An augmented manifest file
     * is a labeled dataset that is produced by Amazon SageMaker Ground Truth.
     * </p>
     * <p>
     * This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAugmentedManifests()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of augmented manifest files that provide training data for your custom model. An augmented
     *         manifest file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.</p>
     *         <p>
     *         This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
     */
    public final List<AugmentedManifestsListItem> augmentedManifests() {
        return augmentedManifests;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dataFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasEntityTypes() ? entityTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(documents());
        hashCode = 31 * hashCode + Objects.hashCode(annotations());
        hashCode = 31 * hashCode + Objects.hashCode(entityList());
        hashCode = 31 * hashCode + Objects.hashCode(hasAugmentedManifests() ? augmentedManifests() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EntityRecognizerInputDataConfig)) {
            return false;
        }
        EntityRecognizerInputDataConfig other = (EntityRecognizerInputDataConfig) obj;
        return Objects.equals(dataFormatAsString(), other.dataFormatAsString()) && hasEntityTypes() == other.hasEntityTypes()
                && Objects.equals(entityTypes(), other.entityTypes()) && Objects.equals(documents(), other.documents())
                && Objects.equals(annotations(), other.annotations()) && Objects.equals(entityList(), other.entityList())
                && hasAugmentedManifests() == other.hasAugmentedManifests()
                && Objects.equals(augmentedManifests(), other.augmentedManifests());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EntityRecognizerInputDataConfig").add("DataFormat", dataFormatAsString())
                .add("EntityTypes", hasEntityTypes() ? entityTypes() : null).add("Documents", documents())
                .add("Annotations", annotations()).add("EntityList", entityList())
                .add("AugmentedManifests", hasAugmentedManifests() ? augmentedManifests() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DataFormat":
            return Optional.ofNullable(clazz.cast(dataFormatAsString()));
        case "EntityTypes":
            return Optional.ofNullable(clazz.cast(entityTypes()));
        case "Documents":
            return Optional.ofNullable(clazz.cast(documents()));
        case "Annotations":
            return Optional.ofNullable(clazz.cast(annotations()));
        case "EntityList":
            return Optional.ofNullable(clazz.cast(entityList()));
        case "AugmentedManifests":
            return Optional.ofNullable(clazz.cast(augmentedManifests()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EntityRecognizerInputDataConfig, T> g) {
        return obj -> g.apply((EntityRecognizerInputDataConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EntityRecognizerInputDataConfig> {
        /**
         * <p>
         * The format of your training data:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPREHEND_CSV</code>: A CSV file that supplements your training documents. The CSV file contains
         * information about the custom entities that your trained model will detect. The required format of the file
         * depends on whether you are providing annotations or an entity list.
         * </p>
         * <p>
         * If you use this value, you must provide your CSV file by using either the <code>Annotations</code> or
         * <code>EntityList</code> parameters. You must provide your training documents by using the
         * <code>Documents</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth. This
         * file is in JSON lines format. Each line is a complete JSON object that contains a training document and its
         * labels. Each label annotates a named entity in the training document.
         * </p>
         * <p>
         * If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
         * </p>
         * 
         * @param dataFormat
         *        The format of your training data:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPREHEND_CSV</code>: A CSV file that supplements your training documents. The CSV file
         *        contains information about the custom entities that your trained model will detect. The required
         *        format of the file depends on whether you are providing annotations or an entity list.
         *        </p>
         *        <p>
         *        If you use this value, you must provide your CSV file by using either the <code>Annotations</code> or
         *        <code>EntityList</code> parameters. You must provide your training documents by using the
         *        <code>Documents</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth.
         *        This file is in JSON lines format. Each line is a complete JSON object that contains a training
         *        document and its labels. Each label annotates a named entity in the training document.
         *        </p>
         *        <p>
         *        If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
         * @see EntityRecognizerDataFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EntityRecognizerDataFormat
         */
        Builder dataFormat(String dataFormat);

        /**
         * <p>
         * The format of your training data:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPREHEND_CSV</code>: A CSV file that supplements your training documents. The CSV file contains
         * information about the custom entities that your trained model will detect. The required format of the file
         * depends on whether you are providing annotations or an entity list.
         * </p>
         * <p>
         * If you use this value, you must provide your CSV file by using either the <code>Annotations</code> or
         * <code>EntityList</code> parameters. You must provide your training documents by using the
         * <code>Documents</code> parameter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth. This
         * file is in JSON lines format. Each line is a complete JSON object that contains a training document and its
         * labels. Each label annotates a named entity in the training document.
         * </p>
         * <p>
         * If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
         * </p>
         * 
         * @param dataFormat
         *        The format of your training data:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPREHEND_CSV</code>: A CSV file that supplements your training documents. The CSV file
         *        contains information about the custom entities that your trained model will detect. The required
         *        format of the file depends on whether you are providing annotations or an entity list.
         *        </p>
         *        <p>
         *        If you use this value, you must provide your CSV file by using either the <code>Annotations</code> or
         *        <code>EntityList</code> parameters. You must provide your training documents by using the
         *        <code>Documents</code> parameter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth.
         *        This file is in JSON lines format. Each line is a complete JSON object that contains a training
         *        document and its labels. Each label annotates a named entity in the training document.
         *        </p>
         *        <p>
         *        If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
         * @see EntityRecognizerDataFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EntityRecognizerDataFormat
         */
        Builder dataFormat(EntityRecognizerDataFormat dataFormat);

        /**
         * <p>
         * The entity types in the labeled training data that Amazon Comprehend uses to train the custom entity
         * recognizer. Any entity types that you don't specify are ignored.
         * </p>
         * <p>
         * A maximum of 25 entity types can be used at one time to train an entity recognizer. Entity types must not
         * contain the following invalid characters: \n (line break), \\n (escaped line break), \r (carriage return),
         * \\r (escaped carriage return), \t (tab), \\t (escaped tab), space, and , (comma).
         * </p>
         * 
         * @param entityTypes
         *        The entity types in the labeled training data that Amazon Comprehend uses to train the custom entity
         *        recognizer. Any entity types that you don't specify are ignored.</p>
         *        <p>
         *        A maximum of 25 entity types can be used at one time to train an entity recognizer. Entity types must
         *        not contain the following invalid characters: \n (line break), \\n (escaped line break), \r (carriage
         *        return), \\r (escaped carriage return), \t (tab), \\t (escaped tab), space, and , (comma).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypes(Collection<EntityTypesListItem> entityTypes);

        /**
         * <p>
         * The entity types in the labeled training data that Amazon Comprehend uses to train the custom entity
         * recognizer. Any entity types that you don't specify are ignored.
         * </p>
         * <p>
         * A maximum of 25 entity types can be used at one time to train an entity recognizer. Entity types must not
         * contain the following invalid characters: \n (line break), \\n (escaped line break), \r (carriage return),
         * \\r (escaped carriage return), \t (tab), \\t (escaped tab), space, and , (comma).
         * </p>
         * 
         * @param entityTypes
         *        The entity types in the labeled training data that Amazon Comprehend uses to train the custom entity
         *        recognizer. Any entity types that you don't specify are ignored.</p>
         *        <p>
         *        A maximum of 25 entity types can be used at one time to train an entity recognizer. Entity types must
         *        not contain the following invalid characters: \n (line break), \\n (escaped line break), \r (carriage
         *        return), \\r (escaped carriage return), \t (tab), \\t (escaped tab), space, and , (comma).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityTypes(EntityTypesListItem... entityTypes);

        /**
         * <p>
         * The entity types in the labeled training data that Amazon Comprehend uses to train the custom entity
         * recognizer. Any entity types that you don't specify are ignored.
         * </p>
         * <p>
         * A maximum of 25 entity types can be used at one time to train an entity recognizer. Entity types must not
         * contain the following invalid characters: \n (line break), \\n (escaped line break), \r (carriage return),
         * \\r (escaped carriage return), \t (tab), \\t (escaped tab), space, and , (comma).
         * </p>
         * This is a convenience that creates an instance of the {@link List<EntityTypesListItem>.Builder} avoiding the
         * need to create one manually via {@link List<EntityTypesListItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EntityTypesListItem>.Builder#build()} is called immediately
         * and its result is passed to {@link #entityTypes(List<EntityTypesListItem>)}.
         * 
         * @param entityTypes
         *        a consumer that will call methods on {@link List<EntityTypesListItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entityTypes(List<EntityTypesListItem>)
         */
        Builder entityTypes(Consumer<EntityTypesListItem.Builder>... entityTypes);

        /**
         * <p>
         * The S3 location of the folder that contains the training documents for your custom entity recognizer.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
         * </p>
         * 
         * @param documents
         *        The S3 location of the folder that contains the training documents for your custom entity
         *        recognizer.</p>
         *        <p>
         *        This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documents(EntityRecognizerDocuments documents);

        /**
         * <p>
         * The S3 location of the folder that contains the training documents for your custom entity recognizer.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link EntityRecognizerDocuments.Builder} avoiding the
         * need to create one manually via {@link EntityRecognizerDocuments#builder()}.
         *
         * When the {@link Consumer} completes, {@link EntityRecognizerDocuments.Builder#build()} is called immediately
         * and its result is passed to {@link #documents(EntityRecognizerDocuments)}.
         * 
         * @param documents
         *        a consumer that will call methods on {@link EntityRecognizerDocuments.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #documents(EntityRecognizerDocuments)
         */
        default Builder documents(Consumer<EntityRecognizerDocuments.Builder> documents) {
            return documents(EntityRecognizerDocuments.builder().applyMutation(documents).build());
        }

        /**
         * <p>
         * The S3 location of the CSV file that annotates your training documents.
         * </p>
         * 
         * @param annotations
         *        The S3 location of the CSV file that annotates your training documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder annotations(EntityRecognizerAnnotations annotations);

        /**
         * <p>
         * The S3 location of the CSV file that annotates your training documents.
         * </p>
         * This is a convenience that creates an instance of the {@link EntityRecognizerAnnotations.Builder} avoiding
         * the need to create one manually via {@link EntityRecognizerAnnotations#builder()}.
         *
         * When the {@link Consumer} completes, {@link EntityRecognizerAnnotations.Builder#build()} is called
         * immediately and its result is passed to {@link #annotations(EntityRecognizerAnnotations)}.
         * 
         * @param annotations
         *        a consumer that will call methods on {@link EntityRecognizerAnnotations.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #annotations(EntityRecognizerAnnotations)
         */
        default Builder annotations(Consumer<EntityRecognizerAnnotations.Builder> annotations) {
            return annotations(EntityRecognizerAnnotations.builder().applyMutation(annotations).build());
        }

        /**
         * <p>
         * The S3 location of the CSV file that has the entity list for your custom entity recognizer.
         * </p>
         * 
         * @param entityList
         *        The S3 location of the CSV file that has the entity list for your custom entity recognizer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityList(EntityRecognizerEntityList entityList);

        /**
         * <p>
         * The S3 location of the CSV file that has the entity list for your custom entity recognizer.
         * </p>
         * This is a convenience that creates an instance of the {@link EntityRecognizerEntityList.Builder} avoiding the
         * need to create one manually via {@link EntityRecognizerEntityList#builder()}.
         *
         * When the {@link Consumer} completes, {@link EntityRecognizerEntityList.Builder#build()} is called immediately
         * and its result is passed to {@link #entityList(EntityRecognizerEntityList)}.
         * 
         * @param entityList
         *        a consumer that will call methods on {@link EntityRecognizerEntityList.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #entityList(EntityRecognizerEntityList)
         */
        default Builder entityList(Consumer<EntityRecognizerEntityList.Builder> entityList) {
            return entityList(EntityRecognizerEntityList.builder().applyMutation(entityList).build());
        }

        /**
         * <p>
         * A list of augmented manifest files that provide training data for your custom model. An augmented manifest
         * file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * </p>
         * 
         * @param augmentedManifests
         *        A list of augmented manifest files that provide training data for your custom model. An augmented
         *        manifest file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.</p>
         *        <p>
         *        This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder augmentedManifests(Collection<AugmentedManifestsListItem> augmentedManifests);

        /**
         * <p>
         * A list of augmented manifest files that provide training data for your custom model. An augmented manifest
         * file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * </p>
         * 
         * @param augmentedManifests
         *        A list of augmented manifest files that provide training data for your custom model. An augmented
         *        manifest file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.</p>
         *        <p>
         *        This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder augmentedManifests(AugmentedManifestsListItem... augmentedManifests);

        /**
         * <p>
         * A list of augmented manifest files that provide training data for your custom model. An augmented manifest
         * file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AugmentedManifestsListItem>.Builder}
         * avoiding the need to create one manually via {@link List<AugmentedManifestsListItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AugmentedManifestsListItem>.Builder#build()} is called
         * immediately and its result is passed to {@link #augmentedManifests(List<AugmentedManifestsListItem>)}.
         * 
         * @param augmentedManifests
         *        a consumer that will call methods on {@link List<AugmentedManifestsListItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #augmentedManifests(List<AugmentedManifestsListItem>)
         */
        Builder augmentedManifests(Consumer<AugmentedManifestsListItem.Builder>... augmentedManifests);
    }

    static final class BuilderImpl implements Builder {
        private String dataFormat;

        private List<EntityTypesListItem> entityTypes = DefaultSdkAutoConstructList.getInstance();

        private EntityRecognizerDocuments documents;

        private EntityRecognizerAnnotations annotations;

        private EntityRecognizerEntityList entityList;

        private List<AugmentedManifestsListItem> augmentedManifests = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EntityRecognizerInputDataConfig model) {
            dataFormat(model.dataFormat);
            entityTypes(model.entityTypes);
            documents(model.documents);
            annotations(model.annotations);
            entityList(model.entityList);
            augmentedManifests(model.augmentedManifests);
        }

        public final String getDataFormat() {
            return dataFormat;
        }

        @Override
        public final Builder dataFormat(String dataFormat) {
            this.dataFormat = dataFormat;
            return this;
        }

        @Override
        public final Builder dataFormat(EntityRecognizerDataFormat dataFormat) {
            this.dataFormat(dataFormat == null ? null : dataFormat.toString());
            return this;
        }

        public final void setDataFormat(String dataFormat) {
            this.dataFormat = dataFormat;
        }

        public final List<EntityTypesListItem.Builder> getEntityTypes() {
            List<EntityTypesListItem.Builder> result = EntityTypesListCopier.copyToBuilder(this.entityTypes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder entityTypes(Collection<EntityTypesListItem> entityTypes) {
            this.entityTypes = EntityTypesListCopier.copy(entityTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityTypes(EntityTypesListItem... entityTypes) {
            entityTypes(Arrays.asList(entityTypes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder entityTypes(Consumer<EntityTypesListItem.Builder>... entityTypes) {
            entityTypes(Stream.of(entityTypes).map(c -> EntityTypesListItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEntityTypes(Collection<EntityTypesListItem.BuilderImpl> entityTypes) {
            this.entityTypes = EntityTypesListCopier.copyFromBuilder(entityTypes);
        }

        public final EntityRecognizerDocuments.Builder getDocuments() {
            return documents != null ? documents.toBuilder() : null;
        }

        @Override
        public final Builder documents(EntityRecognizerDocuments documents) {
            this.documents = documents;
            return this;
        }

        public final void setDocuments(EntityRecognizerDocuments.BuilderImpl documents) {
            this.documents = documents != null ? documents.build() : null;
        }

        public final EntityRecognizerAnnotations.Builder getAnnotations() {
            return annotations != null ? annotations.toBuilder() : null;
        }

        @Override
        public final Builder annotations(EntityRecognizerAnnotations annotations) {
            this.annotations = annotations;
            return this;
        }

        public final void setAnnotations(EntityRecognizerAnnotations.BuilderImpl annotations) {
            this.annotations = annotations != null ? annotations.build() : null;
        }

        public final EntityRecognizerEntityList.Builder getEntityList() {
            return entityList != null ? entityList.toBuilder() : null;
        }

        @Override
        public final Builder entityList(EntityRecognizerEntityList entityList) {
            this.entityList = entityList;
            return this;
        }

        public final void setEntityList(EntityRecognizerEntityList.BuilderImpl entityList) {
            this.entityList = entityList != null ? entityList.build() : null;
        }

        public final List<AugmentedManifestsListItem.Builder> getAugmentedManifests() {
            List<AugmentedManifestsListItem.Builder> result = EntityRecognizerAugmentedManifestsListCopier
                    .copyToBuilder(this.augmentedManifests);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder augmentedManifests(Collection<AugmentedManifestsListItem> augmentedManifests) {
            this.augmentedManifests = EntityRecognizerAugmentedManifestsListCopier.copy(augmentedManifests);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder augmentedManifests(AugmentedManifestsListItem... augmentedManifests) {
            augmentedManifests(Arrays.asList(augmentedManifests));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder augmentedManifests(Consumer<AugmentedManifestsListItem.Builder>... augmentedManifests) {
            augmentedManifests(Stream.of(augmentedManifests)
                    .map(c -> AugmentedManifestsListItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAugmentedManifests(Collection<AugmentedManifestsListItem.BuilderImpl> augmentedManifests) {
            this.augmentedManifests = EntityRecognizerAugmentedManifestsListCopier.copyFromBuilder(augmentedManifests);
        }

        @Override
        public EntityRecognizerInputDataConfig build() {
            return new EntityRecognizerInputDataConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
