/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The input properties for training a document classifier.
 * </p>
 * <p>
 * For more information on how the input file is formatted, see <a
 * href="https://docs.aws.amazon.com/comprehend/latest/dg/prep-classifier-data.html">Preparing training data</a> in the
 * Comprehend Developer Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentClassifierInputDataConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentClassifierInputDataConfig.Builder, DocumentClassifierInputDataConfig> {
    private static final SdkField<String> DATA_FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataFormat").getter(getter(DocumentClassifierInputDataConfig::dataFormatAsString))
            .setter(setter(Builder::dataFormat))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataFormat").build()).build();

    private static final SdkField<String> S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S3Uri")
            .getter(getter(DocumentClassifierInputDataConfig::s3Uri)).setter(setter(Builder::s3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Uri").build()).build();

    private static final SdkField<String> TEST_S3_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TestS3Uri").getter(getter(DocumentClassifierInputDataConfig::testS3Uri))
            .setter(setter(Builder::testS3Uri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TestS3Uri").build()).build();

    private static final SdkField<String> LABEL_DELIMITER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LabelDelimiter").getter(getter(DocumentClassifierInputDataConfig::labelDelimiter))
            .setter(setter(Builder::labelDelimiter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LabelDelimiter").build()).build();

    private static final SdkField<List<AugmentedManifestsListItem>> AUGMENTED_MANIFESTS_FIELD = SdkField
            .<List<AugmentedManifestsListItem>> builder(MarshallingType.LIST)
            .memberName("AugmentedManifests")
            .getter(getter(DocumentClassifierInputDataConfig::augmentedManifests))
            .setter(setter(Builder::augmentedManifests))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AugmentedManifests").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AugmentedManifestsListItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(AugmentedManifestsListItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DOCUMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentType").getter(getter(DocumentClassifierInputDataConfig::documentTypeAsString))
            .setter(setter(Builder::documentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentType").build()).build();

    private static final SdkField<DocumentClassifierDocuments> DOCUMENTS_FIELD = SdkField
            .<DocumentClassifierDocuments> builder(MarshallingType.SDK_POJO).memberName("Documents")
            .getter(getter(DocumentClassifierInputDataConfig::documents)).setter(setter(Builder::documents))
            .constructor(DocumentClassifierDocuments::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Documents").build()).build();

    private static final SdkField<DocumentReaderConfig> DOCUMENT_READER_CONFIG_FIELD = SdkField
            .<DocumentReaderConfig> builder(MarshallingType.SDK_POJO).memberName("DocumentReaderConfig")
            .getter(getter(DocumentClassifierInputDataConfig::documentReaderConfig))
            .setter(setter(Builder::documentReaderConfig)).constructor(DocumentReaderConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentReaderConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_FORMAT_FIELD,
            S3_URI_FIELD, TEST_S3_URI_FIELD, LABEL_DELIMITER_FIELD, AUGMENTED_MANIFESTS_FIELD, DOCUMENT_TYPE_FIELD,
            DOCUMENTS_FIELD, DOCUMENT_READER_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dataFormat;

    private final String s3Uri;

    private final String testS3Uri;

    private final String labelDelimiter;

    private final List<AugmentedManifestsListItem> augmentedManifests;

    private final String documentType;

    private final DocumentClassifierDocuments documents;

    private final DocumentReaderConfig documentReaderConfig;

    private DocumentClassifierInputDataConfig(BuilderImpl builder) {
        this.dataFormat = builder.dataFormat;
        this.s3Uri = builder.s3Uri;
        this.testS3Uri = builder.testS3Uri;
        this.labelDelimiter = builder.labelDelimiter;
        this.augmentedManifests = builder.augmentedManifests;
        this.documentType = builder.documentType;
        this.documents = builder.documents;
        this.documentReaderConfig = builder.documentReaderConfig;
    }

    /**
     * <p>
     * The format of your training data:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPREHEND_CSV</code>: A two-column CSV file, where labels are provided in the first column, and documents
     * are provided in the second. If you use this value, you must provide the <code>S3Uri</code> parameter in your
     * request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth. This file
     * is in JSON lines format. Each line is a complete JSON object that contains a training document and its associated
     * labels.
     * </p>
     * <p>
     * If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataFormat} will
     * return {@link DocumentClassifierDataFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #dataFormatAsString}.
     * </p>
     * 
     * @return The format of your training data:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPREHEND_CSV</code>: A two-column CSV file, where labels are provided in the first column, and
     *         documents are provided in the second. If you use this value, you must provide the <code>S3Uri</code>
     *         parameter in your request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth.
     *         This file is in JSON lines format. Each line is a complete JSON object that contains a training document
     *         and its associated labels.
     *         </p>
     *         <p>
     *         If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
     * @see DocumentClassifierDataFormat
     */
    public final DocumentClassifierDataFormat dataFormat() {
        return DocumentClassifierDataFormat.fromValue(dataFormat);
    }

    /**
     * <p>
     * The format of your training data:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>COMPREHEND_CSV</code>: A two-column CSV file, where labels are provided in the first column, and documents
     * are provided in the second. If you use this value, you must provide the <code>S3Uri</code> parameter in your
     * request.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth. This file
     * is in JSON lines format. Each line is a complete JSON object that contains a training document and its associated
     * labels.
     * </p>
     * <p>
     * If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #dataFormat} will
     * return {@link DocumentClassifierDataFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #dataFormatAsString}.
     * </p>
     * 
     * @return The format of your training data:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>COMPREHEND_CSV</code>: A two-column CSV file, where labels are provided in the first column, and
     *         documents are provided in the second. If you use this value, you must provide the <code>S3Uri</code>
     *         parameter in your request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth.
     *         This file is in JSON lines format. Each line is a complete JSON object that contains a training document
     *         and its associated labels.
     *         </p>
     *         <p>
     *         If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
     * @see DocumentClassifierDataFormat
     */
    public final String dataFormatAsString() {
        return dataFormat;
    }

    /**
     * <p>
     * The Amazon S3 URI for the input data. The S3 bucket must be in the same Region as the API endpoint that you are
     * calling. The URI can point to a single input file or it can provide the prefix for a collection of input files.
     * </p>
     * <p>
     * For example, if you use the URI <code>S3://bucketName/prefix</code>, if the prefix is a single file, Amazon
     * Comprehend uses that file as input. If more than one file begins with the prefix, Amazon Comprehend uses all of
     * them as input.
     * </p>
     * <p>
     * This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
     * </p>
     * 
     * @return The Amazon S3 URI for the input data. The S3 bucket must be in the same Region as the API endpoint that
     *         you are calling. The URI can point to a single input file or it can provide the prefix for a collection
     *         of input files.</p>
     *         <p>
     *         For example, if you use the URI <code>S3://bucketName/prefix</code>, if the prefix is a single file,
     *         Amazon Comprehend uses that file as input. If more than one file begins with the prefix, Amazon
     *         Comprehend uses all of them as input.
     *         </p>
     *         <p>
     *         This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
     */
    public final String s3Uri() {
        return s3Uri;
    }

    /**
     * <p>
     * This specifies the Amazon S3 location that contains the test annotations for the document classifier. The URI
     * must be in the same Amazon Web Services Region as the API endpoint that you are calling.
     * </p>
     * 
     * @return This specifies the Amazon S3 location that contains the test annotations for the document classifier. The
     *         URI must be in the same Amazon Web Services Region as the API endpoint that you are calling.
     */
    public final String testS3Uri() {
        return testS3Uri;
    }

    /**
     * <p>
     * Indicates the delimiter used to separate each label for training a multi-label classifier. The default delimiter
     * between labels is a pipe (|). You can use a different character as a delimiter (if it's an allowed character) by
     * specifying it under Delimiter for labels. If the training documents use a delimiter other than the default or the
     * delimiter you specify, the labels on that line will be combined to make a single unique label, such as
     * LABELLABELLABEL.
     * </p>
     * 
     * @return Indicates the delimiter used to separate each label for training a multi-label classifier. The default
     *         delimiter between labels is a pipe (|). You can use a different character as a delimiter (if it's an
     *         allowed character) by specifying it under Delimiter for labels. If the training documents use a delimiter
     *         other than the default or the delimiter you specify, the labels on that line will be combined to make a
     *         single unique label, such as LABELLABELLABEL.
     */
    public final String labelDelimiter() {
        return labelDelimiter;
    }

    /**
     * For responses, this returns true if the service returned a value for the AugmentedManifests property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAugmentedManifests() {
        return augmentedManifests != null && !(augmentedManifests instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of augmented manifest files that provide training data for your custom model. An augmented manifest file
     * is a labeled dataset that is produced by Amazon SageMaker Ground Truth.
     * </p>
     * <p>
     * This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAugmentedManifests} method.
     * </p>
     * 
     * @return A list of augmented manifest files that provide training data for your custom model. An augmented
     *         manifest file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.</p>
     *         <p>
     *         This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
     */
    public final List<AugmentedManifestsListItem> augmentedManifests() {
        return augmentedManifests;
    }

    /**
     * <p>
     * The type of input documents for training the model. Provide plain-text documents to create a plain-text model,
     * and provide semi-structured documents to create a native document model.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentClassifierDocumentTypeFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The type of input documents for training the model. Provide plain-text documents to create a plain-text
     *         model, and provide semi-structured documents to create a native document model.
     * @see DocumentClassifierDocumentTypeFormat
     */
    public final DocumentClassifierDocumentTypeFormat documentType() {
        return DocumentClassifierDocumentTypeFormat.fromValue(documentType);
    }

    /**
     * <p>
     * The type of input documents for training the model. Provide plain-text documents to create a plain-text model,
     * and provide semi-structured documents to create a native document model.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #documentType} will
     * return {@link DocumentClassifierDocumentTypeFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #documentTypeAsString}.
     * </p>
     * 
     * @return The type of input documents for training the model. Provide plain-text documents to create a plain-text
     *         model, and provide semi-structured documents to create a native document model.
     * @see DocumentClassifierDocumentTypeFormat
     */
    public final String documentTypeAsString() {
        return documentType;
    }

    /**
     * <p>
     * The S3 location of the training documents. This parameter is required in a request to create a native document
     * model.
     * </p>
     * 
     * @return The S3 location of the training documents. This parameter is required in a request to create a native
     *         document model.
     */
    public final DocumentClassifierDocuments documents() {
        return documents;
    }

    /**
     * Returns the value of the DocumentReaderConfig property for this object.
     * 
     * @return The value of the DocumentReaderConfig property for this object.
     */
    public final DocumentReaderConfig documentReaderConfig() {
        return documentReaderConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dataFormatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(s3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(testS3Uri());
        hashCode = 31 * hashCode + Objects.hashCode(labelDelimiter());
        hashCode = 31 * hashCode + Objects.hashCode(hasAugmentedManifests() ? augmentedManifests() : null);
        hashCode = 31 * hashCode + Objects.hashCode(documentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(documents());
        hashCode = 31 * hashCode + Objects.hashCode(documentReaderConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentClassifierInputDataConfig)) {
            return false;
        }
        DocumentClassifierInputDataConfig other = (DocumentClassifierInputDataConfig) obj;
        return Objects.equals(dataFormatAsString(), other.dataFormatAsString()) && Objects.equals(s3Uri(), other.s3Uri())
                && Objects.equals(testS3Uri(), other.testS3Uri()) && Objects.equals(labelDelimiter(), other.labelDelimiter())
                && hasAugmentedManifests() == other.hasAugmentedManifests()
                && Objects.equals(augmentedManifests(), other.augmentedManifests())
                && Objects.equals(documentTypeAsString(), other.documentTypeAsString())
                && Objects.equals(documents(), other.documents())
                && Objects.equals(documentReaderConfig(), other.documentReaderConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentClassifierInputDataConfig").add("DataFormat", dataFormatAsString())
                .add("S3Uri", s3Uri()).add("TestS3Uri", testS3Uri()).add("LabelDelimiter", labelDelimiter())
                .add("AugmentedManifests", hasAugmentedManifests() ? augmentedManifests() : null)
                .add("DocumentType", documentTypeAsString()).add("Documents", documents())
                .add("DocumentReaderConfig", documentReaderConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DataFormat":
            return Optional.ofNullable(clazz.cast(dataFormatAsString()));
        case "S3Uri":
            return Optional.ofNullable(clazz.cast(s3Uri()));
        case "TestS3Uri":
            return Optional.ofNullable(clazz.cast(testS3Uri()));
        case "LabelDelimiter":
            return Optional.ofNullable(clazz.cast(labelDelimiter()));
        case "AugmentedManifests":
            return Optional.ofNullable(clazz.cast(augmentedManifests()));
        case "DocumentType":
            return Optional.ofNullable(clazz.cast(documentTypeAsString()));
        case "Documents":
            return Optional.ofNullable(clazz.cast(documents()));
        case "DocumentReaderConfig":
            return Optional.ofNullable(clazz.cast(documentReaderConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentClassifierInputDataConfig, T> g) {
        return obj -> g.apply((DocumentClassifierInputDataConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentClassifierInputDataConfig> {
        /**
         * <p>
         * The format of your training data:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPREHEND_CSV</code>: A two-column CSV file, where labels are provided in the first column, and
         * documents are provided in the second. If you use this value, you must provide the <code>S3Uri</code>
         * parameter in your request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth. This
         * file is in JSON lines format. Each line is a complete JSON object that contains a training document and its
         * associated labels.
         * </p>
         * <p>
         * If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
         * </p>
         * 
         * @param dataFormat
         *        The format of your training data:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPREHEND_CSV</code>: A two-column CSV file, where labels are provided in the first column, and
         *        documents are provided in the second. If you use this value, you must provide the <code>S3Uri</code>
         *        parameter in your request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth.
         *        This file is in JSON lines format. Each line is a complete JSON object that contains a training
         *        document and its associated labels.
         *        </p>
         *        <p>
         *        If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
         * @see DocumentClassifierDataFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentClassifierDataFormat
         */
        Builder dataFormat(String dataFormat);

        /**
         * <p>
         * The format of your training data:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>COMPREHEND_CSV</code>: A two-column CSV file, where labels are provided in the first column, and
         * documents are provided in the second. If you use this value, you must provide the <code>S3Uri</code>
         * parameter in your request.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth. This
         * file is in JSON lines format. Each line is a complete JSON object that contains a training document and its
         * associated labels.
         * </p>
         * <p>
         * If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
         * </p>
         * 
         * @param dataFormat
         *        The format of your training data:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>COMPREHEND_CSV</code>: A two-column CSV file, where labels are provided in the first column, and
         *        documents are provided in the second. If you use this value, you must provide the <code>S3Uri</code>
         *        parameter in your request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AUGMENTED_MANIFEST</code>: A labeled dataset that is produced by Amazon SageMaker Ground Truth.
         *        This file is in JSON lines format. Each line is a complete JSON object that contains a training
         *        document and its associated labels.
         *        </p>
         *        <p>
         *        If you use this value, you must provide the <code>AugmentedManifests</code> parameter in your request.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you don't specify a value, Amazon Comprehend uses <code>COMPREHEND_CSV</code> as the default.
         * @see DocumentClassifierDataFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentClassifierDataFormat
         */
        Builder dataFormat(DocumentClassifierDataFormat dataFormat);

        /**
         * <p>
         * The Amazon S3 URI for the input data. The S3 bucket must be in the same Region as the API endpoint that you
         * are calling. The URI can point to a single input file or it can provide the prefix for a collection of input
         * files.
         * </p>
         * <p>
         * For example, if you use the URI <code>S3://bucketName/prefix</code>, if the prefix is a single file, Amazon
         * Comprehend uses that file as input. If more than one file begins with the prefix, Amazon Comprehend uses all
         * of them as input.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
         * </p>
         * 
         * @param s3Uri
         *        The Amazon S3 URI for the input data. The S3 bucket must be in the same Region as the API endpoint
         *        that you are calling. The URI can point to a single input file or it can provide the prefix for a
         *        collection of input files.</p>
         *        <p>
         *        For example, if you use the URI <code>S3://bucketName/prefix</code>, if the prefix is a single file,
         *        Amazon Comprehend uses that file as input. If more than one file begins with the prefix, Amazon
         *        Comprehend uses all of them as input.
         *        </p>
         *        <p>
         *        This parameter is required if you set <code>DataFormat</code> to <code>COMPREHEND_CSV</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Uri(String s3Uri);

        /**
         * <p>
         * This specifies the Amazon S3 location that contains the test annotations for the document classifier. The URI
         * must be in the same Amazon Web Services Region as the API endpoint that you are calling.
         * </p>
         * 
         * @param testS3Uri
         *        This specifies the Amazon S3 location that contains the test annotations for the document classifier.
         *        The URI must be in the same Amazon Web Services Region as the API endpoint that you are calling.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder testS3Uri(String testS3Uri);

        /**
         * <p>
         * Indicates the delimiter used to separate each label for training a multi-label classifier. The default
         * delimiter between labels is a pipe (|). You can use a different character as a delimiter (if it's an allowed
         * character) by specifying it under Delimiter for labels. If the training documents use a delimiter other than
         * the default or the delimiter you specify, the labels on that line will be combined to make a single unique
         * label, such as LABELLABELLABEL.
         * </p>
         * 
         * @param labelDelimiter
         *        Indicates the delimiter used to separate each label for training a multi-label classifier. The default
         *        delimiter between labels is a pipe (|). You can use a different character as a delimiter (if it's an
         *        allowed character) by specifying it under Delimiter for labels. If the training documents use a
         *        delimiter other than the default or the delimiter you specify, the labels on that line will be
         *        combined to make a single unique label, such as LABELLABELLABEL.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labelDelimiter(String labelDelimiter);

        /**
         * <p>
         * A list of augmented manifest files that provide training data for your custom model. An augmented manifest
         * file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * </p>
         * 
         * @param augmentedManifests
         *        A list of augmented manifest files that provide training data for your custom model. An augmented
         *        manifest file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.</p>
         *        <p>
         *        This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder augmentedManifests(Collection<AugmentedManifestsListItem> augmentedManifests);

        /**
         * <p>
         * A list of augmented manifest files that provide training data for your custom model. An augmented manifest
         * file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * </p>
         * 
         * @param augmentedManifests
         *        A list of augmented manifest files that provide training data for your custom model. An augmented
         *        manifest file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.</p>
         *        <p>
         *        This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder augmentedManifests(AugmentedManifestsListItem... augmentedManifests);

        /**
         * <p>
         * A list of augmented manifest files that provide training data for your custom model. An augmented manifest
         * file is a labeled dataset that is produced by Amazon SageMaker Ground Truth.
         * </p>
         * <p>
         * This parameter is required if you set <code>DataFormat</code> to <code>AUGMENTED_MANIFEST</code>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.comprehend.model.AugmentedManifestsListItem.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.comprehend.model.AugmentedManifestsListItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.comprehend.model.AugmentedManifestsListItem.Builder#build()} is called
         * immediately and its result is passed to {@link #augmentedManifests(List<AugmentedManifestsListItem>)}.
         * 
         * @param augmentedManifests
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.comprehend.model.AugmentedManifestsListItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #augmentedManifests(java.util.Collection<AugmentedManifestsListItem>)
         */
        Builder augmentedManifests(Consumer<AugmentedManifestsListItem.Builder>... augmentedManifests);

        /**
         * <p>
         * The type of input documents for training the model. Provide plain-text documents to create a plain-text
         * model, and provide semi-structured documents to create a native document model.
         * </p>
         * 
         * @param documentType
         *        The type of input documents for training the model. Provide plain-text documents to create a
         *        plain-text model, and provide semi-structured documents to create a native document model.
         * @see DocumentClassifierDocumentTypeFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentClassifierDocumentTypeFormat
         */
        Builder documentType(String documentType);

        /**
         * <p>
         * The type of input documents for training the model. Provide plain-text documents to create a plain-text
         * model, and provide semi-structured documents to create a native document model.
         * </p>
         * 
         * @param documentType
         *        The type of input documents for training the model. Provide plain-text documents to create a
         *        plain-text model, and provide semi-structured documents to create a native document model.
         * @see DocumentClassifierDocumentTypeFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentClassifierDocumentTypeFormat
         */
        Builder documentType(DocumentClassifierDocumentTypeFormat documentType);

        /**
         * <p>
         * The S3 location of the training documents. This parameter is required in a request to create a native
         * document model.
         * </p>
         * 
         * @param documents
         *        The S3 location of the training documents. This parameter is required in a request to create a native
         *        document model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documents(DocumentClassifierDocuments documents);

        /**
         * <p>
         * The S3 location of the training documents. This parameter is required in a request to create a native
         * document model.
         * </p>
         * This is a convenience method that creates an instance of the {@link DocumentClassifierDocuments.Builder}
         * avoiding the need to create one manually via {@link DocumentClassifierDocuments#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DocumentClassifierDocuments.Builder#build()} is called
         * immediately and its result is passed to {@link #documents(DocumentClassifierDocuments)}.
         * 
         * @param documents
         *        a consumer that will call methods on {@link DocumentClassifierDocuments.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #documents(DocumentClassifierDocuments)
         */
        default Builder documents(Consumer<DocumentClassifierDocuments.Builder> documents) {
            return documents(DocumentClassifierDocuments.builder().applyMutation(documents).build());
        }

        /**
         * Sets the value of the DocumentReaderConfig property for this object.
         *
         * @param documentReaderConfig
         *        The new value for the DocumentReaderConfig property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentReaderConfig(DocumentReaderConfig documentReaderConfig);

        /**
         * Sets the value of the DocumentReaderConfig property for this object.
         *
         * This is a convenience method that creates an instance of the {@link DocumentReaderConfig.Builder} avoiding
         * the need to create one manually via {@link DocumentReaderConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DocumentReaderConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #documentReaderConfig(DocumentReaderConfig)}.
         * 
         * @param documentReaderConfig
         *        a consumer that will call methods on {@link DocumentReaderConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #documentReaderConfig(DocumentReaderConfig)
         */
        default Builder documentReaderConfig(Consumer<DocumentReaderConfig.Builder> documentReaderConfig) {
            return documentReaderConfig(DocumentReaderConfig.builder().applyMutation(documentReaderConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String dataFormat;

        private String s3Uri;

        private String testS3Uri;

        private String labelDelimiter;

        private List<AugmentedManifestsListItem> augmentedManifests = DefaultSdkAutoConstructList.getInstance();

        private String documentType;

        private DocumentClassifierDocuments documents;

        private DocumentReaderConfig documentReaderConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentClassifierInputDataConfig model) {
            dataFormat(model.dataFormat);
            s3Uri(model.s3Uri);
            testS3Uri(model.testS3Uri);
            labelDelimiter(model.labelDelimiter);
            augmentedManifests(model.augmentedManifests);
            documentType(model.documentType);
            documents(model.documents);
            documentReaderConfig(model.documentReaderConfig);
        }

        public final String getDataFormat() {
            return dataFormat;
        }

        public final void setDataFormat(String dataFormat) {
            this.dataFormat = dataFormat;
        }

        @Override
        public final Builder dataFormat(String dataFormat) {
            this.dataFormat = dataFormat;
            return this;
        }

        @Override
        public final Builder dataFormat(DocumentClassifierDataFormat dataFormat) {
            this.dataFormat(dataFormat == null ? null : dataFormat.toString());
            return this;
        }

        public final String getS3Uri() {
            return s3Uri;
        }

        public final void setS3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
        }

        @Override
        public final Builder s3Uri(String s3Uri) {
            this.s3Uri = s3Uri;
            return this;
        }

        public final String getTestS3Uri() {
            return testS3Uri;
        }

        public final void setTestS3Uri(String testS3Uri) {
            this.testS3Uri = testS3Uri;
        }

        @Override
        public final Builder testS3Uri(String testS3Uri) {
            this.testS3Uri = testS3Uri;
            return this;
        }

        public final String getLabelDelimiter() {
            return labelDelimiter;
        }

        public final void setLabelDelimiter(String labelDelimiter) {
            this.labelDelimiter = labelDelimiter;
        }

        @Override
        public final Builder labelDelimiter(String labelDelimiter) {
            this.labelDelimiter = labelDelimiter;
            return this;
        }

        public final List<AugmentedManifestsListItem.Builder> getAugmentedManifests() {
            List<AugmentedManifestsListItem.Builder> result = DocumentClassifierAugmentedManifestsListCopier
                    .copyToBuilder(this.augmentedManifests);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAugmentedManifests(Collection<AugmentedManifestsListItem.BuilderImpl> augmentedManifests) {
            this.augmentedManifests = DocumentClassifierAugmentedManifestsListCopier.copyFromBuilder(augmentedManifests);
        }

        @Override
        public final Builder augmentedManifests(Collection<AugmentedManifestsListItem> augmentedManifests) {
            this.augmentedManifests = DocumentClassifierAugmentedManifestsListCopier.copy(augmentedManifests);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder augmentedManifests(AugmentedManifestsListItem... augmentedManifests) {
            augmentedManifests(Arrays.asList(augmentedManifests));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder augmentedManifests(Consumer<AugmentedManifestsListItem.Builder>... augmentedManifests) {
            augmentedManifests(Stream.of(augmentedManifests)
                    .map(c -> AugmentedManifestsListItem.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getDocumentType() {
            return documentType;
        }

        public final void setDocumentType(String documentType) {
            this.documentType = documentType;
        }

        @Override
        public final Builder documentType(String documentType) {
            this.documentType = documentType;
            return this;
        }

        @Override
        public final Builder documentType(DocumentClassifierDocumentTypeFormat documentType) {
            this.documentType(documentType == null ? null : documentType.toString());
            return this;
        }

        public final DocumentClassifierDocuments.Builder getDocuments() {
            return documents != null ? documents.toBuilder() : null;
        }

        public final void setDocuments(DocumentClassifierDocuments.BuilderImpl documents) {
            this.documents = documents != null ? documents.build() : null;
        }

        @Override
        public final Builder documents(DocumentClassifierDocuments documents) {
            this.documents = documents;
            return this;
        }

        public final DocumentReaderConfig.Builder getDocumentReaderConfig() {
            return documentReaderConfig != null ? documentReaderConfig.toBuilder() : null;
        }

        public final void setDocumentReaderConfig(DocumentReaderConfig.BuilderImpl documentReaderConfig) {
            this.documentReaderConfig = documentReaderConfig != null ? documentReaderConfig.build() : null;
        }

        @Override
        public final Builder documentReaderConfig(DocumentReaderConfig documentReaderConfig) {
            this.documentReaderConfig = documentReaderConfig;
            return this;
        }

        @Override
        public DocumentClassifierInputDataConfig build() {
            return new DocumentClassifierInputDataConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
