/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.config.transform.ConfigSnapshotDeliveryPropertiesMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides options for how often AWS Config delivers configuration snapshots to the Amazon S3 bucket in your delivery
 * channel.
 * </p>
 * <note>
 * <p>
 * If you want to create a rule that triggers evaluations for your resources when AWS Config delivers the configuration
 * snapshot, see the following:
 * </p>
 * </note>
 * <p>
 * The frequency for a rule that triggers evaluations for your resources when AWS Config delivers the configuration
 * snapshot is set by one of two values, depending on which is less frequent:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The value for the <code>deliveryFrequency</code> parameter within the delivery channel configuration, which sets how
 * often AWS Config delivers configuration snapshots. This value also sets how often AWS Config invokes evaluations for
 * Config rules.
 * </p>
 * </li>
 * <li>
 * <p>
 * The value for the <code>MaximumExecutionFrequency</code> parameter, which sets the maximum frequency with which AWS
 * Config invokes evaluations for the rule. For more information, see <a>ConfigRule</a>.
 * </p>
 * </li>
 * </ul>
 * <p>
 * If the <code>deliveryFrequency</code> value is less frequent than the <code>MaximumExecutionFrequency</code> value
 * for a rule, AWS Config invokes the rule only as often as the <code>deliveryFrequency</code> value.
 * </p>
 * <ol>
 * <li>
 * <p>
 * For example, you want your rule to run evaluations when AWS Config delivers the configuration snapshot.
 * </p>
 * </li>
 * <li>
 * <p>
 * You specify the <code>MaximumExecutionFrequency</code> value for <code>Six_Hours</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * You then specify the delivery channel <code>deliveryFrequency</code> value for <code>TwentyFour_Hours</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * Because the value for <code>deliveryFrequency</code> is less frequent than <code>MaximumExecutionFrequency</code>,
 * AWS Config invokes evaluations for the rule every 24 hours.
 * </p>
 * </li>
 * </ol>
 * <p>
 * You should set the <code>MaximumExecutionFrequency</code> value to be at least as frequent as the
 * <code>deliveryFrequency</code> value. You can view the <code>deliveryFrequency</code> value by using the
 * <code>DescribeDeliveryChannnels</code> action.
 * </p>
 * <p>
 * To update the <code>deliveryFrequency</code> with which AWS Config delivers your configuration snapshots, use the
 * <code>PutDeliveryChannel</code> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ConfigSnapshotDeliveryProperties implements StructuredPojo,
        ToCopyableBuilder<ConfigSnapshotDeliveryProperties.Builder, ConfigSnapshotDeliveryProperties> {
    private final String deliveryFrequency;

    private ConfigSnapshotDeliveryProperties(BuilderImpl builder) {
        this.deliveryFrequency = builder.deliveryFrequency;
    }

    /**
     * <p>
     * The frequency with which AWS Config delivers configuration snapshots.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deliveryFrequency}
     * will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deliveryFrequencyAsString}.
     * </p>
     * 
     * @return The frequency with which AWS Config delivers configuration snapshots.
     * @see MaximumExecutionFrequency
     */
    public MaximumExecutionFrequency deliveryFrequency() {
        return MaximumExecutionFrequency.fromValue(deliveryFrequency);
    }

    /**
     * <p>
     * The frequency with which AWS Config delivers configuration snapshots.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #deliveryFrequency}
     * will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #deliveryFrequencyAsString}.
     * </p>
     * 
     * @return The frequency with which AWS Config delivers configuration snapshots.
     * @see MaximumExecutionFrequency
     */
    public String deliveryFrequencyAsString() {
        return deliveryFrequency;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deliveryFrequencyAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfigSnapshotDeliveryProperties)) {
            return false;
        }
        ConfigSnapshotDeliveryProperties other = (ConfigSnapshotDeliveryProperties) obj;
        return Objects.equals(deliveryFrequencyAsString(), other.deliveryFrequencyAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("ConfigSnapshotDeliveryProperties").add("DeliveryFrequency", deliveryFrequencyAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "deliveryFrequency":
            return Optional.of(clazz.cast(deliveryFrequencyAsString()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ConfigSnapshotDeliveryPropertiesMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ConfigSnapshotDeliveryProperties> {
        /**
         * <p>
         * The frequency with which AWS Config delivers configuration snapshots.
         * </p>
         * 
         * @param deliveryFrequency
         *        The frequency with which AWS Config delivers configuration snapshots.
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder deliveryFrequency(String deliveryFrequency);

        /**
         * <p>
         * The frequency with which AWS Config delivers configuration snapshots.
         * </p>
         * 
         * @param deliveryFrequency
         *        The frequency with which AWS Config delivers configuration snapshots.
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder deliveryFrequency(MaximumExecutionFrequency deliveryFrequency);
    }

    static final class BuilderImpl implements Builder {
        private String deliveryFrequency;

        private BuilderImpl() {
        }

        private BuilderImpl(ConfigSnapshotDeliveryProperties model) {
            deliveryFrequency(model.deliveryFrequency);
        }

        public final String getDeliveryFrequency() {
            return deliveryFrequency;
        }

        @Override
        public final Builder deliveryFrequency(String deliveryFrequency) {
            this.deliveryFrequency = deliveryFrequency;
            return this;
        }

        @Override
        public final Builder deliveryFrequency(MaximumExecutionFrequency deliveryFrequency) {
            this.deliveryFrequency(deliveryFrequency.toString());
            return this;
        }

        public final void setDeliveryFrequency(String deliveryFrequency) {
            this.deliveryFrequency = deliveryFrequency;
        }

        @Override
        public ConfigSnapshotDeliveryProperties build() {
            return new ConfigSnapshotDeliveryProperties(this);
        }
    }
}
