/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.config.transform.ScopeMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines which resources trigger an evaluation for an AWS Config rule. The scope can include one or more resource
 * types, a combination of a tag key and value, or a combination of one resource type and one resource ID. Specify a
 * scope to constrain which resources trigger an evaluation for a rule. Otherwise, evaluations for the rule are
 * triggered when any resource in your recording group changes in configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Scope implements StructuredPojo, ToCopyableBuilder<Scope.Builder, Scope> {
    private final List<String> complianceResourceTypes;

    private final String tagKey;

    private final String tagValue;

    private final String complianceResourceId;

    private Scope(BuilderImpl builder) {
        this.complianceResourceTypes = builder.complianceResourceTypes;
        this.tagKey = builder.tagKey;
        this.tagValue = builder.tagValue;
        this.complianceResourceId = builder.complianceResourceId;
    }

    /**
     * <p>
     * The resource types of only those AWS resources that you want to trigger an evaluation for the rule. You can only
     * specify one type if you also specify a resource ID for <code>ComplianceResourceId</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The resource types of only those AWS resources that you want to trigger an evaluation for the rule. You
     *         can only specify one type if you also specify a resource ID for <code>ComplianceResourceId</code>.
     */
    public List<String> complianceResourceTypes() {
        return complianceResourceTypes;
    }

    /**
     * <p>
     * The tag key that is applied to only those AWS resources that you want to trigger an evaluation for the rule.
     * </p>
     * 
     * @return The tag key that is applied to only those AWS resources that you want to trigger an evaluation for the
     *         rule.
     */
    public String tagKey() {
        return tagKey;
    }

    /**
     * <p>
     * The tag value applied to only those AWS resources that you want to trigger an evaluation for the rule. If you
     * specify a value for <code>TagValue</code>, you must also specify a value for <code>TagKey</code>.
     * </p>
     * 
     * @return The tag value applied to only those AWS resources that you want to trigger an evaluation for the rule. If
     *         you specify a value for <code>TagValue</code>, you must also specify a value for <code>TagKey</code>.
     */
    public String tagValue() {
        return tagValue;
    }

    /**
     * <p>
     * The IDs of the only AWS resource that you want to trigger an evaluation for the rule. If you specify a resource
     * ID, you must specify one resource type for <code>ComplianceResourceTypes</code>.
     * </p>
     * 
     * @return The IDs of the only AWS resource that you want to trigger an evaluation for the rule. If you specify a
     *         resource ID, you must specify one resource type for <code>ComplianceResourceTypes</code>.
     */
    public String complianceResourceId() {
        return complianceResourceId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(complianceResourceTypes());
        hashCode = 31 * hashCode + Objects.hashCode(tagKey());
        hashCode = 31 * hashCode + Objects.hashCode(tagValue());
        hashCode = 31 * hashCode + Objects.hashCode(complianceResourceId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Scope)) {
            return false;
        }
        Scope other = (Scope) obj;
        return Objects.equals(complianceResourceTypes(), other.complianceResourceTypes())
                && Objects.equals(tagKey(), other.tagKey()) && Objects.equals(tagValue(), other.tagValue())
                && Objects.equals(complianceResourceId(), other.complianceResourceId());
    }

    @Override
    public String toString() {
        return ToString.builder("Scope").add("ComplianceResourceTypes", complianceResourceTypes()).add("TagKey", tagKey())
                .add("TagValue", tagValue()).add("ComplianceResourceId", complianceResourceId()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ComplianceResourceTypes":
            return Optional.ofNullable(clazz.cast(complianceResourceTypes()));
        case "TagKey":
            return Optional.ofNullable(clazz.cast(tagKey()));
        case "TagValue":
            return Optional.ofNullable(clazz.cast(tagValue()));
        case "ComplianceResourceId":
            return Optional.ofNullable(clazz.cast(complianceResourceId()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ScopeMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Scope> {
        /**
         * <p>
         * The resource types of only those AWS resources that you want to trigger an evaluation for the rule. You can
         * only specify one type if you also specify a resource ID for <code>ComplianceResourceId</code>.
         * </p>
         * 
         * @param complianceResourceTypes
         *        The resource types of only those AWS resources that you want to trigger an evaluation for the rule.
         *        You can only specify one type if you also specify a resource ID for <code>ComplianceResourceId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder complianceResourceTypes(Collection<String> complianceResourceTypes);

        /**
         * <p>
         * The resource types of only those AWS resources that you want to trigger an evaluation for the rule. You can
         * only specify one type if you also specify a resource ID for <code>ComplianceResourceId</code>.
         * </p>
         * 
         * @param complianceResourceTypes
         *        The resource types of only those AWS resources that you want to trigger an evaluation for the rule.
         *        You can only specify one type if you also specify a resource ID for <code>ComplianceResourceId</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder complianceResourceTypes(String... complianceResourceTypes);

        /**
         * <p>
         * The tag key that is applied to only those AWS resources that you want to trigger an evaluation for the rule.
         * </p>
         * 
         * @param tagKey
         *        The tag key that is applied to only those AWS resources that you want to trigger an evaluation for the
         *        rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagKey(String tagKey);

        /**
         * <p>
         * The tag value applied to only those AWS resources that you want to trigger an evaluation for the rule. If you
         * specify a value for <code>TagValue</code>, you must also specify a value for <code>TagKey</code>.
         * </p>
         * 
         * @param tagValue
         *        The tag value applied to only those AWS resources that you want to trigger an evaluation for the rule.
         *        If you specify a value for <code>TagValue</code>, you must also specify a value for
         *        <code>TagKey</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagValue(String tagValue);

        /**
         * <p>
         * The IDs of the only AWS resource that you want to trigger an evaluation for the rule. If you specify a
         * resource ID, you must specify one resource type for <code>ComplianceResourceTypes</code>.
         * </p>
         * 
         * @param complianceResourceId
         *        The IDs of the only AWS resource that you want to trigger an evaluation for the rule. If you specify a
         *        resource ID, you must specify one resource type for <code>ComplianceResourceTypes</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder complianceResourceId(String complianceResourceId);
    }

    static final class BuilderImpl implements Builder {
        private List<String> complianceResourceTypes = DefaultSdkAutoConstructList.getInstance();

        private String tagKey;

        private String tagValue;

        private String complianceResourceId;

        private BuilderImpl() {
        }

        private BuilderImpl(Scope model) {
            complianceResourceTypes(model.complianceResourceTypes);
            tagKey(model.tagKey);
            tagValue(model.tagValue);
            complianceResourceId(model.complianceResourceId);
        }

        public final Collection<String> getComplianceResourceTypes() {
            return complianceResourceTypes;
        }

        @Override
        public final Builder complianceResourceTypes(Collection<String> complianceResourceTypes) {
            this.complianceResourceTypes = ComplianceResourceTypesCopier.copy(complianceResourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder complianceResourceTypes(String... complianceResourceTypes) {
            complianceResourceTypes(Arrays.asList(complianceResourceTypes));
            return this;
        }

        public final void setComplianceResourceTypes(Collection<String> complianceResourceTypes) {
            this.complianceResourceTypes = ComplianceResourceTypesCopier.copy(complianceResourceTypes);
        }

        public final String getTagKey() {
            return tagKey;
        }

        @Override
        public final Builder tagKey(String tagKey) {
            this.tagKey = tagKey;
            return this;
        }

        public final void setTagKey(String tagKey) {
            this.tagKey = tagKey;
        }

        public final String getTagValue() {
            return tagValue;
        }

        @Override
        public final Builder tagValue(String tagValue) {
            this.tagValue = tagValue;
            return this;
        }

        public final void setTagValue(String tagValue) {
            this.tagValue = tagValue;
        }

        public final String getComplianceResourceId() {
            return complianceResourceId;
        }

        @Override
        public final Builder complianceResourceId(String complianceResourceId) {
            this.complianceResourceId = complianceResourceId;
            return this;
        }

        public final void setComplianceResourceId(String complianceResourceId) {
            this.complianceResourceId = complianceResourceId;
        }

        @Override
        public Scope build() {
            return new Scope(this);
        }
    }
}
