/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.config.transform.AggregateEvaluationResultMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details of an AWS Config evaluation for an account ID and region in an aggregator. Provides the AWS resource that
 * was evaluated, the compliance of the resource, related time stamps, and supplementary information.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AggregateEvaluationResult implements StructuredPojo,
        ToCopyableBuilder<AggregateEvaluationResult.Builder, AggregateEvaluationResult> {
    private final EvaluationResultIdentifier evaluationResultIdentifier;

    private final String complianceType;

    private final Instant resultRecordedTime;

    private final Instant configRuleInvokedTime;

    private final String annotation;

    private final String accountId;

    private final String awsRegion;

    private AggregateEvaluationResult(BuilderImpl builder) {
        this.evaluationResultIdentifier = builder.evaluationResultIdentifier;
        this.complianceType = builder.complianceType;
        this.resultRecordedTime = builder.resultRecordedTime;
        this.configRuleInvokedTime = builder.configRuleInvokedTime;
        this.annotation = builder.annotation;
        this.accountId = builder.accountId;
        this.awsRegion = builder.awsRegion;
    }

    /**
     * <p>
     * Uniquely identifies the evaluation result.
     * </p>
     * 
     * @return Uniquely identifies the evaluation result.
     */
    public EvaluationResultIdentifier evaluationResultIdentifier() {
        return evaluationResultIdentifier;
    }

    /**
     * <p>
     * The resource compliance status.
     * </p>
     * <p>
     * For the <code>AggregationEvaluationResult</code> data type, AWS Config supports only the <code>COMPLIANT</code>
     * and <code>NON_COMPLIANT</code>. AWS Config does not support the <code>NOT_APPLICABLE</code> and
     * <code>INSUFFICIENT_DATA</code> value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #complianceType}
     * will return {@link ComplianceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #complianceTypeAsString}.
     * </p>
     * 
     * @return The resource compliance status.</p>
     *         <p>
     *         For the <code>AggregationEvaluationResult</code> data type, AWS Config supports only the
     *         <code>COMPLIANT</code> and <code>NON_COMPLIANT</code>. AWS Config does not support the
     *         <code>NOT_APPLICABLE</code> and <code>INSUFFICIENT_DATA</code> value.
     * @see ComplianceType
     */
    public ComplianceType complianceType() {
        return ComplianceType.fromValue(complianceType);
    }

    /**
     * <p>
     * The resource compliance status.
     * </p>
     * <p>
     * For the <code>AggregationEvaluationResult</code> data type, AWS Config supports only the <code>COMPLIANT</code>
     * and <code>NON_COMPLIANT</code>. AWS Config does not support the <code>NOT_APPLICABLE</code> and
     * <code>INSUFFICIENT_DATA</code> value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #complianceType}
     * will return {@link ComplianceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #complianceTypeAsString}.
     * </p>
     * 
     * @return The resource compliance status.</p>
     *         <p>
     *         For the <code>AggregationEvaluationResult</code> data type, AWS Config supports only the
     *         <code>COMPLIANT</code> and <code>NON_COMPLIANT</code>. AWS Config does not support the
     *         <code>NOT_APPLICABLE</code> and <code>INSUFFICIENT_DATA</code> value.
     * @see ComplianceType
     */
    public String complianceTypeAsString() {
        return complianceType;
    }

    /**
     * <p>
     * The time when AWS Config recorded the aggregate evaluation result.
     * </p>
     * 
     * @return The time when AWS Config recorded the aggregate evaluation result.
     */
    public Instant resultRecordedTime() {
        return resultRecordedTime;
    }

    /**
     * <p>
     * The time when the AWS Config rule evaluated the AWS resource.
     * </p>
     * 
     * @return The time when the AWS Config rule evaluated the AWS resource.
     */
    public Instant configRuleInvokedTime() {
        return configRuleInvokedTime;
    }

    /**
     * <p>
     * Supplementary information about how the agrregate evaluation determined the compliance.
     * </p>
     * 
     * @return Supplementary information about how the agrregate evaluation determined the compliance.
     */
    public String annotation() {
        return annotation;
    }

    /**
     * <p>
     * The 12-digit account ID of the source account.
     * </p>
     * 
     * @return The 12-digit account ID of the source account.
     */
    public String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The source region from where the data is aggregated.
     * </p>
     * 
     * @return The source region from where the data is aggregated.
     */
    public String awsRegion() {
        return awsRegion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(evaluationResultIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(complianceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resultRecordedTime());
        hashCode = 31 * hashCode + Objects.hashCode(configRuleInvokedTime());
        hashCode = 31 * hashCode + Objects.hashCode(annotation());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(awsRegion());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AggregateEvaluationResult)) {
            return false;
        }
        AggregateEvaluationResult other = (AggregateEvaluationResult) obj;
        return Objects.equals(evaluationResultIdentifier(), other.evaluationResultIdentifier())
                && Objects.equals(complianceTypeAsString(), other.complianceTypeAsString())
                && Objects.equals(resultRecordedTime(), other.resultRecordedTime())
                && Objects.equals(configRuleInvokedTime(), other.configRuleInvokedTime())
                && Objects.equals(annotation(), other.annotation()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(awsRegion(), other.awsRegion());
    }

    @Override
    public String toString() {
        return ToString.builder("AggregateEvaluationResult").add("EvaluationResultIdentifier", evaluationResultIdentifier())
                .add("ComplianceType", complianceTypeAsString()).add("ResultRecordedTime", resultRecordedTime())
                .add("ConfigRuleInvokedTime", configRuleInvokedTime()).add("Annotation", annotation())
                .add("AccountId", accountId()).add("AwsRegion", awsRegion()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EvaluationResultIdentifier":
            return Optional.ofNullable(clazz.cast(evaluationResultIdentifier()));
        case "ComplianceType":
            return Optional.ofNullable(clazz.cast(complianceTypeAsString()));
        case "ResultRecordedTime":
            return Optional.ofNullable(clazz.cast(resultRecordedTime()));
        case "ConfigRuleInvokedTime":
            return Optional.ofNullable(clazz.cast(configRuleInvokedTime()));
        case "Annotation":
            return Optional.ofNullable(clazz.cast(annotation()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "AwsRegion":
            return Optional.ofNullable(clazz.cast(awsRegion()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        AggregateEvaluationResultMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, AggregateEvaluationResult> {
        /**
         * <p>
         * Uniquely identifies the evaluation result.
         * </p>
         * 
         * @param evaluationResultIdentifier
         *        Uniquely identifies the evaluation result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationResultIdentifier(EvaluationResultIdentifier evaluationResultIdentifier);

        /**
         * <p>
         * Uniquely identifies the evaluation result.
         * </p>
         * This is a convenience that creates an instance of the {@link EvaluationResultIdentifier.Builder} avoiding the
         * need to create one manually via {@link EvaluationResultIdentifier#builder()}.
         *
         * When the {@link Consumer} completes, {@link EvaluationResultIdentifier.Builder#build()} is called immediately
         * and its result is passed to {@link #evaluationResultIdentifier(EvaluationResultIdentifier)}.
         * 
         * @param evaluationResultIdentifier
         *        a consumer that will call methods on {@link EvaluationResultIdentifier.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #evaluationResultIdentifier(EvaluationResultIdentifier)
         */
        default Builder evaluationResultIdentifier(Consumer<EvaluationResultIdentifier.Builder> evaluationResultIdentifier) {
            return evaluationResultIdentifier(EvaluationResultIdentifier.builder().applyMutation(evaluationResultIdentifier)
                    .build());
        }

        /**
         * <p>
         * The resource compliance status.
         * </p>
         * <p>
         * For the <code>AggregationEvaluationResult</code> data type, AWS Config supports only the
         * <code>COMPLIANT</code> and <code>NON_COMPLIANT</code>. AWS Config does not support the
         * <code>NOT_APPLICABLE</code> and <code>INSUFFICIENT_DATA</code> value.
         * </p>
         * 
         * @param complianceType
         *        The resource compliance status.</p>
         *        <p>
         *        For the <code>AggregationEvaluationResult</code> data type, AWS Config supports only the
         *        <code>COMPLIANT</code> and <code>NON_COMPLIANT</code>. AWS Config does not support the
         *        <code>NOT_APPLICABLE</code> and <code>INSUFFICIENT_DATA</code> value.
         * @see ComplianceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceType
         */
        Builder complianceType(String complianceType);

        /**
         * <p>
         * The resource compliance status.
         * </p>
         * <p>
         * For the <code>AggregationEvaluationResult</code> data type, AWS Config supports only the
         * <code>COMPLIANT</code> and <code>NON_COMPLIANT</code>. AWS Config does not support the
         * <code>NOT_APPLICABLE</code> and <code>INSUFFICIENT_DATA</code> value.
         * </p>
         * 
         * @param complianceType
         *        The resource compliance status.</p>
         *        <p>
         *        For the <code>AggregationEvaluationResult</code> data type, AWS Config supports only the
         *        <code>COMPLIANT</code> and <code>NON_COMPLIANT</code>. AWS Config does not support the
         *        <code>NOT_APPLICABLE</code> and <code>INSUFFICIENT_DATA</code> value.
         * @see ComplianceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComplianceType
         */
        Builder complianceType(ComplianceType complianceType);

        /**
         * <p>
         * The time when AWS Config recorded the aggregate evaluation result.
         * </p>
         * 
         * @param resultRecordedTime
         *        The time when AWS Config recorded the aggregate evaluation result.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resultRecordedTime(Instant resultRecordedTime);

        /**
         * <p>
         * The time when the AWS Config rule evaluated the AWS resource.
         * </p>
         * 
         * @param configRuleInvokedTime
         *        The time when the AWS Config rule evaluated the AWS resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configRuleInvokedTime(Instant configRuleInvokedTime);

        /**
         * <p>
         * Supplementary information about how the agrregate evaluation determined the compliance.
         * </p>
         * 
         * @param annotation
         *        Supplementary information about how the agrregate evaluation determined the compliance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder annotation(String annotation);

        /**
         * <p>
         * The 12-digit account ID of the source account.
         * </p>
         * 
         * @param accountId
         *        The 12-digit account ID of the source account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The source region from where the data is aggregated.
         * </p>
         * 
         * @param awsRegion
         *        The source region from where the data is aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsRegion(String awsRegion);
    }

    static final class BuilderImpl implements Builder {
        private EvaluationResultIdentifier evaluationResultIdentifier;

        private String complianceType;

        private Instant resultRecordedTime;

        private Instant configRuleInvokedTime;

        private String annotation;

        private String accountId;

        private String awsRegion;

        private BuilderImpl() {
        }

        private BuilderImpl(AggregateEvaluationResult model) {
            evaluationResultIdentifier(model.evaluationResultIdentifier);
            complianceType(model.complianceType);
            resultRecordedTime(model.resultRecordedTime);
            configRuleInvokedTime(model.configRuleInvokedTime);
            annotation(model.annotation);
            accountId(model.accountId);
            awsRegion(model.awsRegion);
        }

        public final EvaluationResultIdentifier.Builder getEvaluationResultIdentifier() {
            return evaluationResultIdentifier != null ? evaluationResultIdentifier.toBuilder() : null;
        }

        @Override
        public final Builder evaluationResultIdentifier(EvaluationResultIdentifier evaluationResultIdentifier) {
            this.evaluationResultIdentifier = evaluationResultIdentifier;
            return this;
        }

        public final void setEvaluationResultIdentifier(EvaluationResultIdentifier.BuilderImpl evaluationResultIdentifier) {
            this.evaluationResultIdentifier = evaluationResultIdentifier != null ? evaluationResultIdentifier.build() : null;
        }

        public final String getComplianceType() {
            return complianceType;
        }

        @Override
        public final Builder complianceType(String complianceType) {
            this.complianceType = complianceType;
            return this;
        }

        @Override
        public final Builder complianceType(ComplianceType complianceType) {
            this.complianceType(complianceType.toString());
            return this;
        }

        public final void setComplianceType(String complianceType) {
            this.complianceType = complianceType;
        }

        public final Instant getResultRecordedTime() {
            return resultRecordedTime;
        }

        @Override
        public final Builder resultRecordedTime(Instant resultRecordedTime) {
            this.resultRecordedTime = resultRecordedTime;
            return this;
        }

        public final void setResultRecordedTime(Instant resultRecordedTime) {
            this.resultRecordedTime = resultRecordedTime;
        }

        public final Instant getConfigRuleInvokedTime() {
            return configRuleInvokedTime;
        }

        @Override
        public final Builder configRuleInvokedTime(Instant configRuleInvokedTime) {
            this.configRuleInvokedTime = configRuleInvokedTime;
            return this;
        }

        public final void setConfigRuleInvokedTime(Instant configRuleInvokedTime) {
            this.configRuleInvokedTime = configRuleInvokedTime;
        }

        public final String getAnnotation() {
            return annotation;
        }

        @Override
        public final Builder annotation(String annotation) {
            this.annotation = annotation;
            return this;
        }

        public final void setAnnotation(String annotation) {
            this.annotation = annotation;
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final String getAwsRegion() {
            return awsRegion;
        }

        @Override
        public final Builder awsRegion(String awsRegion) {
            this.awsRegion = awsRegion;
            return this;
        }

        public final void setAwsRegion(String awsRegion) {
            this.awsRegion = awsRegion;
        }

        @Override
        public AggregateEvaluationResult build() {
            return new AggregateEvaluationResult(this);
        }
    }
}
