/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.config.transform.ConfigurationAggregatorMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details about the configuration aggregator, including information about source accounts, regions, and metadata of
 * the aggregator.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfigurationAggregator implements StructuredPojo,
        ToCopyableBuilder<ConfigurationAggregator.Builder, ConfigurationAggregator> {
    private final String configurationAggregatorName;

    private final String configurationAggregatorArn;

    private final List<AccountAggregationSource> accountAggregationSources;

    private final OrganizationAggregationSource organizationAggregationSource;

    private final Instant creationTime;

    private final Instant lastUpdatedTime;

    private ConfigurationAggregator(BuilderImpl builder) {
        this.configurationAggregatorName = builder.configurationAggregatorName;
        this.configurationAggregatorArn = builder.configurationAggregatorArn;
        this.accountAggregationSources = builder.accountAggregationSources;
        this.organizationAggregationSource = builder.organizationAggregationSource;
        this.creationTime = builder.creationTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * The name of the aggregator.
     * </p>
     * 
     * @return The name of the aggregator.
     */
    public String configurationAggregatorName() {
        return configurationAggregatorName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the aggregator.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the aggregator.
     */
    public String configurationAggregatorArn() {
        return configurationAggregatorArn;
    }

    /**
     * <p>
     * Provides a list of source accounts and regions to be aggregated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Provides a list of source accounts and regions to be aggregated.
     */
    public List<AccountAggregationSource> accountAggregationSources() {
        return accountAggregationSources;
    }

    /**
     * <p>
     * Provides an organization and list of regions to be aggregated.
     * </p>
     * 
     * @return Provides an organization and list of regions to be aggregated.
     */
    public OrganizationAggregationSource organizationAggregationSource() {
        return organizationAggregationSource;
    }

    /**
     * <p>
     * The time stamp when the configuration aggregator was created.
     * </p>
     * 
     * @return The time stamp when the configuration aggregator was created.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time of the last update.
     * </p>
     * 
     * @return The time of the last update.
     */
    public Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(configurationAggregatorName());
        hashCode = 31 * hashCode + Objects.hashCode(configurationAggregatorArn());
        hashCode = 31 * hashCode + Objects.hashCode(accountAggregationSources());
        hashCode = 31 * hashCode + Objects.hashCode(organizationAggregationSource());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfigurationAggregator)) {
            return false;
        }
        ConfigurationAggregator other = (ConfigurationAggregator) obj;
        return Objects.equals(configurationAggregatorName(), other.configurationAggregatorName())
                && Objects.equals(configurationAggregatorArn(), other.configurationAggregatorArn())
                && Objects.equals(accountAggregationSources(), other.accountAggregationSources())
                && Objects.equals(organizationAggregationSource(), other.organizationAggregationSource())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    @Override
    public String toString() {
        return ToString.builder("ConfigurationAggregator").add("ConfigurationAggregatorName", configurationAggregatorName())
                .add("ConfigurationAggregatorArn", configurationAggregatorArn())
                .add("AccountAggregationSources", accountAggregationSources())
                .add("OrganizationAggregationSource", organizationAggregationSource()).add("CreationTime", creationTime())
                .add("LastUpdatedTime", lastUpdatedTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConfigurationAggregatorName":
            return Optional.ofNullable(clazz.cast(configurationAggregatorName()));
        case "ConfigurationAggregatorArn":
            return Optional.ofNullable(clazz.cast(configurationAggregatorArn()));
        case "AccountAggregationSources":
            return Optional.ofNullable(clazz.cast(accountAggregationSources()));
        case "OrganizationAggregationSource":
            return Optional.ofNullable(clazz.cast(organizationAggregationSource()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ConfigurationAggregatorMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ConfigurationAggregator> {
        /**
         * <p>
         * The name of the aggregator.
         * </p>
         * 
         * @param configurationAggregatorName
         *        The name of the aggregator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationAggregatorName(String configurationAggregatorName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the aggregator.
         * </p>
         * 
         * @param configurationAggregatorArn
         *        The Amazon Resource Name (ARN) of the aggregator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationAggregatorArn(String configurationAggregatorArn);

        /**
         * <p>
         * Provides a list of source accounts and regions to be aggregated.
         * </p>
         * 
         * @param accountAggregationSources
         *        Provides a list of source accounts and regions to be aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountAggregationSources(Collection<AccountAggregationSource> accountAggregationSources);

        /**
         * <p>
         * Provides a list of source accounts and regions to be aggregated.
         * </p>
         * 
         * @param accountAggregationSources
         *        Provides a list of source accounts and regions to be aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountAggregationSources(AccountAggregationSource... accountAggregationSources);

        /**
         * <p>
         * Provides a list of source accounts and regions to be aggregated.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AccountAggregationSource>.Builder} avoiding
         * the need to create one manually via {@link List<AccountAggregationSource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AccountAggregationSource>.Builder#build()} is called
         * immediately and its result is passed to {@link #accountAggregationSources(List<AccountAggregationSource>)}.
         * 
         * @param accountAggregationSources
         *        a consumer that will call methods on {@link List<AccountAggregationSource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accountAggregationSources(List<AccountAggregationSource>)
         */
        Builder accountAggregationSources(Consumer<AccountAggregationSource.Builder>... accountAggregationSources);

        /**
         * <p>
         * Provides an organization and list of regions to be aggregated.
         * </p>
         * 
         * @param organizationAggregationSource
         *        Provides an organization and list of regions to be aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationAggregationSource(OrganizationAggregationSource organizationAggregationSource);

        /**
         * <p>
         * Provides an organization and list of regions to be aggregated.
         * </p>
         * This is a convenience that creates an instance of the {@link OrganizationAggregationSource.Builder} avoiding
         * the need to create one manually via {@link OrganizationAggregationSource#builder()}.
         *
         * When the {@link Consumer} completes, {@link OrganizationAggregationSource.Builder#build()} is called
         * immediately and its result is passed to {@link #organizationAggregationSource(OrganizationAggregationSource)}
         * .
         * 
         * @param organizationAggregationSource
         *        a consumer that will call methods on {@link OrganizationAggregationSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #organizationAggregationSource(OrganizationAggregationSource)
         */
        default Builder organizationAggregationSource(
                Consumer<OrganizationAggregationSource.Builder> organizationAggregationSource) {
            return organizationAggregationSource(OrganizationAggregationSource.builder()
                    .applyMutation(organizationAggregationSource).build());
        }

        /**
         * <p>
         * The time stamp when the configuration aggregator was created.
         * </p>
         * 
         * @param creationTime
         *        The time stamp when the configuration aggregator was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time of the last update.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The time of the last update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String configurationAggregatorName;

        private String configurationAggregatorArn;

        private List<AccountAggregationSource> accountAggregationSources = DefaultSdkAutoConstructList.getInstance();

        private OrganizationAggregationSource organizationAggregationSource;

        private Instant creationTime;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(ConfigurationAggregator model) {
            configurationAggregatorName(model.configurationAggregatorName);
            configurationAggregatorArn(model.configurationAggregatorArn);
            accountAggregationSources(model.accountAggregationSources);
            organizationAggregationSource(model.organizationAggregationSource);
            creationTime(model.creationTime);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final String getConfigurationAggregatorName() {
            return configurationAggregatorName;
        }

        @Override
        public final Builder configurationAggregatorName(String configurationAggregatorName) {
            this.configurationAggregatorName = configurationAggregatorName;
            return this;
        }

        public final void setConfigurationAggregatorName(String configurationAggregatorName) {
            this.configurationAggregatorName = configurationAggregatorName;
        }

        public final String getConfigurationAggregatorArn() {
            return configurationAggregatorArn;
        }

        @Override
        public final Builder configurationAggregatorArn(String configurationAggregatorArn) {
            this.configurationAggregatorArn = configurationAggregatorArn;
            return this;
        }

        public final void setConfigurationAggregatorArn(String configurationAggregatorArn) {
            this.configurationAggregatorArn = configurationAggregatorArn;
        }

        public final Collection<AccountAggregationSource.Builder> getAccountAggregationSources() {
            return accountAggregationSources != null ? accountAggregationSources.stream()
                    .map(AccountAggregationSource::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder accountAggregationSources(Collection<AccountAggregationSource> accountAggregationSources) {
            this.accountAggregationSources = AccountAggregationSourceListCopier.copy(accountAggregationSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accountAggregationSources(AccountAggregationSource... accountAggregationSources) {
            accountAggregationSources(Arrays.asList(accountAggregationSources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accountAggregationSources(Consumer<AccountAggregationSource.Builder>... accountAggregationSources) {
            accountAggregationSources(Stream.of(accountAggregationSources)
                    .map(c -> AccountAggregationSource.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setAccountAggregationSources(Collection<AccountAggregationSource.BuilderImpl> accountAggregationSources) {
            this.accountAggregationSources = AccountAggregationSourceListCopier.copyFromBuilder(accountAggregationSources);
        }

        public final OrganizationAggregationSource.Builder getOrganizationAggregationSource() {
            return organizationAggregationSource != null ? organizationAggregationSource.toBuilder() : null;
        }

        @Override
        public final Builder organizationAggregationSource(OrganizationAggregationSource organizationAggregationSource) {
            this.organizationAggregationSource = organizationAggregationSource;
            return this;
        }

        public final void setOrganizationAggregationSource(OrganizationAggregationSource.BuilderImpl organizationAggregationSource) {
            this.organizationAggregationSource = organizationAggregationSource != null ? organizationAggregationSource.build()
                    : null;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public ConfigurationAggregator build() {
            return new ConfigurationAggregator(this);
        }
    }
}
