/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.config.transform.SourceMarshaller;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the AWS Config rule owner (AWS or customer), the rule identifier, and the events that trigger the evaluation
 * of your AWS resources.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class Source implements StructuredPojo, ToCopyableBuilder<Source.Builder, Source> {
    private final String owner;

    private final String sourceIdentifier;

    private final List<SourceDetail> sourceDetails;

    private Source(BuilderImpl builder) {
        this.owner = builder.owner;
        this.sourceIdentifier = builder.sourceIdentifier;
        this.sourceDetails = builder.sourceDetails;
    }

    /**
     * <p>
     * Indicates whether AWS or the customer owns and manages the AWS Config rule.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #owner} will return
     * {@link Owner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ownerString}.
     * </p>
     * 
     * @return Indicates whether AWS or the customer owns and manages the AWS Config rule.
     * @see Owner
     */
    public Owner owner() {
        return Owner.fromValue(owner);
    }

    /**
     * <p>
     * Indicates whether AWS or the customer owns and manages the AWS Config rule.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #owner} will return
     * {@link Owner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ownerString}.
     * </p>
     * 
     * @return Indicates whether AWS or the customer owns and manages the AWS Config rule.
     * @see Owner
     */
    public String ownerString() {
        return owner;
    }

    /**
     * <p>
     * For AWS Config managed rules, a predefined identifier from a list. For example, <code>IAM_PASSWORD_POLICY</code>
     * is a managed rule. To reference a managed rule, see <a
     * href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html">Using AWS
     * Managed Config Rules</a>.
     * </p>
     * <p>
     * For custom rules, the identifier is the Amazon Resource Name (ARN) of the rule's AWS Lambda function, such as
     * <code>arn:aws:lambda:us-east-1:123456789012:function:custom_rule_name</code>.
     * </p>
     * 
     * @return For AWS Config managed rules, a predefined identifier from a list. For example,
     *         <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
     *         href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"
     *         >Using AWS Managed Config Rules</a>.</p>
     *         <p>
     *         For custom rules, the identifier is the Amazon Resource Name (ARN) of the rule's AWS Lambda function,
     *         such as <code>arn:aws:lambda:us-east-1:123456789012:function:custom_rule_name</code>.
     */
    public String sourceIdentifier() {
        return sourceIdentifier;
    }

    /**
     * <p>
     * Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
     */
    public List<SourceDetail> sourceDetails() {
        return sourceDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + ((ownerString() == null) ? 0 : ownerString().hashCode());
        hashCode = 31 * hashCode + ((sourceIdentifier() == null) ? 0 : sourceIdentifier().hashCode());
        hashCode = 31 * hashCode + ((sourceDetails() == null) ? 0 : sourceDetails().hashCode());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Source)) {
            return false;
        }
        Source other = (Source) obj;
        if (other.ownerString() == null ^ this.ownerString() == null) {
            return false;
        }
        if (other.ownerString() != null && !other.ownerString().equals(this.ownerString())) {
            return false;
        }
        if (other.sourceIdentifier() == null ^ this.sourceIdentifier() == null) {
            return false;
        }
        if (other.sourceIdentifier() != null && !other.sourceIdentifier().equals(this.sourceIdentifier())) {
            return false;
        }
        if (other.sourceDetails() == null ^ this.sourceDetails() == null) {
            return false;
        }
        if (other.sourceDetails() != null && !other.sourceDetails().equals(this.sourceDetails())) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (ownerString() != null) {
            sb.append("Owner: ").append(ownerString()).append(",");
        }
        if (sourceIdentifier() != null) {
            sb.append("SourceIdentifier: ").append(sourceIdentifier()).append(",");
        }
        if (sourceDetails() != null) {
            sb.append("SourceDetails: ").append(sourceDetails()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Owner":
            return Optional.of(clazz.cast(ownerString()));
        case "SourceIdentifier":
            return Optional.of(clazz.cast(sourceIdentifier()));
        case "SourceDetails":
            return Optional.of(clazz.cast(sourceDetails()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        SourceMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Source> {
        /**
         * <p>
         * Indicates whether AWS or the customer owns and manages the AWS Config rule.
         * </p>
         * 
         * @param owner
         *        Indicates whether AWS or the customer owns and manages the AWS Config rule.
         * @see Owner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Owner
         */
        Builder owner(String owner);

        /**
         * <p>
         * Indicates whether AWS or the customer owns and manages the AWS Config rule.
         * </p>
         * 
         * @param owner
         *        Indicates whether AWS or the customer owns and manages the AWS Config rule.
         * @see Owner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Owner
         */
        Builder owner(Owner owner);

        /**
         * <p>
         * For AWS Config managed rules, a predefined identifier from a list. For example,
         * <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
         * href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html">Using
         * AWS Managed Config Rules</a>.
         * </p>
         * <p>
         * For custom rules, the identifier is the Amazon Resource Name (ARN) of the rule's AWS Lambda function, such as
         * <code>arn:aws:lambda:us-east-1:123456789012:function:custom_rule_name</code>.
         * </p>
         * 
         * @param sourceIdentifier
         *        For AWS Config managed rules, a predefined identifier from a list. For example,
         *        <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
         *        href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"
         *        >Using AWS Managed Config Rules</a>.</p>
         *        <p>
         *        For custom rules, the identifier is the Amazon Resource Name (ARN) of the rule's AWS Lambda function,
         *        such as <code>arn:aws:lambda:us-east-1:123456789012:function:custom_rule_name</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIdentifier(String sourceIdentifier);

        /**
         * <p>
         * Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * </p>
         * 
         * @param sourceDetails
         *        Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDetails(Collection<SourceDetail> sourceDetails);

        /**
         * <p>
         * Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * </p>
         * 
         * @param sourceDetails
         *        Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDetails(SourceDetail... sourceDetails);
    }

    static final class BuilderImpl implements Builder {
        private String owner;

        private String sourceIdentifier;

        private List<SourceDetail> sourceDetails;

        private BuilderImpl() {
        }

        private BuilderImpl(Source model) {
            owner(model.owner);
            sourceIdentifier(model.sourceIdentifier);
            sourceDetails(model.sourceDetails);
        }

        public final String getOwner() {
            return owner;
        }

        @Override
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        @Override
        public final Builder owner(Owner owner) {
            this.owner(owner.toString());
            return this;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        public final String getSourceIdentifier() {
            return sourceIdentifier;
        }

        @Override
        public final Builder sourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
            return this;
        }

        public final void setSourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
        }

        public final Collection<SourceDetail.Builder> getSourceDetails() {
            return sourceDetails != null ? sourceDetails.stream().map(SourceDetail::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder sourceDetails(Collection<SourceDetail> sourceDetails) {
            this.sourceDetails = SourceDetailsCopier.copy(sourceDetails);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceDetails(SourceDetail... sourceDetails) {
            sourceDetails(Arrays.asList(sourceDetails));
            return this;
        }

        public final void setSourceDetails(Collection<SourceDetail.BuilderImpl> sourceDetails) {
            this.sourceDetails = SourceDetailsCopier.copyFromBuilder(sourceDetails);
        }

        @Override
        public Source build() {
            return new Source(this);
        }
    }
}
