/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.config.transform.ConfigurationItemMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list that contains detailed configurations of a specified resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ConfigurationItem implements StructuredPojo, ToCopyableBuilder<ConfigurationItem.Builder, ConfigurationItem> {
    private final String version;

    private final String accountId;

    private final Instant configurationItemCaptureTime;

    private final String configurationItemStatus;

    private final String configurationStateId;

    private final String configurationItemMD5Hash;

    private final String arn;

    private final String resourceType;

    private final String resourceId;

    private final String resourceName;

    private final String awsRegion;

    private final String availabilityZone;

    private final Instant resourceCreationTime;

    private final Map<String, String> tags;

    private final List<String> relatedEvents;

    private final List<Relationship> relationships;

    private final String configuration;

    private final Map<String, String> supplementaryConfiguration;

    private ConfigurationItem(BuilderImpl builder) {
        this.version = builder.version;
        this.accountId = builder.accountId;
        this.configurationItemCaptureTime = builder.configurationItemCaptureTime;
        this.configurationItemStatus = builder.configurationItemStatus;
        this.configurationStateId = builder.configurationStateId;
        this.configurationItemMD5Hash = builder.configurationItemMD5Hash;
        this.arn = builder.arn;
        this.resourceType = builder.resourceType;
        this.resourceId = builder.resourceId;
        this.resourceName = builder.resourceName;
        this.awsRegion = builder.awsRegion;
        this.availabilityZone = builder.availabilityZone;
        this.resourceCreationTime = builder.resourceCreationTime;
        this.tags = builder.tags;
        this.relatedEvents = builder.relatedEvents;
        this.relationships = builder.relationships;
        this.configuration = builder.configuration;
        this.supplementaryConfiguration = builder.supplementaryConfiguration;
    }

    /**
     * <p>
     * The version number of the resource configuration.
     * </p>
     * 
     * @return The version number of the resource configuration.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * The 12 digit AWS account ID associated with the resource.
     * </p>
     * 
     * @return The 12 digit AWS account ID associated with the resource.
     */
    public String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The time when the configuration recording was initiated.
     * </p>
     * 
     * @return The time when the configuration recording was initiated.
     */
    public Instant configurationItemCaptureTime() {
        return configurationItemCaptureTime;
    }

    /**
     * <p>
     * The configuration item status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #configurationItemStatus} will return {@link ConfigurationItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #configurationItemStatusString}.
     * </p>
     * 
     * @return The configuration item status.
     * @see ConfigurationItemStatus
     */
    public ConfigurationItemStatus configurationItemStatus() {
        return ConfigurationItemStatus.fromValue(configurationItemStatus);
    }

    /**
     * <p>
     * The configuration item status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #configurationItemStatus} will return {@link ConfigurationItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #configurationItemStatusString}.
     * </p>
     * 
     * @return The configuration item status.
     * @see ConfigurationItemStatus
     */
    public String configurationItemStatusString() {
        return configurationItemStatus;
    }

    /**
     * <p>
     * An identifier that indicates the ordering of the configuration items of a resource.
     * </p>
     * 
     * @return An identifier that indicates the ordering of the configuration items of a resource.
     */
    public String configurationStateId() {
        return configurationStateId;
    }

    /**
     * <p>
     * Unique MD5 hash that represents the configuration item's state.
     * </p>
     * <p>
     * You can use MD5 hash to compare the states of two or more configuration items that are associated with the same
     * resource.
     * </p>
     * 
     * @return Unique MD5 hash that represents the configuration item's state.</p>
     *         <p>
     *         You can use MD5 hash to compare the states of two or more configuration items that are associated with
     *         the same resource.
     */
    public String configurationItemMD5Hash() {
        return configurationItemMD5Hash;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the resource.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The type of AWS resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeString}.
     * </p>
     * 
     * @return The type of AWS resource.
     * @see ResourceType
     */
    public ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of AWS resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeString}.
     * </p>
     * 
     * @return The type of AWS resource.
     * @see ResourceType
     */
    public String resourceTypeString() {
        return resourceType;
    }

    /**
     * <p>
     * The ID of the resource (for example., <code>sg-xxxxxx</code>).
     * </p>
     * 
     * @return The ID of the resource (for example., <code>sg-xxxxxx</code>).
     */
    public String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The custom name of the resource, if available.
     * </p>
     * 
     * @return The custom name of the resource, if available.
     */
    public String resourceName() {
        return resourceName;
    }

    /**
     * <p>
     * The region where the resource resides.
     * </p>
     * 
     * @return The region where the resource resides.
     */
    public String awsRegion() {
        return awsRegion;
    }

    /**
     * <p>
     * The Availability Zone associated with the resource.
     * </p>
     * 
     * @return The Availability Zone associated with the resource.
     */
    public String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The time stamp when the resource was created.
     * </p>
     * 
     * @return The time stamp when the resource was created.
     */
    public Instant resourceCreationTime() {
        return resourceCreationTime;
    }

    /**
     * <p>
     * A mapping of key value tags associated with the resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A mapping of key value tags associated with the resource.
     */
    public Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * A list of CloudTrail event IDs.
     * </p>
     * <p>
     * A populated field indicates that the current configuration was initiated by the events recorded in the CloudTrail
     * log. For more information about CloudTrail, see <a
     * href="http://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html">What is AWS
     * CloudTrail?</a>.
     * </p>
     * <p>
     * An empty field indicates that the current configuration was not initiated by any event.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of CloudTrail event IDs.</p>
     *         <p>
     *         A populated field indicates that the current configuration was initiated by the events recorded in the
     *         CloudTrail log. For more information about CloudTrail, see <a
     *         href="http://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html">What
     *         is AWS CloudTrail?</a>.
     *         </p>
     *         <p>
     *         An empty field indicates that the current configuration was not initiated by any event.
     */
    public List<String> relatedEvents() {
        return relatedEvents;
    }

    /**
     * <p>
     * A list of related AWS resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of related AWS resources.
     */
    public List<Relationship> relationships() {
        return relationships;
    }

    /**
     * <p>
     * The description of the resource configuration.
     * </p>
     * 
     * @return The description of the resource configuration.
     */
    public String configuration() {
        return configuration;
    }

    /**
     * <p>
     * Configuration attributes that AWS Config returns for certain resource types to supplement the information
     * returned for the <code>configuration</code> parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Configuration attributes that AWS Config returns for certain resource types to supplement the information
     *         returned for the <code>configuration</code> parameter.
     */
    public Map<String, String> supplementaryConfiguration() {
        return supplementaryConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(configurationItemCaptureTime());
        hashCode = 31 * hashCode + Objects.hashCode(configurationItemStatusString());
        hashCode = 31 * hashCode + Objects.hashCode(configurationStateId());
        hashCode = 31 * hashCode + Objects.hashCode(configurationItemMD5Hash());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceName());
        hashCode = 31 * hashCode + Objects.hashCode(awsRegion());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(resourceCreationTime());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(relatedEvents());
        hashCode = 31 * hashCode + Objects.hashCode(relationships());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(supplementaryConfiguration());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfigurationItem)) {
            return false;
        }
        ConfigurationItem other = (ConfigurationItem) obj;
        return Objects.equals(version(), other.version()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(configurationItemCaptureTime(), other.configurationItemCaptureTime())
                && Objects.equals(configurationItemStatusString(), other.configurationItemStatusString())
                && Objects.equals(configurationStateId(), other.configurationStateId())
                && Objects.equals(configurationItemMD5Hash(), other.configurationItemMD5Hash())
                && Objects.equals(arn(), other.arn()) && Objects.equals(resourceTypeString(), other.resourceTypeString())
                && Objects.equals(resourceId(), other.resourceId()) && Objects.equals(resourceName(), other.resourceName())
                && Objects.equals(awsRegion(), other.awsRegion()) && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(resourceCreationTime(), other.resourceCreationTime()) && Objects.equals(tags(), other.tags())
                && Objects.equals(relatedEvents(), other.relatedEvents())
                && Objects.equals(relationships(), other.relationships())
                && Objects.equals(configuration(), other.configuration())
                && Objects.equals(supplementaryConfiguration(), other.supplementaryConfiguration());
    }

    @Override
    public String toString() {
        return ToString.builder("ConfigurationItem").add("Version", version()).add("AccountId", accountId())
                .add("ConfigurationItemCaptureTime", configurationItemCaptureTime())
                .add("ConfigurationItemStatus", configurationItemStatusString())
                .add("ConfigurationStateId", configurationStateId()).add("ConfigurationItemMD5Hash", configurationItemMD5Hash())
                .add("Arn", arn()).add("ResourceType", resourceTypeString()).add("ResourceId", resourceId())
                .add("ResourceName", resourceName()).add("AwsRegion", awsRegion()).add("AvailabilityZone", availabilityZone())
                .add("ResourceCreationTime", resourceCreationTime()).add("Tags", tags()).add("RelatedEvents", relatedEvents())
                .add("Relationships", relationships()).add("Configuration", configuration())
                .add("SupplementaryConfiguration", supplementaryConfiguration()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "version":
            return Optional.of(clazz.cast(version()));
        case "accountId":
            return Optional.of(clazz.cast(accountId()));
        case "configurationItemCaptureTime":
            return Optional.of(clazz.cast(configurationItemCaptureTime()));
        case "configurationItemStatus":
            return Optional.of(clazz.cast(configurationItemStatusString()));
        case "configurationStateId":
            return Optional.of(clazz.cast(configurationStateId()));
        case "configurationItemMD5Hash":
            return Optional.of(clazz.cast(configurationItemMD5Hash()));
        case "arn":
            return Optional.of(clazz.cast(arn()));
        case "resourceType":
            return Optional.of(clazz.cast(resourceTypeString()));
        case "resourceId":
            return Optional.of(clazz.cast(resourceId()));
        case "resourceName":
            return Optional.of(clazz.cast(resourceName()));
        case "awsRegion":
            return Optional.of(clazz.cast(awsRegion()));
        case "availabilityZone":
            return Optional.of(clazz.cast(availabilityZone()));
        case "resourceCreationTime":
            return Optional.of(clazz.cast(resourceCreationTime()));
        case "tags":
            return Optional.of(clazz.cast(tags()));
        case "relatedEvents":
            return Optional.of(clazz.cast(relatedEvents()));
        case "relationships":
            return Optional.of(clazz.cast(relationships()));
        case "configuration":
            return Optional.of(clazz.cast(configuration()));
        case "supplementaryConfiguration":
            return Optional.of(clazz.cast(supplementaryConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ConfigurationItemMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ConfigurationItem> {
        /**
         * <p>
         * The version number of the resource configuration.
         * </p>
         * 
         * @param version
         *        The version number of the resource configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The 12 digit AWS account ID associated with the resource.
         * </p>
         * 
         * @param accountId
         *        The 12 digit AWS account ID associated with the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The time when the configuration recording was initiated.
         * </p>
         * 
         * @param configurationItemCaptureTime
         *        The time when the configuration recording was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationItemCaptureTime(Instant configurationItemCaptureTime);

        /**
         * <p>
         * The configuration item status.
         * </p>
         * 
         * @param configurationItemStatus
         *        The configuration item status.
         * @see ConfigurationItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationItemStatus
         */
        Builder configurationItemStatus(String configurationItemStatus);

        /**
         * <p>
         * The configuration item status.
         * </p>
         * 
         * @param configurationItemStatus
         *        The configuration item status.
         * @see ConfigurationItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationItemStatus
         */
        Builder configurationItemStatus(ConfigurationItemStatus configurationItemStatus);

        /**
         * <p>
         * An identifier that indicates the ordering of the configuration items of a resource.
         * </p>
         * 
         * @param configurationStateId
         *        An identifier that indicates the ordering of the configuration items of a resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationStateId(String configurationStateId);

        /**
         * <p>
         * Unique MD5 hash that represents the configuration item's state.
         * </p>
         * <p>
         * You can use MD5 hash to compare the states of two or more configuration items that are associated with the
         * same resource.
         * </p>
         * 
         * @param configurationItemMD5Hash
         *        Unique MD5 hash that represents the configuration item's state.</p>
         *        <p>
         *        You can use MD5 hash to compare the states of two or more configuration items that are associated with
         *        the same resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationItemMD5Hash(String configurationItemMD5Hash);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the resource.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The type of AWS resource.
         * </p>
         * 
         * @param resourceType
         *        The type of AWS resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of AWS resource.
         * </p>
         * 
         * @param resourceType
         *        The type of AWS resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The ID of the resource (for example., <code>sg-xxxxxx</code>).
         * </p>
         * 
         * @param resourceId
         *        The ID of the resource (for example., <code>sg-xxxxxx</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The custom name of the resource, if available.
         * </p>
         * 
         * @param resourceName
         *        The custom name of the resource, if available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceName(String resourceName);

        /**
         * <p>
         * The region where the resource resides.
         * </p>
         * 
         * @param awsRegion
         *        The region where the resource resides.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsRegion(String awsRegion);

        /**
         * <p>
         * The Availability Zone associated with the resource.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone associated with the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The time stamp when the resource was created.
         * </p>
         * 
         * @param resourceCreationTime
         *        The time stamp when the resource was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceCreationTime(Instant resourceCreationTime);

        /**
         * <p>
         * A mapping of key value tags associated with the resource.
         * </p>
         * 
         * @param tags
         *        A mapping of key value tags associated with the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * A list of CloudTrail event IDs.
         * </p>
         * <p>
         * A populated field indicates that the current configuration was initiated by the events recorded in the
         * CloudTrail log. For more information about CloudTrail, see <a
         * href="http://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html">What is
         * AWS CloudTrail?</a>.
         * </p>
         * <p>
         * An empty field indicates that the current configuration was not initiated by any event.
         * </p>
         * 
         * @param relatedEvents
         *        A list of CloudTrail event IDs.</p>
         *        <p>
         *        A populated field indicates that the current configuration was initiated by the events recorded in the
         *        CloudTrail log. For more information about CloudTrail, see <a
         *        href="http://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html"
         *        >What is AWS CloudTrail?</a>.
         *        </p>
         *        <p>
         *        An empty field indicates that the current configuration was not initiated by any event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedEvents(Collection<String> relatedEvents);

        /**
         * <p>
         * A list of CloudTrail event IDs.
         * </p>
         * <p>
         * A populated field indicates that the current configuration was initiated by the events recorded in the
         * CloudTrail log. For more information about CloudTrail, see <a
         * href="http://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html">What is
         * AWS CloudTrail?</a>.
         * </p>
         * <p>
         * An empty field indicates that the current configuration was not initiated by any event.
         * </p>
         * 
         * @param relatedEvents
         *        A list of CloudTrail event IDs.</p>
         *        <p>
         *        A populated field indicates that the current configuration was initiated by the events recorded in the
         *        CloudTrail log. For more information about CloudTrail, see <a
         *        href="http://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html"
         *        >What is AWS CloudTrail?</a>.
         *        </p>
         *        <p>
         *        An empty field indicates that the current configuration was not initiated by any event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedEvents(String... relatedEvents);

        /**
         * <p>
         * A list of related AWS resources.
         * </p>
         * 
         * @param relationships
         *        A list of related AWS resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Collection<Relationship> relationships);

        /**
         * <p>
         * A list of related AWS resources.
         * </p>
         * 
         * @param relationships
         *        A list of related AWS resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Relationship... relationships);

        /**
         * <p>
         * The description of the resource configuration.
         * </p>
         * 
         * @param configuration
         *        The description of the resource configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(String configuration);

        /**
         * <p>
         * Configuration attributes that AWS Config returns for certain resource types to supplement the information
         * returned for the <code>configuration</code> parameter.
         * </p>
         * 
         * @param supplementaryConfiguration
         *        Configuration attributes that AWS Config returns for certain resource types to supplement the
         *        information returned for the <code>configuration</code> parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supplementaryConfiguration(Map<String, String> supplementaryConfiguration);
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private String accountId;

        private Instant configurationItemCaptureTime;

        private String configurationItemStatus;

        private String configurationStateId;

        private String configurationItemMD5Hash;

        private String arn;

        private String resourceType;

        private String resourceId;

        private String resourceName;

        private String awsRegion;

        private String availabilityZone;

        private Instant resourceCreationTime;

        private Map<String, String> tags;

        private List<String> relatedEvents;

        private List<Relationship> relationships;

        private String configuration;

        private Map<String, String> supplementaryConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(ConfigurationItem model) {
            version(model.version);
            accountId(model.accountId);
            configurationItemCaptureTime(model.configurationItemCaptureTime);
            configurationItemStatus(model.configurationItemStatus);
            configurationStateId(model.configurationStateId);
            configurationItemMD5Hash(model.configurationItemMD5Hash);
            arn(model.arn);
            resourceType(model.resourceType);
            resourceId(model.resourceId);
            resourceName(model.resourceName);
            awsRegion(model.awsRegion);
            availabilityZone(model.availabilityZone);
            resourceCreationTime(model.resourceCreationTime);
            tags(model.tags);
            relatedEvents(model.relatedEvents);
            relationships(model.relationships);
            configuration(model.configuration);
            supplementaryConfiguration(model.supplementaryConfiguration);
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final Instant getConfigurationItemCaptureTime() {
            return configurationItemCaptureTime;
        }

        @Override
        public final Builder configurationItemCaptureTime(Instant configurationItemCaptureTime) {
            this.configurationItemCaptureTime = configurationItemCaptureTime;
            return this;
        }

        public final void setConfigurationItemCaptureTime(Instant configurationItemCaptureTime) {
            this.configurationItemCaptureTime = configurationItemCaptureTime;
        }

        public final String getConfigurationItemStatus() {
            return configurationItemStatus;
        }

        @Override
        public final Builder configurationItemStatus(String configurationItemStatus) {
            this.configurationItemStatus = configurationItemStatus;
            return this;
        }

        @Override
        public final Builder configurationItemStatus(ConfigurationItemStatus configurationItemStatus) {
            this.configurationItemStatus(configurationItemStatus.toString());
            return this;
        }

        public final void setConfigurationItemStatus(String configurationItemStatus) {
            this.configurationItemStatus = configurationItemStatus;
        }

        public final String getConfigurationStateId() {
            return configurationStateId;
        }

        @Override
        public final Builder configurationStateId(String configurationStateId) {
            this.configurationStateId = configurationStateId;
            return this;
        }

        public final void setConfigurationStateId(String configurationStateId) {
            this.configurationStateId = configurationStateId;
        }

        public final String getConfigurationItemMD5Hash() {
            return configurationItemMD5Hash;
        }

        @Override
        public final Builder configurationItemMD5Hash(String configurationItemMD5Hash) {
            this.configurationItemMD5Hash = configurationItemMD5Hash;
            return this;
        }

        public final void setConfigurationItemMD5Hash(String configurationItemMD5Hash) {
            this.configurationItemMD5Hash = configurationItemMD5Hash;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType.toString());
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final String getResourceId() {
            return resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        public final String getResourceName() {
            return resourceName;
        }

        @Override
        public final Builder resourceName(String resourceName) {
            this.resourceName = resourceName;
            return this;
        }

        public final void setResourceName(String resourceName) {
            this.resourceName = resourceName;
        }

        public final String getAwsRegion() {
            return awsRegion;
        }

        @Override
        public final Builder awsRegion(String awsRegion) {
            this.awsRegion = awsRegion;
            return this;
        }

        public final void setAwsRegion(String awsRegion) {
            this.awsRegion = awsRegion;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        public final Instant getResourceCreationTime() {
            return resourceCreationTime;
        }

        @Override
        public final Builder resourceCreationTime(Instant resourceCreationTime) {
            this.resourceCreationTime = resourceCreationTime;
            return this;
        }

        public final void setResourceCreationTime(Instant resourceCreationTime) {
            this.resourceCreationTime = resourceCreationTime;
        }

        public final Map<String, String> getTags() {
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        public final Collection<String> getRelatedEvents() {
            return relatedEvents;
        }

        @Override
        public final Builder relatedEvents(Collection<String> relatedEvents) {
            this.relatedEvents = RelatedEventListCopier.copy(relatedEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedEvents(String... relatedEvents) {
            relatedEvents(Arrays.asList(relatedEvents));
            return this;
        }

        public final void setRelatedEvents(Collection<String> relatedEvents) {
            this.relatedEvents = RelatedEventListCopier.copy(relatedEvents);
        }

        public final Collection<Relationship.Builder> getRelationships() {
            return relationships != null ? relationships.stream().map(Relationship::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder relationships(Collection<Relationship> relationships) {
            this.relationships = RelationshipListCopier.copy(relationships);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(Relationship... relationships) {
            relationships(Arrays.asList(relationships));
            return this;
        }

        public final void setRelationships(Collection<Relationship.BuilderImpl> relationships) {
            this.relationships = RelationshipListCopier.copyFromBuilder(relationships);
        }

        public final String getConfiguration() {
            return configuration;
        }

        @Override
        public final Builder configuration(String configuration) {
            this.configuration = configuration;
            return this;
        }

        public final void setConfiguration(String configuration) {
            this.configuration = configuration;
        }

        public final Map<String, String> getSupplementaryConfiguration() {
            return supplementaryConfiguration;
        }

        @Override
        public final Builder supplementaryConfiguration(Map<String, String> supplementaryConfiguration) {
            this.supplementaryConfiguration = SupplementaryConfigurationCopier.copy(supplementaryConfiguration);
            return this;
        }

        public final void setSupplementaryConfiguration(Map<String, String> supplementaryConfiguration) {
            this.supplementaryConfiguration = SupplementaryConfigurationCopier.copy(supplementaryConfiguration);
        }

        @Override
        public ConfigurationItem build() {
            return new ConfigurationItem(this);
        }
    }
}
