/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that specifies organization custom rule metadata such as resource type, resource ID of AWS resource, Lamdba
 * function ARN, and organization trigger types that trigger AWS Config to evaluate your AWS resources against a rule.
 * It also provides the frequency with which you want AWS Config to run evaluations for the rule if the trigger type is
 * periodic.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OrganizationCustomRuleMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<OrganizationCustomRuleMetadata.Builder, OrganizationCustomRuleMetadata> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(OrganizationCustomRuleMetadata::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> LAMBDA_FUNCTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LambdaFunctionArn").getter(getter(OrganizationCustomRuleMetadata::lambdaFunctionArn))
            .setter(setter(Builder::lambdaFunctionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LambdaFunctionArn").build()).build();

    private static final SdkField<List<String>> ORGANIZATION_CONFIG_RULE_TRIGGER_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("OrganizationConfigRuleTriggerTypes")
            .getter(getter(OrganizationCustomRuleMetadata::organizationConfigRuleTriggerTypesAsStrings))
            .setter(setter(Builder::organizationConfigRuleTriggerTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrganizationConfigRuleTriggerTypes")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INPUT_PARAMETERS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputParameters").getter(getter(OrganizationCustomRuleMetadata::inputParameters))
            .setter(setter(Builder::inputParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputParameters").build()).build();

    private static final SdkField<String> MAXIMUM_EXECUTION_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaximumExecutionFrequency")
            .getter(getter(OrganizationCustomRuleMetadata::maximumExecutionFrequencyAsString))
            .setter(setter(Builder::maximumExecutionFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumExecutionFrequency").build())
            .build();

    private static final SdkField<List<String>> RESOURCE_TYPES_SCOPE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceTypesScope")
            .getter(getter(OrganizationCustomRuleMetadata::resourceTypesScope))
            .setter(setter(Builder::resourceTypesScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypesScope").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_ID_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceIdScope").getter(getter(OrganizationCustomRuleMetadata::resourceIdScope))
            .setter(setter(Builder::resourceIdScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdScope").build()).build();

    private static final SdkField<String> TAG_KEY_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TagKeyScope").getter(getter(OrganizationCustomRuleMetadata::tagKeyScope))
            .setter(setter(Builder::tagKeyScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagKeyScope").build()).build();

    private static final SdkField<String> TAG_VALUE_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TagValueScope").getter(getter(OrganizationCustomRuleMetadata::tagValueScope))
            .setter(setter(Builder::tagValueScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagValueScope").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            LAMBDA_FUNCTION_ARN_FIELD, ORGANIZATION_CONFIG_RULE_TRIGGER_TYPES_FIELD, INPUT_PARAMETERS_FIELD,
            MAXIMUM_EXECUTION_FREQUENCY_FIELD, RESOURCE_TYPES_SCOPE_FIELD, RESOURCE_ID_SCOPE_FIELD, TAG_KEY_SCOPE_FIELD,
            TAG_VALUE_SCOPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String description;

    private final String lambdaFunctionArn;

    private final List<String> organizationConfigRuleTriggerTypes;

    private final String inputParameters;

    private final String maximumExecutionFrequency;

    private final List<String> resourceTypesScope;

    private final String resourceIdScope;

    private final String tagKeyScope;

    private final String tagValueScope;

    private OrganizationCustomRuleMetadata(BuilderImpl builder) {
        this.description = builder.description;
        this.lambdaFunctionArn = builder.lambdaFunctionArn;
        this.organizationConfigRuleTriggerTypes = builder.organizationConfigRuleTriggerTypes;
        this.inputParameters = builder.inputParameters;
        this.maximumExecutionFrequency = builder.maximumExecutionFrequency;
        this.resourceTypesScope = builder.resourceTypesScope;
        this.resourceIdScope = builder.resourceIdScope;
        this.tagKeyScope = builder.tagKeyScope;
        this.tagValueScope = builder.tagValueScope;
    }

    /**
     * <p>
     * The description that you provide for organization config rule.
     * </p>
     * 
     * @return The description that you provide for organization config rule.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The lambda function ARN.
     * </p>
     * 
     * @return The lambda function ARN.
     */
    public String lambdaFunctionArn() {
        return lambdaFunctionArn;
    }

    /**
     * <p>
     * The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the following
     * notification types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
     * configuration item as a result of a resource change.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers an
     * oversized configuration item. AWS Config may generate this notification type when a resource changes and the
     * notification exceeds the maximum size allowed by Amazon SNS.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
     * <code>MaximumExecutionFrequency</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOrganizationConfigRuleTriggerTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
     *         following notification types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
     *         configuration item as a result of a resource change.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config
     *         delivers an oversized configuration item. AWS Config may generate this notification type when a resource
     *         changes and the notification exceeds the maximum size allowed by Amazon SNS.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
     *         <code>MaximumExecutionFrequency</code>.
     *         </p>
     *         </li>
     */
    public List<OrganizationConfigRuleTriggerType> organizationConfigRuleTriggerTypes() {
        return OrganizationConfigRuleTriggerTypesCopier.copyStringToEnum(organizationConfigRuleTriggerTypes);
    }

    /**
     * Returns true if the OrganizationConfigRuleTriggerTypes property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public boolean hasOrganizationConfigRuleTriggerTypes() {
        return organizationConfigRuleTriggerTypes != null
                && !(organizationConfigRuleTriggerTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the following
     * notification types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
     * configuration item as a result of a resource change.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers an
     * oversized configuration item. AWS Config may generate this notification type when a resource changes and the
     * notification exceeds the maximum size allowed by Amazon SNS.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
     * <code>MaximumExecutionFrequency</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasOrganizationConfigRuleTriggerTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
     *         following notification types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
     *         configuration item as a result of a resource change.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config
     *         delivers an oversized configuration item. AWS Config may generate this notification type when a resource
     *         changes and the notification exceeds the maximum size allowed by Amazon SNS.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
     *         <code>MaximumExecutionFrequency</code>.
     *         </p>
     *         </li>
     */
    public List<String> organizationConfigRuleTriggerTypesAsStrings() {
        return organizationConfigRuleTriggerTypes;
    }

    /**
     * <p>
     * A string, in JSON format, that is passed to organization config rule Lambda function.
     * </p>
     * 
     * @return A string, in JSON format, that is passed to organization config rule Lambda function.
     */
    public String inputParameters() {
        return inputParameters;
    }

    /**
     * <p>
     * The maximum frequency with which AWS Config runs evaluations for a rule. Your custom rule is triggered when AWS
     * Config delivers the configuration snapshot. For more information, see <a>ConfigSnapshotDeliveryProperties</a>.
     * </p>
     * <note>
     * <p>
     * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a valid
     * value for the <code>MaximumExecutionFrequency</code> parameter.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maximumExecutionFrequency} will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maximumExecutionFrequencyAsString}.
     * </p>
     * 
     * @return The maximum frequency with which AWS Config runs evaluations for a rule. Your custom rule is triggered
     *         when AWS Config delivers the configuration snapshot. For more information, see
     *         <a>ConfigSnapshotDeliveryProperties</a>.</p> <note>
     *         <p>
     *         By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify
     *         a valid value for the <code>MaximumExecutionFrequency</code> parameter.
     *         </p>
     * @see MaximumExecutionFrequency
     */
    public MaximumExecutionFrequency maximumExecutionFrequency() {
        return MaximumExecutionFrequency.fromValue(maximumExecutionFrequency);
    }

    /**
     * <p>
     * The maximum frequency with which AWS Config runs evaluations for a rule. Your custom rule is triggered when AWS
     * Config delivers the configuration snapshot. For more information, see <a>ConfigSnapshotDeliveryProperties</a>.
     * </p>
     * <note>
     * <p>
     * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a valid
     * value for the <code>MaximumExecutionFrequency</code> parameter.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maximumExecutionFrequency} will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maximumExecutionFrequencyAsString}.
     * </p>
     * 
     * @return The maximum frequency with which AWS Config runs evaluations for a rule. Your custom rule is triggered
     *         when AWS Config delivers the configuration snapshot. For more information, see
     *         <a>ConfigSnapshotDeliveryProperties</a>.</p> <note>
     *         <p>
     *         By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify
     *         a valid value for the <code>MaximumExecutionFrequency</code> parameter.
     *         </p>
     * @see MaximumExecutionFrequency
     */
    public String maximumExecutionFrequencyAsString() {
        return maximumExecutionFrequency;
    }

    /**
     * Returns true if the ResourceTypesScope property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasResourceTypesScope() {
        return resourceTypesScope != null && !(resourceTypesScope instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of the AWS resource that was evaluated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceTypesScope()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The type of the AWS resource that was evaluated.
     */
    public List<String> resourceTypesScope() {
        return resourceTypesScope;
    }

    /**
     * <p>
     * The ID of the AWS resource that was evaluated.
     * </p>
     * 
     * @return The ID of the AWS resource that was evaluated.
     */
    public String resourceIdScope() {
        return resourceIdScope;
    }

    /**
     * <p>
     * One part of a key-value pair that make up a tag. A key is a general label that acts like a category for more
     * specific tag values.
     * </p>
     * 
     * @return One part of a key-value pair that make up a tag. A key is a general label that acts like a category for
     *         more specific tag values.
     */
    public String tagKeyScope() {
        return tagKeyScope;
    }

    /**
     * <p>
     * The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag category
     * (key).
     * </p>
     * 
     * @return The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag
     *         category (key).
     */
    public String tagValueScope() {
        return tagValueScope;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(lambdaFunctionArn());
        hashCode = 31 * hashCode + Objects.hashCode(organizationConfigRuleTriggerTypesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(inputParameters());
        hashCode = 31 * hashCode + Objects.hashCode(maximumExecutionFrequencyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypesScope());
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdScope());
        hashCode = 31 * hashCode + Objects.hashCode(tagKeyScope());
        hashCode = 31 * hashCode + Objects.hashCode(tagValueScope());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OrganizationCustomRuleMetadata)) {
            return false;
        }
        OrganizationCustomRuleMetadata other = (OrganizationCustomRuleMetadata) obj;
        return Objects.equals(description(), other.description())
                && Objects.equals(lambdaFunctionArn(), other.lambdaFunctionArn())
                && Objects.equals(organizationConfigRuleTriggerTypesAsStrings(),
                        other.organizationConfigRuleTriggerTypesAsStrings())
                && Objects.equals(inputParameters(), other.inputParameters())
                && Objects.equals(maximumExecutionFrequencyAsString(), other.maximumExecutionFrequencyAsString())
                && Objects.equals(resourceTypesScope(), other.resourceTypesScope())
                && Objects.equals(resourceIdScope(), other.resourceIdScope())
                && Objects.equals(tagKeyScope(), other.tagKeyScope()) && Objects.equals(tagValueScope(), other.tagValueScope());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("OrganizationCustomRuleMetadata").add("Description", description())
                .add("LambdaFunctionArn", lambdaFunctionArn())
                .add("OrganizationConfigRuleTriggerTypes", organizationConfigRuleTriggerTypesAsStrings())
                .add("InputParameters", inputParameters()).add("MaximumExecutionFrequency", maximumExecutionFrequencyAsString())
                .add("ResourceTypesScope", resourceTypesScope()).add("ResourceIdScope", resourceIdScope())
                .add("TagKeyScope", tagKeyScope()).add("TagValueScope", tagValueScope()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "LambdaFunctionArn":
            return Optional.ofNullable(clazz.cast(lambdaFunctionArn()));
        case "OrganizationConfigRuleTriggerTypes":
            return Optional.ofNullable(clazz.cast(organizationConfigRuleTriggerTypesAsStrings()));
        case "InputParameters":
            return Optional.ofNullable(clazz.cast(inputParameters()));
        case "MaximumExecutionFrequency":
            return Optional.ofNullable(clazz.cast(maximumExecutionFrequencyAsString()));
        case "ResourceTypesScope":
            return Optional.ofNullable(clazz.cast(resourceTypesScope()));
        case "ResourceIdScope":
            return Optional.ofNullable(clazz.cast(resourceIdScope()));
        case "TagKeyScope":
            return Optional.ofNullable(clazz.cast(tagKeyScope()));
        case "TagValueScope":
            return Optional.ofNullable(clazz.cast(tagValueScope()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OrganizationCustomRuleMetadata, T> g) {
        return obj -> g.apply((OrganizationCustomRuleMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OrganizationCustomRuleMetadata> {
        /**
         * <p>
         * The description that you provide for organization config rule.
         * </p>
         * 
         * @param description
         *        The description that you provide for organization config rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The lambda function ARN.
         * </p>
         * 
         * @param lambdaFunctionArn
         *        The lambda function ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lambdaFunctionArn(String lambdaFunctionArn);

        /**
         * <p>
         * The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
         * following notification types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
         * configuration item as a result of a resource change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers
         * an oversized configuration item. AWS Config may generate this notification type when a resource changes and
         * the notification exceeds the maximum size allowed by Amazon SNS.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
         * <code>MaximumExecutionFrequency</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param organizationConfigRuleTriggerTypes
         *        The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
         *        following notification types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
         *        configuration item as a result of a resource change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config
         *        delivers an oversized configuration item. AWS Config may generate this notification type when a
         *        resource changes and the notification exceeds the maximum size allowed by Amazon SNS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
         *        <code>MaximumExecutionFrequency</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationConfigRuleTriggerTypesWithStrings(Collection<String> organizationConfigRuleTriggerTypes);

        /**
         * <p>
         * The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
         * following notification types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
         * configuration item as a result of a resource change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers
         * an oversized configuration item. AWS Config may generate this notification type when a resource changes and
         * the notification exceeds the maximum size allowed by Amazon SNS.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
         * <code>MaximumExecutionFrequency</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param organizationConfigRuleTriggerTypes
         *        The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
         *        following notification types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
         *        configuration item as a result of a resource change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config
         *        delivers an oversized configuration item. AWS Config may generate this notification type when a
         *        resource changes and the notification exceeds the maximum size allowed by Amazon SNS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
         *        <code>MaximumExecutionFrequency</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationConfigRuleTriggerTypesWithStrings(String... organizationConfigRuleTriggerTypes);

        /**
         * <p>
         * The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
         * following notification types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
         * configuration item as a result of a resource change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers
         * an oversized configuration item. AWS Config may generate this notification type when a resource changes and
         * the notification exceeds the maximum size allowed by Amazon SNS.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
         * <code>MaximumExecutionFrequency</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param organizationConfigRuleTriggerTypes
         *        The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
         *        following notification types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
         *        configuration item as a result of a resource change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config
         *        delivers an oversized configuration item. AWS Config may generate this notification type when a
         *        resource changes and the notification exceeds the maximum size allowed by Amazon SNS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
         *        <code>MaximumExecutionFrequency</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationConfigRuleTriggerTypes(
                Collection<OrganizationConfigRuleTriggerType> organizationConfigRuleTriggerTypes);

        /**
         * <p>
         * The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
         * following notification types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
         * configuration item as a result of a resource change.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers
         * an oversized configuration item. AWS Config may generate this notification type when a resource changes and
         * the notification exceeds the maximum size allowed by Amazon SNS.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
         * <code>MaximumExecutionFrequency</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param organizationConfigRuleTriggerTypes
         *        The type of notification that triggers AWS Config to run an evaluation for a rule. You can specify the
         *        following notification types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config delivers a
         *        configuration item as a result of a resource change.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OversizedConfigurationItemChangeNotification</code> - Triggers an evaluation when AWS Config
         *        delivers an oversized configuration item. AWS Config may generate this notification type when a
         *        resource changes and the notification exceeds the maximum size allowed by Amazon SNS.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ScheduledNotification</code> - Triggers a periodic evaluation at the frequency specified for
         *        <code>MaximumExecutionFrequency</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationConfigRuleTriggerTypes(OrganizationConfigRuleTriggerType... organizationConfigRuleTriggerTypes);

        /**
         * <p>
         * A string, in JSON format, that is passed to organization config rule Lambda function.
         * </p>
         * 
         * @param inputParameters
         *        A string, in JSON format, that is passed to organization config rule Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputParameters(String inputParameters);

        /**
         * <p>
         * The maximum frequency with which AWS Config runs evaluations for a rule. Your custom rule is triggered when
         * AWS Config delivers the configuration snapshot. For more information, see
         * <a>ConfigSnapshotDeliveryProperties</a>.
         * </p>
         * <note>
         * <p>
         * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a
         * valid value for the <code>MaximumExecutionFrequency</code> parameter.
         * </p>
         * </note>
         * 
         * @param maximumExecutionFrequency
         *        The maximum frequency with which AWS Config runs evaluations for a rule. Your custom rule is triggered
         *        when AWS Config delivers the configuration snapshot. For more information, see
         *        <a>ConfigSnapshotDeliveryProperties</a>.</p> <note>
         *        <p>
         *        By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency,
         *        specify a valid value for the <code>MaximumExecutionFrequency</code> parameter.
         *        </p>
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder maximumExecutionFrequency(String maximumExecutionFrequency);

        /**
         * <p>
         * The maximum frequency with which AWS Config runs evaluations for a rule. Your custom rule is triggered when
         * AWS Config delivers the configuration snapshot. For more information, see
         * <a>ConfigSnapshotDeliveryProperties</a>.
         * </p>
         * <note>
         * <p>
         * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a
         * valid value for the <code>MaximumExecutionFrequency</code> parameter.
         * </p>
         * </note>
         * 
         * @param maximumExecutionFrequency
         *        The maximum frequency with which AWS Config runs evaluations for a rule. Your custom rule is triggered
         *        when AWS Config delivers the configuration snapshot. For more information, see
         *        <a>ConfigSnapshotDeliveryProperties</a>.</p> <note>
         *        <p>
         *        By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency,
         *        specify a valid value for the <code>MaximumExecutionFrequency</code> parameter.
         *        </p>
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder maximumExecutionFrequency(MaximumExecutionFrequency maximumExecutionFrequency);

        /**
         * <p>
         * The type of the AWS resource that was evaluated.
         * </p>
         * 
         * @param resourceTypesScope
         *        The type of the AWS resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesScope(Collection<String> resourceTypesScope);

        /**
         * <p>
         * The type of the AWS resource that was evaluated.
         * </p>
         * 
         * @param resourceTypesScope
         *        The type of the AWS resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesScope(String... resourceTypesScope);

        /**
         * <p>
         * The ID of the AWS resource that was evaluated.
         * </p>
         * 
         * @param resourceIdScope
         *        The ID of the AWS resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdScope(String resourceIdScope);

        /**
         * <p>
         * One part of a key-value pair that make up a tag. A key is a general label that acts like a category for more
         * specific tag values.
         * </p>
         * 
         * @param tagKeyScope
         *        One part of a key-value pair that make up a tag. A key is a general label that acts like a category
         *        for more specific tag values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagKeyScope(String tagKeyScope);

        /**
         * <p>
         * The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag category
         * (key).
         * </p>
         * 
         * @param tagValueScope
         *        The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag
         *        category (key).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagValueScope(String tagValueScope);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private String lambdaFunctionArn;

        private List<String> organizationConfigRuleTriggerTypes = DefaultSdkAutoConstructList.getInstance();

        private String inputParameters;

        private String maximumExecutionFrequency;

        private List<String> resourceTypesScope = DefaultSdkAutoConstructList.getInstance();

        private String resourceIdScope;

        private String tagKeyScope;

        private String tagValueScope;

        private BuilderImpl() {
        }

        private BuilderImpl(OrganizationCustomRuleMetadata model) {
            description(model.description);
            lambdaFunctionArn(model.lambdaFunctionArn);
            organizationConfigRuleTriggerTypesWithStrings(model.organizationConfigRuleTriggerTypes);
            inputParameters(model.inputParameters);
            maximumExecutionFrequency(model.maximumExecutionFrequency);
            resourceTypesScope(model.resourceTypesScope);
            resourceIdScope(model.resourceIdScope);
            tagKeyScope(model.tagKeyScope);
            tagValueScope(model.tagValueScope);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getLambdaFunctionArn() {
            return lambdaFunctionArn;
        }

        @Override
        public final Builder lambdaFunctionArn(String lambdaFunctionArn) {
            this.lambdaFunctionArn = lambdaFunctionArn;
            return this;
        }

        public final void setLambdaFunctionArn(String lambdaFunctionArn) {
            this.lambdaFunctionArn = lambdaFunctionArn;
        }

        public final Collection<String> getOrganizationConfigRuleTriggerTypes() {
            return organizationConfigRuleTriggerTypes;
        }

        @Override
        public final Builder organizationConfigRuleTriggerTypesWithStrings(Collection<String> organizationConfigRuleTriggerTypes) {
            this.organizationConfigRuleTriggerTypes = OrganizationConfigRuleTriggerTypesCopier
                    .copy(organizationConfigRuleTriggerTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationConfigRuleTriggerTypesWithStrings(String... organizationConfigRuleTriggerTypes) {
            organizationConfigRuleTriggerTypesWithStrings(Arrays.asList(organizationConfigRuleTriggerTypes));
            return this;
        }

        @Override
        public final Builder organizationConfigRuleTriggerTypes(
                Collection<OrganizationConfigRuleTriggerType> organizationConfigRuleTriggerTypes) {
            this.organizationConfigRuleTriggerTypes = OrganizationConfigRuleTriggerTypesCopier
                    .copyEnumToString(organizationConfigRuleTriggerTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationConfigRuleTriggerTypes(
                OrganizationConfigRuleTriggerType... organizationConfigRuleTriggerTypes) {
            organizationConfigRuleTriggerTypes(Arrays.asList(organizationConfigRuleTriggerTypes));
            return this;
        }

        public final void setOrganizationConfigRuleTriggerTypes(Collection<String> organizationConfigRuleTriggerTypes) {
            this.organizationConfigRuleTriggerTypes = OrganizationConfigRuleTriggerTypesCopier
                    .copy(organizationConfigRuleTriggerTypes);
        }

        public final String getInputParameters() {
            return inputParameters;
        }

        @Override
        public final Builder inputParameters(String inputParameters) {
            this.inputParameters = inputParameters;
            return this;
        }

        public final void setInputParameters(String inputParameters) {
            this.inputParameters = inputParameters;
        }

        public final String getMaximumExecutionFrequency() {
            return maximumExecutionFrequency;
        }

        @Override
        public final Builder maximumExecutionFrequency(String maximumExecutionFrequency) {
            this.maximumExecutionFrequency = maximumExecutionFrequency;
            return this;
        }

        @Override
        public final Builder maximumExecutionFrequency(MaximumExecutionFrequency maximumExecutionFrequency) {
            this.maximumExecutionFrequency(maximumExecutionFrequency == null ? null : maximumExecutionFrequency.toString());
            return this;
        }

        public final void setMaximumExecutionFrequency(String maximumExecutionFrequency) {
            this.maximumExecutionFrequency = maximumExecutionFrequency;
        }

        public final Collection<String> getResourceTypesScope() {
            return resourceTypesScope;
        }

        @Override
        public final Builder resourceTypesScope(Collection<String> resourceTypesScope) {
            this.resourceTypesScope = ResourceTypesScopeCopier.copy(resourceTypesScope);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesScope(String... resourceTypesScope) {
            resourceTypesScope(Arrays.asList(resourceTypesScope));
            return this;
        }

        public final void setResourceTypesScope(Collection<String> resourceTypesScope) {
            this.resourceTypesScope = ResourceTypesScopeCopier.copy(resourceTypesScope);
        }

        public final String getResourceIdScope() {
            return resourceIdScope;
        }

        @Override
        public final Builder resourceIdScope(String resourceIdScope) {
            this.resourceIdScope = resourceIdScope;
            return this;
        }

        public final void setResourceIdScope(String resourceIdScope) {
            this.resourceIdScope = resourceIdScope;
        }

        public final String getTagKeyScope() {
            return tagKeyScope;
        }

        @Override
        public final Builder tagKeyScope(String tagKeyScope) {
            this.tagKeyScope = tagKeyScope;
            return this;
        }

        public final void setTagKeyScope(String tagKeyScope) {
            this.tagKeyScope = tagKeyScope;
        }

        public final String getTagValueScope() {
            return tagValueScope;
        }

        @Override
        public final Builder tagValueScope(String tagValueScope) {
            this.tagValueScope = tagValueScope;
            return this;
        }

        public final void setTagValueScope(String tagValueScope) {
            this.tagValueScope = tagValueScope;
        }

        @Override
        public OrganizationCustomRuleMetadata build() {
            return new OrganizationCustomRuleMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
