/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * This object contains regions to set up the aggregator and an IAM role to retrieve organization details.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OrganizationAggregationSource implements SdkPojo, Serializable,
        ToCopyableBuilder<OrganizationAggregationSource.Builder, OrganizationAggregationSource> {
    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(OrganizationAggregationSource::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<List<String>> AWS_REGIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AwsRegions")
            .getter(getter(OrganizationAggregationSource::awsRegions))
            .setter(setter(Builder::awsRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AwsRegions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> ALL_AWS_REGIONS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AllAwsRegions").getter(getter(OrganizationAggregationSource::allAwsRegions))
            .setter(setter(Builder::allAwsRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllAwsRegions").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROLE_ARN_FIELD,
            AWS_REGIONS_FIELD, ALL_AWS_REGIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String roleArn;

    private final List<String> awsRegions;

    private final Boolean allAwsRegions;

    private OrganizationAggregationSource(BuilderImpl builder) {
        this.roleArn = builder.roleArn;
        this.awsRegions = builder.awsRegions;
        this.allAwsRegions = builder.allAwsRegions;
    }

    /**
     * <p>
     * ARN of the IAM role used to retrieve Amazon Web Services Organization details associated with the aggregator
     * account.
     * </p>
     * 
     * @return ARN of the IAM role used to retrieve Amazon Web Services Organization details associated with the
     *         aggregator account.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the AwsRegions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAwsRegions() {
        return awsRegions != null && !(awsRegions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The source regions being aggregated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAwsRegions} method.
     * </p>
     * 
     * @return The source regions being aggregated.
     */
    public final List<String> awsRegions() {
        return awsRegions;
    }

    /**
     * <p>
     * If true, aggregate existing Config regions and future regions.
     * </p>
     * 
     * @return If true, aggregate existing Config regions and future regions.
     */
    public final Boolean allAwsRegions() {
        return allAwsRegions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasAwsRegions() ? awsRegions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(allAwsRegions());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OrganizationAggregationSource)) {
            return false;
        }
        OrganizationAggregationSource other = (OrganizationAggregationSource) obj;
        return Objects.equals(roleArn(), other.roleArn()) && hasAwsRegions() == other.hasAwsRegions()
                && Objects.equals(awsRegions(), other.awsRegions()) && Objects.equals(allAwsRegions(), other.allAwsRegions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OrganizationAggregationSource").add("RoleArn", roleArn())
                .add("AwsRegions", hasAwsRegions() ? awsRegions() : null).add("AllAwsRegions", allAwsRegions()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "AwsRegions":
            return Optional.ofNullable(clazz.cast(awsRegions()));
        case "AllAwsRegions":
            return Optional.ofNullable(clazz.cast(allAwsRegions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OrganizationAggregationSource, T> g) {
        return obj -> g.apply((OrganizationAggregationSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OrganizationAggregationSource> {
        /**
         * <p>
         * ARN of the IAM role used to retrieve Amazon Web Services Organization details associated with the aggregator
         * account.
         * </p>
         * 
         * @param roleArn
         *        ARN of the IAM role used to retrieve Amazon Web Services Organization details associated with the
         *        aggregator account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The source regions being aggregated.
         * </p>
         * 
         * @param awsRegions
         *        The source regions being aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsRegions(Collection<String> awsRegions);

        /**
         * <p>
         * The source regions being aggregated.
         * </p>
         * 
         * @param awsRegions
         *        The source regions being aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsRegions(String... awsRegions);

        /**
         * <p>
         * If true, aggregate existing Config regions and future regions.
         * </p>
         * 
         * @param allAwsRegions
         *        If true, aggregate existing Config regions and future regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allAwsRegions(Boolean allAwsRegions);
    }

    static final class BuilderImpl implements Builder {
        private String roleArn;

        private List<String> awsRegions = DefaultSdkAutoConstructList.getInstance();

        private Boolean allAwsRegions;

        private BuilderImpl() {
        }

        private BuilderImpl(OrganizationAggregationSource model) {
            roleArn(model.roleArn);
            awsRegions(model.awsRegions);
            allAwsRegions(model.allAwsRegions);
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        @Transient
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final Collection<String> getAwsRegions() {
            if (awsRegions instanceof SdkAutoConstructList) {
                return null;
            }
            return awsRegions;
        }

        public final void setAwsRegions(Collection<String> awsRegions) {
            this.awsRegions = AggregatorRegionListCopier.copy(awsRegions);
        }

        @Override
        @Transient
        public final Builder awsRegions(Collection<String> awsRegions) {
            this.awsRegions = AggregatorRegionListCopier.copy(awsRegions);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder awsRegions(String... awsRegions) {
            awsRegions(Arrays.asList(awsRegions));
            return this;
        }

        public final Boolean getAllAwsRegions() {
            return allAwsRegions;
        }

        public final void setAllAwsRegions(Boolean allAwsRegions) {
            this.allAwsRegions = allAwsRegions;
        }

        @Override
        @Transient
        public final Builder allAwsRegions(Boolean allAwsRegions) {
            this.allAwsRegions = allAwsRegions;
            return this;
        }

        @Override
        public OrganizationAggregationSource build() {
            return new OrganizationAggregationSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
