/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides the AWS Config rule owner (AWS or customer), the rule identifier, and the events that trigger the evaluation
 * of your AWS resources.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Source implements SdkPojo, Serializable, ToCopyableBuilder<Source.Builder, Source> {
    private static final SdkField<String> OWNER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Owner")
            .getter(getter(Source::ownerAsString)).setter(setter(Builder::owner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Owner").build()).build();

    private static final SdkField<String> SOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceIdentifier").getter(getter(Source::sourceIdentifier)).setter(setter(Builder::sourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceIdentifier").build()).build();

    private static final SdkField<List<SourceDetail>> SOURCE_DETAILS_FIELD = SdkField
            .<List<SourceDetail>> builder(MarshallingType.LIST)
            .memberName("SourceDetails")
            .getter(getter(Source::sourceDetails))
            .setter(setter(Builder::sourceDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDetails").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SourceDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(SourceDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_FIELD,
            SOURCE_IDENTIFIER_FIELD, SOURCE_DETAILS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String owner;

    private final String sourceIdentifier;

    private final List<SourceDetail> sourceDetails;

    private Source(BuilderImpl builder) {
        this.owner = builder.owner;
        this.sourceIdentifier = builder.sourceIdentifier;
        this.sourceDetails = builder.sourceDetails;
    }

    /**
     * <p>
     * Indicates whether AWS or the customer owns and manages the AWS Config rule.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #owner} will return
     * {@link Owner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ownerAsString}.
     * </p>
     * 
     * @return Indicates whether AWS or the customer owns and manages the AWS Config rule.
     * @see Owner
     */
    public final Owner owner() {
        return Owner.fromValue(owner);
    }

    /**
     * <p>
     * Indicates whether AWS or the customer owns and manages the AWS Config rule.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #owner} will return
     * {@link Owner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ownerAsString}.
     * </p>
     * 
     * @return Indicates whether AWS or the customer owns and manages the AWS Config rule.
     * @see Owner
     */
    public final String ownerAsString() {
        return owner;
    }

    /**
     * <p>
     * For AWS Config managed rules, a predefined identifier from a list. For example, <code>IAM_PASSWORD_POLICY</code>
     * is a managed rule. To reference a managed rule, see <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html">Using AWS
     * Managed Config Rules</a>.
     * </p>
     * <p>
     * For custom rules, the identifier is the Amazon Resource Name (ARN) of the rule's AWS Lambda function, such as
     * <code>arn:aws:lambda:us-east-2:123456789012:function:custom_rule_name</code>.
     * </p>
     * 
     * @return For AWS Config managed rules, a predefined identifier from a list. For example,
     *         <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"
     *         >Using AWS Managed Config Rules</a>.</p>
     *         <p>
     *         For custom rules, the identifier is the Amazon Resource Name (ARN) of the rule's AWS Lambda function,
     *         such as <code>arn:aws:lambda:us-east-2:123456789012:function:custom_rule_name</code>.
     */
    public final String sourceIdentifier() {
        return sourceIdentifier;
    }

    /**
     * Returns true if the SourceDetails property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSourceDetails() {
        return sourceDetails != null && !(sourceDetails instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSourceDetails()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
     */
    public final List<SourceDetail> sourceDetails() {
        return sourceDetails;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceDetails() ? sourceDetails() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Source)) {
            return false;
        }
        Source other = (Source) obj;
        return Objects.equals(ownerAsString(), other.ownerAsString())
                && Objects.equals(sourceIdentifier(), other.sourceIdentifier()) && hasSourceDetails() == other.hasSourceDetails()
                && Objects.equals(sourceDetails(), other.sourceDetails());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Source").add("Owner", ownerAsString()).add("SourceIdentifier", sourceIdentifier())
                .add("SourceDetails", hasSourceDetails() ? sourceDetails() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Owner":
            return Optional.ofNullable(clazz.cast(ownerAsString()));
        case "SourceIdentifier":
            return Optional.ofNullable(clazz.cast(sourceIdentifier()));
        case "SourceDetails":
            return Optional.ofNullable(clazz.cast(sourceDetails()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Source, T> g) {
        return obj -> g.apply((Source) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Source> {
        /**
         * <p>
         * Indicates whether AWS or the customer owns and manages the AWS Config rule.
         * </p>
         * 
         * @param owner
         *        Indicates whether AWS or the customer owns and manages the AWS Config rule.
         * @see Owner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Owner
         */
        Builder owner(String owner);

        /**
         * <p>
         * Indicates whether AWS or the customer owns and manages the AWS Config rule.
         * </p>
         * 
         * @param owner
         *        Indicates whether AWS or the customer owns and manages the AWS Config rule.
         * @see Owner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Owner
         */
        Builder owner(Owner owner);

        /**
         * <p>
         * For AWS Config managed rules, a predefined identifier from a list. For example,
         * <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html">Using
         * AWS Managed Config Rules</a>.
         * </p>
         * <p>
         * For custom rules, the identifier is the Amazon Resource Name (ARN) of the rule's AWS Lambda function, such as
         * <code>arn:aws:lambda:us-east-2:123456789012:function:custom_rule_name</code>.
         * </p>
         * 
         * @param sourceIdentifier
         *        For AWS Config managed rules, a predefined identifier from a list. For example,
         *        <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
         *        href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"
         *        >Using AWS Managed Config Rules</a>.</p>
         *        <p>
         *        For custom rules, the identifier is the Amazon Resource Name (ARN) of the rule's AWS Lambda function,
         *        such as <code>arn:aws:lambda:us-east-2:123456789012:function:custom_rule_name</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIdentifier(String sourceIdentifier);

        /**
         * <p>
         * Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * </p>
         * 
         * @param sourceDetails
         *        Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDetails(Collection<SourceDetail> sourceDetails);

        /**
         * <p>
         * Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * </p>
         * 
         * @param sourceDetails
         *        Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDetails(SourceDetail... sourceDetails);

        /**
         * <p>
         * Provides the source and type of the event that causes AWS Config to evaluate your AWS resources.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SourceDetail>.Builder} avoiding the need to
         * create one manually via {@link List<SourceDetail>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SourceDetail>.Builder#build()} is called immediately and its
         * result is passed to {@link #sourceDetails(List<SourceDetail>)}.
         * 
         * @param sourceDetails
         *        a consumer that will call methods on {@link List<SourceDetail>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceDetails(List<SourceDetail>)
         */
        Builder sourceDetails(Consumer<SourceDetail.Builder>... sourceDetails);
    }

    static final class BuilderImpl implements Builder {
        private String owner;

        private String sourceIdentifier;

        private List<SourceDetail> sourceDetails = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Source model) {
            owner(model.owner);
            sourceIdentifier(model.sourceIdentifier);
            sourceDetails(model.sourceDetails);
        }

        public final String getOwner() {
            return owner;
        }

        public final void setOwner(String owner) {
            this.owner = owner;
        }

        @Override
        @Transient
        public final Builder owner(String owner) {
            this.owner = owner;
            return this;
        }

        @Override
        @Transient
        public final Builder owner(Owner owner) {
            this.owner(owner == null ? null : owner.toString());
            return this;
        }

        public final String getSourceIdentifier() {
            return sourceIdentifier;
        }

        public final void setSourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
        }

        @Override
        @Transient
        public final Builder sourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
            return this;
        }

        public final List<SourceDetail.Builder> getSourceDetails() {
            List<SourceDetail.Builder> result = SourceDetailsCopier.copyToBuilder(this.sourceDetails);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSourceDetails(Collection<SourceDetail.BuilderImpl> sourceDetails) {
            this.sourceDetails = SourceDetailsCopier.copyFromBuilder(sourceDetails);
        }

        @Override
        @Transient
        public final Builder sourceDetails(Collection<SourceDetail> sourceDetails) {
            this.sourceDetails = SourceDetailsCopier.copy(sourceDetails);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder sourceDetails(SourceDetail... sourceDetails) {
            sourceDetails(Arrays.asList(sourceDetails));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder sourceDetails(Consumer<SourceDetail.Builder>... sourceDetails) {
            sourceDetails(Stream.of(sourceDetails).map(c -> SourceDetail.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Source build() {
            return new Source(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
