/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that specifies organization managed rule metadata such as resource type and ID of AWS resource along with
 * the rule identifier. It also provides the frequency with which you want AWS Config to run evaluations for the rule if
 * the trigger type is periodic.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OrganizationManagedRuleMetadata implements SdkPojo, Serializable,
        ToCopyableBuilder<OrganizationManagedRuleMetadata.Builder, OrganizationManagedRuleMetadata> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(OrganizationManagedRuleMetadata::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> RULE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleIdentifier").getter(getter(OrganizationManagedRuleMetadata::ruleIdentifier))
            .setter(setter(Builder::ruleIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleIdentifier").build()).build();

    private static final SdkField<String> INPUT_PARAMETERS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputParameters").getter(getter(OrganizationManagedRuleMetadata::inputParameters))
            .setter(setter(Builder::inputParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputParameters").build()).build();

    private static final SdkField<String> MAXIMUM_EXECUTION_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaximumExecutionFrequency")
            .getter(getter(OrganizationManagedRuleMetadata::maximumExecutionFrequencyAsString))
            .setter(setter(Builder::maximumExecutionFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumExecutionFrequency").build())
            .build();

    private static final SdkField<List<String>> RESOURCE_TYPES_SCOPE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceTypesScope")
            .getter(getter(OrganizationManagedRuleMetadata::resourceTypesScope))
            .setter(setter(Builder::resourceTypesScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTypesScope").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_ID_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceIdScope").getter(getter(OrganizationManagedRuleMetadata::resourceIdScope))
            .setter(setter(Builder::resourceIdScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdScope").build()).build();

    private static final SdkField<String> TAG_KEY_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TagKeyScope").getter(getter(OrganizationManagedRuleMetadata::tagKeyScope))
            .setter(setter(Builder::tagKeyScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagKeyScope").build()).build();

    private static final SdkField<String> TAG_VALUE_SCOPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TagValueScope").getter(getter(OrganizationManagedRuleMetadata::tagValueScope))
            .setter(setter(Builder::tagValueScope))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagValueScope").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            RULE_IDENTIFIER_FIELD, INPUT_PARAMETERS_FIELD, MAXIMUM_EXECUTION_FREQUENCY_FIELD, RESOURCE_TYPES_SCOPE_FIELD,
            RESOURCE_ID_SCOPE_FIELD, TAG_KEY_SCOPE_FIELD, TAG_VALUE_SCOPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String description;

    private final String ruleIdentifier;

    private final String inputParameters;

    private final String maximumExecutionFrequency;

    private final List<String> resourceTypesScope;

    private final String resourceIdScope;

    private final String tagKeyScope;

    private final String tagValueScope;

    private OrganizationManagedRuleMetadata(BuilderImpl builder) {
        this.description = builder.description;
        this.ruleIdentifier = builder.ruleIdentifier;
        this.inputParameters = builder.inputParameters;
        this.maximumExecutionFrequency = builder.maximumExecutionFrequency;
        this.resourceTypesScope = builder.resourceTypesScope;
        this.resourceIdScope = builder.resourceIdScope;
        this.tagKeyScope = builder.tagKeyScope;
        this.tagValueScope = builder.tagValueScope;
    }

    /**
     * <p>
     * The description that you provide for organization config rule.
     * </p>
     * 
     * @return The description that you provide for organization config rule.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * For organization config managed rules, a predefined identifier from a list. For example,
     * <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
     * href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html">Using AWS
     * Managed Config Rules</a>.
     * </p>
     * 
     * @return For organization config managed rules, a predefined identifier from a list. For example,
     *         <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
     *         href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"
     *         >Using AWS Managed Config Rules</a>.
     */
    public final String ruleIdentifier() {
        return ruleIdentifier;
    }

    /**
     * <p>
     * A string, in JSON format, that is passed to organization config rule Lambda function.
     * </p>
     * 
     * @return A string, in JSON format, that is passed to organization config rule Lambda function.
     */
    public final String inputParameters() {
        return inputParameters;
    }

    /**
     * <p>
     * The maximum frequency with which AWS Config runs evaluations for a rule. You are using an AWS managed rule that
     * is triggered at a periodic frequency.
     * </p>
     * <note>
     * <p>
     * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a valid
     * value for the <code>MaximumExecutionFrequency</code> parameter.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maximumExecutionFrequency} will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maximumExecutionFrequencyAsString}.
     * </p>
     * 
     * @return The maximum frequency with which AWS Config runs evaluations for a rule. You are using an AWS managed
     *         rule that is triggered at a periodic frequency.</p> <note>
     *         <p>
     *         By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify
     *         a valid value for the <code>MaximumExecutionFrequency</code> parameter.
     *         </p>
     * @see MaximumExecutionFrequency
     */
    public final MaximumExecutionFrequency maximumExecutionFrequency() {
        return MaximumExecutionFrequency.fromValue(maximumExecutionFrequency);
    }

    /**
     * <p>
     * The maximum frequency with which AWS Config runs evaluations for a rule. You are using an AWS managed rule that
     * is triggered at a periodic frequency.
     * </p>
     * <note>
     * <p>
     * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a valid
     * value for the <code>MaximumExecutionFrequency</code> parameter.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maximumExecutionFrequency} will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maximumExecutionFrequencyAsString}.
     * </p>
     * 
     * @return The maximum frequency with which AWS Config runs evaluations for a rule. You are using an AWS managed
     *         rule that is triggered at a periodic frequency.</p> <note>
     *         <p>
     *         By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify
     *         a valid value for the <code>MaximumExecutionFrequency</code> parameter.
     *         </p>
     * @see MaximumExecutionFrequency
     */
    public final String maximumExecutionFrequencyAsString() {
        return maximumExecutionFrequency;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypesScope property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypesScope() {
        return resourceTypesScope != null && !(resourceTypesScope instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of the AWS resource that was evaluated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypesScope} method.
     * </p>
     * 
     * @return The type of the AWS resource that was evaluated.
     */
    public final List<String> resourceTypesScope() {
        return resourceTypesScope;
    }

    /**
     * <p>
     * The ID of the AWS resource that was evaluated.
     * </p>
     * 
     * @return The ID of the AWS resource that was evaluated.
     */
    public final String resourceIdScope() {
        return resourceIdScope;
    }

    /**
     * <p>
     * One part of a key-value pair that make up a tag. A key is a general label that acts like a category for more
     * specific tag values.
     * </p>
     * 
     * @return One part of a key-value pair that make up a tag. A key is a general label that acts like a category for
     *         more specific tag values.
     */
    public final String tagKeyScope() {
        return tagKeyScope;
    }

    /**
     * <p>
     * The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag category
     * (key).
     * </p>
     * 
     * @return The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag
     *         category (key).
     */
    public final String tagValueScope() {
        return tagValueScope;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(ruleIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(inputParameters());
        hashCode = 31 * hashCode + Objects.hashCode(maximumExecutionFrequencyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypesScope() ? resourceTypesScope() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceIdScope());
        hashCode = 31 * hashCode + Objects.hashCode(tagKeyScope());
        hashCode = 31 * hashCode + Objects.hashCode(tagValueScope());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OrganizationManagedRuleMetadata)) {
            return false;
        }
        OrganizationManagedRuleMetadata other = (OrganizationManagedRuleMetadata) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(ruleIdentifier(), other.ruleIdentifier())
                && Objects.equals(inputParameters(), other.inputParameters())
                && Objects.equals(maximumExecutionFrequencyAsString(), other.maximumExecutionFrequencyAsString())
                && hasResourceTypesScope() == other.hasResourceTypesScope()
                && Objects.equals(resourceTypesScope(), other.resourceTypesScope())
                && Objects.equals(resourceIdScope(), other.resourceIdScope())
                && Objects.equals(tagKeyScope(), other.tagKeyScope()) && Objects.equals(tagValueScope(), other.tagValueScope());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OrganizationManagedRuleMetadata").add("Description", description())
                .add("RuleIdentifier", ruleIdentifier()).add("InputParameters", inputParameters())
                .add("MaximumExecutionFrequency", maximumExecutionFrequencyAsString())
                .add("ResourceTypesScope", hasResourceTypesScope() ? resourceTypesScope() : null)
                .add("ResourceIdScope", resourceIdScope()).add("TagKeyScope", tagKeyScope())
                .add("TagValueScope", tagValueScope()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "RuleIdentifier":
            return Optional.ofNullable(clazz.cast(ruleIdentifier()));
        case "InputParameters":
            return Optional.ofNullable(clazz.cast(inputParameters()));
        case "MaximumExecutionFrequency":
            return Optional.ofNullable(clazz.cast(maximumExecutionFrequencyAsString()));
        case "ResourceTypesScope":
            return Optional.ofNullable(clazz.cast(resourceTypesScope()));
        case "ResourceIdScope":
            return Optional.ofNullable(clazz.cast(resourceIdScope()));
        case "TagKeyScope":
            return Optional.ofNullable(clazz.cast(tagKeyScope()));
        case "TagValueScope":
            return Optional.ofNullable(clazz.cast(tagValueScope()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OrganizationManagedRuleMetadata, T> g) {
        return obj -> g.apply((OrganizationManagedRuleMetadata) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OrganizationManagedRuleMetadata> {
        /**
         * <p>
         * The description that you provide for organization config rule.
         * </p>
         * 
         * @param description
         *        The description that you provide for organization config rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * For organization config managed rules, a predefined identifier from a list. For example,
         * <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
         * href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html">Using
         * AWS Managed Config Rules</a>.
         * </p>
         * 
         * @param ruleIdentifier
         *        For organization config managed rules, a predefined identifier from a list. For example,
         *        <code>IAM_PASSWORD_POLICY</code> is a managed rule. To reference a managed rule, see <a
         *        href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"
         *        >Using AWS Managed Config Rules</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ruleIdentifier(String ruleIdentifier);

        /**
         * <p>
         * A string, in JSON format, that is passed to organization config rule Lambda function.
         * </p>
         * 
         * @param inputParameters
         *        A string, in JSON format, that is passed to organization config rule Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputParameters(String inputParameters);

        /**
         * <p>
         * The maximum frequency with which AWS Config runs evaluations for a rule. You are using an AWS managed rule
         * that is triggered at a periodic frequency.
         * </p>
         * <note>
         * <p>
         * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a
         * valid value for the <code>MaximumExecutionFrequency</code> parameter.
         * </p>
         * </note>
         * 
         * @param maximumExecutionFrequency
         *        The maximum frequency with which AWS Config runs evaluations for a rule. You are using an AWS managed
         *        rule that is triggered at a periodic frequency.</p> <note>
         *        <p>
         *        By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency,
         *        specify a valid value for the <code>MaximumExecutionFrequency</code> parameter.
         *        </p>
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder maximumExecutionFrequency(String maximumExecutionFrequency);

        /**
         * <p>
         * The maximum frequency with which AWS Config runs evaluations for a rule. You are using an AWS managed rule
         * that is triggered at a periodic frequency.
         * </p>
         * <note>
         * <p>
         * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a
         * valid value for the <code>MaximumExecutionFrequency</code> parameter.
         * </p>
         * </note>
         * 
         * @param maximumExecutionFrequency
         *        The maximum frequency with which AWS Config runs evaluations for a rule. You are using an AWS managed
         *        rule that is triggered at a periodic frequency.</p> <note>
         *        <p>
         *        By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency,
         *        specify a valid value for the <code>MaximumExecutionFrequency</code> parameter.
         *        </p>
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder maximumExecutionFrequency(MaximumExecutionFrequency maximumExecutionFrequency);

        /**
         * <p>
         * The type of the AWS resource that was evaluated.
         * </p>
         * 
         * @param resourceTypesScope
         *        The type of the AWS resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesScope(Collection<String> resourceTypesScope);

        /**
         * <p>
         * The type of the AWS resource that was evaluated.
         * </p>
         * 
         * @param resourceTypesScope
         *        The type of the AWS resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesScope(String... resourceTypesScope);

        /**
         * <p>
         * The ID of the AWS resource that was evaluated.
         * </p>
         * 
         * @param resourceIdScope
         *        The ID of the AWS resource that was evaluated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdScope(String resourceIdScope);

        /**
         * <p>
         * One part of a key-value pair that make up a tag. A key is a general label that acts like a category for more
         * specific tag values.
         * </p>
         * 
         * @param tagKeyScope
         *        One part of a key-value pair that make up a tag. A key is a general label that acts like a category
         *        for more specific tag values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagKeyScope(String tagKeyScope);

        /**
         * <p>
         * The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag category
         * (key).
         * </p>
         * 
         * @param tagValueScope
         *        The optional part of a key-value pair that make up a tag. A value acts as a descriptor within a tag
         *        category (key).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagValueScope(String tagValueScope);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private String ruleIdentifier;

        private String inputParameters;

        private String maximumExecutionFrequency;

        private List<String> resourceTypesScope = DefaultSdkAutoConstructList.getInstance();

        private String resourceIdScope;

        private String tagKeyScope;

        private String tagValueScope;

        private BuilderImpl() {
        }

        private BuilderImpl(OrganizationManagedRuleMetadata model) {
            description(model.description);
            ruleIdentifier(model.ruleIdentifier);
            inputParameters(model.inputParameters);
            maximumExecutionFrequency(model.maximumExecutionFrequency);
            resourceTypesScope(model.resourceTypesScope);
            resourceIdScope(model.resourceIdScope);
            tagKeyScope(model.tagKeyScope);
            tagValueScope(model.tagValueScope);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        @Transient
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getRuleIdentifier() {
            return ruleIdentifier;
        }

        public final void setRuleIdentifier(String ruleIdentifier) {
            this.ruleIdentifier = ruleIdentifier;
        }

        @Override
        @Transient
        public final Builder ruleIdentifier(String ruleIdentifier) {
            this.ruleIdentifier = ruleIdentifier;
            return this;
        }

        public final String getInputParameters() {
            return inputParameters;
        }

        public final void setInputParameters(String inputParameters) {
            this.inputParameters = inputParameters;
        }

        @Override
        @Transient
        public final Builder inputParameters(String inputParameters) {
            this.inputParameters = inputParameters;
            return this;
        }

        public final String getMaximumExecutionFrequency() {
            return maximumExecutionFrequency;
        }

        public final void setMaximumExecutionFrequency(String maximumExecutionFrequency) {
            this.maximumExecutionFrequency = maximumExecutionFrequency;
        }

        @Override
        @Transient
        public final Builder maximumExecutionFrequency(String maximumExecutionFrequency) {
            this.maximumExecutionFrequency = maximumExecutionFrequency;
            return this;
        }

        @Override
        @Transient
        public final Builder maximumExecutionFrequency(MaximumExecutionFrequency maximumExecutionFrequency) {
            this.maximumExecutionFrequency(maximumExecutionFrequency == null ? null : maximumExecutionFrequency.toString());
            return this;
        }

        public final Collection<String> getResourceTypesScope() {
            if (resourceTypesScope instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypesScope;
        }

        public final void setResourceTypesScope(Collection<String> resourceTypesScope) {
            this.resourceTypesScope = ResourceTypesScopeCopier.copy(resourceTypesScope);
        }

        @Override
        @Transient
        public final Builder resourceTypesScope(Collection<String> resourceTypesScope) {
            this.resourceTypesScope = ResourceTypesScopeCopier.copy(resourceTypesScope);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder resourceTypesScope(String... resourceTypesScope) {
            resourceTypesScope(Arrays.asList(resourceTypesScope));
            return this;
        }

        public final String getResourceIdScope() {
            return resourceIdScope;
        }

        public final void setResourceIdScope(String resourceIdScope) {
            this.resourceIdScope = resourceIdScope;
        }

        @Override
        @Transient
        public final Builder resourceIdScope(String resourceIdScope) {
            this.resourceIdScope = resourceIdScope;
            return this;
        }

        public final String getTagKeyScope() {
            return tagKeyScope;
        }

        public final void setTagKeyScope(String tagKeyScope) {
            this.tagKeyScope = tagKeyScope;
        }

        @Override
        @Transient
        public final Builder tagKeyScope(String tagKeyScope) {
            this.tagKeyScope = tagKeyScope;
            return this;
        }

        public final String getTagValueScope() {
            return tagValueScope;
        }

        public final void setTagValueScope(String tagValueScope) {
            this.tagValueScope = tagValueScope;
        }

        @Override
        @Transient
        public final Builder tagValueScope(String tagValueScope) {
            this.tagValueScope = tagValueScope;
            return this;
        }

        @Override
        public OrganizationManagedRuleMetadata build() {
            return new OrganizationManagedRuleMetadata(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
