/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list that contains detailed configurations of a specified resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfigurationItem implements SdkPojo, Serializable,
        ToCopyableBuilder<ConfigurationItem.Builder, ConfigurationItem> {
    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(ConfigurationItem::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(ConfigurationItem::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<Instant> CONFIGURATION_ITEM_CAPTURE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("configurationItemCaptureTime")
            .getter(getter(ConfigurationItem::configurationItemCaptureTime))
            .setter(setter(Builder::configurationItemCaptureTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationItemCaptureTime")
                    .build()).build();

    private static final SdkField<String> CONFIGURATION_ITEM_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("configurationItemStatus").getter(getter(ConfigurationItem::configurationItemStatusAsString))
            .setter(setter(Builder::configurationItemStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationItemStatus").build())
            .build();

    private static final SdkField<String> CONFIGURATION_STATE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("configurationStateId").getter(getter(ConfigurationItem::configurationStateId))
            .setter(setter(Builder::configurationStateId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationStateId").build())
            .build();

    private static final SdkField<String> CONFIGURATION_ITEM_MD5_HASH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("configurationItemMD5Hash").getter(getter(ConfigurationItem::configurationItemMD5Hash))
            .setter(setter(Builder::configurationItemMD5Hash))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationItemMD5Hash").build())
            .build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(ConfigurationItem::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(ConfigurationItem::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceId").getter(getter(ConfigurationItem::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceId").build()).build();

    private static final SdkField<String> RESOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceName").getter(getter(ConfigurationItem::resourceName)).setter(setter(Builder::resourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceName").build()).build();

    private static final SdkField<String> AWS_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("awsRegion").getter(getter(ConfigurationItem::awsRegion)).setter(setter(Builder::awsRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsRegion").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("availabilityZone").getter(getter(ConfigurationItem::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("availabilityZone").build()).build();

    private static final SdkField<Instant> RESOURCE_CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("resourceCreationTime").getter(getter(ConfigurationItem::resourceCreationTime))
            .setter(setter(Builder::resourceCreationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceCreationTime").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(ConfigurationItem::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<String>> RELATED_EVENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("relatedEvents")
            .getter(getter(ConfigurationItem::relatedEvents))
            .setter(setter(Builder::relatedEvents))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relatedEvents").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Relationship>> RELATIONSHIPS_FIELD = SdkField
            .<List<Relationship>> builder(MarshallingType.LIST)
            .memberName("relationships")
            .getter(getter(ConfigurationItem::relationships))
            .setter(setter(Builder::relationships))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("relationships").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Relationship> builder(MarshallingType.SDK_POJO)
                                            .constructor(Relationship::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("configuration").getter(getter(ConfigurationItem::configuration)).setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<Map<String, String>> SUPPLEMENTARY_CONFIGURATION_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("supplementaryConfiguration")
            .getter(getter(ConfigurationItem::supplementaryConfiguration))
            .setter(setter(Builder::supplementaryConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("supplementaryConfiguration").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VERSION_FIELD,
            ACCOUNT_ID_FIELD, CONFIGURATION_ITEM_CAPTURE_TIME_FIELD, CONFIGURATION_ITEM_STATUS_FIELD,
            CONFIGURATION_STATE_ID_FIELD, CONFIGURATION_ITEM_MD5_HASH_FIELD, ARN_FIELD, RESOURCE_TYPE_FIELD, RESOURCE_ID_FIELD,
            RESOURCE_NAME_FIELD, AWS_REGION_FIELD, AVAILABILITY_ZONE_FIELD, RESOURCE_CREATION_TIME_FIELD, TAGS_FIELD,
            RELATED_EVENTS_FIELD, RELATIONSHIPS_FIELD, CONFIGURATION_FIELD, SUPPLEMENTARY_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String version;

    private final String accountId;

    private final Instant configurationItemCaptureTime;

    private final String configurationItemStatus;

    private final String configurationStateId;

    private final String configurationItemMD5Hash;

    private final String arn;

    private final String resourceType;

    private final String resourceId;

    private final String resourceName;

    private final String awsRegion;

    private final String availabilityZone;

    private final Instant resourceCreationTime;

    private final Map<String, String> tags;

    private final List<String> relatedEvents;

    private final List<Relationship> relationships;

    private final String configuration;

    private final Map<String, String> supplementaryConfiguration;

    private ConfigurationItem(BuilderImpl builder) {
        this.version = builder.version;
        this.accountId = builder.accountId;
        this.configurationItemCaptureTime = builder.configurationItemCaptureTime;
        this.configurationItemStatus = builder.configurationItemStatus;
        this.configurationStateId = builder.configurationStateId;
        this.configurationItemMD5Hash = builder.configurationItemMD5Hash;
        this.arn = builder.arn;
        this.resourceType = builder.resourceType;
        this.resourceId = builder.resourceId;
        this.resourceName = builder.resourceName;
        this.awsRegion = builder.awsRegion;
        this.availabilityZone = builder.availabilityZone;
        this.resourceCreationTime = builder.resourceCreationTime;
        this.tags = builder.tags;
        this.relatedEvents = builder.relatedEvents;
        this.relationships = builder.relationships;
        this.configuration = builder.configuration;
        this.supplementaryConfiguration = builder.supplementaryConfiguration;
    }

    /**
     * <p>
     * The version number of the resource configuration.
     * </p>
     * 
     * @return The version number of the resource configuration.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The 12-digit Amazon Web Services account ID associated with the resource.
     * </p>
     * 
     * @return The 12-digit Amazon Web Services account ID associated with the resource.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The time when the configuration recording was initiated.
     * </p>
     * 
     * @return The time when the configuration recording was initiated.
     */
    public final Instant configurationItemCaptureTime() {
        return configurationItemCaptureTime;
    }

    /**
     * <p>
     * The configuration item status. The valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * OK – The resource configuration has been updated
     * </p>
     * </li>
     * <li>
     * <p>
     * ResourceDiscovered – The resource was newly discovered
     * </p>
     * </li>
     * <li>
     * <p>
     * ResourceNotRecorded – The resource was discovered but its configuration was not recorded since the recorder
     * excludes the recording of resources of this type
     * </p>
     * </li>
     * <li>
     * <p>
     * ResourceDeleted – The resource was deleted
     * </p>
     * </li>
     * <li>
     * <p>
     * ResourceDeletedNotRecorded – The resource was deleted but its configuration was not recorded since the recorder
     * excludes the recording of resources of this type
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The CIs do not incur any cost.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #configurationItemStatus} will return {@link ConfigurationItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #configurationItemStatusAsString}.
     * </p>
     * 
     * @return The configuration item status. The valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         OK – The resource configuration has been updated
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ResourceDiscovered – The resource was newly discovered
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ResourceNotRecorded – The resource was discovered but its configuration was not recorded since the
     *         recorder excludes the recording of resources of this type
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ResourceDeleted – The resource was deleted
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ResourceDeletedNotRecorded – The resource was deleted but its configuration was not recorded since the
     *         recorder excludes the recording of resources of this type
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The CIs do not incur any cost.
     *         </p>
     * @see ConfigurationItemStatus
     */
    public final ConfigurationItemStatus configurationItemStatus() {
        return ConfigurationItemStatus.fromValue(configurationItemStatus);
    }

    /**
     * <p>
     * The configuration item status. The valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * OK – The resource configuration has been updated
     * </p>
     * </li>
     * <li>
     * <p>
     * ResourceDiscovered – The resource was newly discovered
     * </p>
     * </li>
     * <li>
     * <p>
     * ResourceNotRecorded – The resource was discovered but its configuration was not recorded since the recorder
     * excludes the recording of resources of this type
     * </p>
     * </li>
     * <li>
     * <p>
     * ResourceDeleted – The resource was deleted
     * </p>
     * </li>
     * <li>
     * <p>
     * ResourceDeletedNotRecorded – The resource was deleted but its configuration was not recorded since the recorder
     * excludes the recording of resources of this type
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The CIs do not incur any cost.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #configurationItemStatus} will return {@link ConfigurationItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #configurationItemStatusAsString}.
     * </p>
     * 
     * @return The configuration item status. The valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         OK – The resource configuration has been updated
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ResourceDiscovered – The resource was newly discovered
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ResourceNotRecorded – The resource was discovered but its configuration was not recorded since the
     *         recorder excludes the recording of resources of this type
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ResourceDeleted – The resource was deleted
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ResourceDeletedNotRecorded – The resource was deleted but its configuration was not recorded since the
     *         recorder excludes the recording of resources of this type
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The CIs do not incur any cost.
     *         </p>
     * @see ConfigurationItemStatus
     */
    public final String configurationItemStatusAsString() {
        return configurationItemStatus;
    }

    /**
     * <p>
     * An identifier that indicates the ordering of the configuration items of a resource.
     * </p>
     * 
     * @return An identifier that indicates the ordering of the configuration items of a resource.
     */
    public final String configurationStateId() {
        return configurationStateId;
    }

    /**
     * <p>
     * Unique MD5 hash that represents the configuration item's state.
     * </p>
     * <p>
     * You can use MD5 hash to compare the states of two or more configuration items that are associated with the same
     * resource.
     * </p>
     * 
     * @return Unique MD5 hash that represents the configuration item's state.</p>
     *         <p>
     *         You can use MD5 hash to compare the states of two or more configuration items that are associated with
     *         the same resource.
     */
    public final String configurationItemMD5Hash() {
        return configurationItemMD5Hash;
    }

    /**
     * <p>
     * Amazon Resource Name (ARN) associated with the resource.
     * </p>
     * 
     * @return Amazon Resource Name (ARN) associated with the resource.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The type of Amazon Web Services resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon Web Services resource.
     * @see ResourceType
     */
    public final ResourceType resourceType() {
        return ResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of Amazon Web Services resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link ResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon Web Services resource.
     * @see ResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * <p>
     * The ID of the resource (for example, <code>sg-xxxxxx</code>).
     * </p>
     * 
     * @return The ID of the resource (for example, <code>sg-xxxxxx</code>).
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The custom name of the resource, if available.
     * </p>
     * 
     * @return The custom name of the resource, if available.
     */
    public final String resourceName() {
        return resourceName;
    }

    /**
     * <p>
     * The region where the resource resides.
     * </p>
     * 
     * @return The region where the resource resides.
     */
    public final String awsRegion() {
        return awsRegion;
    }

    /**
     * <p>
     * The Availability Zone associated with the resource.
     * </p>
     * 
     * @return The Availability Zone associated with the resource.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The time stamp when the resource was created.
     * </p>
     * 
     * @return The time stamp when the resource was created.
     */
    public final Instant resourceCreationTime() {
        return resourceCreationTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A mapping of key value tags associated with the resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A mapping of key value tags associated with the resource.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedEvents property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelatedEvents() {
        return relatedEvents != null && !(relatedEvents instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of CloudTrail event IDs.
     * </p>
     * <p>
     * A populated field indicates that the current configuration was initiated by the events recorded in the CloudTrail
     * log. For more information about CloudTrail, see <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html">What Is
     * CloudTrail</a>.
     * </p>
     * <p>
     * An empty field indicates that the current configuration was not initiated by any event. As of Version 1.3, the
     * relatedEvents field is empty. You can access the <a
     * href="https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html">LookupEvents API</a>
     * in the <i>CloudTrail API Reference</i> to retrieve the events for the resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedEvents} method.
     * </p>
     * 
     * @return A list of CloudTrail event IDs.</p>
     *         <p>
     *         A populated field indicates that the current configuration was initiated by the events recorded in the
     *         CloudTrail log. For more information about CloudTrail, see <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html">What
     *         Is CloudTrail</a>.
     *         </p>
     *         <p>
     *         An empty field indicates that the current configuration was not initiated by any event. As of Version
     *         1.3, the relatedEvents field is empty. You can access the <a
     *         href="https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html">LookupEvents
     *         API</a> in the <i>CloudTrail API Reference</i> to retrieve the events for the resource.
     */
    public final List<String> relatedEvents() {
        return relatedEvents;
    }

    /**
     * For responses, this returns true if the service returned a value for the Relationships property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelationships() {
        return relationships != null && !(relationships instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of related Amazon Web Services resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelationships} method.
     * </p>
     * 
     * @return A list of related Amazon Web Services resources.
     */
    public final List<Relationship> relationships() {
        return relationships;
    }

    /**
     * <p>
     * The description of the resource configuration.
     * </p>
     * 
     * @return The description of the resource configuration.
     */
    public final String configuration() {
        return configuration;
    }

    /**
     * For responses, this returns true if the service returned a value for the SupplementaryConfiguration property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasSupplementaryConfiguration() {
        return supplementaryConfiguration != null && !(supplementaryConfiguration instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Configuration attributes that Config returns for certain resource types to supplement the information returned
     * for the <code>configuration</code> parameter.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSupplementaryConfiguration} method.
     * </p>
     * 
     * @return Configuration attributes that Config returns for certain resource types to supplement the information
     *         returned for the <code>configuration</code> parameter.
     */
    public final Map<String, String> supplementaryConfiguration() {
        return supplementaryConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(configurationItemCaptureTime());
        hashCode = 31 * hashCode + Objects.hashCode(configurationItemStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(configurationStateId());
        hashCode = 31 * hashCode + Objects.hashCode(configurationItemMD5Hash());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceName());
        hashCode = 31 * hashCode + Objects.hashCode(awsRegion());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(resourceCreationTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedEvents() ? relatedEvents() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRelationships() ? relationships() : null);
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupplementaryConfiguration() ? supplementaryConfiguration() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfigurationItem)) {
            return false;
        }
        ConfigurationItem other = (ConfigurationItem) obj;
        return Objects.equals(version(), other.version()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(configurationItemCaptureTime(), other.configurationItemCaptureTime())
                && Objects.equals(configurationItemStatusAsString(), other.configurationItemStatusAsString())
                && Objects.equals(configurationStateId(), other.configurationStateId())
                && Objects.equals(configurationItemMD5Hash(), other.configurationItemMD5Hash())
                && Objects.equals(arn(), other.arn()) && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && Objects.equals(resourceId(), other.resourceId()) && Objects.equals(resourceName(), other.resourceName())
                && Objects.equals(awsRegion(), other.awsRegion()) && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(resourceCreationTime(), other.resourceCreationTime()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasRelatedEvents() == other.hasRelatedEvents()
                && Objects.equals(relatedEvents(), other.relatedEvents()) && hasRelationships() == other.hasRelationships()
                && Objects.equals(relationships(), other.relationships())
                && Objects.equals(configuration(), other.configuration())
                && hasSupplementaryConfiguration() == other.hasSupplementaryConfiguration()
                && Objects.equals(supplementaryConfiguration(), other.supplementaryConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConfigurationItem").add("Version", version()).add("AccountId", accountId())
                .add("ConfigurationItemCaptureTime", configurationItemCaptureTime())
                .add("ConfigurationItemStatus", configurationItemStatusAsString())
                .add("ConfigurationStateId", configurationStateId()).add("ConfigurationItemMD5Hash", configurationItemMD5Hash())
                .add("Arn", arn()).add("ResourceType", resourceTypeAsString()).add("ResourceId", resourceId())
                .add("ResourceName", resourceName()).add("AwsRegion", awsRegion()).add("AvailabilityZone", availabilityZone())
                .add("ResourceCreationTime", resourceCreationTime()).add("Tags", hasTags() ? tags() : null)
                .add("RelatedEvents", hasRelatedEvents() ? relatedEvents() : null)
                .add("Relationships", hasRelationships() ? relationships() : null).add("Configuration", configuration())
                .add("SupplementaryConfiguration", hasSupplementaryConfiguration() ? supplementaryConfiguration() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "configurationItemCaptureTime":
            return Optional.ofNullable(clazz.cast(configurationItemCaptureTime()));
        case "configurationItemStatus":
            return Optional.ofNullable(clazz.cast(configurationItemStatusAsString()));
        case "configurationStateId":
            return Optional.ofNullable(clazz.cast(configurationStateId()));
        case "configurationItemMD5Hash":
            return Optional.ofNullable(clazz.cast(configurationItemMD5Hash()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "resourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "resourceName":
            return Optional.ofNullable(clazz.cast(resourceName()));
        case "awsRegion":
            return Optional.ofNullable(clazz.cast(awsRegion()));
        case "availabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "resourceCreationTime":
            return Optional.ofNullable(clazz.cast(resourceCreationTime()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "relatedEvents":
            return Optional.ofNullable(clazz.cast(relatedEvents()));
        case "relationships":
            return Optional.ofNullable(clazz.cast(relationships()));
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "supplementaryConfiguration":
            return Optional.ofNullable(clazz.cast(supplementaryConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConfigurationItem, T> g) {
        return obj -> g.apply((ConfigurationItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConfigurationItem> {
        /**
         * <p>
         * The version number of the resource configuration.
         * </p>
         * 
         * @param version
         *        The version number of the resource configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The 12-digit Amazon Web Services account ID associated with the resource.
         * </p>
         * 
         * @param accountId
         *        The 12-digit Amazon Web Services account ID associated with the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The time when the configuration recording was initiated.
         * </p>
         * 
         * @param configurationItemCaptureTime
         *        The time when the configuration recording was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationItemCaptureTime(Instant configurationItemCaptureTime);

        /**
         * <p>
         * The configuration item status. The valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * OK – The resource configuration has been updated
         * </p>
         * </li>
         * <li>
         * <p>
         * ResourceDiscovered – The resource was newly discovered
         * </p>
         * </li>
         * <li>
         * <p>
         * ResourceNotRecorded – The resource was discovered but its configuration was not recorded since the recorder
         * excludes the recording of resources of this type
         * </p>
         * </li>
         * <li>
         * <p>
         * ResourceDeleted – The resource was deleted
         * </p>
         * </li>
         * <li>
         * <p>
         * ResourceDeletedNotRecorded – The resource was deleted but its configuration was not recorded since the
         * recorder excludes the recording of resources of this type
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The CIs do not incur any cost.
         * </p>
         * </note>
         * 
         * @param configurationItemStatus
         *        The configuration item status. The valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        OK – The resource configuration has been updated
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ResourceDiscovered – The resource was newly discovered
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ResourceNotRecorded – The resource was discovered but its configuration was not recorded since the
         *        recorder excludes the recording of resources of this type
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ResourceDeleted – The resource was deleted
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ResourceDeletedNotRecorded – The resource was deleted but its configuration was not recorded since the
         *        recorder excludes the recording of resources of this type
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The CIs do not incur any cost.
         *        </p>
         * @see ConfigurationItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationItemStatus
         */
        Builder configurationItemStatus(String configurationItemStatus);

        /**
         * <p>
         * The configuration item status. The valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * OK – The resource configuration has been updated
         * </p>
         * </li>
         * <li>
         * <p>
         * ResourceDiscovered – The resource was newly discovered
         * </p>
         * </li>
         * <li>
         * <p>
         * ResourceNotRecorded – The resource was discovered but its configuration was not recorded since the recorder
         * excludes the recording of resources of this type
         * </p>
         * </li>
         * <li>
         * <p>
         * ResourceDeleted – The resource was deleted
         * </p>
         * </li>
         * <li>
         * <p>
         * ResourceDeletedNotRecorded – The resource was deleted but its configuration was not recorded since the
         * recorder excludes the recording of resources of this type
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The CIs do not incur any cost.
         * </p>
         * </note>
         * 
         * @param configurationItemStatus
         *        The configuration item status. The valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        OK – The resource configuration has been updated
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ResourceDiscovered – The resource was newly discovered
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ResourceNotRecorded – The resource was discovered but its configuration was not recorded since the
         *        recorder excludes the recording of resources of this type
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ResourceDeleted – The resource was deleted
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ResourceDeletedNotRecorded – The resource was deleted but its configuration was not recorded since the
         *        recorder excludes the recording of resources of this type
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The CIs do not incur any cost.
         *        </p>
         * @see ConfigurationItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigurationItemStatus
         */
        Builder configurationItemStatus(ConfigurationItemStatus configurationItemStatus);

        /**
         * <p>
         * An identifier that indicates the ordering of the configuration items of a resource.
         * </p>
         * 
         * @param configurationStateId
         *        An identifier that indicates the ordering of the configuration items of a resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationStateId(String configurationStateId);

        /**
         * <p>
         * Unique MD5 hash that represents the configuration item's state.
         * </p>
         * <p>
         * You can use MD5 hash to compare the states of two or more configuration items that are associated with the
         * same resource.
         * </p>
         * 
         * @param configurationItemMD5Hash
         *        Unique MD5 hash that represents the configuration item's state.</p>
         *        <p>
         *        You can use MD5 hash to compare the states of two or more configuration items that are associated with
         *        the same resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationItemMD5Hash(String configurationItemMD5Hash);

        /**
         * <p>
         * Amazon Resource Name (ARN) associated with the resource.
         * </p>
         * 
         * @param arn
         *        Amazon Resource Name (ARN) associated with the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The type of Amazon Web Services resource.
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of Amazon Web Services resource.
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource.
         * @see ResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResourceType
         */
        Builder resourceType(ResourceType resourceType);

        /**
         * <p>
         * The ID of the resource (for example, <code>sg-xxxxxx</code>).
         * </p>
         * 
         * @param resourceId
         *        The ID of the resource (for example, <code>sg-xxxxxx</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The custom name of the resource, if available.
         * </p>
         * 
         * @param resourceName
         *        The custom name of the resource, if available.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceName(String resourceName);

        /**
         * <p>
         * The region where the resource resides.
         * </p>
         * 
         * @param awsRegion
         *        The region where the resource resides.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsRegion(String awsRegion);

        /**
         * <p>
         * The Availability Zone associated with the resource.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone associated with the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The time stamp when the resource was created.
         * </p>
         * 
         * @param resourceCreationTime
         *        The time stamp when the resource was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceCreationTime(Instant resourceCreationTime);

        /**
         * <p>
         * A mapping of key value tags associated with the resource.
         * </p>
         * 
         * @param tags
         *        A mapping of key value tags associated with the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * A list of CloudTrail event IDs.
         * </p>
         * <p>
         * A populated field indicates that the current configuration was initiated by the events recorded in the
         * CloudTrail log. For more information about CloudTrail, see <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html">What Is
         * CloudTrail</a>.
         * </p>
         * <p>
         * An empty field indicates that the current configuration was not initiated by any event. As of Version 1.3,
         * the relatedEvents field is empty. You can access the <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html">LookupEvents
         * API</a> in the <i>CloudTrail API Reference</i> to retrieve the events for the resource.
         * </p>
         * 
         * @param relatedEvents
         *        A list of CloudTrail event IDs.</p>
         *        <p>
         *        A populated field indicates that the current configuration was initiated by the events recorded in the
         *        CloudTrail log. For more information about CloudTrail, see <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html"
         *        >What Is CloudTrail</a>.
         *        </p>
         *        <p>
         *        An empty field indicates that the current configuration was not initiated by any event. As of Version
         *        1.3, the relatedEvents field is empty. You can access the <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html"
         *        >LookupEvents API</a> in the <i>CloudTrail API Reference</i> to retrieve the events for the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedEvents(Collection<String> relatedEvents);

        /**
         * <p>
         * A list of CloudTrail event IDs.
         * </p>
         * <p>
         * A populated field indicates that the current configuration was initiated by the events recorded in the
         * CloudTrail log. For more information about CloudTrail, see <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html">What Is
         * CloudTrail</a>.
         * </p>
         * <p>
         * An empty field indicates that the current configuration was not initiated by any event. As of Version 1.3,
         * the relatedEvents field is empty. You can access the <a
         * href="https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html">LookupEvents
         * API</a> in the <i>CloudTrail API Reference</i> to retrieve the events for the resource.
         * </p>
         * 
         * @param relatedEvents
         *        A list of CloudTrail event IDs.</p>
         *        <p>
         *        A populated field indicates that the current configuration was initiated by the events recorded in the
         *        CloudTrail log. For more information about CloudTrail, see <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/what_is_cloud_trail_top_level.html"
         *        >What Is CloudTrail</a>.
         *        </p>
         *        <p>
         *        An empty field indicates that the current configuration was not initiated by any event. As of Version
         *        1.3, the relatedEvents field is empty. You can access the <a
         *        href="https://docs.aws.amazon.com/awscloudtrail/latest/APIReference/API_LookupEvents.html"
         *        >LookupEvents API</a> in the <i>CloudTrail API Reference</i> to retrieve the events for the resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedEvents(String... relatedEvents);

        /**
         * <p>
         * A list of related Amazon Web Services resources.
         * </p>
         * 
         * @param relationships
         *        A list of related Amazon Web Services resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Collection<Relationship> relationships);

        /**
         * <p>
         * A list of related Amazon Web Services resources.
         * </p>
         * 
         * @param relationships
         *        A list of related Amazon Web Services resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relationships(Relationship... relationships);

        /**
         * <p>
         * A list of related Amazon Web Services resources.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Relationship>.Builder} avoiding the
         * need to create one manually via {@link List<Relationship>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Relationship>.Builder#build()} is called immediately and its
         * result is passed to {@link #relationships(List<Relationship>)}.
         * 
         * @param relationships
         *        a consumer that will call methods on {@link List<Relationship>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relationships(List<Relationship>)
         */
        Builder relationships(Consumer<Relationship.Builder>... relationships);

        /**
         * <p>
         * The description of the resource configuration.
         * </p>
         * 
         * @param configuration
         *        The description of the resource configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(String configuration);

        /**
         * <p>
         * Configuration attributes that Config returns for certain resource types to supplement the information
         * returned for the <code>configuration</code> parameter.
         * </p>
         * 
         * @param supplementaryConfiguration
         *        Configuration attributes that Config returns for certain resource types to supplement the information
         *        returned for the <code>configuration</code> parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supplementaryConfiguration(Map<String, String> supplementaryConfiguration);
    }

    static final class BuilderImpl implements Builder {
        private String version;

        private String accountId;

        private Instant configurationItemCaptureTime;

        private String configurationItemStatus;

        private String configurationStateId;

        private String configurationItemMD5Hash;

        private String arn;

        private String resourceType;

        private String resourceId;

        private String resourceName;

        private String awsRegion;

        private String availabilityZone;

        private Instant resourceCreationTime;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private List<String> relatedEvents = DefaultSdkAutoConstructList.getInstance();

        private List<Relationship> relationships = DefaultSdkAutoConstructList.getInstance();

        private String configuration;

        private Map<String, String> supplementaryConfiguration = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConfigurationItem model) {
            version(model.version);
            accountId(model.accountId);
            configurationItemCaptureTime(model.configurationItemCaptureTime);
            configurationItemStatus(model.configurationItemStatus);
            configurationStateId(model.configurationStateId);
            configurationItemMD5Hash(model.configurationItemMD5Hash);
            arn(model.arn);
            resourceType(model.resourceType);
            resourceId(model.resourceId);
            resourceName(model.resourceName);
            awsRegion(model.awsRegion);
            availabilityZone(model.availabilityZone);
            resourceCreationTime(model.resourceCreationTime);
            tags(model.tags);
            relatedEvents(model.relatedEvents);
            relationships(model.relationships);
            configuration(model.configuration);
            supplementaryConfiguration(model.supplementaryConfiguration);
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final Instant getConfigurationItemCaptureTime() {
            return configurationItemCaptureTime;
        }

        public final void setConfigurationItemCaptureTime(Instant configurationItemCaptureTime) {
            this.configurationItemCaptureTime = configurationItemCaptureTime;
        }

        @Override
        public final Builder configurationItemCaptureTime(Instant configurationItemCaptureTime) {
            this.configurationItemCaptureTime = configurationItemCaptureTime;
            return this;
        }

        public final String getConfigurationItemStatus() {
            return configurationItemStatus;
        }

        public final void setConfigurationItemStatus(String configurationItemStatus) {
            this.configurationItemStatus = configurationItemStatus;
        }

        @Override
        public final Builder configurationItemStatus(String configurationItemStatus) {
            this.configurationItemStatus = configurationItemStatus;
            return this;
        }

        @Override
        public final Builder configurationItemStatus(ConfigurationItemStatus configurationItemStatus) {
            this.configurationItemStatus(configurationItemStatus == null ? null : configurationItemStatus.toString());
            return this;
        }

        public final String getConfigurationStateId() {
            return configurationStateId;
        }

        public final void setConfigurationStateId(String configurationStateId) {
            this.configurationStateId = configurationStateId;
        }

        @Override
        public final Builder configurationStateId(String configurationStateId) {
            this.configurationStateId = configurationStateId;
            return this;
        }

        public final String getConfigurationItemMD5Hash() {
            return configurationItemMD5Hash;
        }

        public final void setConfigurationItemMD5Hash(String configurationItemMD5Hash) {
            this.configurationItemMD5Hash = configurationItemMD5Hash;
        }

        @Override
        public final Builder configurationItemMD5Hash(String configurationItemMD5Hash) {
            this.configurationItemMD5Hash = configurationItemMD5Hash;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(ResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getResourceName() {
            return resourceName;
        }

        public final void setResourceName(String resourceName) {
            this.resourceName = resourceName;
        }

        @Override
        public final Builder resourceName(String resourceName) {
            this.resourceName = resourceName;
            return this;
        }

        public final String getAwsRegion() {
            return awsRegion;
        }

        public final void setAwsRegion(String awsRegion) {
            this.awsRegion = awsRegion;
        }

        @Override
        public final Builder awsRegion(String awsRegion) {
            this.awsRegion = awsRegion;
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final Instant getResourceCreationTime() {
            return resourceCreationTime;
        }

        public final void setResourceCreationTime(Instant resourceCreationTime) {
            this.resourceCreationTime = resourceCreationTime;
        }

        @Override
        public final Builder resourceCreationTime(Instant resourceCreationTime) {
            this.resourceCreationTime = resourceCreationTime;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        public final Collection<String> getRelatedEvents() {
            if (relatedEvents instanceof SdkAutoConstructList) {
                return null;
            }
            return relatedEvents;
        }

        public final void setRelatedEvents(Collection<String> relatedEvents) {
            this.relatedEvents = RelatedEventListCopier.copy(relatedEvents);
        }

        @Override
        public final Builder relatedEvents(Collection<String> relatedEvents) {
            this.relatedEvents = RelatedEventListCopier.copy(relatedEvents);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedEvents(String... relatedEvents) {
            relatedEvents(Arrays.asList(relatedEvents));
            return this;
        }

        public final List<Relationship.Builder> getRelationships() {
            List<Relationship.Builder> result = RelationshipListCopier.copyToBuilder(this.relationships);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRelationships(Collection<Relationship.BuilderImpl> relationships) {
            this.relationships = RelationshipListCopier.copyFromBuilder(relationships);
        }

        @Override
        public final Builder relationships(Collection<Relationship> relationships) {
            this.relationships = RelationshipListCopier.copy(relationships);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(Relationship... relationships) {
            relationships(Arrays.asList(relationships));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relationships(Consumer<Relationship.Builder>... relationships) {
            relationships(Stream.of(relationships).map(c -> Relationship.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getConfiguration() {
            return configuration;
        }

        public final void setConfiguration(String configuration) {
            this.configuration = configuration;
        }

        @Override
        public final Builder configuration(String configuration) {
            this.configuration = configuration;
            return this;
        }

        public final Map<String, String> getSupplementaryConfiguration() {
            if (supplementaryConfiguration instanceof SdkAutoConstructMap) {
                return null;
            }
            return supplementaryConfiguration;
        }

        public final void setSupplementaryConfiguration(Map<String, String> supplementaryConfiguration) {
            this.supplementaryConfiguration = SupplementaryConfigurationCopier.copy(supplementaryConfiguration);
        }

        @Override
        public final Builder supplementaryConfiguration(Map<String, String> supplementaryConfiguration) {
            this.supplementaryConfiguration = SupplementaryConfigurationCopier.copy(supplementaryConfiguration);
            return this;
        }

        @Override
        public ConfigurationItem build() {
            return new ConfigurationItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
