/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object for you to specify your overrides for the recording mode.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RecordingModeOverride implements SdkPojo, Serializable,
        ToCopyableBuilder<RecordingModeOverride.Builder, RecordingModeOverride> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(RecordingModeOverride::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> RESOURCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("resourceTypes")
            .getter(getter(RecordingModeOverride::resourceTypesAsStrings))
            .setter(setter(Builder::resourceTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RECORDING_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("recordingFrequency").getter(getter(RecordingModeOverride::recordingFrequencyAsString))
            .setter(setter(Builder::recordingFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recordingFrequency").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            RESOURCE_TYPES_FIELD, RECORDING_FREQUENCY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String description;

    private final List<String> resourceTypes;

    private final String recordingFrequency;

    private RecordingModeOverride(BuilderImpl builder) {
        this.description = builder.description;
        this.resourceTypes = builder.resourceTypes;
        this.recordingFrequency = builder.recordingFrequency;
    }

    /**
     * <p>
     * A description that you provide for the override.
     * </p>
     * 
     * @return A description that you provide for the override.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A comma-separated list that specifies which resource types Config includes in the override.
     * </p>
     * <important>
     * <p>
     * Daily recording is not supported for the following resource types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::Config::ResourceCompliance</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::Config::ConformancePackCompliance</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::Config::ConfigurationRecorder</code>
     * </p>
     * </li>
     * </ul>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return A comma-separated list that specifies which resource types Config includes in the override.</p>
     *         <important>
     *         <p>
     *         Daily recording is not supported for the following resource types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ResourceCompliance</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ConformancePackCompliance</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ConfigurationRecorder</code>
     *         </p>
     *         </li>
     *         </ul>
     */
    public final List<ResourceType> resourceTypes() {
        return RecordingModeResourceTypesListCopier.copyStringToEnum(resourceTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTypes() {
        return resourceTypes != null && !(resourceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A comma-separated list that specifies which resource types Config includes in the override.
     * </p>
     * <important>
     * <p>
     * Daily recording is not supported for the following resource types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AWS::Config::ResourceCompliance</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::Config::ConformancePackCompliance</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AWS::Config::ConfigurationRecorder</code>
     * </p>
     * </li>
     * </ul>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTypes} method.
     * </p>
     * 
     * @return A comma-separated list that specifies which resource types Config includes in the override.</p>
     *         <important>
     *         <p>
     *         Daily recording is not supported for the following resource types:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ResourceCompliance</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ConformancePackCompliance</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>AWS::Config::ConfigurationRecorder</code>
     *         </p>
     *         </li>
     *         </ul>
     */
    public final List<String> resourceTypesAsStrings() {
        return resourceTypes;
    }

    /**
     * <p>
     * The recording frequency that will be applied to all the resource types specified in the override.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Continuous recording allows you to record configuration changes continuously whenever a change occurs.
     * </p>
     * </li>
     * <li>
     * <p>
     * Daily recording allows you record configuration data once every 24 hours, only if a change has occurred.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall Manager, it
     * is recommended that you set the recording frequency to Continuous.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recordingFrequency} will return {@link RecordingFrequency#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #recordingFrequencyAsString}.
     * </p>
     * 
     * @return The recording frequency that will be applied to all the resource types specified in the override.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Continuous recording allows you to record configuration changes continuously whenever a change occurs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Daily recording allows you record configuration data once every 24 hours, only if a change has occurred.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall
     *         Manager, it is recommended that you set the recording frequency to Continuous.
     *         </p>
     * @see RecordingFrequency
     */
    public final RecordingFrequency recordingFrequency() {
        return RecordingFrequency.fromValue(recordingFrequency);
    }

    /**
     * <p>
     * The recording frequency that will be applied to all the resource types specified in the override.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Continuous recording allows you to record configuration changes continuously whenever a change occurs.
     * </p>
     * </li>
     * <li>
     * <p>
     * Daily recording allows you record configuration data once every 24 hours, only if a change has occurred.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall Manager, it
     * is recommended that you set the recording frequency to Continuous.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #recordingFrequency} will return {@link RecordingFrequency#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #recordingFrequencyAsString}.
     * </p>
     * 
     * @return The recording frequency that will be applied to all the resource types specified in the override.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Continuous recording allows you to record configuration changes continuously whenever a change occurs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Daily recording allows you record configuration data once every 24 hours, only if a change has occurred.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall
     *         Manager, it is recommended that you set the recording frequency to Continuous.
     *         </p>
     * @see RecordingFrequency
     */
    public final String recordingFrequencyAsString() {
        return recordingFrequency;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTypes() ? resourceTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recordingFrequencyAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RecordingModeOverride)) {
            return false;
        }
        RecordingModeOverride other = (RecordingModeOverride) obj;
        return Objects.equals(description(), other.description()) && hasResourceTypes() == other.hasResourceTypes()
                && Objects.equals(resourceTypesAsStrings(), other.resourceTypesAsStrings())
                && Objects.equals(recordingFrequencyAsString(), other.recordingFrequencyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RecordingModeOverride").add("Description", description())
                .add("ResourceTypes", hasResourceTypes() ? resourceTypesAsStrings() : null)
                .add("RecordingFrequency", recordingFrequencyAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "resourceTypes":
            return Optional.ofNullable(clazz.cast(resourceTypesAsStrings()));
        case "recordingFrequency":
            return Optional.ofNullable(clazz.cast(recordingFrequencyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RecordingModeOverride, T> g) {
        return obj -> g.apply((RecordingModeOverride) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RecordingModeOverride> {
        /**
         * <p>
         * A description that you provide for the override.
         * </p>
         * 
         * @param description
         *        A description that you provide for the override.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A comma-separated list that specifies which resource types Config includes in the override.
         * </p>
         * <important>
         * <p>
         * Daily recording is not supported for the following resource types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS::Config::ResourceCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConformancePackCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConfigurationRecorder</code>
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies which resource types Config includes in the override.</p>
         *        <important>
         *        <p>
         *        Daily recording is not supported for the following resource types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ResourceCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConformancePackCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConfigurationRecorder</code>
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(Collection<String> resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies which resource types Config includes in the override.
         * </p>
         * <important>
         * <p>
         * Daily recording is not supported for the following resource types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS::Config::ResourceCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConformancePackCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConfigurationRecorder</code>
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies which resource types Config includes in the override.</p>
         *        <important>
         *        <p>
         *        Daily recording is not supported for the following resource types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ResourceCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConformancePackCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConfigurationRecorder</code>
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypesWithStrings(String... resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies which resource types Config includes in the override.
         * </p>
         * <important>
         * <p>
         * Daily recording is not supported for the following resource types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS::Config::ResourceCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConformancePackCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConfigurationRecorder</code>
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies which resource types Config includes in the override.</p>
         *        <important>
         *        <p>
         *        Daily recording is not supported for the following resource types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ResourceCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConformancePackCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConfigurationRecorder</code>
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(Collection<ResourceType> resourceTypes);

        /**
         * <p>
         * A comma-separated list that specifies which resource types Config includes in the override.
         * </p>
         * <important>
         * <p>
         * Daily recording is not supported for the following resource types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AWS::Config::ResourceCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConformancePackCompliance</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>AWS::Config::ConfigurationRecorder</code>
         * </p>
         * </li>
         * </ul>
         * </important>
         * 
         * @param resourceTypes
         *        A comma-separated list that specifies which resource types Config includes in the override.</p>
         *        <important>
         *        <p>
         *        Daily recording is not supported for the following resource types:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ResourceCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConformancePackCompliance</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>AWS::Config::ConfigurationRecorder</code>
         *        </p>
         *        </li>
         *        </ul>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTypes(ResourceType... resourceTypes);

        /**
         * <p>
         * The recording frequency that will be applied to all the resource types specified in the override.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Continuous recording allows you to record configuration changes continuously whenever a change occurs.
         * </p>
         * </li>
         * <li>
         * <p>
         * Daily recording allows you record configuration data once every 24 hours, only if a change has occurred.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall
         * Manager, it is recommended that you set the recording frequency to Continuous.
         * </p>
         * </note>
         * 
         * @param recordingFrequency
         *        The recording frequency that will be applied to all the resource types specified in the override.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Continuous recording allows you to record configuration changes continuously whenever a change occurs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Daily recording allows you record configuration data once every 24 hours, only if a change has
         *        occurred.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall
         *        Manager, it is recommended that you set the recording frequency to Continuous.
         *        </p>
         * @see RecordingFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordingFrequency
         */
        Builder recordingFrequency(String recordingFrequency);

        /**
         * <p>
         * The recording frequency that will be applied to all the resource types specified in the override.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Continuous recording allows you to record configuration changes continuously whenever a change occurs.
         * </p>
         * </li>
         * <li>
         * <p>
         * Daily recording allows you record configuration data once every 24 hours, only if a change has occurred.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall
         * Manager, it is recommended that you set the recording frequency to Continuous.
         * </p>
         * </note>
         * 
         * @param recordingFrequency
         *        The recording frequency that will be applied to all the resource types specified in the override.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Continuous recording allows you to record configuration changes continuously whenever a change occurs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Daily recording allows you record configuration data once every 24 hours, only if a change has
         *        occurred.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Firewall Manager depends on continuous recording to monitor your resources. If you are using Firewall
         *        Manager, it is recommended that you set the recording frequency to Continuous.
         *        </p>
         * @see RecordingFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RecordingFrequency
         */
        Builder recordingFrequency(RecordingFrequency recordingFrequency);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private List<String> resourceTypes = DefaultSdkAutoConstructList.getInstance();

        private String recordingFrequency;

        private BuilderImpl() {
        }

        private BuilderImpl(RecordingModeOverride model) {
            description(model.description);
            resourceTypesWithStrings(model.resourceTypes);
            recordingFrequency(model.recordingFrequency);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getResourceTypes() {
            if (resourceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceTypes;
        }

        public final void setResourceTypes(Collection<String> resourceTypes) {
            this.resourceTypes = RecordingModeResourceTypesListCopier.copy(resourceTypes);
        }

        @Override
        public final Builder resourceTypesWithStrings(Collection<String> resourceTypes) {
            this.resourceTypes = RecordingModeResourceTypesListCopier.copy(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypesWithStrings(String... resourceTypes) {
            resourceTypesWithStrings(Arrays.asList(resourceTypes));
            return this;
        }

        @Override
        public final Builder resourceTypes(Collection<ResourceType> resourceTypes) {
            this.resourceTypes = RecordingModeResourceTypesListCopier.copyEnumToString(resourceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTypes(ResourceType... resourceTypes) {
            resourceTypes(Arrays.asList(resourceTypes));
            return this;
        }

        public final String getRecordingFrequency() {
            return recordingFrequency;
        }

        public final void setRecordingFrequency(String recordingFrequency) {
            this.recordingFrequency = recordingFrequency;
        }

        @Override
        public final Builder recordingFrequency(String recordingFrequency) {
            this.recordingFrequency = recordingFrequency;
            return this;
        }

        @Override
        public final Builder recordingFrequency(RecordingFrequency recordingFrequency) {
            this.recordingFrequency(recordingFrequency == null ? null : recordingFrequency.toString());
            return this;
        }

        @Override
        public RecordingModeOverride build() {
            return new RecordingModeOverride(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
